/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/cfg_parse.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/browser.h"
#include "xine-toolkit/window.h"
#include "skins.h"
#include "download.h"
#include "actions.h"
#include "event.h"
#include "errors.h"
#include "mediamark.h"

#define MAX_DISP_ENTRIES  5

#define WINDOW_WIDTH      630
#define WINDOW_HEIGHT     446

#define PREVIEW_WIDTH     (WINDOW_WIDTH - 30)
#define PREVIEW_HEIGHT    220

typedef enum {
  _SLX_name = 0,
  _SLX_author,
  _SLX_preview,
  _SLX_href,
  _SLX_LAST
} _slx_strings_t;

typedef struct {
  char *s[_SLX_LAST];
} slx_entry_t;

struct xui_skdloader_s {
  gui_new_window_t  nw;

  xitk_widget_t     *browser;
  xitk_image_t      *preview_image;

  const char       **names, **info;
#define _BUF_CHUNK_SIZE (1 << 12)
  char              *buf;
  unsigned int       bused, slxused;
  int                num;
  int                sel;
#define _SLX_CHUNK_SIZE (16)
  slx_entry_t       *slxs;

  void             (*exit) (xui_skdloader_t *skd);
};

static char *_skd_buf_add (xui_skdloader_t *skd, const char *s) {
  unsigned int len, bsize = (skd->bused + _BUF_CHUNK_SIZE - 1) & ~(_BUF_CHUNK_SIZE - 1);
  char *ret;
  if (!s)
    s = "";
  len = xitk_find_byte (s, 0) + 1;
  if (skd->bused + len > bsize) {
    unsigned int u;
    char *nbuf = realloc (skd->buf, (skd->bused + len + _BUF_CHUNK_SIZE - 1) & ~(_BUF_CHUNK_SIZE - 1));

    if (!nbuf)
      return NULL;
    for (u = 0; u < skd->slxused; u++) {
      char **s = &skd->slxs[0].s[0] + u;
      *s = nbuf + (*s - skd->buf);
    }
    skd->buf = nbuf;
  }
  ret = skd->buf + skd->bused;
  memcpy (ret, s, len);
  skd->bused += len;
  skd->slxused++;
  return ret;
}

static slx_entry_t *_skd_get_slx (xui_skdloader_t *skd, const char *url) {
  gGui_t *gui = skd->nw.gui;
  int num = 0;
  slx_entry_t *slxs = NULL;
  download_t download = { .gui = gui };

  do {
    xml_parser_t *xml;
    xml_node_t *tree;
    mrl_buf_t base;

    if (!mrl_buf_set (&base, NULL, url))
      break;
    if (!network_download (base.start, &download)) {
      gui_msg (gui, XUI_MSG_ERROR, _("Unable to retrieve skin list from %s: %s"), base.start, download.error);
      break;
    }
    xml_parser_init_R (xml, download.buf, download.size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree_R (xml, &tree) != XML_PARSER_OK)
      break;

    do {
      xml_node_t *entry;
      xml_property_t *prop;
      int  v0, v1 = 0;

      if (strcasecmp (tree->name, "SLX"))
        break;
      for (prop = tree->props; prop && strcasecmp (prop->name, "VERSION"); prop = prop->next) ;
      if (!prop)
        break;
      if ((sscanf (prop->value, "%d.%d", &v0, &v1) != 2) && (sscanf (prop->value, "%d", &v0)) != 1)
        break;
      if ((v0 < 2) || (v1 < 0))
        break;

      for (entry = tree->child; entry; entry = entry->next) {
        mrl_buf_t item, preview, href;
        xml_node_t *ref;
        slx_entry_t slx = { .s[0] = NULL };
        const char *author = "", *email = "";

        if (strcasecmp (entry->name, "SKIN"))
          continue;

        v0 = v1 = 0; /* skin version, maintained */
        for (ref = entry->child; ref; ref = ref->next) {
          if (!strcasecmp (ref->name, "NAME")) {
            slx.s[_SLX_name] = ref->data;
          } else if (!strcasecmp (ref->name, "AUTHOR")) {
            for (prop = ref->props; prop; prop = prop->next) {
              if (!strcasecmp (prop->name, "NAME")) {
                author = prop->value;
              } else if (!strcasecmp (prop->name, "EMAIL")) {
                email = prop->value;
              }
            }
          } else if (!strcasecmp (ref->name, "REF")) {
            for (prop = ref->props; prop; prop = prop->next) {
              if (!strcasecmp (prop->name, "HREF")) {
                if (mrl_buf_set (&item, &base, prop->value)) {
                  mrl_buf_merge (&href, &base, &item);
                  slx.s[_SLX_href] = href.start;
                }
              } else if (!strcasecmp (prop->name, "VERSION")) {
                v0 = atoi (prop->value);
              } else if (!strcasecmp (prop->name, "MAINTAINED")) {
                v1 = xitk_get_bool_value (prop->value);
              }
            }
          } else if (!strcasecmp (ref->name, "PREVIEW")) {
            for (prop = ref->props; prop; prop = prop->next) {
              if (!strcasecmp (prop->name, "HREF")) {
                if (mrl_buf_set (&item, &base, prop->value)) {
                  mrl_buf_merge (&preview, &base, &item);
                  slx.s[_SLX_preview] = preview.start;
                }
              }
            }
          }
        }

        if (slx.s[_SLX_name] && slx.s[_SLX_href] && ((v0 >= SKIN_IFACE_VERSION) && v1)) {
          char buf[1024];
          if (gui->verbosity >= 2)
            printf ("gui.skin.download.get slx[%d]: %s.\n", num, slx.s[_SLX_href]);
#if 0
          printf("  Name: %s\n", slx.s[_SLX_name]);
          printf("  Author Name: %s\n", author);
          printf("  Author email: %s\n", email);
          printf("  Preview: %s\n", slx.s[_SLX_preview]);
          printf("  Version: %d\n", v0);
          printf("  Maintained: %d\n", v1);
          printf("--\n");
#endif
          if (!(num & (_SLX_CHUNK_SIZE - 1))) {
            slx_entry_t *nslxs = realloc (slxs, sizeof (slx_entry_t) * (num + _SLX_CHUNK_SIZE));
            if (!nslxs)
              continue;
            skd->slxs = slxs = nslxs;
          }
          snprintf (buf, sizeof (buf), "%s <%s>", author, email);
          slxs[num].s[_SLX_name]    = _skd_buf_add (skd, slx.s[_SLX_name]);
          slxs[num].s[_SLX_author]  = _skd_buf_add (skd, buf);
          slxs[num].s[_SLX_preview] = _skd_buf_add (skd, slx.s[_SLX_preview]);
          slxs[num].s[_SLX_href]    = _skd_buf_add (skd, slx.s[_SLX_href]);
          skd->num = ++num;
        } else {
          continue;
        }
      }
    } while (0);

    skd->names = malloc (sizeof (const char *) * 2 * (num + 1));
    if (skd->names) {
      unsigned int u;
      skd->info = skd->names + (num + 1);
      for (u = 0; u < (unsigned int)num; u++) {
        skd->names[u] = skd->slxs[u].s[_SLX_name];
        skd->info[u] = skd->slxs[u].s[_SLX_author];
      }
      skd->names[num] = NULL;
      skd->info[num] = NULL;
    }

    xml_parser_free_tree (tree);
    xml_parser_finalize_R (xml);
  } while (0);

  skd->slxs = slxs;
  free(download.buf);
  free(download.error);

  return slxs;
}

/*
 * Remote skin loader
 */
static void _skd_exit_1 (xui_skdloader_t *skd) {
  gGui_t *gui = skd->nw.gui;

  gui->skdloader = NULL;

  gui_window_delete (&skd->nw);

  xitk_image_free_image (&skd->preview_image);

  free (skd->names);
  free (skd->buf);
  free (skd->slxs);

  skd->num = 0;

  free (skd);
}

static void _skd_exit_4 (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_skdloader_t *skd = data;

  (void)w;
  (void)state;
  (void)modifier;
  if (skd)
    skd->exit (skd);
}

static void _skd_blank_preview (xui_skdloader_t *skd) {

  xitk_image_t *p = xitk_image_new (skd->nw.gui->xitk, NULL, 0, PREVIEW_WIDTH, PREVIEW_HEIGHT);
  xitk_image_fill_rectangle (p, 0, 0, PREVIEW_WIDTH, PREVIEW_HEIGHT,
    xitk_color_db_get (skd->nw.gui->xitk, (52 << 16) + (52 << 8) + 52));
  xitk_image_draw_image (skd->nw.wl, p, 0, 0, PREVIEW_WIDTH, PREVIEW_HEIGHT, 15, 34, 0);
  xitk_image_free_image (&p);
}

static void _skd_preview (xui_skdloader_t *skd) {
  _skd_blank_preview (skd);

  if (skd->preview_image) {
    int img_width = xitk_image_width(skd->preview_image);
    int img_height = xitk_image_height(skd->preview_image);
    xitk_image_draw_image (skd->nw.wl, skd->preview_image,
        0, 0, img_width, img_height,
        15 + ((PREVIEW_WIDTH - img_width) >> 1), 34 + ((PREVIEW_HEIGHT - img_height) >> 1), 0);
  }
}

static void _skd_get_preview (xitk_widget_t *w, void *data, int selected, unsigned int modifier) {
  xui_skdloader_t *skd = data;
  gGui_t          *gui = skd->nw.gui;
  download_t       download = { .gui = gui };

  (void)w;
  (void)modifier;
  skd->sel = selected;
  if (gui->verbosity >= 2)
    printf ("gui.skin.download.preview (%d).\n", selected);
  if (selected < 0)
    return;
  if (skd->slxs[selected].s[_SLX_preview] == NULL)
    return;

  xitk_window_define_window_cursor (skd->nw.xwin, xitk_cursor_watch);

  if (gui->verbosity >= 2)
    printf ("gui.skin.download.preview (%s).\n", skd->slxs[selected].s[_SLX_preview]);
  if ((network_download(skd->slxs[selected].s[_SLX_preview], &download))) {
    xitk_image_t *ximg;

    ximg = xitk_image_new (gui->xitk, download.buf, download.size, PREVIEW_WIDTH, PREVIEW_HEIGHT);
    if (ximg) {
      xitk_image_t *oimg = skd->preview_image;
      skd->preview_image = ximg;
      if(oimg)
        xitk_image_free_image (&oimg);
    }

    _skd_preview(skd);
  }
  else {
    gui_msg (gui, XUI_MSG_ERROR, _("Unable to download '%s': %s"),
      skd->slxs[selected].s[_SLX_preview], download.error);
    _skd_blank_preview(skd);
  }

  xitk_window_define_window_cursor (skd->nw.xwin, xitk_cursor_default);

  free(download.buf);
  free(download.error);
}

static void _skd_select (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_skdloader_t *skd = data;
  gGui_t *gui = skd->nw.gui;
  download_t download = { .gui = gui };

  (void)w;
  (void)state;
  (void)modifier;
  if (skd->sel < 0)
    return;

  xitk_window_define_window_cursor (skd->nw.xwin, xitk_cursor_watch);

  do {
    char buf[2048], *e = buf + sizeof (buf) - 32, *q, *cmd;
    const char *name = skd->slxs[skd->sel].s[_SLX_href];
    int err;
    unsigned int u, v;

    /* get last path component. */
    u = v = 0;
    while (name[v]) {
      u = v;
      v += xitk_find_0_or_byte (name + v + 1, '/') + 1;
    }
    if (name[u] == '/')
      u++;
    if (v - u <= strlen (".tar.gz"))
      break;

    /* build local skin path. */
    q = buf + strlcpy (buf, xine_get_homedir (), e - buf);
    if (q >= e)
      break;
    memcpy (q, "/.xine/skins", 13); q += 12; e += 12;
    /* Make sure to have that dir. */
    err = mkdir_safe (buf);
    if (!err) {
      /* xprintf (gui->xine, XINE_VERBOSITY_DEBUG, "skins_download: created dir %s.\n", skindir) */;
    } else if (err != EEXIST) {
      gui_msg (gui, XUI_MSG_ERROR, _("Unable to create '%s' directory: %s."), buf, strerror (err));
      break;
    }
    if (gui->verbosity >= 2)
      printf ("gui.skin.download.dir (%s).\n", buf);

    /* try download. */
    if (!network_download (name, &download)) {
      gui_msg (gui, XUI_MSG_ERROR, _("Unable to download '%s': %s"), name, download.error);
      break;
    }

    /* write download to temp file. */
    *q++ = '/';
    if (q + (v - u) > e)
      break;
    memcpy (q, name + u, v - u + 1);
    if (gui->verbosity >= 2)
      printf ("gui.skin.download.temp_file (%s).\n", buf);
    {
      FILE *fd = fopen (buf, "w+b");
      if (!fd) {
        gui_msg (gui, XUI_MSG_ERROR, _("Unable to create '%s'."), buf);
        break;
      }
      fwrite (download.buf, download.size, 1, fd);
      fclose (fd);
    }

    /* unpack temp file. */
    q[-1] = 0;
    cmd = q + (v - u) + 1;
    snprintf (cmd, e - cmd, "which tar > /dev/null 2>&1 && cd  %s  && gunzip -c %s | tar xf -", buf, q);
    if (gui->verbosity >= 2)
      printf ("gui.skin.download.run (%s).\n", cmd);
    xitk_system (0, cmd);
    q[-1] = '/';
    unlink (buf);

    /* append last path component minus ".tar.gz" to the skin dir path. */
    v -= strlen (".tar.gz");
    q += v - u;
    /* *q = 0; */

    /* run installer script if found.
      * FIXME: Tremendous security risk! We should at least scan the script to contain only allowed commands. */
    memcpy (q, "/doinst.sh", 11);
    cmd = q + 11;
    if (xitk_filetype (buf) == XITK_FILETYPE_FILE) {
      if (!(gui->flags & XUI_FLAG_trust_skin)) {
        size_t ilen = 20 << 10;
        char *inst = xitk_cfg_load (buf, &ilen);
        *q = 0;
        *cmd = 0;
        if (inst) {
          if (strstr (inst, "font_install"))
            snprintf (cmd, e - cmd, "cd %s && xine-font-install", buf);
          else
            *cmd = 0;
          xitk_cfg_unload (inst);
        }
      } else {
        *q = 0;
        snprintf (cmd, e - cmd, "cd %s && ./doinst.sh", buf);
      }
      if (*cmd) {
        if (gui->verbosity >= 2)
          printf ("gui.skin.download.run (%s).\n", cmd);
        xitk_system (0, cmd);
      }
    } else {
      *q = 0;
    }

    /* add this skin. */
    {
      int r = skin_add_1 (gui, buf, q - (v - u), q);
      gui_msg (gui, XUI_MSG_INFO, _("Skin %s correctly installed"), q - (v - u));
      if (r < 0)
        r = ~r;
      /* Okay, load this skin */
      skin_select (gui, r);
    }
  } while (0);

  xitk_window_define_window_cursor (skd->nw.xwin, xitk_cursor_default);

  free(download.buf);
  free(download.error);

  skd->exit (skd);
}

static int _skd_event (void *data, const xitk_be_event_t *e) {
  xui_skdloader_t *skd = data;

  if (e->type == XITK_EV_EXPOSE) {
    _skd_preview (skd);
    return 1;
  } else if (((e->type == XITK_EV_KEY_DOWN) && (e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE))
    || (e->type == XITK_EV_DEL_WIN)) {
    skd->exit (skd);
    return 1;
  }
  return gui_handle_be_event (skd->nw.gui, e);
}

void skin_download_end (xui_skdloader_t *skd) {
  if (skd)
    skd->exit (skd);
}

void skin_download (gGui_t *gui, const char *url) {
  slx_entry_t *slxs;
  xui_skdloader_t *skd;
  int x, y;

  if (gui->skdloader) {
    xitk_window_t *win = gui->skdloader->nw.xwin;
    xitk_window_raise_window (win);
    xitk_window_set_input_focus (win);
    return;
  }

  skd = calloc (1, sizeof (*skd));
  if (!skd)
    return;
  skd->nw.gui = gui;

  if (xitk_init_NULL ()) {
    skd->nw.skin = NULL;
    skd->nw.wfskin = NULL;
    skd->nw.adjust = NULL;
    skd->slxs = NULL;
    skd->buf = NULL;
    skd->preview_image = NULL;
  }
  {
    xitk_register_key_t downloading_key = xitk_window_dialog_3 (gui->xitk, NULL,
      gui_layer_above (gui, NULL), 400, _("Be patient..."), NULL, NULL,
      NULL, NULL, NULL, NULL, 0, ALIGN_CENTER, _("Retrieving skin list from %s"), url);

    slxs = _skd_get_slx (skd, url);
    xitk_unregister_event_handler (gui->xitk, &downloading_key);
  }

  if (!slxs) {
    free (skd);
    return;
  }

#if 0
  for (i = 0; i < skd->num; i++) {
    printf("download skins: Skin number %d:\n", i);
    printf("  Name: %s\n", slxs[i].name);
    printf("  Author Name: %s\n", slxs[i].author.name);
    printf("  Author email: %s\n", slxs[i].author.email);
    printf("  Href: %s\n", slxs[i].skin.href);
    printf("  Version: %d\n", slxs[i].skin.version);
    printf("  Maintained: %d\n", slxs[i].skin.maintained);
    printf("--\n");
  }
#endif

  skd->nw.title = _("Choose a skin to download...");
  skd->nw.id = "skin_download";
  skd->nw.wr.x = XITK_WINDOW_POS_CENTER;
  skd->nw.wr.y = XITK_WINDOW_POS_CENTER;
  skd->nw.wr.width = WINDOW_WIDTH;
  skd->nw.wr.height = WINDOW_HEIGHT;
  {
    int r = gui_window_new (&skd->nw);
    if (r < 0) {
      free (skd->names);
      free (skd->buf);
      free (skd->slxs);
      free (skd);
      return;
    }
  }
  skd->exit = _skd_exit_1;

  x = 15;
  y = 34 + PREVIEW_HEIGHT + 14;

  {
    xitk_browser_widget_t br = {
      .nw = { .wl = skd->nw.wl, .userdata = skd, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .browser = {
        .max_displayed_entries = MAX_DISP_ENTRIES,
        .num_entries           = skd->num,
        .entries               = (const char *const *)skd->names,
        .shortcuts             = (const char *const *)skd->info
      },
      .callback = _skd_get_preview
    };

    skd->browser = xitk_noskin_browser_create (&br,
      x + 5, y + 5, WINDOW_WIDTH - (30 + 10 + 16), 20, 16, br_fontname);
  }

  xitk_image_draw_rectangular_box (skd->nw.bg, x, y, WINDOW_WIDTH - 30, MAX_DISP_ENTRIES * 20 + 16 + 10, XITK_DRAW_INNER);
  xitk_window_set_background_image (skd->nw.xwin, skd->nw.bg);

  y = WINDOW_HEIGHT - (23 + 15);
  x = 15;

  {
    xitk_labelbutton_widget_t lb = {
      .nw = { .wl = skd->nw.wl, .userdata = skd, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .button_type = CLICK_BUTTON,
      .align = ALIGN_CENTER,
    };

    lb.label = _("Load");
    lb.callback = _skd_select;
    xitk_noskin_labelbutton_create (&lb,
      x, y, 100, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    x = WINDOW_WIDTH - (100 + 15);

    lb.label    = _("Cancel");
    lb.callback = _skd_exit_4;
    xitk_noskin_labelbutton_create (&lb,
      x, y, 100, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);
  }

  skd->nw.key = xitk_be_register_event_handler ("skdloader", skd->nw.xwin, _skd_event, skd, NULL, NULL);

  xitk_window_flags (skd->nw.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (skd->nw.xwin);
  gui_layer_above (skd->nw.gui, skd->nw.xwin);
  xitk_window_set_input_focus (skd->nw.xwin);
  _skd_blank_preview (skd);

  gui->skdloader = skd;
}
