# -*- coding: utf-8 -*-
=begin rdoc
String extensions.
=end

class String

  # Replace a string's accented characters with unaccented characters.
  #
  # @example
  #   s = "Å Ç ß"
  #   s.unaccent = > "AA C ss"
  #
  # @return [String] a string that has no accents

  def unaccent
    unaccent_via_scan
  end

  # Replace a string's accented characters with unaccented characters,
  # by using string `#scan` to iterate on characters.
  #
  # @example
  #   s = "Å Ç ß"
  #   s.unaccent_via_scan = > "AA C ss"
  #
  # @return [String] a string that has no accents

  def unaccent_via_scan
    result=""; scan(/./){|c| result += (ACCENTMAP[c] || c) }; result
  end

  # Replace a string's accented characters with unaccented characters,
  # by using string `#each_char` to iterate on characters.
  #
  # @example
  #   s = "Å Ç ß"
  #   s.unaccent_via_each_char = > "AA C ss"
  #
  # @return [String] a string that has no accents

  def unaccent_via_each_char
    result=""; each_char{|c| result += (ACCENTMAP[c] || c) }; result
  end

  # Replace a string's accented characters with unaccented characters,
  # by using string `#split` and `#map` to iterate on characters.
  #
  # @example
  #   s = "Å Ç ß"
  #   s.unaccent_via_split_map = > "AA C ss"
  #
  # @return [String] a string that has no accents

  def unaccent_via_split_map
    split(//u).map{|c| ACCENTMAP[c] || c }.join("")
  end

  ACCENTMAP = {

    # 00A0 NO-BREAK SPACE
    # ->   0020 SPACE
    "\u{A0}" => " ",

    # 00A8 DIAERESIS
    # ->   0020 SPACE
    "\u{A8}" => " ",

    # 00AA FEMININE ORDINAL INDICATOR
    # ->   0061 LATIN SMALL LETTER A
    "\u{AA}" => "a",

    # 00AF MACRON
    # ->   0020 SPACE
    "\u{AF}" => " ",

    # 00B2 SUPERSCRIPT TWO
    # ->   0032 DIGIT TWO
    "\u{B2}" => "2",

    # 00B3 SUPERSCRIPT THREE
    # ->   0033 DIGIT THREE
    "\u{B3}" => "3",

    # 00B4 ACUTE ACCENT
    # ->   0020 SPACE
    "\u{B4}" => " ",

    # 00B5 MICRO SIGN
    # ->   03BC GREEK SMALL LETTER MU
    "\u{B5}" => "\u{03BC}",

    # 00B8 CEDILLA
    # ->   0020 SPACE
    "\u{B8}" => " ",

    # 00B9 SUPERSCRIPT ONE
    # ->   0031 DIGIT ONE
    "\u{B9}" => "1",

    # 00BA MASCULINE ORDINAL INDICATOR
    # ->   006F LATIN SMALL LETTER O
    "\u{BA}" => "o",

    # 00BC VULGAR FRACTION ONE QUARTER
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0034 DIGIT FOUR
    "\u{BC}" => "1\u{2044}4",

    # 00BD VULGAR FRACTION ONE HALF
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0032 DIGIT TWO
    "\u{BD}" => "1\u{2044}2",

    # 00BE VULGAR FRACTION THREE QUARTERS
    # ->   0033 DIGIT THREE
    #    + 2044 FRACTION SLASH
    #    + 0034 DIGIT FOUR
    "\u{BE}" => "3\u{2044}4",

    # 00C0 LATIN CAPITAL LETTER A WITH GRAVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C0}" => "A",

    # 00C1 LATIN CAPITAL LETTER A WITH ACUTE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C1}" => "A",

    # 00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C2}" => "A",

    # 00C3 LATIN CAPITAL LETTER A WITH TILDE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C3}" => "A",

    # 00C4 LATIN CAPITAL LETTER A WITH DIAERESIS
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C4}" => "A",

    # 00C5 LATIN CAPITAL LETTER A WITH RING ABOVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{C5}" => "A",

    # 00C6 LATIN CAPITAL LETTER AE
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 0045 LATIN CAPITAL LETTER E
    "\u{C6}" => "AE",

    # 00C7 LATIN CAPITAL LETTER C WITH CEDILLA
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{C7}" => "C",

    # 00C8 LATIN CAPITAL LETTER E WITH GRAVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{C8}" => "E",

    # 00C9 LATIN CAPITAL LETTER E WITH ACUTE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{C9}" => "E",

    # 00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{CA}" => "E",

    # 00CB LATIN CAPITAL LETTER E WITH DIAERESIS
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{CB}" => "E",

    # 00CC LATIN CAPITAL LETTER I WITH GRAVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{CC}" => "I",

    # 00CD LATIN CAPITAL LETTER I WITH ACUTE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{CD}" => "I",

    # 00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{CE}" => "I",

    # 00CF LATIN CAPITAL LETTER I WITH DIAERESIS
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{CF}" => "I",

    # 00D1 LATIN CAPITAL LETTER N WITH TILDE
    # ->   004E LATIN CAPITAL LETTER N
    "\u{D1}" => "N",

    # 00D2 LATIN CAPITAL LETTER O WITH GRAVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D2}" => "O",

    # 00D3 LATIN CAPITAL LETTER O WITH ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D3}" => "O",

    # 00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D4}" => "O",

    # 00D5 LATIN CAPITAL LETTER O WITH TILDE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D5}" => "O",

    # 00D6 LATIN CAPITAL LETTER O WITH DIAERESIS
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D6}" => "O",

    # 00D8 LATIN CAPITAL LETTER O WITH STROKE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{D8}" => "O",

    # 00D9 LATIN CAPITAL LETTER U WITH GRAVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{D9}" => "U",

    # 00DA LATIN CAPITAL LETTER U WITH ACUTE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{DA}" => "U",

    # 00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{DB}" => "U",

    # 00DC LATIN CAPITAL LETTER U WITH DIAERESIS
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{DC}" => "U",

    # 00DD LATIN CAPITAL LETTER Y WITH ACUTE
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{DD}" => "Y",

    # 00E0 LATIN SMALL LETTER A WITH GRAVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{E0}" => "a",

    # 00E1 LATIN SMALL LETTER A WITH ACUTE
    # ->   0061 LATIN SMALL LETTER A
    "\u{E1}" => "a",

    # 00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX
    # ->   0061 LATIN SMALL LETTER A
    "\u{E2}" => "a",

    # 00E3 LATIN SMALL LETTER A WITH TILDE
    # ->   0061 LATIN SMALL LETTER A
    "\u{E3}" => "a",

    # 00E4 LATIN SMALL LETTER A WITH DIAERESIS
    # ->   0061 LATIN SMALL LETTER A
    "\u{E4}" => "a",

    # 00E5 LATIN SMALL LETTER A WITH RING ABOVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{E5}" => "a",

    # 00E6 LATIN SMALL LETTER AE
    # ->   0061 LATIN SMALL LETTER A
    #    + 0065 LATIN SMALL LETTER E
    "\u{E6}" => "ae",

    # 00E7 LATIN SMALL LETTER C WITH CEDILLA
    # ->   0063 LATIN SMALL LETTER C
    "\u{E7}" => "c",

    # 00E8 LATIN SMALL LETTER E WITH GRAVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{E8}" => "e",

    # 00E9 LATIN SMALL LETTER E WITH ACUTE
    # ->   0065 LATIN SMALL LETTER E
    "\u{E9}" => "e",

    # 00EA LATIN SMALL LETTER E WITH CIRCUMFLEX
    # ->   0065 LATIN SMALL LETTER E
    "\u{EA}" => "e",

    # 00EB LATIN SMALL LETTER E WITH DIAERESIS
    # ->   0065 LATIN SMALL LETTER E
    "\u{EB}" => "e",

    # 00EC LATIN SMALL LETTER I WITH GRAVE
    # ->   0069 LATIN SMALL LETTER I
    "\u{EC}" => "i",

    # 00ED LATIN SMALL LETTER I WITH ACUTE
    # ->   0069 LATIN SMALL LETTER I
    "\u{ED}" => "i",

    # 00EE LATIN SMALL LETTER I WITH CIRCUMFLEX
    # ->   0069 LATIN SMALL LETTER I
    "\u{EE}" => "i",

    # 00EF LATIN SMALL LETTER I WITH DIAERESIS
    # ->   0069 LATIN SMALL LETTER I
    "\u{EF}" => "i",

    # 00F1 LATIN SMALL LETTER N WITH TILDE
    # ->   006E LATIN SMALL LETTER N
    "\u{F1}" => "n",

    # 00F2 LATIN SMALL LETTER O WITH GRAVE
    # ->   006F LATIN SMALL LETTER O
    "\u{F2}" => "o",

    # 00F3 LATIN SMALL LETTER O WITH ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{F3}" => "o",

    # 00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX
    # ->   006F LATIN SMALL LETTER O
    "\u{F4}" => "o",

    # 00F5 LATIN SMALL LETTER O WITH TILDE
    # ->   006F LATIN SMALL LETTER O
    "\u{F5}" => "o",

    # 00F6 LATIN SMALL LETTER O WITH DIAERESIS
    # ->   006F LATIN SMALL LETTER O
    "\u{F6}" => "o",

    # 00F8 LATIN SMALL LETTER O WITH STROKE
    # ->   006F LATIN SMALL LETTER O
    "\u{F8}" => "o",

    # 00F9 LATIN SMALL LETTER U WITH GRAVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{F9}" => "u",

    # 00FA LATIN SMALL LETTER U WITH ACUTE
    # ->   0075 LATIN SMALL LETTER U
    "\u{FA}" => "u",

    # 00FB LATIN SMALL LETTER U WITH CIRCUMFLEX
    # ->   0075 LATIN SMALL LETTER U
    "\u{FB}" => "u",

    # 00FC LATIN SMALL LETTER U WITH DIAERESIS
    # ->   0075 LATIN SMALL LETTER U
    "\u{FC}" => "u",

    # 00FD LATIN SMALL LETTER Y WITH ACUTE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{FD}" => "y",

    # 00FF LATIN SMALL LETTER Y WITH DIAERESIS
    # ->   0079 LATIN SMALL LETTER Y
    "\u{FF}" => "y",

    # 0100 LATIN CAPITAL LETTER A WITH MACRON
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0100}" => "A",

    # 0101 LATIN SMALL LETTER A WITH MACRON
    # ->   0061 LATIN SMALL LETTER A
    "\u{0101}" => "a",

    # 0102 LATIN CAPITAL LETTER A WITH BREVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0102}" => "A",

    # 0103 LATIN SMALL LETTER A WITH BREVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{0103}" => "a",

    # 0104 LATIN CAPITAL LETTER A WITH OGONEK
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0104}" => "A",

    # 0105 LATIN SMALL LETTER A WITH OGONEK
    # ->   0061 LATIN SMALL LETTER A
    "\u{0105}" => "a",

    # 0106 LATIN CAPITAL LETTER C WITH ACUTE
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{0106}" => "C",

    # 0107 LATIN SMALL LETTER C WITH ACUTE
    # ->   0063 LATIN SMALL LETTER C
    "\u{0107}" => "c",

    # 0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{0108}" => "C",

    # 0109 LATIN SMALL LETTER C WITH CIRCUMFLEX
    # ->   0063 LATIN SMALL LETTER C
    "\u{0109}" => "c",

    # 010A LATIN CAPITAL LETTER C WITH DOT ABOVE
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{010A}" => "C",

    # 010B LATIN SMALL LETTER C WITH DOT ABOVE
    # ->   0063 LATIN SMALL LETTER C
    "\u{010B}" => "c",

    # 010C LATIN CAPITAL LETTER C WITH CARON
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{010C}" => "C",

    # 010D LATIN SMALL LETTER C WITH CARON
    # ->   0063 LATIN SMALL LETTER C
    "\u{010D}" => "c",

    # 010E LATIN CAPITAL LETTER D WITH CARON
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{010E}" => "D",

    # 010F LATIN SMALL LETTER D WITH CARON
    # ->   0064 LATIN SMALL LETTER D
    "\u{010F}" => "d",

    # 0110 LATIN CAPITAL LETTER D WITH STROKE
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{0110}" => "D",

    # 0111 LATIN SMALL LETTER D WITH STROKE
    # ->   0064 LATIN SMALL LETTER D
    "\u{0111}" => "d",

    # 0112 LATIN CAPITAL LETTER E WITH MACRON
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0112}" => "E",

    # 0113 LATIN SMALL LETTER E WITH MACRON
    # ->   0065 LATIN SMALL LETTER E
    "\u{0113}" => "e",

    # 0114 LATIN CAPITAL LETTER E WITH BREVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0114}" => "E",

    # 0115 LATIN SMALL LETTER E WITH BREVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{0115}" => "e",

    # 0116 LATIN CAPITAL LETTER E WITH DOT ABOVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0116}" => "E",

    # 0117 LATIN SMALL LETTER E WITH DOT ABOVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{0117}" => "e",

    # 0118 LATIN CAPITAL LETTER E WITH OGONEK
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0118}" => "E",

    # 0119 LATIN SMALL LETTER E WITH OGONEK
    # ->   0065 LATIN SMALL LETTER E
    "\u{0119}" => "e",

    # 011A LATIN CAPITAL LETTER E WITH CARON
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{011A}" => "E",

    # 011B LATIN SMALL LETTER E WITH CARON
    # ->   0065 LATIN SMALL LETTER E
    "\u{011B}" => "e",

    # 011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{011C}" => "G",

    # 011D LATIN SMALL LETTER G WITH CIRCUMFLEX
    # ->   0067 LATIN SMALL LETTER G
    "\u{011D}" => "g",

    # 011E LATIN CAPITAL LETTER G WITH BREVE
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{011E}" => "G",

    # 011F LATIN SMALL LETTER G WITH BREVE
    # ->   0067 LATIN SMALL LETTER G
    "\u{011F}" => "g",

    # 0120 LATIN CAPITAL LETTER G WITH DOT ABOVE
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{0120}" => "G",

    # 0121 LATIN SMALL LETTER G WITH DOT ABOVE
    # ->   0067 LATIN SMALL LETTER G
    "\u{0121}" => "g",

    # 0122 LATIN CAPITAL LETTER G WITH CEDILLA
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{0122}" => "G",

    # 0123 LATIN SMALL LETTER G WITH CEDILLA
    # ->   0067 LATIN SMALL LETTER G
    "\u{0123}" => "g",

    # 0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{0124}" => "H",

    # 0125 LATIN SMALL LETTER H WITH CIRCUMFLEX
    # ->   0068 LATIN SMALL LETTER H
    "\u{0125}" => "h",

    # 0126 LATIN CAPITAL LETTER H WITH STROKE
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{0126}" => "H",

    # 0127 LATIN SMALL LETTER H WITH STROKE
    # ->   0068 LATIN SMALL LETTER H
    "\u{0127}" => "h",

    # 0128 LATIN CAPITAL LETTER I WITH TILDE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{0128}" => "I",

    # 0129 LATIN SMALL LETTER I WITH TILDE
    # ->   0069 LATIN SMALL LETTER I
    "\u{0129}" => "i",

    # 012A LATIN CAPITAL LETTER I WITH MACRON
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{012A}" => "I",

    # 012B LATIN SMALL LETTER I WITH MACRON
    # ->   0069 LATIN SMALL LETTER I
    "\u{012B}" => "i",

    # 012C LATIN CAPITAL LETTER I WITH BREVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{012C}" => "I",

    # 012D LATIN SMALL LETTER I WITH BREVE
    # ->   0069 LATIN SMALL LETTER I
    "\u{012D}" => "i",

    # 012E LATIN CAPITAL LETTER I WITH OGONEK
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{012E}" => "I",

    # 012F LATIN SMALL LETTER I WITH OGONEK
    # ->   0069 LATIN SMALL LETTER I
    "\u{012F}" => "i",

    # 0130 LATIN CAPITAL LETTER I WITH DOT ABOVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{0130}" => "I",

    # 0132 LATIN CAPITAL LIGATURE IJ
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 004A LATIN CAPITAL LETTER J
    "\u{0132}" => "IJ",

    # 0133 LATIN SMALL LIGATURE IJ
    # ->   0069 LATIN SMALL LETTER I
    #    + 006A LATIN SMALL LETTER J
    "\u{0133}" => "ij",

    # 0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX
    # ->   004A LATIN CAPITAL LETTER J
    "\u{0134}" => "J",

    # 0135 LATIN SMALL LETTER J WITH CIRCUMFLEX
    # ->   006A LATIN SMALL LETTER J
    "\u{0135}" => "j",

    # 0136 LATIN CAPITAL LETTER K WITH CEDILLA
    # ->   004B LATIN CAPITAL LETTER K
    "\u{0136}" => "K",

    # 0137 LATIN SMALL LETTER K WITH CEDILLA
    # ->   006B LATIN SMALL LETTER K
    "\u{0137}" => "k",

    # 0139 LATIN CAPITAL LETTER L WITH ACUTE
    # ->   004C LATIN CAPITAL LETTER L
    "\u{0139}" => "L",

    # 013A LATIN SMALL LETTER L WITH ACUTE
    # ->   006C LATIN SMALL LETTER L
    "\u{013A}" => "l",

    # 013B LATIN CAPITAL LETTER L WITH CEDILLA
    # ->   004C LATIN CAPITAL LETTER L
    "\u{013B}" => "L",

    # 013C LATIN SMALL LETTER L WITH CEDILLA
    # ->   006C LATIN SMALL LETTER L
    "\u{013C}" => "l",

    # 013D LATIN CAPITAL LETTER L WITH CARON
    # ->   004C LATIN CAPITAL LETTER L
    "\u{013D}" => "L",

    # 013E LATIN SMALL LETTER L WITH CARON
    # ->   006C LATIN SMALL LETTER L
    "\u{013E}" => "l",

    # 013F LATIN CAPITAL LETTER L WITH MIDDLE DOT
    # ->   004C LATIN CAPITAL LETTER L
    #    + 00B7 MIDDLE DOT
    "\u{013F}" => "L\u{B7}",

    # 0140 LATIN SMALL LETTER L WITH MIDDLE DOT
    # ->   006C LATIN SMALL LETTER L
    #    + 00B7 MIDDLE DOT
    "\u{0140}" => "l\u{B7}",

    # 0141 LATIN CAPITAL LETTER L WITH STROKE
    # ->   004C LATIN CAPITAL LETTER L
    "\u{0141}" => "L",

    # 0142 LATIN SMALL LETTER L WITH STROKE
    # ->   006C LATIN SMALL LETTER L
    "\u{0142}" => "l",

    # 0143 LATIN CAPITAL LETTER N WITH ACUTE
    # ->   004E LATIN CAPITAL LETTER N
    "\u{0143}" => "N",

    # 0144 LATIN SMALL LETTER N WITH ACUTE
    # ->   006E LATIN SMALL LETTER N
    "\u{0144}" => "n",

    # 0145 LATIN CAPITAL LETTER N WITH CEDILLA
    # ->   004E LATIN CAPITAL LETTER N
    "\u{0145}" => "N",

    # 0146 LATIN SMALL LETTER N WITH CEDILLA
    # ->   006E LATIN SMALL LETTER N
    "\u{0146}" => "n",

    # 0147 LATIN CAPITAL LETTER N WITH CARON
    # ->   004E LATIN CAPITAL LETTER N
    "\u{0147}" => "N",

    # 0148 LATIN SMALL LETTER N WITH CARON
    # ->   006E LATIN SMALL LETTER N
    "\u{0148}" => "n",

    # 0149 LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
    # ->   02BC MODIFIER LETTER APOSTROPHE
    #    + 006E LATIN SMALL LETTER N
    "\u{0149}" => "\u{02BC}n",

    # 014C LATIN CAPITAL LETTER O WITH MACRON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{014C}" => "O",

    # 014D LATIN SMALL LETTER O WITH MACRON
    # ->   006F LATIN SMALL LETTER O
    "\u{014D}" => "o",

    # 014E LATIN CAPITAL LETTER O WITH BREVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{014E}" => "O",

    # 014F LATIN SMALL LETTER O WITH BREVE
    # ->   006F LATIN SMALL LETTER O
    "\u{014F}" => "o",

    # 0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{0150}" => "O",

    # 0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{0151}" => "o",

    # 0154 LATIN CAPITAL LETTER R WITH ACUTE
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{0154}" => "R",

    # 0155 LATIN SMALL LETTER R WITH ACUTE
    # ->   0072 LATIN SMALL LETTER R
    "\u{0155}" => "r",

    # 0156 LATIN CAPITAL LETTER R WITH CEDILLA
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{0156}" => "R",

    # 0157 LATIN SMALL LETTER R WITH CEDILLA
    # ->   0072 LATIN SMALL LETTER R
    "\u{0157}" => "r",

    # 0158 LATIN CAPITAL LETTER R WITH CARON
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{0158}" => "R",

    # 0159 LATIN SMALL LETTER R WITH CARON
    # ->   0072 LATIN SMALL LETTER R
    "\u{0159}" => "r",

    # 015A LATIN CAPITAL LETTER S WITH ACUTE
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{015A}" => "S",

    # 015B LATIN SMALL LETTER S WITH ACUTE
    # ->   0073 LATIN SMALL LETTER S
    "\u{015B}" => "s",

    # 015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{015C}" => "S",

    # 015D LATIN SMALL LETTER S WITH CIRCUMFLEX
    # ->   0073 LATIN SMALL LETTER S
    "\u{015D}" => "s",

    # 015E LATIN CAPITAL LETTER S WITH CEDILLA
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{015E}" => "S",

    # 015F LATIN SMALL LETTER S WITH CEDILLA
    # ->   0073 LATIN SMALL LETTER S
    "\u{015F}" => "s",

    # 0160 LATIN CAPITAL LETTER S WITH CARON
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{0160}" => "S",

    # 0161 LATIN SMALL LETTER S WITH CARON
    # ->   0073 LATIN SMALL LETTER S
    "\u{0161}" => "s",

    # 0162 LATIN CAPITAL LETTER T WITH CEDILLA
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{0162}" => "T",

    # 0163 LATIN SMALL LETTER T WITH CEDILLA
    # ->   0074 LATIN SMALL LETTER T
    "\u{0163}" => "t",

    # 0164 LATIN CAPITAL LETTER T WITH CARON
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{0164}" => "T",

    # 0165 LATIN SMALL LETTER T WITH CARON
    # ->   0074 LATIN SMALL LETTER T
    "\u{0165}" => "t",

    # 0166 LATIN CAPITAL LETTER T WITH STROKE
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{0166}" => "T",

    # 0167 LATIN SMALL LETTER T WITH STROKE
    # ->   0074 LATIN SMALL LETTER T
    "\u{0167}" => "t",

    # 0168 LATIN CAPITAL LETTER U WITH TILDE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{0168}" => "U",

    # 0169 LATIN SMALL LETTER U WITH TILDE
    # ->   0075 LATIN SMALL LETTER U
    "\u{0169}" => "u",

    # 016A LATIN CAPITAL LETTER U WITH MACRON
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{016A}" => "U",

    # 016B LATIN SMALL LETTER U WITH MACRON
    # ->   0075 LATIN SMALL LETTER U
    "\u{016B}" => "u",

    # 016C LATIN CAPITAL LETTER U WITH BREVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{016C}" => "U",

    # 016D LATIN SMALL LETTER U WITH BREVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{016D}" => "u",

    # 016E LATIN CAPITAL LETTER U WITH RING ABOVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{016E}" => "U",

    # 016F LATIN SMALL LETTER U WITH RING ABOVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{016F}" => "u",

    # 0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{0170}" => "U",

    # 0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE
    # ->   0075 LATIN SMALL LETTER U
    "\u{0171}" => "u",

    # 0172 LATIN CAPITAL LETTER U WITH OGONEK
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{0172}" => "U",

    # 0173 LATIN SMALL LETTER U WITH OGONEK
    # ->   0075 LATIN SMALL LETTER U
    "\u{0173}" => "u",

    # 0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{0174}" => "W",

    # 0175 LATIN SMALL LETTER W WITH CIRCUMFLEX
    # ->   0077 LATIN SMALL LETTER W
    "\u{0175}" => "w",

    # 0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{0176}" => "Y",

    # 0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX
    # ->   0079 LATIN SMALL LETTER Y
    "\u{0177}" => "y",

    # 0178 LATIN CAPITAL LETTER Y WITH DIAERESIS
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{0178}" => "Y",

    # 0179 LATIN CAPITAL LETTER Z WITH ACUTE
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{0179}" => "Z",

    # 017A LATIN SMALL LETTER Z WITH ACUTE
    # ->   007A LATIN SMALL LETTER Z
    "\u{017A}" => "z",

    # 017B LATIN CAPITAL LETTER Z WITH DOT ABOVE
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{017B}" => "Z",

    # 017C LATIN SMALL LETTER Z WITH DOT ABOVE
    # ->   007A LATIN SMALL LETTER Z
    "\u{017C}" => "z",

    # 017D LATIN CAPITAL LETTER Z WITH CARON
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{017D}" => "Z",

    # 017E LATIN SMALL LETTER Z WITH CARON
    # ->   007A LATIN SMALL LETTER Z
    "\u{017E}" => "z",

    # 017F LATIN SMALL LETTER LONG S
    # ->   0073 LATIN SMALL LETTER S
    "\u{017F}" => "s",

    # 0180 LATIN SMALL LETTER B WITH STROKE
    # ->   0062 LATIN SMALL LETTER B
    "\u{0180}" => "b",

    # 0181 LATIN CAPITAL LETTER B WITH HOOK
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{0181}" => "B",

    # 0182 LATIN CAPITAL LETTER B WITH TOPBAR
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{0182}" => "B",

    # 0183 LATIN SMALL LETTER B WITH TOPBAR
    # ->   0062 LATIN SMALL LETTER B
    "\u{0183}" => "b",

    # 0187 LATIN CAPITAL LETTER C WITH HOOK
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{0187}" => "C",

    # 0188 LATIN SMALL LETTER C WITH HOOK
    # ->   0063 LATIN SMALL LETTER C
    "\u{0188}" => "c",

    # 018A LATIN CAPITAL LETTER D WITH HOOK
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{018A}" => "D",

    # 018B LATIN CAPITAL LETTER D WITH TOPBAR
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{018B}" => "D",

    # 018C LATIN SMALL LETTER D WITH TOPBAR
    # ->   0064 LATIN SMALL LETTER D
    "\u{018C}" => "d",

    # 0191 LATIN CAPITAL LETTER F WITH HOOK
    # ->   0046 LATIN CAPITAL LETTER F
    "\u{0191}" => "F",

    # 0192 LATIN SMALL LETTER F WITH HOOK
    # ->   0066 LATIN SMALL LETTER F
    "\u{0192}" => "f",

    # 0193 LATIN CAPITAL LETTER G WITH HOOK
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{0193}" => "G",

    # 0197 LATIN CAPITAL LETTER I WITH STROKE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{0197}" => "I",

    # 0198 LATIN CAPITAL LETTER K WITH HOOK
    # ->   004B LATIN CAPITAL LETTER K
    "\u{0198}" => "K",

    # 0199 LATIN SMALL LETTER K WITH HOOK
    # ->   006B LATIN SMALL LETTER K
    "\u{0199}" => "k",

    # 019A LATIN SMALL LETTER L WITH BAR
    # ->   006C LATIN SMALL LETTER L
    "\u{019A}" => "l",

    # 019D LATIN CAPITAL LETTER N WITH LEFT HOOK
    # ->   004E LATIN CAPITAL LETTER N
    "\u{019D}" => "N",

    # 019E LATIN SMALL LETTER N WITH LONG RIGHT LEG
    # ->   006E LATIN SMALL LETTER N
    "\u{019E}" => "n",

    # 019F LATIN CAPITAL LETTER O WITH MIDDLE TILDE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{019F}" => "O",

    # 01A0 LATIN CAPITAL LETTER O WITH HORN
    # ->   004F LATIN CAPITAL LETTER O
    "\u{01A0}" => "O",

    # 01A1 LATIN SMALL LETTER O WITH HORN
    # ->   006F LATIN SMALL LETTER O
    "\u{01A1}" => "o",

    # 01A4 LATIN CAPITAL LETTER P WITH HOOK
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{01A4}" => "P",

    # 01A5 LATIN SMALL LETTER P WITH HOOK
    # ->   0070 LATIN SMALL LETTER P
    "\u{01A5}" => "p",

    # 01AB LATIN SMALL LETTER T WITH PALATAL HOOK
    # ->   0074 LATIN SMALL LETTER T
    "\u{01AB}" => "t",

    # 01AC LATIN CAPITAL LETTER T WITH HOOK
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{01AC}" => "T",

    # 01AD LATIN SMALL LETTER T WITH HOOK
    # ->   0074 LATIN SMALL LETTER T
    "\u{01AD}" => "t",

    # 01AE LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{01AE}" => "T",

    # 01AF LATIN CAPITAL LETTER U WITH HORN
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01AF}" => "U",

    # 01B0 LATIN SMALL LETTER U WITH HORN
    # ->   0075 LATIN SMALL LETTER U
    "\u{01B0}" => "u",

    # 01B2 LATIN CAPITAL LETTER V WITH HOOK
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{01B2}" => "V",

    # 01B3 LATIN CAPITAL LETTER Y WITH HOOK
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{01B3}" => "Y",

    # 01B4 LATIN SMALL LETTER Y WITH HOOK
    # ->   0079 LATIN SMALL LETTER Y
    "\u{01B4}" => "y",

    # 01B5 LATIN CAPITAL LETTER Z WITH STROKE
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{01B5}" => "Z",

    # 01B6 LATIN SMALL LETTER Z WITH STROKE
    # ->   007A LATIN SMALL LETTER Z
    "\u{01B6}" => "z",

    # 01BA LATIN SMALL LETTER EZH WITH TAIL
    # ->   0292 LATIN SMALL LETTER EZH
    "\u{01BA}" => "\u{0292}",

    # 01BE LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE
    # ->   0296 LATIN LETTER INVERTED GLOTTAL STOP
    "\u{01BE}" => "\u{0296}",

    # 01C4 LATIN CAPITAL LETTER DZ WITH CARON
    # ->   0044 LATIN CAPITAL LETTER D
    #    + 005A LATIN CAPITAL LETTER Z
    "\u{01C4}" => "DZ",

    # 01C5 LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
    # ->   0044 LATIN CAPITAL LETTER D
    #    + 007A LATIN SMALL LETTER Z
    "\u{01C5}" => "Dz",

    # 01C6 LATIN SMALL LETTER DZ WITH CARON
    # ->   0064 LATIN SMALL LETTER D
    #    + 007A LATIN SMALL LETTER Z
    "\u{01C6}" => "dz",

    # 01C7 LATIN CAPITAL LETTER LJ
    # ->   004C LATIN CAPITAL LETTER L
    #    + 004A LATIN CAPITAL LETTER J
    "\u{01C7}" => "LJ",

    # 01C8 LATIN CAPITAL LETTER L WITH SMALL LETTER J
    # ->   004C LATIN CAPITAL LETTER L
    #    + 006A LATIN SMALL LETTER J
    "\u{01C8}" => "Lj",

    # 01C9 LATIN SMALL LETTER LJ
    # ->   006C LATIN SMALL LETTER L
    #    + 006A LATIN SMALL LETTER J
    "\u{01C9}" => "lj",

    # 01CA LATIN CAPITAL LETTER NJ
    # ->   004E LATIN CAPITAL LETTER N
    #    + 004A LATIN CAPITAL LETTER J
    "\u{01CA}" => "NJ",

    # 01CB LATIN CAPITAL LETTER N WITH SMALL LETTER J
    # ->   004E LATIN CAPITAL LETTER N
    #    + 006A LATIN SMALL LETTER J
    "\u{01CB}" => "Nj",

    # 01CC LATIN SMALL LETTER NJ
    # ->   006E LATIN SMALL LETTER N
    #    + 006A LATIN SMALL LETTER J
    "\u{01CC}" => "nj",

    # 01CD LATIN CAPITAL LETTER A WITH CARON
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{01CD}" => "A",

    # 01CE LATIN SMALL LETTER A WITH CARON
    # ->   0061 LATIN SMALL LETTER A
    "\u{01CE}" => "a",

    # 01CF LATIN CAPITAL LETTER I WITH CARON
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{01CF}" => "I",

    # 01D0 LATIN SMALL LETTER I WITH CARON
    # ->   0069 LATIN SMALL LETTER I
    "\u{01D0}" => "i",

    # 01D1 LATIN CAPITAL LETTER O WITH CARON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{01D1}" => "O",

    # 01D2 LATIN SMALL LETTER O WITH CARON
    # ->   006F LATIN SMALL LETTER O
    "\u{01D2}" => "o",

    # 01D3 LATIN CAPITAL LETTER U WITH CARON
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01D3}" => "U",

    # 01D4 LATIN SMALL LETTER U WITH CARON
    # ->   0075 LATIN SMALL LETTER U
    "\u{01D4}" => "u",

    # 01D5 LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01D5}" => "U",

    # 01D6 LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
    # ->   0075 LATIN SMALL LETTER U
    "\u{01D6}" => "u",

    # 01D7 LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01D7}" => "U",

    # 01D8 LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
    # ->   0075 LATIN SMALL LETTER U
    "\u{01D8}" => "u",

    # 01D9 LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01D9}" => "U",

    # 01DA LATIN SMALL LETTER U WITH DIAERESIS AND CARON
    # ->   0075 LATIN SMALL LETTER U
    "\u{01DA}" => "u",

    # 01DB LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{01DB}" => "U",

    # 01DC LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{01DC}" => "u",

    # 01DE LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{01DE}" => "A",

    # 01DF LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
    # ->   0061 LATIN SMALL LETTER A
    "\u{01DF}" => "a",

    # 01E0 LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{01E0}" => "A",

    # 01E1 LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
    # ->   0061 LATIN SMALL LETTER A
    "\u{01E1}" => "a",

    # 01E2 LATIN CAPITAL LETTER AE WITH MACRON
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 0045 LATIN CAPITAL LETTER E
    "\u{01E2}" => "AE",

    # 01E3 LATIN SMALL LETTER AE WITH MACRON
    # ->   0061 LATIN SMALL LETTER A
    #    + 0065 LATIN SMALL LETTER E
    "\u{01E3}" => "ae",

    # 01E4 LATIN CAPITAL LETTER G WITH STROKE
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{01E4}" => "G",

    # 01E5 LATIN SMALL LETTER G WITH STROKE
    # ->   0067 LATIN SMALL LETTER G
    "\u{01E5}" => "g",

    # 01E6 LATIN CAPITAL LETTER G WITH CARON
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{01E6}" => "G",

    # 01E7 LATIN SMALL LETTER G WITH CARON
    # ->   0067 LATIN SMALL LETTER G
    "\u{01E7}" => "g",

    # 01E8 LATIN CAPITAL LETTER K WITH CARON
    # ->   004B LATIN CAPITAL LETTER K
    "\u{01E8}" => "K",

    # 01E9 LATIN SMALL LETTER K WITH CARON
    # ->   006B LATIN SMALL LETTER K
    "\u{01E9}" => "k",

    # 01EA LATIN CAPITAL LETTER O WITH OGONEK
    # ->   004F LATIN CAPITAL LETTER O
    "\u{01EA}" => "O",

    # 01EB LATIN SMALL LETTER O WITH OGONEK
    # ->   006F LATIN SMALL LETTER O
    "\u{01EB}" => "o",

    # 01EC LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{01EC}" => "O",

    # 01ED LATIN SMALL LETTER O WITH OGONEK AND MACRON
    # ->   006F LATIN SMALL LETTER O
    "\u{01ED}" => "o",

    # 01EE LATIN CAPITAL LETTER EZH WITH CARON
    # ->   01B7 LATIN CAPITAL LETTER EZH
    "\u{01EE}" => "\u{01B7}",

    # 01EF LATIN SMALL LETTER EZH WITH CARON
    # ->   0292 LATIN SMALL LETTER EZH
    "\u{01EF}" => "\u{0292}",

    # 01F0 LATIN SMALL LETTER J WITH CARON
    # ->   006A LATIN SMALL LETTER J
    "\u{01F0}" => "j",

    # 01F1 LATIN CAPITAL LETTER DZ
    # ->   0044 LATIN CAPITAL LETTER D
    #    + 005A LATIN CAPITAL LETTER Z
    "\u{01F1}" => "DZ",

    # 01F2 LATIN CAPITAL LETTER D WITH SMALL LETTER Z
    # ->   0044 LATIN CAPITAL LETTER D
    #    + 007A LATIN SMALL LETTER Z
    "\u{01F2}" => "Dz",

    # 01F3 LATIN SMALL LETTER DZ
    # ->   0064 LATIN SMALL LETTER D
    #    + 007A LATIN SMALL LETTER Z
    "\u{01F3}" => "dz",

    # 01F4 LATIN CAPITAL LETTER G WITH ACUTE
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{01F4}" => "G",

    # 01F5 LATIN SMALL LETTER G WITH ACUTE
    # ->   0067 LATIN SMALL LETTER G
    "\u{01F5}" => "g",

    # 01F8 LATIN CAPITAL LETTER N WITH GRAVE
    # ->   004E LATIN CAPITAL LETTER N
    "\u{01F8}" => "N",

    # 01F9 LATIN SMALL LETTER N WITH GRAVE
    # ->   006E LATIN SMALL LETTER N
    "\u{01F9}" => "n",

    # 01FA LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{01FA}" => "A",

    # 01FB LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
    # ->   0061 LATIN SMALL LETTER A
    "\u{01FB}" => "a",

    # 01FC LATIN CAPITAL LETTER AE WITH ACUTE
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 0045 LATIN CAPITAL LETTER E
    "\u{01FC}" => "AE",

    # 01FD LATIN SMALL LETTER AE WITH ACUTE
    # ->   0061 LATIN SMALL LETTER A
    #    + 0065 LATIN SMALL LETTER E
    "\u{01FD}" => "ae",

    # 01FE LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{01FE}" => "O",

    # 01FF LATIN SMALL LETTER O WITH STROKE AND ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{01FF}" => "o",

    # 0200 LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0200}" => "A",

    # 0201 LATIN SMALL LETTER A WITH DOUBLE GRAVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{0201}" => "a",

    # 0202 LATIN CAPITAL LETTER A WITH INVERTED BREVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0202}" => "A",

    # 0203 LATIN SMALL LETTER A WITH INVERTED BREVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{0203}" => "a",

    # 0204 LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0204}" => "E",

    # 0205 LATIN SMALL LETTER E WITH DOUBLE GRAVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{0205}" => "e",

    # 0206 LATIN CAPITAL LETTER E WITH INVERTED BREVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0206}" => "E",

    # 0207 LATIN SMALL LETTER E WITH INVERTED BREVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{0207}" => "e",

    # 0208 LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{0208}" => "I",

    # 0209 LATIN SMALL LETTER I WITH DOUBLE GRAVE
    # ->   0069 LATIN SMALL LETTER I
    "\u{0209}" => "i",

    # 020A LATIN CAPITAL LETTER I WITH INVERTED BREVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{020A}" => "I",

    # 020B LATIN SMALL LETTER I WITH INVERTED BREVE
    # ->   0069 LATIN SMALL LETTER I
    "\u{020B}" => "i",

    # 020C LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{020C}" => "O",

    # 020D LATIN SMALL LETTER O WITH DOUBLE GRAVE
    # ->   006F LATIN SMALL LETTER O
    "\u{020D}" => "o",

    # 020E LATIN CAPITAL LETTER O WITH INVERTED BREVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{020E}" => "O",

    # 020F LATIN SMALL LETTER O WITH INVERTED BREVE
    # ->   006F LATIN SMALL LETTER O
    "\u{020F}" => "o",

    # 0210 LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{0210}" => "R",

    # 0211 LATIN SMALL LETTER R WITH DOUBLE GRAVE
    # ->   0072 LATIN SMALL LETTER R
    "\u{0211}" => "r",

    # 0212 LATIN CAPITAL LETTER R WITH INVERTED BREVE
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{0212}" => "R",

    # 0213 LATIN SMALL LETTER R WITH INVERTED BREVE
    # ->   0072 LATIN SMALL LETTER R
    "\u{0213}" => "r",

    # 0214 LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{0214}" => "U",

    # 0215 LATIN SMALL LETTER U WITH DOUBLE GRAVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{0215}" => "u",

    # 0216 LATIN CAPITAL LETTER U WITH INVERTED BREVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{0216}" => "U",

    # 0217 LATIN SMALL LETTER U WITH INVERTED BREVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{0217}" => "u",

    # 0218 LATIN CAPITAL LETTER S WITH COMMA BELOW
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{0218}" => "S",

    # 0219 LATIN SMALL LETTER S WITH COMMA BELOW
    # ->   0073 LATIN SMALL LETTER S
    "\u{0219}" => "s",

    # 021A LATIN CAPITAL LETTER T WITH COMMA BELOW
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{021A}" => "T",

    # 021B LATIN SMALL LETTER T WITH COMMA BELOW
    # ->   0074 LATIN SMALL LETTER T
    "\u{021B}" => "t",

    # 021E LATIN CAPITAL LETTER H WITH CARON
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{021E}" => "H",

    # 021F LATIN SMALL LETTER H WITH CARON
    # ->   0068 LATIN SMALL LETTER H
    "\u{021F}" => "h",

    # 0220 LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
    # ->   004E LATIN CAPITAL LETTER N
    "\u{0220}" => "N",

    # 0221 LATIN SMALL LETTER D WITH CURL
    # ->   0064 LATIN SMALL LETTER D
    "\u{0221}" => "d",

    # 0224 LATIN CAPITAL LETTER Z WITH HOOK
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{0224}" => "Z",

    # 0225 LATIN SMALL LETTER Z WITH HOOK
    # ->   007A LATIN SMALL LETTER Z
    "\u{0225}" => "z",

    # 0226 LATIN CAPITAL LETTER A WITH DOT ABOVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{0226}" => "A",

    # 0227 LATIN SMALL LETTER A WITH DOT ABOVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{0227}" => "a",

    # 0228 LATIN CAPITAL LETTER E WITH CEDILLA
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{0228}" => "E",

    # 0229 LATIN SMALL LETTER E WITH CEDILLA
    # ->   0065 LATIN SMALL LETTER E
    "\u{0229}" => "e",

    # 022A LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{022A}" => "O",

    # 022B LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
    # ->   006F LATIN SMALL LETTER O
    "\u{022B}" => "o",

    # 022C LATIN CAPITAL LETTER O WITH TILDE AND MACRON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{022C}" => "O",

    # 022D LATIN SMALL LETTER O WITH TILDE AND MACRON
    # ->   006F LATIN SMALL LETTER O
    "\u{022D}" => "o",

    # 022E LATIN CAPITAL LETTER O WITH DOT ABOVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{022E}" => "O",

    # 022F LATIN SMALL LETTER O WITH DOT ABOVE
    # ->   006F LATIN SMALL LETTER O
    "\u{022F}" => "o",

    # 0230 LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
    # ->   004F LATIN CAPITAL LETTER O
    "\u{0230}" => "O",

    # 0231 LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
    # ->   006F LATIN SMALL LETTER O
    "\u{0231}" => "o",

    # 0232 LATIN CAPITAL LETTER Y WITH MACRON
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{0232}" => "Y",

    # 0233 LATIN SMALL LETTER Y WITH MACRON
    # ->   0079 LATIN SMALL LETTER Y
    "\u{0233}" => "y",

    # 0234 LATIN SMALL LETTER L WITH CURL
    # ->   006C LATIN SMALL LETTER L
    "\u{0234}" => "l",

    # 0235 LATIN SMALL LETTER N WITH CURL
    # ->   006E LATIN SMALL LETTER N
    "\u{0235}" => "n",

    # 0236 LATIN SMALL LETTER T WITH CURL
    # ->   0074 LATIN SMALL LETTER T
    "\u{0236}" => "t",

    # 0253 LATIN SMALL LETTER B WITH HOOK
    # ->   0062 LATIN SMALL LETTER B
    "\u{0253}" => "b",

    # 0255 LATIN SMALL LETTER C WITH CURL
    # ->   0063 LATIN SMALL LETTER C
    "\u{0255}" => "c",

    # 0256 LATIN SMALL LETTER D WITH TAIL
    # ->   0064 LATIN SMALL LETTER D
    "\u{0256}" => "d",

    # 0257 LATIN SMALL LETTER D WITH HOOK
    # ->   0064 LATIN SMALL LETTER D
    "\u{0257}" => "d",

    # 025A LATIN SMALL LETTER SCHWA WITH HOOK
    # ->   0259 LATIN SMALL LETTER SCHWA
    "\u{025A}" => "\u{0259}",

    # 025D LATIN SMALL LETTER REVERSED OPEN E WITH HOOK
    # ->   025C LATIN SMALL LETTER REVERSED OPEN E
    "\u{025D}" => "\u{025C}",

    # 0260 LATIN SMALL LETTER G WITH HOOK
    # ->   0067 LATIN SMALL LETTER G
    "\u{0260}" => "g",

    # 0266 LATIN SMALL LETTER H WITH HOOK
    # ->   0068 LATIN SMALL LETTER H
    "\u{0266}" => "h",

    # 0268 LATIN SMALL LETTER I WITH STROKE
    # ->   0069 LATIN SMALL LETTER I
    "\u{0268}" => "i",

    # 026B LATIN SMALL LETTER L WITH MIDDLE TILDE
    # ->   006C LATIN SMALL LETTER L
    "\u{026B}" => "l",

    # 026C LATIN SMALL LETTER L WITH BELT
    # ->   006C LATIN SMALL LETTER L
    "\u{026C}" => "l",

    # 026D LATIN SMALL LETTER L WITH RETROFLEX HOOK
    # ->   006C LATIN SMALL LETTER L
    "\u{026D}" => "l",

    # 0270 LATIN SMALL LETTER TURNED M WITH LONG LEG
    # ->   026F LATIN SMALL LETTER TURNED M
    "\u{0270}" => "\u{026F}",

    # 0271 LATIN SMALL LETTER M WITH HOOK
    # ->   006D LATIN SMALL LETTER M
    "\u{0271}" => "m",

    # 0272 LATIN SMALL LETTER N WITH LEFT HOOK
    # ->   006E LATIN SMALL LETTER N
    "\u{0272}" => "n",

    # 0273 LATIN SMALL LETTER N WITH RETROFLEX HOOK
    # ->   006E LATIN SMALL LETTER N
    "\u{0273}" => "n",

    # 027A LATIN SMALL LETTER TURNED R WITH LONG LEG
    # ->   0279 LATIN SMALL LETTER TURNED R
    "\u{027A}" => "\u{0279}",

    # 027B LATIN SMALL LETTER TURNED R WITH HOOK
    # ->   0279 LATIN SMALL LETTER TURNED R
    "\u{027B}" => "\u{0279}",

    # 027C LATIN SMALL LETTER R WITH LONG LEG
    # ->   0072 LATIN SMALL LETTER R
    "\u{027C}" => "r",

    # 027D LATIN SMALL LETTER R WITH TAIL
    # ->   0072 LATIN SMALL LETTER R
    "\u{027D}" => "r",

    # 027E LATIN SMALL LETTER R WITH FISHHOOK
    # ->   0072 LATIN SMALL LETTER R
    "\u{027E}" => "r",

    # 0282 LATIN SMALL LETTER S WITH HOOK
    # ->   0073 LATIN SMALL LETTER S
    "\u{0282}" => "s",

    # 0286 LATIN SMALL LETTER ESH WITH CURL
    # ->   0283 LATIN SMALL LETTER ESH
    "\u{0286}" => "\u{0283}",

    # 0288 LATIN SMALL LETTER T WITH RETROFLEX HOOK
    # ->   0074 LATIN SMALL LETTER T
    "\u{0288}" => "t",

    # 028B LATIN SMALL LETTER V WITH HOOK
    # ->   0076 LATIN SMALL LETTER V
    "\u{028B}" => "v",

    # 0290 LATIN SMALL LETTER Z WITH RETROFLEX HOOK
    # ->   007A LATIN SMALL LETTER Z
    "\u{0290}" => "z",

    # 0291 LATIN SMALL LETTER Z WITH CURL
    # ->   007A LATIN SMALL LETTER Z
    "\u{0291}" => "z",

    # 0293 LATIN SMALL LETTER EZH WITH CURL
    # ->   0292 LATIN SMALL LETTER EZH
    "\u{0293}" => "\u{0292}",

    # 029B LATIN LETTER SMALL CAPITAL G WITH HOOK
    # ->   0262 LATIN LETTER SMALL CAPITAL G
    "\u{029B}" => "\u{0262}",

    # 029D LATIN SMALL LETTER J WITH CROSSED-TAIL
    # ->   006A LATIN SMALL LETTER J
    "\u{029D}" => "j",

    # 02A0 LATIN SMALL LETTER Q WITH HOOK
    # ->   0071 LATIN SMALL LETTER Q
    "\u{02A0}" => "q",

    # 02A1 LATIN LETTER GLOTTAL STOP WITH STROKE
    # ->   0294 LATIN LETTER GLOTTAL STOP
    "\u{02A1}" => "\u{0294}",

    # 02A5 LATIN SMALL LETTER DZ DIGRAPH WITH CURL
    # ->   02A3 LATIN SMALL LETTER DZ DIGRAPH
    "\u{02A5}" => "\u{02A3}",

    # 02AE LATIN SMALL LETTER TURNED H WITH FISHHOOK
    # ->   0265 LATIN SMALL LETTER TURNED H
    "\u{02AE}" => "\u{0265}",

    # 02AF LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    # ->   0265 LATIN SMALL LETTER TURNED H
    "\u{02AF}" => "\u{0265}",

    # 02B0 MODIFIER LETTER SMALL H
    # ->   0068 LATIN SMALL LETTER H
    "\u{02B0}" => "h",

    # 02B1 MODIFIER LETTER SMALL H WITH HOOK
    # ->   0068 LATIN SMALL LETTER H
    "\u{02B1}" => "h",

    # 02B2 MODIFIER LETTER SMALL J
    # ->   006A LATIN SMALL LETTER J
    "\u{02B2}" => "j",

    # 02B3 MODIFIER LETTER SMALL R
    # ->   0072 LATIN SMALL LETTER R
    "\u{02B3}" => "r",

    # 02B4 MODIFIER LETTER SMALL TURNED R
    # ->   0279 LATIN SMALL LETTER TURNED R
    "\u{02B4}" => "\u{0279}",

    # 02B5 MODIFIER LETTER SMALL TURNED R WITH HOOK
    # ->   0279 LATIN SMALL LETTER TURNED R
    "\u{02B5}" => "\u{0279}",

    # 02B6 MODIFIER LETTER SMALL CAPITAL INVERTED R
    # ->   0281 LATIN LETTER SMALL CAPITAL INVERTED R
    "\u{02B6}" => "\u{0281}",

    # 02B7 MODIFIER LETTER SMALL W
    # ->   0077 LATIN SMALL LETTER W
    "\u{02B7}" => "w",

    # 02B8 MODIFIER LETTER SMALL Y
    # ->   0079 LATIN SMALL LETTER Y
    "\u{02B8}" => "y",

    # 02D8 BREVE
    # ->   0020 SPACE
    "\u{02D8}" => " ",

    # 02D9 DOT ABOVE
    # ->   0020 SPACE
    "\u{02D9}" => " ",

    # 02DA RING ABOVE
    # ->   0020 SPACE
    "\u{02DA}" => " ",

    # 02DB OGONEK
    # ->   0020 SPACE
    "\u{02DB}" => " ",

    # 02DC SMALL TILDE
    # ->   0020 SPACE
    "\u{02DC}" => " ",

    # 02DD DOUBLE ACUTE ACCENT
    # ->   0020 SPACE
    "\u{02DD}" => " ",

    # 02E0 MODIFIER LETTER SMALL GAMMA
    # ->   0263 LATIN SMALL LETTER GAMMA
    "\u{02E0}" => "\u{0263}",

    # 02E1 MODIFIER LETTER SMALL L
    # ->   006C LATIN SMALL LETTER L
    "\u{02E1}" => "l",

    # 02E2 MODIFIER LETTER SMALL S
    # ->   0073 LATIN SMALL LETTER S
    "\u{02E2}" => "s",

    # 02E3 MODIFIER LETTER SMALL X
    # ->   0078 LATIN SMALL LETTER X
    "\u{02E3}" => "x",

    # 02E4 MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    # ->   0295 LATIN LETTER PHARYNGEAL VOICED FRICATIVE
    "\u{02E4}" => "\u{0295}",

    # 0374 GREEK NUMERAL SIGN
    # ->   02B9 MODIFIER LETTER PRIME
    "\u{0374}" => "\u{02B9}",

    # 037A GREEK YPOGEGRAMMENI
    # ->   0020 SPACE
    "\u{037A}" => " ",

    # 037E GREEK QUESTION MARK
    # ->   003B SEMICOLON
    "\u{037E}" => ";",

    # 0384 GREEK TONOS
    # ->   0020 SPACE
    "\u{0384}" => " ",

    # 0385 GREEK DIALYTIKA TONOS
    # ->   0020 SPACE
    "\u{0385}" => " ",

    # 0386 GREEK CAPITAL LETTER ALPHA WITH TONOS
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{0386}" => "\u{0391}",

    # 0387 GREEK ANO TELEIA
    # ->   00B7 MIDDLE DOT
    "\u{0387}" => "\u{B7}",

    # 0388 GREEK CAPITAL LETTER EPSILON WITH TONOS
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{0388}" => "\u{0395}",

    # 0389 GREEK CAPITAL LETTER ETA WITH TONOS
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{0389}" => "\u{0397}",

    # 038A GREEK CAPITAL LETTER IOTA WITH TONOS
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{038A}" => "\u{0399}",

    # 038C GREEK CAPITAL LETTER OMICRON WITH TONOS
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{038C}" => "\u{039F}",

    # 038E GREEK CAPITAL LETTER UPSILON WITH TONOS
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{038E}" => "\u{03A5}",

    # 038F GREEK CAPITAL LETTER OMEGA WITH TONOS
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{038F}" => "\u{03A9}",

    # 0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{0390}" => "\u{03B9}",

    # 03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{03AA}" => "\u{0399}",

    # 03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{03AB}" => "\u{03A5}",

    # 03AC GREEK SMALL LETTER ALPHA WITH TONOS
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{03AC}" => "\u{03B1}",

    # 03AD GREEK SMALL LETTER EPSILON WITH TONOS
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{03AD}" => "\u{03B5}",

    # 03AE GREEK SMALL LETTER ETA WITH TONOS
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{03AE}" => "\u{03B7}",

    # 03AF GREEK SMALL LETTER IOTA WITH TONOS
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{03AF}" => "\u{03B9}",

    # 03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{03B0}" => "\u{03C5}",

    # 03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{03CA}" => "\u{03B9}",

    # 03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{03CB}" => "\u{03C5}",

    # 03CC GREEK SMALL LETTER OMICRON WITH TONOS
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{03CC}" => "\u{03BF}",

    # 03CD GREEK SMALL LETTER UPSILON WITH TONOS
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{03CD}" => "\u{03C5}",

    # 03CE GREEK SMALL LETTER OMEGA WITH TONOS
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{03CE}" => "\u{03C9}",

    # 03D0 GREEK BETA SYMBOL
    # ->   03B2 GREEK SMALL LETTER BETA
    "\u{03D0}" => "\u{03B2}",

    # 03D1 GREEK THETA SYMBOL
    # ->   03B8 GREEK SMALL LETTER THETA
    "\u{03D1}" => "\u{03B8}",

    # 03D2 GREEK UPSILON WITH HOOK SYMBOL
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{03D2}" => "\u{03A5}",

    # 03D3 GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{03D3}" => "\u{03A5}",

    # 03D4 GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{03D4}" => "\u{03A5}",

    # 03D5 GREEK PHI SYMBOL
    # ->   03C6 GREEK SMALL LETTER PHI
    "\u{03D5}" => "\u{03C6}",

    # 03D6 GREEK PI SYMBOL
    # ->   03C0 GREEK SMALL LETTER PI
    "\u{03D6}" => "\u{03C0}",

    # 03F0 GREEK KAPPA SYMBOL
    # ->   03BA GREEK SMALL LETTER KAPPA
    "\u{03F0}" => "\u{03BA}",

    # 03F1 GREEK RHO SYMBOL
    # ->   03C1 GREEK SMALL LETTER RHO
    "\u{03F1}" => "\u{03C1}",

    # 03F2 GREEK LUNATE SIGMA SYMBOL
    # ->   03C2 GREEK SMALL LETTER FINAL SIGMA
    "\u{03F2}" => "\u{03C2}",

    # 03F4 GREEK CAPITAL THETA SYMBOL
    # ->   0398 GREEK CAPITAL LETTER THETA
    "\u{03F4}" => "\u{0398}",

    # 03F5 GREEK LUNATE EPSILON SYMBOL
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{03F5}" => "\u{03B5}",

    # 03F9 GREEK CAPITAL LUNATE SIGMA SYMBOL
    # ->   03A3 GREEK CAPITAL LETTER SIGMA
    "\u{03F9}" => "\u{03A3}",

    # 0400 CYRILLIC CAPITAL LETTER IE WITH GRAVE
    # ->   0415 CYRILLIC CAPITAL LETTER IE
    "\u{0400}" => "\u{0415}",

    # 0401 CYRILLIC CAPITAL LETTER IO
    # ->   0415 CYRILLIC CAPITAL LETTER IE
    "\u{0401}" => "\u{0415}",

    # 0403 CYRILLIC CAPITAL LETTER GJE
    # ->   0413 CYRILLIC CAPITAL LETTER GHE
    "\u{0403}" => "\u{0413}",

    # 0407 CYRILLIC CAPITAL LETTER YI
    # ->   0406 CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
    "\u{0407}" => "\u{0406}",

    # 040C CYRILLIC CAPITAL LETTER KJE
    # ->   041A CYRILLIC CAPITAL LETTER KA
    "\u{040C}" => "\u{041A}",

    # 040D CYRILLIC CAPITAL LETTER I WITH GRAVE
    # ->   0418 CYRILLIC CAPITAL LETTER I
    "\u{040D}" => "\u{0418}",

    # 040E CYRILLIC CAPITAL LETTER SHORT U
    # ->   0423 CYRILLIC CAPITAL LETTER U
    "\u{040E}" => "\u{0423}",

    # 0419 CYRILLIC CAPITAL LETTER SHORT I
    # ->   0418 CYRILLIC CAPITAL LETTER I
    "\u{0419}" => "\u{0418}",

    # 0439 CYRILLIC SMALL LETTER SHORT I
    # ->   0438 CYRILLIC SMALL LETTER I
    "\u{0439}" => "\u{0438}",

    # 0450 CYRILLIC SMALL LETTER IE WITH GRAVE
    # ->   0435 CYRILLIC SMALL LETTER IE
    "\u{0450}" => "\u{0435}",

    # 0451 CYRILLIC SMALL LETTER IO
    # ->   0435 CYRILLIC SMALL LETTER IE
    "\u{0451}" => "\u{0435}",

    # 0453 CYRILLIC SMALL LETTER GJE
    # ->   0433 CYRILLIC SMALL LETTER GHE
    "\u{0453}" => "\u{0433}",

    # 0457 CYRILLIC SMALL LETTER YI
    # ->   0456 CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    "\u{0457}" => "\u{0456}",

    # 045C CYRILLIC SMALL LETTER KJE
    # ->   043A CYRILLIC SMALL LETTER KA
    "\u{045C}" => "\u{043A}",

    # 045D CYRILLIC SMALL LETTER I WITH GRAVE
    # ->   0438 CYRILLIC SMALL LETTER I
    "\u{045D}" => "\u{0438}",

    # 045E CYRILLIC SMALL LETTER SHORT U
    # ->   0443 CYRILLIC SMALL LETTER U
    "\u{045E}" => "\u{0443}",

    # 0476 CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    # ->   0474 CYRILLIC CAPITAL LETTER IZHITSA
    "\u{0476}" => "\u{0474}",

    # 0477 CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    # ->   0475 CYRILLIC SMALL LETTER IZHITSA
    "\u{0477}" => "\u{0475}",

    # 047C CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
    # ->   0460 CYRILLIC CAPITAL LETTER OMEGA
    "\u{047C}" => "\u{0460}",

    # 047D CYRILLIC SMALL LETTER OMEGA WITH TITLO
    # ->   0461 CYRILLIC SMALL LETTER OMEGA
    "\u{047D}" => "\u{0461}",

    # 048A CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
    # ->   0418 CYRILLIC CAPITAL LETTER I
    "\u{048A}" => "\u{0418}",

    # 048B CYRILLIC SMALL LETTER SHORT I WITH TAIL
    # ->   0438 CYRILLIC SMALL LETTER I
    "\u{048B}" => "\u{0438}",

    # 048E CYRILLIC CAPITAL LETTER ER WITH TICK
    # ->   0420 CYRILLIC CAPITAL LETTER ER
    "\u{048E}" => "\u{0420}",

    # 048F CYRILLIC SMALL LETTER ER WITH TICK
    # ->   0440 CYRILLIC SMALL LETTER ER
    "\u{048F}" => "\u{0440}",

    # 0490 CYRILLIC CAPITAL LETTER GHE WITH UPTURN
    # ->   0413 CYRILLIC CAPITAL LETTER GHE
    "\u{0490}" => "\u{0413}",

    # 0491 CYRILLIC SMALL LETTER GHE WITH UPTURN
    # ->   0433 CYRILLIC SMALL LETTER GHE
    "\u{0491}" => "\u{0433}",

    # 0492 CYRILLIC CAPITAL LETTER GHE WITH STROKE
    # ->   0413 CYRILLIC CAPITAL LETTER GHE
    "\u{0492}" => "\u{0413}",

    # 0493 CYRILLIC SMALL LETTER GHE WITH STROKE
    # ->   0433 CYRILLIC SMALL LETTER GHE
    "\u{0493}" => "\u{0433}",

    # 0494 CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
    # ->   0413 CYRILLIC CAPITAL LETTER GHE
    "\u{0494}" => "\u{0413}",

    # 0495 CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
    # ->   0433 CYRILLIC SMALL LETTER GHE
    "\u{0495}" => "\u{0433}",

    # 0496 CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
    # ->   0416 CYRILLIC CAPITAL LETTER ZHE
    "\u{0496}" => "\u{0416}",

    # 0497 CYRILLIC SMALL LETTER ZHE WITH DESCENDER
    # ->   0436 CYRILLIC SMALL LETTER ZHE
    "\u{0497}" => "\u{0436}",

    # 0498 CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
    # ->   0417 CYRILLIC CAPITAL LETTER ZE
    "\u{0498}" => "\u{0417}",

    # 0499 CYRILLIC SMALL LETTER ZE WITH DESCENDER
    # ->   0437 CYRILLIC SMALL LETTER ZE
    "\u{0499}" => "\u{0437}",

    # 049A CYRILLIC CAPITAL LETTER KA WITH DESCENDER
    # ->   041A CYRILLIC CAPITAL LETTER KA
    "\u{049A}" => "\u{041A}",

    # 049B CYRILLIC SMALL LETTER KA WITH DESCENDER
    # ->   043A CYRILLIC SMALL LETTER KA
    "\u{049B}" => "\u{043A}",

    # 049C CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
    # ->   041A CYRILLIC CAPITAL LETTER KA
    "\u{049C}" => "\u{041A}",

    # 049D CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
    # ->   043A CYRILLIC SMALL LETTER KA
    "\u{049D}" => "\u{043A}",

    # 049E CYRILLIC CAPITAL LETTER KA WITH STROKE
    # ->   041A CYRILLIC CAPITAL LETTER KA
    "\u{049E}" => "\u{041A}",

    # 049F CYRILLIC SMALL LETTER KA WITH STROKE
    # ->   043A CYRILLIC SMALL LETTER KA
    "\u{049F}" => "\u{043A}",

    # 04A2 CYRILLIC CAPITAL LETTER EN WITH DESCENDER
    # ->   041D CYRILLIC CAPITAL LETTER EN
    "\u{04A2}" => "\u{041D}",

    # 04A3 CYRILLIC SMALL LETTER EN WITH DESCENDER
    # ->   043D CYRILLIC SMALL LETTER EN
    "\u{04A3}" => "\u{043D}",

    # 04A6 CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
    # ->   041F CYRILLIC CAPITAL LETTER PE
    "\u{04A6}" => "\u{041F}",

    # 04A7 CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
    # ->   043F CYRILLIC SMALL LETTER PE
    "\u{04A7}" => "\u{043F}",

    # 04AA CYRILLIC CAPITAL LETTER ES WITH DESCENDER
    # ->   0421 CYRILLIC CAPITAL LETTER ES
    "\u{04AA}" => "\u{0421}",

    # 04AB CYRILLIC SMALL LETTER ES WITH DESCENDER
    # ->   0441 CYRILLIC SMALL LETTER ES
    "\u{04AB}" => "\u{0441}",

    # 04AC CYRILLIC CAPITAL LETTER TE WITH DESCENDER
    # ->   0422 CYRILLIC CAPITAL LETTER TE
    "\u{04AC}" => "\u{0422}",

    # 04AD CYRILLIC SMALL LETTER TE WITH DESCENDER
    # ->   0442 CYRILLIC SMALL LETTER TE
    "\u{04AD}" => "\u{0442}",

    # 04B0 CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
    # ->   04AE CYRILLIC CAPITAL LETTER STRAIGHT U
    "\u{04B0}" => "\u{04AE}",

    # 04B1 CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
    # ->   04AF CYRILLIC SMALL LETTER STRAIGHT U
    "\u{04B1}" => "\u{04AF}",

    # 04B2 CYRILLIC CAPITAL LETTER HA WITH DESCENDER
    # ->   0425 CYRILLIC CAPITAL LETTER HA
    "\u{04B2}" => "\u{0425}",

    # 04B3 CYRILLIC SMALL LETTER HA WITH DESCENDER
    # ->   0445 CYRILLIC SMALL LETTER HA
    "\u{04B3}" => "\u{0445}",

    # 04B6 CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
    # ->   0427 CYRILLIC CAPITAL LETTER CHE
    "\u{04B6}" => "\u{0427}",

    # 04B7 CYRILLIC SMALL LETTER CHE WITH DESCENDER
    # ->   0447 CYRILLIC SMALL LETTER CHE
    "\u{04B7}" => "\u{0447}",

    # 04B8 CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
    # ->   0427 CYRILLIC CAPITAL LETTER CHE
    "\u{04B8}" => "\u{0427}",

    # 04B9 CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
    # ->   0447 CYRILLIC SMALL LETTER CHE
    "\u{04B9}" => "\u{0447}",

    # 04BE CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
    # ->   04BC CYRILLIC CAPITAL LETTER ABKHASIAN CHE
    "\u{04BE}" => "\u{04BC}",

    # 04BF CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
    # ->   04BD CYRILLIC SMALL LETTER ABKHASIAN CHE
    "\u{04BF}" => "\u{04BD}",

    # 04C1 CYRILLIC CAPITAL LETTER ZHE WITH BREVE
    # ->   0416 CYRILLIC CAPITAL LETTER ZHE
    "\u{04C1}" => "\u{0416}",

    # 04C2 CYRILLIC SMALL LETTER ZHE WITH BREVE
    # ->   0436 CYRILLIC SMALL LETTER ZHE
    "\u{04C2}" => "\u{0436}",

    # 04C3 CYRILLIC CAPITAL LETTER KA WITH HOOK
    # ->   041A CYRILLIC CAPITAL LETTER KA
    "\u{04C3}" => "\u{041A}",

    # 04C4 CYRILLIC SMALL LETTER KA WITH HOOK
    # ->   043A CYRILLIC SMALL LETTER KA
    "\u{04C4}" => "\u{043A}",

    # 04C5 CYRILLIC CAPITAL LETTER EL WITH TAIL
    # ->   041B CYRILLIC CAPITAL LETTER EL
    "\u{04C5}" => "\u{041B}",

    # 04C6 CYRILLIC SMALL LETTER EL WITH TAIL
    # ->   043B CYRILLIC SMALL LETTER EL
    "\u{04C6}" => "\u{043B}",

    # 04C7 CYRILLIC CAPITAL LETTER EN WITH HOOK
    # ->   041D CYRILLIC CAPITAL LETTER EN
    "\u{04C7}" => "\u{041D}",

    # 04C8 CYRILLIC SMALL LETTER EN WITH HOOK
    # ->   043D CYRILLIC SMALL LETTER EN
    "\u{04C8}" => "\u{043D}",

    # 04C9 CYRILLIC CAPITAL LETTER EN WITH TAIL
    # ->   041D CYRILLIC CAPITAL LETTER EN
    "\u{04C9}" => "\u{041D}",

    # 04CA CYRILLIC SMALL LETTER EN WITH TAIL
    # ->   043D CYRILLIC SMALL LETTER EN
    "\u{04CA}" => "\u{043D}",

    # 04CD CYRILLIC CAPITAL LETTER EM WITH TAIL
    # ->   041C CYRILLIC CAPITAL LETTER EM
    "\u{04CD}" => "\u{041C}",

    # 04CE CYRILLIC SMALL LETTER EM WITH TAIL
    # ->   043C CYRILLIC SMALL LETTER EM
    "\u{04CE}" => "\u{043C}",

    # 04D0 CYRILLIC CAPITAL LETTER A WITH BREVE
    # ->   0410 CYRILLIC CAPITAL LETTER A
    "\u{04D0}" => "\u{0410}",

    # 04D1 CYRILLIC SMALL LETTER A WITH BREVE
    # ->   0430 CYRILLIC SMALL LETTER A
    "\u{04D1}" => "\u{0430}",

    # 04D2 CYRILLIC CAPITAL LETTER A WITH DIAERESIS
    # ->   0410 CYRILLIC CAPITAL LETTER A
    "\u{04D2}" => "\u{0410}",

    # 04D3 CYRILLIC SMALL LETTER A WITH DIAERESIS
    # ->   0430 CYRILLIC SMALL LETTER A
    "\u{04D3}" => "\u{0430}",

    # 04D6 CYRILLIC CAPITAL LETTER IE WITH BREVE
    # ->   0415 CYRILLIC CAPITAL LETTER IE
    "\u{04D6}" => "\u{0415}",

    # 04D7 CYRILLIC SMALL LETTER IE WITH BREVE
    # ->   0435 CYRILLIC SMALL LETTER IE
    "\u{04D7}" => "\u{0435}",

    # 04DA CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
    # ->   04D8 CYRILLIC CAPITAL LETTER SCHWA
    "\u{04DA}" => "\u{04D8}",

    # 04DB CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
    # ->   04D9 CYRILLIC SMALL LETTER SCHWA
    "\u{04DB}" => "\u{04D9}",

    # 04DC CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
    # ->   0416 CYRILLIC CAPITAL LETTER ZHE
    "\u{04DC}" => "\u{0416}",

    # 04DD CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
    # ->   0436 CYRILLIC SMALL LETTER ZHE
    "\u{04DD}" => "\u{0436}",

    # 04DE CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
    # ->   0417 CYRILLIC CAPITAL LETTER ZE
    "\u{04DE}" => "\u{0417}",

    # 04DF CYRILLIC SMALL LETTER ZE WITH DIAERESIS
    # ->   0437 CYRILLIC SMALL LETTER ZE
    "\u{04DF}" => "\u{0437}",

    # 04E2 CYRILLIC CAPITAL LETTER I WITH MACRON
    # ->   0418 CYRILLIC CAPITAL LETTER I
    "\u{04E2}" => "\u{0418}",

    # 04E3 CYRILLIC SMALL LETTER I WITH MACRON
    # ->   0438 CYRILLIC SMALL LETTER I
    "\u{04E3}" => "\u{0438}",

    # 04E4 CYRILLIC CAPITAL LETTER I WITH DIAERESIS
    # ->   0418 CYRILLIC CAPITAL LETTER I
    "\u{04E4}" => "\u{0418}",

    # 04E5 CYRILLIC SMALL LETTER I WITH DIAERESIS
    # ->   0438 CYRILLIC SMALL LETTER I
    "\u{04E5}" => "\u{0438}",

    # 04E6 CYRILLIC CAPITAL LETTER O WITH DIAERESIS
    # ->   041E CYRILLIC CAPITAL LETTER O
    "\u{04E6}" => "\u{041E}",

    # 04E7 CYRILLIC SMALL LETTER O WITH DIAERESIS
    # ->   043E CYRILLIC SMALL LETTER O
    "\u{04E7}" => "\u{043E}",

    # 04EA CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
    # ->   04E8 CYRILLIC CAPITAL LETTER BARRED O
    "\u{04EA}" => "\u{04E8}",

    # 04EB CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
    # ->   04E9 CYRILLIC SMALL LETTER BARRED O
    "\u{04EB}" => "\u{04E9}",

    # 04EC CYRILLIC CAPITAL LETTER E WITH DIAERESIS
    # ->   042D CYRILLIC CAPITAL LETTER E
    "\u{04EC}" => "\u{042D}",

    # 04ED CYRILLIC SMALL LETTER E WITH DIAERESIS
    # ->   044D CYRILLIC SMALL LETTER E
    "\u{04ED}" => "\u{044D}",

    # 04EE CYRILLIC CAPITAL LETTER U WITH MACRON
    # ->   0423 CYRILLIC CAPITAL LETTER U
    "\u{04EE}" => "\u{0423}",

    # 04EF CYRILLIC SMALL LETTER U WITH MACRON
    # ->   0443 CYRILLIC SMALL LETTER U
    "\u{04EF}" => "\u{0443}",

    # 04F0 CYRILLIC CAPITAL LETTER U WITH DIAERESIS
    # ->   0423 CYRILLIC CAPITAL LETTER U
    "\u{04F0}" => "\u{0423}",

    # 04F1 CYRILLIC SMALL LETTER U WITH DIAERESIS
    # ->   0443 CYRILLIC SMALL LETTER U
    "\u{04F1}" => "\u{0443}",

    # 04F2 CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
    # ->   0423 CYRILLIC CAPITAL LETTER U
    "\u{04F2}" => "\u{0423}",

    # 04F3 CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
    # ->   0443 CYRILLIC SMALL LETTER U
    "\u{04F3}" => "\u{0443}",

    # 04F4 CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
    # ->   0427 CYRILLIC CAPITAL LETTER CHE
    "\u{04F4}" => "\u{0427}",

    # 04F5 CYRILLIC SMALL LETTER CHE WITH DIAERESIS
    # ->   0447 CYRILLIC SMALL LETTER CHE
    "\u{04F5}" => "\u{0447}",

    # 04F8 CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
    # ->   042B CYRILLIC CAPITAL LETTER YERU
    "\u{04F8}" => "\u{042B}",

    # 04F9 CYRILLIC SMALL LETTER YERU WITH DIAERESIS
    # ->   044B CYRILLIC SMALL LETTER YERU
    "\u{04F9}" => "\u{044B}",

    # 0587 ARMENIAN SMALL LIGATURE ECH YIWN
    # ->   0565 ARMENIAN SMALL LETTER ECH
    #    + 0582 ARMENIAN SMALL LETTER YIWN
    "\u{0587}" => "\u{0565}\u{0582}",

    # 0622 ARABIC LETTER ALEF WITH MADDA ABOVE
    # ->   0627 ARABIC LETTER ALEF
    "\u{0622}" => "\u{0627}",

    # 0623 ARABIC LETTER ALEF WITH HAMZA ABOVE
    # ->   0627 ARABIC LETTER ALEF
    "\u{0623}" => "\u{0627}",

    # 0624 ARABIC LETTER WAW WITH HAMZA ABOVE
    # ->   0648 ARABIC LETTER WAW
    "\u{0624}" => "\u{0648}",

    # 0625 ARABIC LETTER ALEF WITH HAMZA BELOW
    # ->   0627 ARABIC LETTER ALEF
    "\u{0625}" => "\u{0627}",

    # 0626 ARABIC LETTER YEH WITH HAMZA ABOVE
    # ->   064A ARABIC LETTER YEH
    "\u{0626}" => "\u{064A}",

    # 0672 ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE
    # ->   0627 ARABIC LETTER ALEF
    "\u{0672}" => "\u{0627}",

    # 0673 ARABIC LETTER ALEF WITH WAVY HAMZA BELOW
    # ->   0627 ARABIC LETTER ALEF
    "\u{0673}" => "\u{0627}",

    # 0675 ARABIC LETTER HIGH HAMZA ALEF
    # ->   0627 ARABIC LETTER ALEF
    #    + 0674 ARABIC LETTER HIGH HAMZA
    "\u{0675}" => "\u{0627}\u{0674}",

    # 0676 ARABIC LETTER HIGH HAMZA WAW
    # ->   0648 ARABIC LETTER WAW
    #    + 0674 ARABIC LETTER HIGH HAMZA
    "\u{0676}" => "\u{0648}\u{0674}",

    # 0677 ARABIC LETTER U WITH HAMZA ABOVE
    # ->   06C7 ARABIC LETTER U
    #    + 0674 ARABIC LETTER HIGH HAMZA
    "\u{0677}" => "\u{06C7}\u{0674}",

    # 0678 ARABIC LETTER HIGH HAMZA YEH
    # ->   064A ARABIC LETTER YEH
    #    + 0674 ARABIC LETTER HIGH HAMZA
    "\u{0678}" => "\u{064A}\u{0674}",

    # 067C ARABIC LETTER TEH WITH RING
    # ->   062A ARABIC LETTER TEH
    "\u{067C}" => "\u{062A}",

    # 067D ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS
    # ->   062A ARABIC LETTER TEH
    "\u{067D}" => "\u{062A}",

    # 0681 ARABIC LETTER HAH WITH HAMZA ABOVE
    # ->   062D ARABIC LETTER HAH
    "\u{0681}" => "\u{062D}",

    # 0682 ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE
    # ->   062D ARABIC LETTER HAH
    "\u{0682}" => "\u{062D}",

    # 0685 ARABIC LETTER HAH WITH THREE DOTS ABOVE
    # ->   062D ARABIC LETTER HAH
    "\u{0685}" => "\u{062D}",

    # 0689 ARABIC LETTER DAL WITH RING
    # ->   062F ARABIC LETTER DAL
    "\u{0689}" => "\u{062F}",

    # 068A ARABIC LETTER DAL WITH DOT BELOW
    # ->   062F ARABIC LETTER DAL
    "\u{068A}" => "\u{062F}",

    # 068B ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH
    # ->   062F ARABIC LETTER DAL
    "\u{068B}" => "\u{062F}",

    # 068F ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS
    # ->   062F ARABIC LETTER DAL
    "\u{068F}" => "\u{062F}",

    # 0690 ARABIC LETTER DAL WITH FOUR DOTS ABOVE
    # ->   062F ARABIC LETTER DAL
    "\u{0690}" => "\u{062F}",

    # 0692 ARABIC LETTER REH WITH SMALL V
    # ->   0631 ARABIC LETTER REH
    "\u{0692}" => "\u{0631}",

    # 0693 ARABIC LETTER REH WITH RING
    # ->   0631 ARABIC LETTER REH
    "\u{0693}" => "\u{0631}",

    # 0694 ARABIC LETTER REH WITH DOT BELOW
    # ->   0631 ARABIC LETTER REH
    "\u{0694}" => "\u{0631}",

    # 0695 ARABIC LETTER REH WITH SMALL V BELOW
    # ->   0631 ARABIC LETTER REH
    "\u{0695}" => "\u{0631}",

    # 0696 ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE
    # ->   0631 ARABIC LETTER REH
    "\u{0696}" => "\u{0631}",

    # 0697 ARABIC LETTER REH WITH TWO DOTS ABOVE
    # ->   0631 ARABIC LETTER REH
    "\u{0697}" => "\u{0631}",

    # 0699 ARABIC LETTER REH WITH FOUR DOTS ABOVE
    # ->   0631 ARABIC LETTER REH
    "\u{0699}" => "\u{0631}",

    # 069A ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE
    # ->   0633 ARABIC LETTER SEEN
    "\u{069A}" => "\u{0633}",

    # 069B ARABIC LETTER SEEN WITH THREE DOTS BELOW
    # ->   0633 ARABIC LETTER SEEN
    "\u{069B}" => "\u{0633}",

    # 069C ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE
    # ->   0633 ARABIC LETTER SEEN
    "\u{069C}" => "\u{0633}",

    # 069D ARABIC LETTER SAD WITH TWO DOTS BELOW
    # ->   0635 ARABIC LETTER SAD
    "\u{069D}" => "\u{0635}",

    # 069E ARABIC LETTER SAD WITH THREE DOTS ABOVE
    # ->   0635 ARABIC LETTER SAD
    "\u{069E}" => "\u{0635}",

    # 069F ARABIC LETTER TAH WITH THREE DOTS ABOVE
    # ->   0637 ARABIC LETTER TAH
    "\u{069F}" => "\u{0637}",

    # 06A0 ARABIC LETTER AIN WITH THREE DOTS ABOVE
    # ->   0639 ARABIC LETTER AIN
    "\u{06A0}" => "\u{0639}",

    # 06A2 ARABIC LETTER FEH WITH DOT MOVED BELOW
    # ->   0641 ARABIC LETTER FEH
    "\u{06A2}" => "\u{0641}",

    # 06A3 ARABIC LETTER FEH WITH DOT BELOW
    # ->   0641 ARABIC LETTER FEH
    "\u{06A3}" => "\u{0641}",

    # 06A5 ARABIC LETTER FEH WITH THREE DOTS BELOW
    # ->   0641 ARABIC LETTER FEH
    "\u{06A5}" => "\u{0641}",

    # 06A7 ARABIC LETTER QAF WITH DOT ABOVE
    # ->   0642 ARABIC LETTER QAF
    "\u{06A7}" => "\u{0642}",

    # 06A8 ARABIC LETTER QAF WITH THREE DOTS ABOVE
    # ->   0642 ARABIC LETTER QAF
    "\u{06A8}" => "\u{0642}",

    # 06AB ARABIC LETTER KAF WITH RING
    # ->   0643 ARABIC LETTER KAF
    "\u{06AB}" => "\u{0643}",

    # 06AC ARABIC LETTER KAF WITH DOT ABOVE
    # ->   0643 ARABIC LETTER KAF
    "\u{06AC}" => "\u{0643}",

    # 06AE ARABIC LETTER KAF WITH THREE DOTS BELOW
    # ->   0643 ARABIC LETTER KAF
    "\u{06AE}" => "\u{0643}",

    # 06B0 ARABIC LETTER GAF WITH RING
    # ->   06AF ARABIC LETTER GAF
    "\u{06B0}" => "\u{06AF}",

    # 06B2 ARABIC LETTER GAF WITH TWO DOTS BELOW
    # ->   06AF ARABIC LETTER GAF
    "\u{06B2}" => "\u{06AF}",

    # 06B4 ARABIC LETTER GAF WITH THREE DOTS ABOVE
    # ->   06AF ARABIC LETTER GAF
    "\u{06B4}" => "\u{06AF}",

    # 06B5 ARABIC LETTER LAM WITH SMALL V
    # ->   0644 ARABIC LETTER LAM
    "\u{06B5}" => "\u{0644}",

    # 06B6 ARABIC LETTER LAM WITH DOT ABOVE
    # ->   0644 ARABIC LETTER LAM
    "\u{06B6}" => "\u{0644}",

    # 06B7 ARABIC LETTER LAM WITH THREE DOTS ABOVE
    # ->   0644 ARABIC LETTER LAM
    "\u{06B7}" => "\u{0644}",

    # 06B8 ARABIC LETTER LAM WITH THREE DOTS BELOW
    # ->   0644 ARABIC LETTER LAM
    "\u{06B8}" => "\u{0644}",

    # 06B9 ARABIC LETTER NOON WITH DOT BELOW
    # ->   0646 ARABIC LETTER NOON
    "\u{06B9}" => "\u{0646}",

    # 06BC ARABIC LETTER NOON WITH RING
    # ->   0646 ARABIC LETTER NOON
    "\u{06BC}" => "\u{0646}",

    # 06BD ARABIC LETTER NOON WITH THREE DOTS ABOVE
    # ->   0646 ARABIC LETTER NOON
    "\u{06BD}" => "\u{0646}",

    # 06BF ARABIC LETTER TCHEH WITH DOT ABOVE
    # ->   0686 ARABIC LETTER TCHEH
    "\u{06BF}" => "\u{0686}",

    # 06C0 ARABIC LETTER HEH WITH YEH ABOVE
    # ->   06D5 ARABIC LETTER AE
    "\u{06C0}" => "\u{06D5}",

    # 06C2 ARABIC LETTER HEH GOAL WITH HAMZA ABOVE
    # ->   06C1 ARABIC LETTER HEH GOAL
    "\u{06C2}" => "\u{06C1}",

    # 06C4 ARABIC LETTER WAW WITH RING
    # ->   0648 ARABIC LETTER WAW
    "\u{06C4}" => "\u{0648}",

    # 06CA ARABIC LETTER WAW WITH TWO DOTS ABOVE
    # ->   0648 ARABIC LETTER WAW
    "\u{06CA}" => "\u{0648}",

    # 06CD ARABIC LETTER YEH WITH TAIL
    # ->   064A ARABIC LETTER YEH
    "\u{06CD}" => "\u{064A}",

    # 06CE ARABIC LETTER YEH WITH SMALL V
    # ->   064A ARABIC LETTER YEH
    "\u{06CE}" => "\u{064A}",

    # 06CF ARABIC LETTER WAW WITH DOT ABOVE
    # ->   0648 ARABIC LETTER WAW
    "\u{06CF}" => "\u{0648}",

    # 06D1 ARABIC LETTER YEH WITH THREE DOTS BELOW
    # ->   064A ARABIC LETTER YEH
    "\u{06D1}" => "\u{064A}",

    # 06D3 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    # ->   06D2 ARABIC LETTER YEH BARREE
    "\u{06D3}" => "\u{06D2}",

    # 06EE ARABIC LETTER DAL WITH INVERTED V
    # ->   062F ARABIC LETTER DAL
    "\u{06EE}" => "\u{062F}",

    # 06EF ARABIC LETTER REH WITH INVERTED V
    # ->   0631 ARABIC LETTER REH
    "\u{06EF}" => "\u{0631}",

    # 06FA ARABIC LETTER SHEEN WITH DOT BELOW
    # ->   0634 ARABIC LETTER SHEEN
    "\u{06FA}" => "\u{0634}",

    # 06FB ARABIC LETTER DAD WITH DOT BELOW
    # ->   0636 ARABIC LETTER DAD
    "\u{06FB}" => "\u{0636}",

    # 06FC ARABIC LETTER GHAIN WITH DOT BELOW
    # ->   063A ARABIC LETTER GHAIN
    "\u{06FC}" => "\u{063A}",

    # 06FF ARABIC LETTER HEH WITH INVERTED V
    # ->   0647 ARABIC LETTER HEH
    "\u{06FF}" => "\u{0647}",

    # 0929 DEVANAGARI LETTER NNNA
    # ->   0928 DEVANAGARI LETTER NA
    "\u{0929}" => "\u{0928}",

    # 0931 DEVANAGARI LETTER RRA
    # ->   0930 DEVANAGARI LETTER RA
    "\u{0931}" => "\u{0930}",

    # 0934 DEVANAGARI LETTER LLLA
    # ->   0933 DEVANAGARI LETTER LLA
    "\u{0934}" => "\u{0933}",

    # 0958 DEVANAGARI LETTER QA
    # ->   0915 DEVANAGARI LETTER KA
    "\u{0958}" => "\u{0915}",

    # 0959 DEVANAGARI LETTER KHHA
    # ->   0916 DEVANAGARI LETTER KHA
    "\u{0959}" => "\u{0916}",

    # 095A DEVANAGARI LETTER GHHA
    # ->   0917 DEVANAGARI LETTER GA
    "\u{095A}" => "\u{0917}",

    # 095B DEVANAGARI LETTER ZA
    # ->   091C DEVANAGARI LETTER JA
    "\u{095B}" => "\u{091C}",

    # 095C DEVANAGARI LETTER DDDHA
    # ->   0921 DEVANAGARI LETTER DDA
    "\u{095C}" => "\u{0921}",

    # 095D DEVANAGARI LETTER RHA
    # ->   0922 DEVANAGARI LETTER DDHA
    "\u{095D}" => "\u{0922}",

    # 095E DEVANAGARI LETTER FA
    # ->   092B DEVANAGARI LETTER PHA
    "\u{095E}" => "\u{092B}",

    # 095F DEVANAGARI LETTER YYA
    # ->   092F DEVANAGARI LETTER YA
    "\u{095F}" => "\u{092F}",

    # 09DC BENGALI LETTER RRA
    # ->   09A1 BENGALI LETTER DDA
    "\u{09DC}" => "\u{09A1}",

    # 09DD BENGALI LETTER RHA
    # ->   09A2 BENGALI LETTER DDHA
    "\u{09DD}" => "\u{09A2}",

    # 09DF BENGALI LETTER YYA
    # ->   09AF BENGALI LETTER YA
    "\u{09DF}" => "\u{09AF}",

    # 09F0 BENGALI LETTER RA WITH MIDDLE DIAGONAL
    # ->   09B0 BENGALI LETTER RA
    "\u{09F0}" => "\u{09B0}",

    # 09F1 BENGALI LETTER RA WITH LOWER DIAGONAL
    # ->   09B0 BENGALI LETTER RA
    "\u{09F1}" => "\u{09B0}",

    # 0A33 GURMUKHI LETTER LLA
    # ->   0A32 GURMUKHI LETTER LA
    "\u{0A33}" => "\u{0A32}",

    # 0A36 GURMUKHI LETTER SHA
    # ->   0A38 GURMUKHI LETTER SA
    "\u{0A36}" => "\u{0A38}",

    # 0A59 GURMUKHI LETTER KHHA
    # ->   0A16 GURMUKHI LETTER KHA
    "\u{0A59}" => "\u{0A16}",

    # 0A5A GURMUKHI LETTER GHHA
    # ->   0A17 GURMUKHI LETTER GA
    "\u{0A5A}" => "\u{0A17}",

    # 0A5B GURMUKHI LETTER ZA
    # ->   0A1C GURMUKHI LETTER JA
    "\u{0A5B}" => "\u{0A1C}",

    # 0A5E GURMUKHI LETTER FA
    # ->   0A2B GURMUKHI LETTER PHA
    "\u{0A5E}" => "\u{0A2B}",

    # 0B5C ORIYA LETTER RRA
    # ->   0B21 ORIYA LETTER DDA
    "\u{0B5C}" => "\u{0B21}",

    # 0B5D ORIYA LETTER RHA
    # ->   0B22 ORIYA LETTER DDHA
    "\u{0B5D}" => "\u{0B22}",

    # 0B94 TAMIL LETTER AU
    # ->   0B92 TAMIL LETTER O
    "\u{0B94}" => "\u{0B92}",

    # 0E33 THAI CHARACTER SARA AM
    # ->   0E32 THAI CHARACTER SARA AA
    "\u{0E33}" => "\u{0E32}",

    # 0EB3 LAO VOWEL SIGN AM
    # ->   0EB2 LAO VOWEL SIGN AA
    "\u{0EB3}" => "\u{0EB2}",

    # 0EDC LAO HO NO
    # ->   0EAB LAO LETTER HO SUNG
    #    + 0E99 LAO LETTER NO
    "\u{0EDC}" => "\u{0EAB}\u{0E99}",

    # 0EDD LAO HO MO
    # ->   0EAB LAO LETTER HO SUNG
    #    + 0EA1 LAO LETTER MO
    "\u{0EDD}" => "\u{0EAB}\u{0EA1}",

    # 0F0C TIBETAN MARK DELIMITER TSHEG BSTAR
    # ->   0F0B TIBETAN MARK INTERSYLLABIC TSHEG
    "\u{0F0C}" => "\u{0F0B}",

    # 0F43 TIBETAN LETTER GHA
    # ->   0F42 TIBETAN LETTER GA
    "\u{0F43}" => "\u{0F42}",

    # 0F4D TIBETAN LETTER DDHA
    # ->   0F4C TIBETAN LETTER DDA
    "\u{0F4D}" => "\u{0F4C}",

    # 0F52 TIBETAN LETTER DHA
    # ->   0F51 TIBETAN LETTER DA
    "\u{0F52}" => "\u{0F51}",

    # 0F57 TIBETAN LETTER BHA
    # ->   0F56 TIBETAN LETTER BA
    "\u{0F57}" => "\u{0F56}",

    # 0F5C TIBETAN LETTER DZHA
    # ->   0F5B TIBETAN LETTER DZA
    "\u{0F5C}" => "\u{0F5B}",

    # 0F69 TIBETAN LETTER KSSA
    # ->   0F40 TIBETAN LETTER KA
    "\u{0F69}" => "\u{0F40}",

    # 1026 MYANMAR LETTER UU
    # ->   1025 MYANMAR LETTER U
    "\u{1026}" => "\u{1025}",

    # 1D0C LATIN LETTER SMALL CAPITAL L WITH STROKE
    # ->   029F LATIN LETTER SMALL CAPITAL L
    "\u{1D0C}" => "\u{029F}",

    # 1D13 LATIN SMALL LETTER SIDEWAYS O WITH STROKE
    # ->   1D11 LATIN SMALL LETTER SIDEWAYS O
    "\u{1D13}" => "\u{1D11}",

    # 1D2C MODIFIER LETTER CAPITAL A
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1D2C}" => "A",

    # 1D2D MODIFIER LETTER CAPITAL AE
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 0045 LATIN CAPITAL LETTER E
    "\u{1D2D}" => "AE",

    # 1D2E MODIFIER LETTER CAPITAL B
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{1D2E}" => "B",

    # 1D30 MODIFIER LETTER CAPITAL D
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1D30}" => "D",

    # 1D31 MODIFIER LETTER CAPITAL E
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1D31}" => "E",

    # 1D32 MODIFIER LETTER CAPITAL REVERSED E
    # ->   018E LATIN CAPITAL LETTER REVERSED E
    "\u{1D32}" => "\u{018E}",

    # 1D33 MODIFIER LETTER CAPITAL G
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{1D33}" => "G",

    # 1D34 MODIFIER LETTER CAPITAL H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1D34}" => "H",

    # 1D35 MODIFIER LETTER CAPITAL I
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{1D35}" => "I",

    # 1D36 MODIFIER LETTER CAPITAL J
    # ->   004A LATIN CAPITAL LETTER J
    "\u{1D36}" => "J",

    # 1D37 MODIFIER LETTER CAPITAL K
    # ->   004B LATIN CAPITAL LETTER K
    "\u{1D37}" => "K",

    # 1D38 MODIFIER LETTER CAPITAL L
    # ->   004C LATIN CAPITAL LETTER L
    "\u{1D38}" => "L",

    # 1D39 MODIFIER LETTER CAPITAL M
    # ->   004D LATIN CAPITAL LETTER M
    "\u{1D39}" => "M",

    # 1D3A MODIFIER LETTER CAPITAL N
    # ->   004E LATIN CAPITAL LETTER N
    "\u{1D3A}" => "N",

    # 1D3C MODIFIER LETTER CAPITAL O
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1D3C}" => "O",

    # 1D3D MODIFIER LETTER CAPITAL OU
    # ->   0222 LATIN CAPITAL LETTER OU
    "\u{1D3D}" => "\u{0222}",

    # 1D3E MODIFIER LETTER CAPITAL P
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{1D3E}" => "P",

    # 1D3F MODIFIER LETTER CAPITAL R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{1D3F}" => "R",

    # 1D40 MODIFIER LETTER CAPITAL T
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{1D40}" => "T",

    # 1D41 MODIFIER LETTER CAPITAL U
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1D41}" => "U",

    # 1D42 MODIFIER LETTER CAPITAL W
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1D42}" => "W",

    # 1D43 MODIFIER LETTER SMALL A
    # ->   0061 LATIN SMALL LETTER A
    "\u{1D43}" => "a",

    # 1D44 MODIFIER LETTER SMALL TURNED A
    # ->   0250 LATIN SMALL LETTER TURNED A
    "\u{1D44}" => "\u{0250}",

    # 1D45 MODIFIER LETTER SMALL ALPHA
    # ->   0251 LATIN SMALL LETTER ALPHA
    "\u{1D45}" => "\u{0251}",

    # 1D46 MODIFIER LETTER SMALL TURNED AE
    # ->   1D02 LATIN SMALL LETTER TURNED AE
    "\u{1D46}" => "\u{1D02}",

    # 1D47 MODIFIER LETTER SMALL B
    # ->   0062 LATIN SMALL LETTER B
    "\u{1D47}" => "b",

    # 1D48 MODIFIER LETTER SMALL D
    # ->   0064 LATIN SMALL LETTER D
    "\u{1D48}" => "d",

    # 1D49 MODIFIER LETTER SMALL E
    # ->   0065 LATIN SMALL LETTER E
    "\u{1D49}" => "e",

    # 1D4A MODIFIER LETTER SMALL SCHWA
    # ->   0259 LATIN SMALL LETTER SCHWA
    "\u{1D4A}" => "\u{0259}",

    # 1D4B MODIFIER LETTER SMALL OPEN E
    # ->   025B LATIN SMALL LETTER OPEN E
    "\u{1D4B}" => "\u{025B}",

    # 1D4C MODIFIER LETTER SMALL TURNED OPEN E
    # ->   025C LATIN SMALL LETTER REVERSED OPEN E
    "\u{1D4C}" => "\u{025C}",

    # 1D4D MODIFIER LETTER SMALL G
    # ->   0067 LATIN SMALL LETTER G
    "\u{1D4D}" => "g",

    # 1D4F MODIFIER LETTER SMALL K
    # ->   006B LATIN SMALL LETTER K
    "\u{1D4F}" => "k",

    # 1D50 MODIFIER LETTER SMALL M
    # ->   006D LATIN SMALL LETTER M
    "\u{1D50}" => "m",

    # 1D51 MODIFIER LETTER SMALL ENG
    # ->   014B LATIN SMALL LETTER ENG
    "\u{1D51}" => "\u{014B}",

    # 1D52 MODIFIER LETTER SMALL O
    # ->   006F LATIN SMALL LETTER O
    "\u{1D52}" => "o",

    # 1D53 MODIFIER LETTER SMALL OPEN O
    # ->   0254 LATIN SMALL LETTER OPEN O
    "\u{1D53}" => "\u{0254}",

    # 1D54 MODIFIER LETTER SMALL TOP HALF O
    # ->   1D16 LATIN SMALL LETTER TOP HALF O
    "\u{1D54}" => "\u{1D16}",

    # 1D55 MODIFIER LETTER SMALL BOTTOM HALF O
    # ->   1D17 LATIN SMALL LETTER BOTTOM HALF O
    "\u{1D55}" => "\u{1D17}",

    # 1D56 MODIFIER LETTER SMALL P
    # ->   0070 LATIN SMALL LETTER P
    "\u{1D56}" => "p",

    # 1D57 MODIFIER LETTER SMALL T
    # ->   0074 LATIN SMALL LETTER T
    "\u{1D57}" => "t",

    # 1D58 MODIFIER LETTER SMALL U
    # ->   0075 LATIN SMALL LETTER U
    "\u{1D58}" => "u",

    # 1D59 MODIFIER LETTER SMALL SIDEWAYS U
    # ->   1D1D LATIN SMALL LETTER SIDEWAYS U
    "\u{1D59}" => "\u{1D1D}",

    # 1D5A MODIFIER LETTER SMALL TURNED M
    # ->   026F LATIN SMALL LETTER TURNED M
    "\u{1D5A}" => "\u{026F}",

    # 1D5B MODIFIER LETTER SMALL V
    # ->   0076 LATIN SMALL LETTER V
    "\u{1D5B}" => "v",

    # 1D5C MODIFIER LETTER SMALL AIN
    # ->   1D25 LATIN LETTER AIN
    "\u{1D5C}" => "\u{1D25}",

    # 1D5D MODIFIER LETTER SMALL BETA
    # ->   03B2 GREEK SMALL LETTER BETA
    "\u{1D5D}" => "\u{03B2}",

    # 1D5E MODIFIER LETTER SMALL GREEK GAMMA
    # ->   03B3 GREEK SMALL LETTER GAMMA
    "\u{1D5E}" => "\u{03B3}",

    # 1D5F MODIFIER LETTER SMALL DELTA
    # ->   03B4 GREEK SMALL LETTER DELTA
    "\u{1D5F}" => "\u{03B4}",

    # 1D60 MODIFIER LETTER SMALL GREEK PHI
    # ->   03C6 GREEK SMALL LETTER PHI
    "\u{1D60}" => "\u{03C6}",

    # 1D61 MODIFIER LETTER SMALL CHI
    # ->   03C7 GREEK SMALL LETTER CHI
    "\u{1D61}" => "\u{03C7}",

    # 1D62 LATIN SUBSCRIPT SMALL LETTER I
    # ->   0069 LATIN SMALL LETTER I
    "\u{1D62}" => "i",

    # 1D63 LATIN SUBSCRIPT SMALL LETTER R
    # ->   0072 LATIN SMALL LETTER R
    "\u{1D63}" => "r",

    # 1D64 LATIN SUBSCRIPT SMALL LETTER U
    # ->   0075 LATIN SMALL LETTER U
    "\u{1D64}" => "u",

    # 1D65 LATIN SUBSCRIPT SMALL LETTER V
    # ->   0076 LATIN SMALL LETTER V
    "\u{1D65}" => "v",

    # 1D66 GREEK SUBSCRIPT SMALL LETTER BETA
    # ->   03B2 GREEK SMALL LETTER BETA
    "\u{1D66}" => "\u{03B2}",

    # 1D67 GREEK SUBSCRIPT SMALL LETTER GAMMA
    # ->   03B3 GREEK SMALL LETTER GAMMA
    "\u{1D67}" => "\u{03B3}",

    # 1D68 GREEK SUBSCRIPT SMALL LETTER RHO
    # ->   03C1 GREEK SMALL LETTER RHO
    "\u{1D68}" => "\u{03C1}",

    # 1D69 GREEK SUBSCRIPT SMALL LETTER PHI
    # ->   03C6 GREEK SMALL LETTER PHI
    "\u{1D69}" => "\u{03C6}",

    # 1D6A GREEK SUBSCRIPT SMALL LETTER CHI
    # ->   03C7 GREEK SMALL LETTER CHI
    "\u{1D6A}" => "\u{03C7}",

    # 1E00 LATIN CAPITAL LETTER A WITH RING BELOW
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1E00}" => "A",

    # 1E01 LATIN SMALL LETTER A WITH RING BELOW
    # ->   0061 LATIN SMALL LETTER A
    "\u{1E01}" => "a",

    # 1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{1E02}" => "B",

    # 1E03 LATIN SMALL LETTER B WITH DOT ABOVE
    # ->   0062 LATIN SMALL LETTER B
    "\u{1E03}" => "b",

    # 1E04 LATIN CAPITAL LETTER B WITH DOT BELOW
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{1E04}" => "B",

    # 1E05 LATIN SMALL LETTER B WITH DOT BELOW
    # ->   0062 LATIN SMALL LETTER B
    "\u{1E05}" => "b",

    # 1E06 LATIN CAPITAL LETTER B WITH LINE BELOW
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{1E06}" => "B",

    # 1E07 LATIN SMALL LETTER B WITH LINE BELOW
    # ->   0062 LATIN SMALL LETTER B
    "\u{1E07}" => "b",

    # 1E08 LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{1E08}" => "C",

    # 1E09 LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
    # ->   0063 LATIN SMALL LETTER C
    "\u{1E09}" => "c",

    # 1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1E0A}" => "D",

    # 1E0B LATIN SMALL LETTER D WITH DOT ABOVE
    # ->   0064 LATIN SMALL LETTER D
    "\u{1E0B}" => "d",

    # 1E0C LATIN CAPITAL LETTER D WITH DOT BELOW
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1E0C}" => "D",

    # 1E0D LATIN SMALL LETTER D WITH DOT BELOW
    # ->   0064 LATIN SMALL LETTER D
    "\u{1E0D}" => "d",

    # 1E0E LATIN CAPITAL LETTER D WITH LINE BELOW
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1E0E}" => "D",

    # 1E0F LATIN SMALL LETTER D WITH LINE BELOW
    # ->   0064 LATIN SMALL LETTER D
    "\u{1E0F}" => "d",

    # 1E10 LATIN CAPITAL LETTER D WITH CEDILLA
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1E10}" => "D",

    # 1E11 LATIN SMALL LETTER D WITH CEDILLA
    # ->   0064 LATIN SMALL LETTER D
    "\u{1E11}" => "d",

    # 1E12 LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{1E12}" => "D",

    # 1E13 LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
    # ->   0064 LATIN SMALL LETTER D
    "\u{1E13}" => "d",

    # 1E14 LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1E14}" => "E",

    # 1E15 LATIN SMALL LETTER E WITH MACRON AND GRAVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1E15}" => "e",

    # 1E16 LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1E16}" => "E",

    # 1E17 LATIN SMALL LETTER E WITH MACRON AND ACUTE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1E17}" => "e",

    # 1E18 LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1E18}" => "E",

    # 1E19 LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
    # ->   0065 LATIN SMALL LETTER E
    "\u{1E19}" => "e",

    # 1E1A LATIN CAPITAL LETTER E WITH TILDE BELOW
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1E1A}" => "E",

    # 1E1B LATIN SMALL LETTER E WITH TILDE BELOW
    # ->   0065 LATIN SMALL LETTER E
    "\u{1E1B}" => "e",

    # 1E1C LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1E1C}" => "E",

    # 1E1D LATIN SMALL LETTER E WITH CEDILLA AND BREVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1E1D}" => "e",

    # 1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE
    # ->   0046 LATIN CAPITAL LETTER F
    "\u{1E1E}" => "F",

    # 1E1F LATIN SMALL LETTER F WITH DOT ABOVE
    # ->   0066 LATIN SMALL LETTER F
    "\u{1E1F}" => "f",

    # 1E20 LATIN CAPITAL LETTER G WITH MACRON
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{1E20}" => "G",

    # 1E21 LATIN SMALL LETTER G WITH MACRON
    # ->   0067 LATIN SMALL LETTER G
    "\u{1E21}" => "g",

    # 1E22 LATIN CAPITAL LETTER H WITH DOT ABOVE
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1E22}" => "H",

    # 1E23 LATIN SMALL LETTER H WITH DOT ABOVE
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E23}" => "h",

    # 1E24 LATIN CAPITAL LETTER H WITH DOT BELOW
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1E24}" => "H",

    # 1E25 LATIN SMALL LETTER H WITH DOT BELOW
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E25}" => "h",

    # 1E26 LATIN CAPITAL LETTER H WITH DIAERESIS
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1E26}" => "H",

    # 1E27 LATIN SMALL LETTER H WITH DIAERESIS
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E27}" => "h",

    # 1E28 LATIN CAPITAL LETTER H WITH CEDILLA
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1E28}" => "H",

    # 1E29 LATIN SMALL LETTER H WITH CEDILLA
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E29}" => "h",

    # 1E2A LATIN CAPITAL LETTER H WITH BREVE BELOW
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{1E2A}" => "H",

    # 1E2B LATIN SMALL LETTER H WITH BREVE BELOW
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E2B}" => "h",

    # 1E2C LATIN CAPITAL LETTER I WITH TILDE BELOW
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{1E2C}" => "I",

    # 1E2D LATIN SMALL LETTER I WITH TILDE BELOW
    # ->   0069 LATIN SMALL LETTER I
    "\u{1E2D}" => "i",

    # 1E2E LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{1E2E}" => "I",

    # 1E2F LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
    # ->   0069 LATIN SMALL LETTER I
    "\u{1E2F}" => "i",

    # 1E30 LATIN CAPITAL LETTER K WITH ACUTE
    # ->   004B LATIN CAPITAL LETTER K
    "\u{1E30}" => "K",

    # 1E31 LATIN SMALL LETTER K WITH ACUTE
    # ->   006B LATIN SMALL LETTER K
    "\u{1E31}" => "k",

    # 1E32 LATIN CAPITAL LETTER K WITH DOT BELOW
    # ->   004B LATIN CAPITAL LETTER K
    "\u{1E32}" => "K",

    # 1E33 LATIN SMALL LETTER K WITH DOT BELOW
    # ->   006B LATIN SMALL LETTER K
    "\u{1E33}" => "k",

    # 1E34 LATIN CAPITAL LETTER K WITH LINE BELOW
    # ->   004B LATIN CAPITAL LETTER K
    "\u{1E34}" => "K",

    # 1E35 LATIN SMALL LETTER K WITH LINE BELOW
    # ->   006B LATIN SMALL LETTER K
    "\u{1E35}" => "k",

    # 1E36 LATIN CAPITAL LETTER L WITH DOT BELOW
    # ->   004C LATIN CAPITAL LETTER L
    "\u{1E36}" => "L",

    # 1E37 LATIN SMALL LETTER L WITH DOT BELOW
    # ->   006C LATIN SMALL LETTER L
    "\u{1E37}" => "l",

    # 1E38 LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
    # ->   004C LATIN CAPITAL LETTER L
    "\u{1E38}" => "L",

    # 1E39 LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
    # ->   006C LATIN SMALL LETTER L
    "\u{1E39}" => "l",

    # 1E3A LATIN CAPITAL LETTER L WITH LINE BELOW
    # ->   004C LATIN CAPITAL LETTER L
    "\u{1E3A}" => "L",

    # 1E3B LATIN SMALL LETTER L WITH LINE BELOW
    # ->   006C LATIN SMALL LETTER L
    "\u{1E3B}" => "l",

    # 1E3C LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
    # ->   004C LATIN CAPITAL LETTER L
    "\u{1E3C}" => "L",

    # 1E3D LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
    # ->   006C LATIN SMALL LETTER L
    "\u{1E3D}" => "l",

    # 1E3E LATIN CAPITAL LETTER M WITH ACUTE
    # ->   004D LATIN CAPITAL LETTER M
    "\u{1E3E}" => "M",

    # 1E3F LATIN SMALL LETTER M WITH ACUTE
    # ->   006D LATIN SMALL LETTER M
    "\u{1E3F}" => "m",

    # 1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE
    # ->   004D LATIN CAPITAL LETTER M
    "\u{1E40}" => "M",

    # 1E41 LATIN SMALL LETTER M WITH DOT ABOVE
    # ->   006D LATIN SMALL LETTER M
    "\u{1E41}" => "m",

    # 1E42 LATIN CAPITAL LETTER M WITH DOT BELOW
    # ->   004D LATIN CAPITAL LETTER M
    "\u{1E42}" => "M",

    # 1E43 LATIN SMALL LETTER M WITH DOT BELOW
    # ->   006D LATIN SMALL LETTER M
    "\u{1E43}" => "m",

    # 1E44 LATIN CAPITAL LETTER N WITH DOT ABOVE
    # ->   004E LATIN CAPITAL LETTER N
    "\u{1E44}" => "N",

    # 1E45 LATIN SMALL LETTER N WITH DOT ABOVE
    # ->   006E LATIN SMALL LETTER N
    "\u{1E45}" => "n",

    # 1E46 LATIN CAPITAL LETTER N WITH DOT BELOW
    # ->   004E LATIN CAPITAL LETTER N
    "\u{1E46}" => "N",

    # 1E47 LATIN SMALL LETTER N WITH DOT BELOW
    # ->   006E LATIN SMALL LETTER N
    "\u{1E47}" => "n",

    # 1E48 LATIN CAPITAL LETTER N WITH LINE BELOW
    # ->   004E LATIN CAPITAL LETTER N
    "\u{1E48}" => "N",

    # 1E49 LATIN SMALL LETTER N WITH LINE BELOW
    # ->   006E LATIN SMALL LETTER N
    "\u{1E49}" => "n",

    # 1E4A LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
    # ->   004E LATIN CAPITAL LETTER N
    "\u{1E4A}" => "N",

    # 1E4B LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
    # ->   006E LATIN SMALL LETTER N
    "\u{1E4B}" => "n",

    # 1E4C LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1E4C}" => "O",

    # 1E4D LATIN SMALL LETTER O WITH TILDE AND ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{1E4D}" => "o",

    # 1E4E LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1E4E}" => "O",

    # 1E4F LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
    # ->   006F LATIN SMALL LETTER O
    "\u{1E4F}" => "o",

    # 1E50 LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1E50}" => "O",

    # 1E51 LATIN SMALL LETTER O WITH MACRON AND GRAVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1E51}" => "o",

    # 1E52 LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1E52}" => "O",

    # 1E53 LATIN SMALL LETTER O WITH MACRON AND ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{1E53}" => "o",

    # 1E54 LATIN CAPITAL LETTER P WITH ACUTE
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{1E54}" => "P",

    # 1E55 LATIN SMALL LETTER P WITH ACUTE
    # ->   0070 LATIN SMALL LETTER P
    "\u{1E55}" => "p",

    # 1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{1E56}" => "P",

    # 1E57 LATIN SMALL LETTER P WITH DOT ABOVE
    # ->   0070 LATIN SMALL LETTER P
    "\u{1E57}" => "p",

    # 1E58 LATIN CAPITAL LETTER R WITH DOT ABOVE
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{1E58}" => "R",

    # 1E59 LATIN SMALL LETTER R WITH DOT ABOVE
    # ->   0072 LATIN SMALL LETTER R
    "\u{1E59}" => "r",

    # 1E5A LATIN CAPITAL LETTER R WITH DOT BELOW
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{1E5A}" => "R",

    # 1E5B LATIN SMALL LETTER R WITH DOT BELOW
    # ->   0072 LATIN SMALL LETTER R
    "\u{1E5B}" => "r",

    # 1E5C LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{1E5C}" => "R",

    # 1E5D LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
    # ->   0072 LATIN SMALL LETTER R
    "\u{1E5D}" => "r",

    # 1E5E LATIN CAPITAL LETTER R WITH LINE BELOW
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{1E5E}" => "R",

    # 1E5F LATIN SMALL LETTER R WITH LINE BELOW
    # ->   0072 LATIN SMALL LETTER R
    "\u{1E5F}" => "r",

    # 1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{1E60}" => "S",

    # 1E61 LATIN SMALL LETTER S WITH DOT ABOVE
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E61}" => "s",

    # 1E62 LATIN CAPITAL LETTER S WITH DOT BELOW
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{1E62}" => "S",

    # 1E63 LATIN SMALL LETTER S WITH DOT BELOW
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E63}" => "s",

    # 1E64 LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{1E64}" => "S",

    # 1E65 LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E65}" => "s",

    # 1E66 LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{1E66}" => "S",

    # 1E67 LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E67}" => "s",

    # 1E68 LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{1E68}" => "S",

    # 1E69 LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E69}" => "s",

    # 1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{1E6A}" => "T",

    # 1E6B LATIN SMALL LETTER T WITH DOT ABOVE
    # ->   0074 LATIN SMALL LETTER T
    "\u{1E6B}" => "t",

    # 1E6C LATIN CAPITAL LETTER T WITH DOT BELOW
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{1E6C}" => "T",

    # 1E6D LATIN SMALL LETTER T WITH DOT BELOW
    # ->   0074 LATIN SMALL LETTER T
    "\u{1E6D}" => "t",

    # 1E6E LATIN CAPITAL LETTER T WITH LINE BELOW
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{1E6E}" => "T",

    # 1E6F LATIN SMALL LETTER T WITH LINE BELOW
    # ->   0074 LATIN SMALL LETTER T
    "\u{1E6F}" => "t",

    # 1E70 LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{1E70}" => "T",

    # 1E71 LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
    # ->   0074 LATIN SMALL LETTER T
    "\u{1E71}" => "t",

    # 1E72 LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1E72}" => "U",

    # 1E73 LATIN SMALL LETTER U WITH DIAERESIS BELOW
    # ->   0075 LATIN SMALL LETTER U
    "\u{1E73}" => "u",

    # 1E74 LATIN CAPITAL LETTER U WITH TILDE BELOW
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1E74}" => "U",

    # 1E75 LATIN SMALL LETTER U WITH TILDE BELOW
    # ->   0075 LATIN SMALL LETTER U
    "\u{1E75}" => "u",

    # 1E76 LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1E76}" => "U",

    # 1E77 LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
    # ->   0075 LATIN SMALL LETTER U
    "\u{1E77}" => "u",

    # 1E78 LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1E78}" => "U",

    # 1E79 LATIN SMALL LETTER U WITH TILDE AND ACUTE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1E79}" => "u",

    # 1E7A LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1E7A}" => "U",

    # 1E7B LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
    # ->   0075 LATIN SMALL LETTER U
    "\u{1E7B}" => "u",

    # 1E7C LATIN CAPITAL LETTER V WITH TILDE
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{1E7C}" => "V",

    # 1E7D LATIN SMALL LETTER V WITH TILDE
    # ->   0076 LATIN SMALL LETTER V
    "\u{1E7D}" => "v",

    # 1E7E LATIN CAPITAL LETTER V WITH DOT BELOW
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{1E7E}" => "V",

    # 1E7F LATIN SMALL LETTER V WITH DOT BELOW
    # ->   0076 LATIN SMALL LETTER V
    "\u{1E7F}" => "v",

    # 1E80 LATIN CAPITAL LETTER W WITH GRAVE
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1E80}" => "W",

    # 1E81 LATIN SMALL LETTER W WITH GRAVE
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E81}" => "w",

    # 1E82 LATIN CAPITAL LETTER W WITH ACUTE
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1E82}" => "W",

    # 1E83 LATIN SMALL LETTER W WITH ACUTE
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E83}" => "w",

    # 1E84 LATIN CAPITAL LETTER W WITH DIAERESIS
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1E84}" => "W",

    # 1E85 LATIN SMALL LETTER W WITH DIAERESIS
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E85}" => "w",

    # 1E86 LATIN CAPITAL LETTER W WITH DOT ABOVE
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1E86}" => "W",

    # 1E87 LATIN SMALL LETTER W WITH DOT ABOVE
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E87}" => "w",

    # 1E88 LATIN CAPITAL LETTER W WITH DOT BELOW
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{1E88}" => "W",

    # 1E89 LATIN SMALL LETTER W WITH DOT BELOW
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E89}" => "w",

    # 1E8A LATIN CAPITAL LETTER X WITH DOT ABOVE
    # ->   0058 LATIN CAPITAL LETTER X
    "\u{1E8A}" => "X",

    # 1E8B LATIN SMALL LETTER X WITH DOT ABOVE
    # ->   0078 LATIN SMALL LETTER X
    "\u{1E8B}" => "x",

    # 1E8C LATIN CAPITAL LETTER X WITH DIAERESIS
    # ->   0058 LATIN CAPITAL LETTER X
    "\u{1E8C}" => "X",

    # 1E8D LATIN SMALL LETTER X WITH DIAERESIS
    # ->   0078 LATIN SMALL LETTER X
    "\u{1E8D}" => "x",

    # 1E8E LATIN CAPITAL LETTER Y WITH DOT ABOVE
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{1E8E}" => "Y",

    # 1E8F LATIN SMALL LETTER Y WITH DOT ABOVE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1E8F}" => "y",

    # 1E90 LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{1E90}" => "Z",

    # 1E91 LATIN SMALL LETTER Z WITH CIRCUMFLEX
    # ->   007A LATIN SMALL LETTER Z
    "\u{1E91}" => "z",

    # 1E92 LATIN CAPITAL LETTER Z WITH DOT BELOW
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{1E92}" => "Z",

    # 1E93 LATIN SMALL LETTER Z WITH DOT BELOW
    # ->   007A LATIN SMALL LETTER Z
    "\u{1E93}" => "z",

    # 1E94 LATIN CAPITAL LETTER Z WITH LINE BELOW
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{1E94}" => "Z",

    # 1E95 LATIN SMALL LETTER Z WITH LINE BELOW
    # ->   007A LATIN SMALL LETTER Z
    "\u{1E95}" => "z",

    # 1E96 LATIN SMALL LETTER H WITH LINE BELOW
    # ->   0068 LATIN SMALL LETTER H
    "\u{1E96}" => "h",

    # 1E97 LATIN SMALL LETTER T WITH DIAERESIS
    # ->   0074 LATIN SMALL LETTER T
    "\u{1E97}" => "t",

    # 1E98 LATIN SMALL LETTER W WITH RING ABOVE
    # ->   0077 LATIN SMALL LETTER W
    "\u{1E98}" => "w",

    # 1E99 LATIN SMALL LETTER Y WITH RING ABOVE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1E99}" => "y",

    # 1E9A LATIN SMALL LETTER A WITH RIGHT HALF RING
    # ->   0061 LATIN SMALL LETTER A
    #    + 02BE MODIFIER LETTER RIGHT HALF RING
    "\u{1E9A}" => "a\u{02BE}",

    # 1E9B LATIN SMALL LETTER LONG S WITH DOT ABOVE
    # ->   0073 LATIN SMALL LETTER S
    "\u{1E9B}" => "s",

    # 1EA0 LATIN CAPITAL LETTER A WITH DOT BELOW
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EA0}" => "A",

    # 1EA1 LATIN SMALL LETTER A WITH DOT BELOW
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EA1}" => "a",

    # 1EA2 LATIN CAPITAL LETTER A WITH HOOK ABOVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EA2}" => "A",

    # 1EA3 LATIN SMALL LETTER A WITH HOOK ABOVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EA3}" => "a",

    # 1EA4 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EA4}" => "A",

    # 1EA5 LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EA5}" => "a",

    # 1EA6 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EA6}" => "A",

    # 1EA7 LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EA7}" => "a",

    # 1EA8 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EA8}" => "A",

    # 1EA9 LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EA9}" => "a",

    # 1EAA LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EAA}" => "A",

    # 1EAB LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EAB}" => "a",

    # 1EAC LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EAC}" => "A",

    # 1EAD LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EAD}" => "a",

    # 1EAE LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EAE}" => "A",

    # 1EAF LATIN SMALL LETTER A WITH BREVE AND ACUTE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EAF}" => "a",

    # 1EB0 LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EB0}" => "A",

    # 1EB1 LATIN SMALL LETTER A WITH BREVE AND GRAVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EB1}" => "a",

    # 1EB2 LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EB2}" => "A",

    # 1EB3 LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EB3}" => "a",

    # 1EB4 LATIN CAPITAL LETTER A WITH BREVE AND TILDE
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EB4}" => "A",

    # 1EB5 LATIN SMALL LETTER A WITH BREVE AND TILDE
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EB5}" => "a",

    # 1EB6 LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{1EB6}" => "A",

    # 1EB7 LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
    # ->   0061 LATIN SMALL LETTER A
    "\u{1EB7}" => "a",

    # 1EB8 LATIN CAPITAL LETTER E WITH DOT BELOW
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EB8}" => "E",

    # 1EB9 LATIN SMALL LETTER E WITH DOT BELOW
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EB9}" => "e",

    # 1EBA LATIN CAPITAL LETTER E WITH HOOK ABOVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EBA}" => "E",

    # 1EBB LATIN SMALL LETTER E WITH HOOK ABOVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EBB}" => "e",

    # 1EBC LATIN CAPITAL LETTER E WITH TILDE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EBC}" => "E",

    # 1EBD LATIN SMALL LETTER E WITH TILDE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EBD}" => "e",

    # 1EBE LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EBE}" => "E",

    # 1EBF LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EBF}" => "e",

    # 1EC0 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EC0}" => "E",

    # 1EC1 LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EC1}" => "e",

    # 1EC2 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EC2}" => "E",

    # 1EC3 LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EC3}" => "e",

    # 1EC4 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EC4}" => "E",

    # 1EC5 LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EC5}" => "e",

    # 1EC6 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{1EC6}" => "E",

    # 1EC7 LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    # ->   0065 LATIN SMALL LETTER E
    "\u{1EC7}" => "e",

    # 1EC8 LATIN CAPITAL LETTER I WITH HOOK ABOVE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{1EC8}" => "I",

    # 1EC9 LATIN SMALL LETTER I WITH HOOK ABOVE
    # ->   0069 LATIN SMALL LETTER I
    "\u{1EC9}" => "i",

    # 1ECA LATIN CAPITAL LETTER I WITH DOT BELOW
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{1ECA}" => "I",

    # 1ECB LATIN SMALL LETTER I WITH DOT BELOW
    # ->   0069 LATIN SMALL LETTER I
    "\u{1ECB}" => "i",

    # 1ECC LATIN CAPITAL LETTER O WITH DOT BELOW
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ECC}" => "O",

    # 1ECD LATIN SMALL LETTER O WITH DOT BELOW
    # ->   006F LATIN SMALL LETTER O
    "\u{1ECD}" => "o",

    # 1ECE LATIN CAPITAL LETTER O WITH HOOK ABOVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ECE}" => "O",

    # 1ECF LATIN SMALL LETTER O WITH HOOK ABOVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1ECF}" => "o",

    # 1ED0 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ED0}" => "O",

    # 1ED1 LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{1ED1}" => "o",

    # 1ED2 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ED2}" => "O",

    # 1ED3 LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1ED3}" => "o",

    # 1ED4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ED4}" => "O",

    # 1ED5 LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1ED5}" => "o",

    # 1ED6 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ED6}" => "O",

    # 1ED7 LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
    # ->   006F LATIN SMALL LETTER O
    "\u{1ED7}" => "o",

    # 1ED8 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1ED8}" => "O",

    # 1ED9 LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    # ->   006F LATIN SMALL LETTER O
    "\u{1ED9}" => "o",

    # 1EDA LATIN CAPITAL LETTER O WITH HORN AND ACUTE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1EDA}" => "O",

    # 1EDB LATIN SMALL LETTER O WITH HORN AND ACUTE
    # ->   006F LATIN SMALL LETTER O
    "\u{1EDB}" => "o",

    # 1EDC LATIN CAPITAL LETTER O WITH HORN AND GRAVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1EDC}" => "O",

    # 1EDD LATIN SMALL LETTER O WITH HORN AND GRAVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1EDD}" => "o",

    # 1EDE LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1EDE}" => "O",

    # 1EDF LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
    # ->   006F LATIN SMALL LETTER O
    "\u{1EDF}" => "o",

    # 1EE0 LATIN CAPITAL LETTER O WITH HORN AND TILDE
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1EE0}" => "O",

    # 1EE1 LATIN SMALL LETTER O WITH HORN AND TILDE
    # ->   006F LATIN SMALL LETTER O
    "\u{1EE1}" => "o",

    # 1EE2 LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
    # ->   004F LATIN CAPITAL LETTER O
    "\u{1EE2}" => "O",

    # 1EE3 LATIN SMALL LETTER O WITH HORN AND DOT BELOW
    # ->   006F LATIN SMALL LETTER O
    "\u{1EE3}" => "o",

    # 1EE4 LATIN CAPITAL LETTER U WITH DOT BELOW
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EE4}" => "U",

    # 1EE5 LATIN SMALL LETTER U WITH DOT BELOW
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EE5}" => "u",

    # 1EE6 LATIN CAPITAL LETTER U WITH HOOK ABOVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EE6}" => "U",

    # 1EE7 LATIN SMALL LETTER U WITH HOOK ABOVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EE7}" => "u",

    # 1EE8 LATIN CAPITAL LETTER U WITH HORN AND ACUTE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EE8}" => "U",

    # 1EE9 LATIN SMALL LETTER U WITH HORN AND ACUTE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EE9}" => "u",

    # 1EEA LATIN CAPITAL LETTER U WITH HORN AND GRAVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EEA}" => "U",

    # 1EEB LATIN SMALL LETTER U WITH HORN AND GRAVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EEB}" => "u",

    # 1EEC LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EEC}" => "U",

    # 1EED LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EED}" => "u",

    # 1EEE LATIN CAPITAL LETTER U WITH HORN AND TILDE
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EEE}" => "U",

    # 1EEF LATIN SMALL LETTER U WITH HORN AND TILDE
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EEF}" => "u",

    # 1EF0 LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{1EF0}" => "U",

    # 1EF1 LATIN SMALL LETTER U WITH HORN AND DOT BELOW
    # ->   0075 LATIN SMALL LETTER U
    "\u{1EF1}" => "u",

    # 1EF2 LATIN CAPITAL LETTER Y WITH GRAVE
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{1EF2}" => "Y",

    # 1EF3 LATIN SMALL LETTER Y WITH GRAVE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1EF3}" => "y",

    # 1EF4 LATIN CAPITAL LETTER Y WITH DOT BELOW
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{1EF4}" => "Y",

    # 1EF5 LATIN SMALL LETTER Y WITH DOT BELOW
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1EF5}" => "y",

    # 1EF6 LATIN CAPITAL LETTER Y WITH HOOK ABOVE
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{1EF6}" => "Y",

    # 1EF7 LATIN SMALL LETTER Y WITH HOOK ABOVE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1EF7}" => "y",

    # 1EF8 LATIN CAPITAL LETTER Y WITH TILDE
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{1EF8}" => "Y",

    # 1EF9 LATIN SMALL LETTER Y WITH TILDE
    # ->   0079 LATIN SMALL LETTER Y
    "\u{1EF9}" => "y",

    # 1F00 GREEK SMALL LETTER ALPHA WITH PSILI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F00}" => "\u{03B1}",

    # 1F01 GREEK SMALL LETTER ALPHA WITH DASIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F01}" => "\u{03B1}",

    # 1F02 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F02}" => "\u{03B1}",

    # 1F03 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F03}" => "\u{03B1}",

    # 1F04 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F04}" => "\u{03B1}",

    # 1F05 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F05}" => "\u{03B1}",

    # 1F06 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F06}" => "\u{03B1}",

    # 1F07 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F07}" => "\u{03B1}",

    # 1F08 GREEK CAPITAL LETTER ALPHA WITH PSILI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F08}" => "\u{0391}",

    # 1F09 GREEK CAPITAL LETTER ALPHA WITH DASIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F09}" => "\u{0391}",

    # 1F0A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0A}" => "\u{0391}",

    # 1F0B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0B}" => "\u{0391}",

    # 1F0C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0C}" => "\u{0391}",

    # 1F0D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0D}" => "\u{0391}",

    # 1F0E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0E}" => "\u{0391}",

    # 1F0F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F0F}" => "\u{0391}",

    # 1F10 GREEK SMALL LETTER EPSILON WITH PSILI
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F10}" => "\u{03B5}",

    # 1F11 GREEK SMALL LETTER EPSILON WITH DASIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F11}" => "\u{03B5}",

    # 1F12 GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F12}" => "\u{03B5}",

    # 1F13 GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F13}" => "\u{03B5}",

    # 1F14 GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F14}" => "\u{03B5}",

    # 1F15 GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F15}" => "\u{03B5}",

    # 1F18 GREEK CAPITAL LETTER EPSILON WITH PSILI
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F18}" => "\u{0395}",

    # 1F19 GREEK CAPITAL LETTER EPSILON WITH DASIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F19}" => "\u{0395}",

    # 1F1A GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F1A}" => "\u{0395}",

    # 1F1B GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F1B}" => "\u{0395}",

    # 1F1C GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F1C}" => "\u{0395}",

    # 1F1D GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1F1D}" => "\u{0395}",

    # 1F20 GREEK SMALL LETTER ETA WITH PSILI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F20}" => "\u{03B7}",

    # 1F21 GREEK SMALL LETTER ETA WITH DASIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F21}" => "\u{03B7}",

    # 1F22 GREEK SMALL LETTER ETA WITH PSILI AND VARIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F22}" => "\u{03B7}",

    # 1F23 GREEK SMALL LETTER ETA WITH DASIA AND VARIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F23}" => "\u{03B7}",

    # 1F24 GREEK SMALL LETTER ETA WITH PSILI AND OXIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F24}" => "\u{03B7}",

    # 1F25 GREEK SMALL LETTER ETA WITH DASIA AND OXIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F25}" => "\u{03B7}",

    # 1F26 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F26}" => "\u{03B7}",

    # 1F27 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F27}" => "\u{03B7}",

    # 1F28 GREEK CAPITAL LETTER ETA WITH PSILI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F28}" => "\u{0397}",

    # 1F29 GREEK CAPITAL LETTER ETA WITH DASIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F29}" => "\u{0397}",

    # 1F2A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2A}" => "\u{0397}",

    # 1F2B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2B}" => "\u{0397}",

    # 1F2C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2C}" => "\u{0397}",

    # 1F2D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2D}" => "\u{0397}",

    # 1F2E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2E}" => "\u{0397}",

    # 1F2F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F2F}" => "\u{0397}",

    # 1F30 GREEK SMALL LETTER IOTA WITH PSILI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F30}" => "\u{03B9}",

    # 1F31 GREEK SMALL LETTER IOTA WITH DASIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F31}" => "\u{03B9}",

    # 1F32 GREEK SMALL LETTER IOTA WITH PSILI AND VARIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F32}" => "\u{03B9}",

    # 1F33 GREEK SMALL LETTER IOTA WITH DASIA AND VARIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F33}" => "\u{03B9}",

    # 1F34 GREEK SMALL LETTER IOTA WITH PSILI AND OXIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F34}" => "\u{03B9}",

    # 1F35 GREEK SMALL LETTER IOTA WITH DASIA AND OXIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F35}" => "\u{03B9}",

    # 1F36 GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F36}" => "\u{03B9}",

    # 1F37 GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F37}" => "\u{03B9}",

    # 1F38 GREEK CAPITAL LETTER IOTA WITH PSILI
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F38}" => "\u{0399}",

    # 1F39 GREEK CAPITAL LETTER IOTA WITH DASIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F39}" => "\u{0399}",

    # 1F3A GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3A}" => "\u{0399}",

    # 1F3B GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3B}" => "\u{0399}",

    # 1F3C GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3C}" => "\u{0399}",

    # 1F3D GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3D}" => "\u{0399}",

    # 1F3E GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3E}" => "\u{0399}",

    # 1F3F GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1F3F}" => "\u{0399}",

    # 1F40 GREEK SMALL LETTER OMICRON WITH PSILI
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F40}" => "\u{03BF}",

    # 1F41 GREEK SMALL LETTER OMICRON WITH DASIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F41}" => "\u{03BF}",

    # 1F42 GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F42}" => "\u{03BF}",

    # 1F43 GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F43}" => "\u{03BF}",

    # 1F44 GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F44}" => "\u{03BF}",

    # 1F45 GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F45}" => "\u{03BF}",

    # 1F48 GREEK CAPITAL LETTER OMICRON WITH PSILI
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F48}" => "\u{039F}",

    # 1F49 GREEK CAPITAL LETTER OMICRON WITH DASIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F49}" => "\u{039F}",

    # 1F4A GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F4A}" => "\u{039F}",

    # 1F4B GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F4B}" => "\u{039F}",

    # 1F4C GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F4C}" => "\u{039F}",

    # 1F4D GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1F4D}" => "\u{039F}",

    # 1F50 GREEK SMALL LETTER UPSILON WITH PSILI
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F50}" => "\u{03C5}",

    # 1F51 GREEK SMALL LETTER UPSILON WITH DASIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F51}" => "\u{03C5}",

    # 1F52 GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F52}" => "\u{03C5}",

    # 1F53 GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F53}" => "\u{03C5}",

    # 1F54 GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F54}" => "\u{03C5}",

    # 1F55 GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F55}" => "\u{03C5}",

    # 1F56 GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F56}" => "\u{03C5}",

    # 1F57 GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F57}" => "\u{03C5}",

    # 1F59 GREEK CAPITAL LETTER UPSILON WITH DASIA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1F59}" => "\u{03A5}",

    # 1F5B GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1F5B}" => "\u{03A5}",

    # 1F5D GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1F5D}" => "\u{03A5}",

    # 1F5F GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1F5F}" => "\u{03A5}",

    # 1F60 GREEK SMALL LETTER OMEGA WITH PSILI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F60}" => "\u{03C9}",

    # 1F61 GREEK SMALL LETTER OMEGA WITH DASIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F61}" => "\u{03C9}",

    # 1F62 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F62}" => "\u{03C9}",

    # 1F63 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F63}" => "\u{03C9}",

    # 1F64 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F64}" => "\u{03C9}",

    # 1F65 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F65}" => "\u{03C9}",

    # 1F66 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F66}" => "\u{03C9}",

    # 1F67 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F67}" => "\u{03C9}",

    # 1F68 GREEK CAPITAL LETTER OMEGA WITH PSILI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F68}" => "\u{03A9}",

    # 1F69 GREEK CAPITAL LETTER OMEGA WITH DASIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F69}" => "\u{03A9}",

    # 1F6A GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6A}" => "\u{03A9}",

    # 1F6B GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6B}" => "\u{03A9}",

    # 1F6C GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6C}" => "\u{03A9}",

    # 1F6D GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6D}" => "\u{03A9}",

    # 1F6E GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6E}" => "\u{03A9}",

    # 1F6F GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1F6F}" => "\u{03A9}",

    # 1F70 GREEK SMALL LETTER ALPHA WITH VARIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F70}" => "\u{03B1}",

    # 1F71 GREEK SMALL LETTER ALPHA WITH OXIA
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F71}" => "\u{03B1}",

    # 1F72 GREEK SMALL LETTER EPSILON WITH VARIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F72}" => "\u{03B5}",

    # 1F73 GREEK SMALL LETTER EPSILON WITH OXIA
    # ->   03B5 GREEK SMALL LETTER EPSILON
    "\u{1F73}" => "\u{03B5}",

    # 1F74 GREEK SMALL LETTER ETA WITH VARIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F74}" => "\u{03B7}",

    # 1F75 GREEK SMALL LETTER ETA WITH OXIA
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F75}" => "\u{03B7}",

    # 1F76 GREEK SMALL LETTER IOTA WITH VARIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F76}" => "\u{03B9}",

    # 1F77 GREEK SMALL LETTER IOTA WITH OXIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1F77}" => "\u{03B9}",

    # 1F78 GREEK SMALL LETTER OMICRON WITH VARIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F78}" => "\u{03BF}",

    # 1F79 GREEK SMALL LETTER OMICRON WITH OXIA
    # ->   03BF GREEK SMALL LETTER OMICRON
    "\u{1F79}" => "\u{03BF}",

    # 1F7A GREEK SMALL LETTER UPSILON WITH VARIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F7A}" => "\u{03C5}",

    # 1F7B GREEK SMALL LETTER UPSILON WITH OXIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1F7B}" => "\u{03C5}",

    # 1F7C GREEK SMALL LETTER OMEGA WITH VARIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F7C}" => "\u{03C9}",

    # 1F7D GREEK SMALL LETTER OMEGA WITH OXIA
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1F7D}" => "\u{03C9}",

    # 1F80 GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F80}" => "\u{03B1}",

    # 1F81 GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F81}" => "\u{03B1}",

    # 1F82 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F82}" => "\u{03B1}",

    # 1F83 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F83}" => "\u{03B1}",

    # 1F84 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F84}" => "\u{03B1}",

    # 1F85 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F85}" => "\u{03B1}",

    # 1F86 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F86}" => "\u{03B1}",

    # 1F87 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1F87}" => "\u{03B1}",

    # 1F88 GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F88}" => "\u{0391}",

    # 1F89 GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F89}" => "\u{0391}",

    # 1F8A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8A}" => "\u{0391}",

    # 1F8B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8B}" => "\u{0391}",

    # 1F8C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8C}" => "\u{0391}",

    # 1F8D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8D}" => "\u{0391}",

    # 1F8E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8E}" => "\u{0391}",

    # 1F8F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1F8F}" => "\u{0391}",

    # 1F90 GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F90}" => "\u{03B7}",

    # 1F91 GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F91}" => "\u{03B7}",

    # 1F92 GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F92}" => "\u{03B7}",

    # 1F93 GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F93}" => "\u{03B7}",

    # 1F94 GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F94}" => "\u{03B7}",

    # 1F95 GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F95}" => "\u{03B7}",

    # 1F96 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F96}" => "\u{03B7}",

    # 1F97 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1F97}" => "\u{03B7}",

    # 1F98 GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F98}" => "\u{0397}",

    # 1F99 GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F99}" => "\u{0397}",

    # 1F9A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9A}" => "\u{0397}",

    # 1F9B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9B}" => "\u{0397}",

    # 1F9C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9C}" => "\u{0397}",

    # 1F9D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9D}" => "\u{0397}",

    # 1F9E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9E}" => "\u{0397}",

    # 1F9F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1F9F}" => "\u{0397}",

    # 1FA0 GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA0}" => "\u{03C9}",

    # 1FA1 GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA1}" => "\u{03C9}",

    # 1FA2 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA2}" => "\u{03C9}",

    # 1FA3 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA3}" => "\u{03C9}",

    # 1FA4 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA4}" => "\u{03C9}",

    # 1FA5 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA5}" => "\u{03C9}",

    # 1FA6 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA6}" => "\u{03C9}",

    # 1FA7 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FA7}" => "\u{03C9}",

    # 1FA8 GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FA8}" => "\u{03A9}",

    # 1FA9 GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FA9}" => "\u{03A9}",

    # 1FAA GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAA}" => "\u{03A9}",

    # 1FAB GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAB}" => "\u{03A9}",

    # 1FAC GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAC}" => "\u{03A9}",

    # 1FAD GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAD}" => "\u{03A9}",

    # 1FAE GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAE}" => "\u{03A9}",

    # 1FAF GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FAF}" => "\u{03A9}",

    # 1FB0 GREEK SMALL LETTER ALPHA WITH VRACHY
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB0}" => "\u{03B1}",

    # 1FB1 GREEK SMALL LETTER ALPHA WITH MACRON
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB1}" => "\u{03B1}",

    # 1FB2 GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB2}" => "\u{03B1}",

    # 1FB3 GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB3}" => "\u{03B1}",

    # 1FB4 GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB4}" => "\u{03B1}",

    # 1FB6 GREEK SMALL LETTER ALPHA WITH PERISPOMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB6}" => "\u{03B1}",

    # 1FB7 GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B1 GREEK SMALL LETTER ALPHA
    "\u{1FB7}" => "\u{03B1}",

    # 1FB8 GREEK CAPITAL LETTER ALPHA WITH VRACHY
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1FB8}" => "\u{0391}",

    # 1FB9 GREEK CAPITAL LETTER ALPHA WITH MACRON
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1FB9}" => "\u{0391}",

    # 1FBA GREEK CAPITAL LETTER ALPHA WITH VARIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1FBA}" => "\u{0391}",

    # 1FBB GREEK CAPITAL LETTER ALPHA WITH OXIA
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1FBB}" => "\u{0391}",

    # 1FBC GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    # ->   0391 GREEK CAPITAL LETTER ALPHA
    "\u{1FBC}" => "\u{0391}",

    # 1FBD GREEK KORONIS
    # ->   0020 SPACE
    "\u{1FBD}" => " ",

    # 1FBE GREEK PROSGEGRAMMENI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FBE}" => "\u{03B9}",

    # 1FBF GREEK PSILI
    # ->   0020 SPACE
    "\u{1FBF}" => " ",

    # 1FC0 GREEK PERISPOMENI
    # ->   0020 SPACE
    "\u{1FC0}" => " ",

    # 1FC1 GREEK DIALYTIKA AND PERISPOMENI
    # ->   0020 SPACE
    "\u{1FC1}" => " ",

    # 1FC2 GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1FC2}" => "\u{03B7}",

    # 1FC3 GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1FC3}" => "\u{03B7}",

    # 1FC4 GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1FC4}" => "\u{03B7}",

    # 1FC6 GREEK SMALL LETTER ETA WITH PERISPOMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1FC6}" => "\u{03B7}",

    # 1FC7 GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
    # ->   03B7 GREEK SMALL LETTER ETA
    "\u{1FC7}" => "\u{03B7}",

    # 1FC8 GREEK CAPITAL LETTER EPSILON WITH VARIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1FC8}" => "\u{0395}",

    # 1FC9 GREEK CAPITAL LETTER EPSILON WITH OXIA
    # ->   0395 GREEK CAPITAL LETTER EPSILON
    "\u{1FC9}" => "\u{0395}",

    # 1FCA GREEK CAPITAL LETTER ETA WITH VARIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1FCA}" => "\u{0397}",

    # 1FCB GREEK CAPITAL LETTER ETA WITH OXIA
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1FCB}" => "\u{0397}",

    # 1FCC GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    # ->   0397 GREEK CAPITAL LETTER ETA
    "\u{1FCC}" => "\u{0397}",

    # 1FCD GREEK PSILI AND VARIA
    # ->   0020 SPACE
    "\u{1FCD}" => " ",

    # 1FCE GREEK PSILI AND OXIA
    # ->   0020 SPACE
    "\u{1FCE}" => " ",

    # 1FCF GREEK PSILI AND PERISPOMENI
    # ->   0020 SPACE
    "\u{1FCF}" => " ",

    # 1FD0 GREEK SMALL LETTER IOTA WITH VRACHY
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD0}" => "\u{03B9}",

    # 1FD1 GREEK SMALL LETTER IOTA WITH MACRON
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD1}" => "\u{03B9}",

    # 1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD2}" => "\u{03B9}",

    # 1FD3 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD3}" => "\u{03B9}",

    # 1FD6 GREEK SMALL LETTER IOTA WITH PERISPOMENI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD6}" => "\u{03B9}",

    # 1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
    # ->   03B9 GREEK SMALL LETTER IOTA
    "\u{1FD7}" => "\u{03B9}",

    # 1FD8 GREEK CAPITAL LETTER IOTA WITH VRACHY
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1FD8}" => "\u{0399}",

    # 1FD9 GREEK CAPITAL LETTER IOTA WITH MACRON
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1FD9}" => "\u{0399}",

    # 1FDA GREEK CAPITAL LETTER IOTA WITH VARIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1FDA}" => "\u{0399}",

    # 1FDB GREEK CAPITAL LETTER IOTA WITH OXIA
    # ->   0399 GREEK CAPITAL LETTER IOTA
    "\u{1FDB}" => "\u{0399}",

    # 1FDD GREEK DASIA AND VARIA
    # ->   0020 SPACE
    "\u{1FDD}" => " ",

    # 1FDE GREEK DASIA AND OXIA
    # ->   0020 SPACE
    "\u{1FDE}" => " ",

    # 1FDF GREEK DASIA AND PERISPOMENI
    # ->   0020 SPACE
    "\u{1FDF}" => " ",

    # 1FE0 GREEK SMALL LETTER UPSILON WITH VRACHY
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE0}" => "\u{03C5}",

    # 1FE1 GREEK SMALL LETTER UPSILON WITH MACRON
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE1}" => "\u{03C5}",

    # 1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE2}" => "\u{03C5}",

    # 1FE3 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE3}" => "\u{03C5}",

    # 1FE4 GREEK SMALL LETTER RHO WITH PSILI
    # ->   03C1 GREEK SMALL LETTER RHO
    "\u{1FE4}" => "\u{03C1}",

    # 1FE5 GREEK SMALL LETTER RHO WITH DASIA
    # ->   03C1 GREEK SMALL LETTER RHO
    "\u{1FE5}" => "\u{03C1}",

    # 1FE6 GREEK SMALL LETTER UPSILON WITH PERISPOMENI
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE6}" => "\u{03C5}",

    # 1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
    # ->   03C5 GREEK SMALL LETTER UPSILON
    "\u{1FE7}" => "\u{03C5}",

    # 1FE8 GREEK CAPITAL LETTER UPSILON WITH VRACHY
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1FE8}" => "\u{03A5}",

    # 1FE9 GREEK CAPITAL LETTER UPSILON WITH MACRON
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1FE9}" => "\u{03A5}",

    # 1FEA GREEK CAPITAL LETTER UPSILON WITH VARIA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1FEA}" => "\u{03A5}",

    # 1FEB GREEK CAPITAL LETTER UPSILON WITH OXIA
    # ->   03A5 GREEK CAPITAL LETTER UPSILON
    "\u{1FEB}" => "\u{03A5}",

    # 1FEC GREEK CAPITAL LETTER RHO WITH DASIA
    # ->   03A1 GREEK CAPITAL LETTER RHO
    "\u{1FEC}" => "\u{03A1}",

    # 1FED GREEK DIALYTIKA AND VARIA
    # ->   0020 SPACE
    "\u{1FED}" => " ",

    # 1FEE GREEK DIALYTIKA AND OXIA
    # ->   0020 SPACE
    "\u{1FEE}" => " ",

    # 1FEF GREEK VARIA
    # ->   0060 GRAVE ACCENT
    "\u{1FEF}" => "`",

    # 1FF2 GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FF2}" => "\u{03C9}",

    # 1FF3 GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FF3}" => "\u{03C9}",

    # 1FF4 GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FF4}" => "\u{03C9}",

    # 1FF6 GREEK SMALL LETTER OMEGA WITH PERISPOMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FF6}" => "\u{03C9}",

    # 1FF7 GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
    # ->   03C9 GREEK SMALL LETTER OMEGA
    "\u{1FF7}" => "\u{03C9}",

    # 1FF8 GREEK CAPITAL LETTER OMICRON WITH VARIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1FF8}" => "\u{039F}",

    # 1FF9 GREEK CAPITAL LETTER OMICRON WITH OXIA
    # ->   039F GREEK CAPITAL LETTER OMICRON
    "\u{1FF9}" => "\u{039F}",

    # 1FFA GREEK CAPITAL LETTER OMEGA WITH VARIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FFA}" => "\u{03A9}",

    # 1FFB GREEK CAPITAL LETTER OMEGA WITH OXIA
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FFB}" => "\u{03A9}",

    # 1FFC GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{1FFC}" => "\u{03A9}",

    # 1FFD GREEK OXIA
    # ->   0020 SPACE
    "\u{1FFD}" => " ",

    # 1FFE GREEK DASIA
    # ->   0020 SPACE
    "\u{1FFE}" => " ",

    # 2000 EN QUAD
    # ->   0020 SPACE
    "\u{2000}" => " ",

    # 2001 EM QUAD
    # ->   0020 SPACE
    "\u{2001}" => " ",

    # 2002 EN SPACE
    # ->   0020 SPACE
    "\u{2002}" => " ",

    # 2003 EM SPACE
    # ->   0020 SPACE
    "\u{2003}" => " ",

    # 2004 THREE-PER-EM SPACE
    # ->   0020 SPACE
    "\u{2004}" => " ",

    # 2005 FOUR-PER-EM SPACE
    # ->   0020 SPACE
    "\u{2005}" => " ",

    # 2006 SIX-PER-EM SPACE
    # ->   0020 SPACE
    "\u{2006}" => " ",

    # 2007 FIGURE SPACE
    # ->   0020 SPACE
    "\u{2007}" => " ",

    # 2008 PUNCTUATION SPACE
    # ->   0020 SPACE
    "\u{2008}" => " ",

    # 2009 THIN SPACE
    # ->   0020 SPACE
    "\u{2009}" => " ",

    # 200A HAIR SPACE
    # ->   0020 SPACE
    "\u{200A}" => " ",

    # 2011 NON-BREAKING HYPHEN
    # ->   2010 HYPHEN
    "\u{2011}" => "\u{2010}",

    # 2017 DOUBLE LOW LINE
    # ->   0020 SPACE
    "\u{2017}" => " ",

    # 2024 ONE DOT LEADER
    # ->   002E FULL STOP
    "\u{2024}" => ".",

    # 2025 TWO DOT LEADER
    # ->   002E FULL STOP
    #    + 002E FULL STOP
    "\u{2025}" => "..",

    # 2026 HORIZONTAL ELLIPSIS
    # ->   002E FULL STOP
    #    + 002E FULL STOP
    #    + 002E FULL STOP
    "\u{2026}" => "...",

    # 202F NARROW NO-BREAK SPACE
    # ->   0020 SPACE
    "\u{202F}" => " ",

    # 2033 DOUBLE PRIME
    # ->   2032 PRIME
    #    + 2032 PRIME
    "\u{2033}" => "\u{2032}\u{2032}",

    # 2034 TRIPLE PRIME
    # ->   2032 PRIME
    #    + 2032 PRIME
    #    + 2032 PRIME
    "\u{2034}" => "\u{2032}\u{2032}\u{2032}",

    # 2036 REVERSED DOUBLE PRIME
    # ->   2035 REVERSED PRIME
    #    + 2035 REVERSED PRIME
    "\u{2036}" => "\u{2035}\u{2035}",

    # 2037 REVERSED TRIPLE PRIME
    # ->   2035 REVERSED PRIME
    #    + 2035 REVERSED PRIME
    #    + 2035 REVERSED PRIME
    "\u{2037}" => "\u{2035}\u{2035}\u{2035}",

    # 203C DOUBLE EXCLAMATION MARK
    # ->   0021 EXCLAMATION MARK
    #    + 0021 EXCLAMATION MARK
    "\u{203C}" => "!!",

    # 203E OVERLINE
    # ->   0020 SPACE
    "\u{203E}" => " ",

    # 2045 LEFT SQUARE BRACKET WITH QUILL
    # ->   005B LEFT SQUARE BRACKET
    "\u{2045}" => "[",

    # 2046 RIGHT SQUARE BRACKET WITH QUILL
    # ->   005D RIGHT SQUARE BRACKET
    "\u{2046}" => "]",

    # 2047 DOUBLE QUESTION MARK
    # ->   003F QUESTION MARK
    #    + 003F QUESTION MARK
    "\u{2047}" => "??",

    # 2048 QUESTION EXCLAMATION MARK
    # ->   003F QUESTION MARK
    #    + 0021 EXCLAMATION MARK
    "\u{2048}" => "?!",

    # 2049 EXCLAMATION QUESTION MARK
    # ->   0021 EXCLAMATION MARK
    #    + 003F QUESTION MARK
    "\u{2049}" => "!?",

    # 2057 QUADRUPLE PRIME
    # ->   2032 PRIME
    #    + 2032 PRIME
    #    + 2032 PRIME
    #    + 2032 PRIME
    "\u{2057}" => "\u{2032}\u{2032}\u{2032}\u{2032}",

    # 205F MEDIUM MATHEMATICAL SPACE
    # ->   0020 SPACE
    "\u{205F}" => " ",

    # 2070 SUPERSCRIPT ZERO
    # ->   0030 DIGIT ZERO
    "\u{2070}" => "0",

    # 2071 SUPERSCRIPT LATIN SMALL LETTER I
    # ->   0069 LATIN SMALL LETTER I
    "\u{2071}" => "i",

    # 2074 SUPERSCRIPT FOUR
    # ->   0034 DIGIT FOUR
    "\u{2074}" => "4",

    # 2075 SUPERSCRIPT FIVE
    # ->   0035 DIGIT FIVE
    "\u{2075}" => "5",

    # 2076 SUPERSCRIPT SIX
    # ->   0036 DIGIT SIX
    "\u{2076}" => "6",

    # 2077 SUPERSCRIPT SEVEN
    # ->   0037 DIGIT SEVEN
    "\u{2077}" => "7",

    # 2078 SUPERSCRIPT EIGHT
    # ->   0038 DIGIT EIGHT
    "\u{2078}" => "8",

    # 2079 SUPERSCRIPT NINE
    # ->   0039 DIGIT NINE
    "\u{2079}" => "9",

    # 207A SUPERSCRIPT PLUS SIGN
    # ->   002B PLUS SIGN
    "\u{207A}" => "+",

    # 207B SUPERSCRIPT MINUS
    # ->   2212 MINUS SIGN
    "\u{207B}" => "\u{2212}",

    # 207C SUPERSCRIPT EQUALS SIGN
    # ->   003D EQUALS SIGN
    "\u{207C}" => "=",

    # 207D SUPERSCRIPT LEFT PARENTHESIS
    # ->   0028 LEFT PARENTHESIS
    "\u{207D}" => "(",

    # 207E SUPERSCRIPT RIGHT PARENTHESIS
    # ->   0029 RIGHT PARENTHESIS
    "\u{207E}" => ")",

    # 207F SUPERSCRIPT LATIN SMALL LETTER N
    # ->   006E LATIN SMALL LETTER N
    "\u{207F}" => "n",

    # 2080 SUBSCRIPT ZERO
    # ->   0030 DIGIT ZERO
    "\u{2080}" => "0",

    # 2081 SUBSCRIPT ONE
    # ->   0031 DIGIT ONE
    "\u{2081}" => "1",

    # 2082 SUBSCRIPT TWO
    # ->   0032 DIGIT TWO
    "\u{2082}" => "2",

    # 2083 SUBSCRIPT THREE
    # ->   0033 DIGIT THREE
    "\u{2083}" => "3",

    # 2084 SUBSCRIPT FOUR
    # ->   0034 DIGIT FOUR
    "\u{2084}" => "4",

    # 2085 SUBSCRIPT FIVE
    # ->   0035 DIGIT FIVE
    "\u{2085}" => "5",

    # 2086 SUBSCRIPT SIX
    # ->   0036 DIGIT SIX
    "\u{2086}" => "6",

    # 2087 SUBSCRIPT SEVEN
    # ->   0037 DIGIT SEVEN
    "\u{2087}" => "7",

    # 2088 SUBSCRIPT EIGHT
    # ->   0038 DIGIT EIGHT
    "\u{2088}" => "8",

    # 2089 SUBSCRIPT NINE
    # ->   0039 DIGIT NINE
    "\u{2089}" => "9",

    # 208A SUBSCRIPT PLUS SIGN
    # ->   002B PLUS SIGN
    "\u{208A}" => "+",

    # 208B SUBSCRIPT MINUS
    # ->   2212 MINUS SIGN
    "\u{208B}" => "\u{2212}",

    # 208C SUBSCRIPT EQUALS SIGN
    # ->   003D EQUALS SIGN
    "\u{208C}" => "=",

    # 208D SUBSCRIPT LEFT PARENTHESIS
    # ->   0028 LEFT PARENTHESIS
    "\u{208D}" => "(",

    # 208E SUBSCRIPT RIGHT PARENTHESIS
    # ->   0029 RIGHT PARENTHESIS
    "\u{208E}" => ")",

    # 20A8 RUPEE SIGN
    # ->   0052 LATIN CAPITAL LETTER R
    #    + 0073 LATIN SMALL LETTER S
    "\u{20A8}" => "Rs",

    # 2100 ACCOUNT OF
    # ->   0061 LATIN SMALL LETTER A
    #    + 002F SOLIDUS
    #    + 0063 LATIN SMALL LETTER C
    "\u{2100}" => "a/c",

    # 2101 ADDRESSED TO THE SUBJECT
    # ->   0061 LATIN SMALL LETTER A
    #    + 002F SOLIDUS
    #    + 0073 LATIN SMALL LETTER S
    "\u{2101}" => "a/s",

    # 2102 DOUBLE-STRUCK CAPITAL C
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{2102}" => "C",

    # 2103 DEGREE CELSIUS
    # ->   00B0 DEGREE SIGN
    #    + 0043 LATIN CAPITAL LETTER C
    "\u{2103}" => "\u{B0C}",

    # 2105 CARE OF
    # ->   0063 LATIN SMALL LETTER C
    #    + 002F SOLIDUS
    #    + 006F LATIN SMALL LETTER O
    "\u{2105}" => "c/o",

    # 2106 CADA UNA
    # ->   0063 LATIN SMALL LETTER C
    #    + 002F SOLIDUS
    #    + 0075 LATIN SMALL LETTER U
    "\u{2106}" => "c/u",

    # 2107 EULER CONSTANT
    # ->   0190 LATIN CAPITAL LETTER OPEN E
    "\u{2107}" => "\u{0190}",

    # 2109 DEGREE FAHRENHEIT
    # ->   00B0 DEGREE SIGN
    #    + 0046 LATIN CAPITAL LETTER F
    "\u{2109}" => "\u{B0F}",

    # 210A SCRIPT SMALL G
    # ->   0067 LATIN SMALL LETTER G
    "\u{210A}" => "g",

    # 210B SCRIPT CAPITAL H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{210B}" => "H",

    # 210C BLACK-LETTER CAPITAL H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{210C}" => "H",

    # 210D DOUBLE-STRUCK CAPITAL H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{210D}" => "H",

    # 210E PLANCK CONSTANT
    # ->   0068 LATIN SMALL LETTER H
    "\u{210E}" => "h",

    # 210F PLANCK CONSTANT OVER TWO PI
    # ->   0068 LATIN SMALL LETTER H
    "\u{210F}" => "h",

    # 2110 SCRIPT CAPITAL I
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{2110}" => "I",

    # 2111 BLACK-LETTER CAPITAL I
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{2111}" => "I",

    # 2112 SCRIPT CAPITAL L
    # ->   004C LATIN CAPITAL LETTER L
    "\u{2112}" => "L",

    # 2113 SCRIPT SMALL L
    # ->   006C LATIN SMALL LETTER L
    "\u{2113}" => "l",

    # 2115 DOUBLE-STRUCK CAPITAL N
    # ->   004E LATIN CAPITAL LETTER N
    "\u{2115}" => "N",

    # 2116 NUMERO SIGN
    # ->   004E LATIN CAPITAL LETTER N
    #    + 006F LATIN SMALL LETTER O
    "\u{2116}" => "No",

    # 2119 DOUBLE-STRUCK CAPITAL P
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{2119}" => "P",

    # 211A DOUBLE-STRUCK CAPITAL Q
    # ->   0051 LATIN CAPITAL LETTER Q
    "\u{211A}" => "Q",

    # 211B SCRIPT CAPITAL R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{211B}" => "R",

    # 211C BLACK-LETTER CAPITAL R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{211C}" => "R",

    # 211D DOUBLE-STRUCK CAPITAL R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{211D}" => "R",

    # 2120 SERVICE MARK
    # ->   0053 LATIN CAPITAL LETTER S
    #    + 004D LATIN CAPITAL LETTER M
    "\u{2120}" => "SM",

    # 2121 TELEPHONE SIGN
    # ->   0054 LATIN CAPITAL LETTER T
    #    + 0045 LATIN CAPITAL LETTER E
    #    + 004C LATIN CAPITAL LETTER L
    "\u{2121}" => "TEL",

    # 2122 TRADE MARK SIGN
    # ->   0054 LATIN CAPITAL LETTER T
    #    + 004D LATIN CAPITAL LETTER M
    "\u{2122}" => "TM",

    # 2124 DOUBLE-STRUCK CAPITAL Z
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{2124}" => "Z",

    # 2126 OHM SIGN
    # ->   03A9 GREEK CAPITAL LETTER OMEGA
    "\u{2126}" => "\u{03A9}",

    # 2128 BLACK-LETTER CAPITAL Z
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{2128}" => "Z",

    # 212A KELVIN SIGN
    # ->   004B LATIN CAPITAL LETTER K
    "\u{212A}" => "K",

    # 212B ANGSTROM SIGN
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{212B}" => "A",

    # 212C SCRIPT CAPITAL B
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{212C}" => "B",

    # 212D BLACK-LETTER CAPITAL C
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{212D}" => "C",

    # 212F SCRIPT SMALL E
    # ->   0065 LATIN SMALL LETTER E
    "\u{212F}" => "e",

    # 2130 SCRIPT CAPITAL E
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{2130}" => "E",

    # 2131 SCRIPT CAPITAL F
    # ->   0046 LATIN CAPITAL LETTER F
    "\u{2131}" => "F",

    # 2133 SCRIPT CAPITAL M
    # ->   004D LATIN CAPITAL LETTER M
    "\u{2133}" => "M",

    # 2134 SCRIPT SMALL O
    # ->   006F LATIN SMALL LETTER O
    "\u{2134}" => "o",

    # 2135 ALEF SYMBOL
    # ->   05D0 HEBREW LETTER ALEF
    "\u{2135}" => "\u{05D0}",

    # 2136 BET SYMBOL
    # ->   05D1 HEBREW LETTER BET
    "\u{2136}" => "\u{05D1}",

    # 2137 GIMEL SYMBOL
    # ->   05D2 HEBREW LETTER GIMEL
    "\u{2137}" => "\u{05D2}",

    # 2138 DALET SYMBOL
    # ->   05D3 HEBREW LETTER DALET
    "\u{2138}" => "\u{05D3}",

    # 2139 INFORMATION SOURCE
    # ->   0069 LATIN SMALL LETTER I
    "\u{2139}" => "i",

    # 213B FACSIMILE SIGN
    # ->   0046 LATIN CAPITAL LETTER F
    #    + 0041 LATIN CAPITAL LETTER A
    #    + 0058 LATIN CAPITAL LETTER X
    "\u{213B}" => "FAX",

    # 213D DOUBLE-STRUCK SMALL GAMMA
    # ->   03B3 GREEK SMALL LETTER GAMMA
    "\u{213D}" => "\u{03B3}",

    # 213E DOUBLE-STRUCK CAPITAL GAMMA
    # ->   0393 GREEK CAPITAL LETTER GAMMA
    "\u{213E}" => "\u{0393}",

    # 213F DOUBLE-STRUCK CAPITAL PI
    # ->   03A0 GREEK CAPITAL LETTER PI
    "\u{213F}" => "\u{03A0}",

    # 2140 DOUBLE-STRUCK N-ARY SUMMATION
    # ->   2211 N-ARY SUMMATION
    "\u{2140}" => "\u{2211}",

    # 2145 DOUBLE-STRUCK ITALIC CAPITAL D
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{2145}" => "D",

    # 2146 DOUBLE-STRUCK ITALIC SMALL D
    # ->   0064 LATIN SMALL LETTER D
    "\u{2146}" => "d",

    # 2147 DOUBLE-STRUCK ITALIC SMALL E
    # ->   0065 LATIN SMALL LETTER E
    "\u{2147}" => "e",

    # 2148 DOUBLE-STRUCK ITALIC SMALL I
    # ->   0069 LATIN SMALL LETTER I
    "\u{2148}" => "i",

    # 2149 DOUBLE-STRUCK ITALIC SMALL J
    # ->   006A LATIN SMALL LETTER J
    "\u{2149}" => "j",

    # 2153 VULGAR FRACTION ONE THIRD
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0033 DIGIT THREE
    "\u{2153}" => "1\u{2044}3",

    # 2154 VULGAR FRACTION TWO THIRDS
    # ->   0032 DIGIT TWO
    #    + 2044 FRACTION SLASH
    #    + 0033 DIGIT THREE
    "\u{2154}" => "2\u{2044}3",

    # 2155 VULGAR FRACTION ONE FIFTH
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0035 DIGIT FIVE
    "\u{2155}" => "1\u{2044}5",

    # 2156 VULGAR FRACTION TWO FIFTHS
    # ->   0032 DIGIT TWO
    #    + 2044 FRACTION SLASH
    #    + 0035 DIGIT FIVE
    "\u{2156}" => "2\u{2044}5",

    # 2157 VULGAR FRACTION THREE FIFTHS
    # ->   0033 DIGIT THREE
    #    + 2044 FRACTION SLASH
    #    + 0035 DIGIT FIVE
    "\u{2157}" => "3\u{2044}5",

    # 2158 VULGAR FRACTION FOUR FIFTHS
    # ->   0034 DIGIT FOUR
    #    + 2044 FRACTION SLASH
    #    + 0035 DIGIT FIVE
    "\u{2158}" => "4\u{2044}5",

    # 2159 VULGAR FRACTION ONE SIXTH
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0036 DIGIT SIX
    "\u{2159}" => "1\u{2044}6",

    # 215A VULGAR FRACTION FIVE SIXTHS
    # ->   0035 DIGIT FIVE
    #    + 2044 FRACTION SLASH
    #    + 0036 DIGIT SIX
    "\u{215A}" => "5\u{2044}6",

    # 215B VULGAR FRACTION ONE EIGHTH
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    #    + 0038 DIGIT EIGHT
    "\u{215B}" => "1\u{2044}8",

    # 215C VULGAR FRACTION THREE EIGHTHS
    # ->   0033 DIGIT THREE
    #    + 2044 FRACTION SLASH
    #    + 0038 DIGIT EIGHT
    "\u{215C}" => "3\u{2044}8",

    # 215D VULGAR FRACTION FIVE EIGHTHS
    # ->   0035 DIGIT FIVE
    #    + 2044 FRACTION SLASH
    #    + 0038 DIGIT EIGHT
    "\u{215D}" => "5\u{2044}8",

    # 215E VULGAR FRACTION SEVEN EIGHTHS
    # ->   0037 DIGIT SEVEN
    #    + 2044 FRACTION SLASH
    #    + 0038 DIGIT EIGHT
    "\u{215E}" => "7\u{2044}8",

    # 215F FRACTION NUMERATOR ONE
    # ->   0031 DIGIT ONE
    #    + 2044 FRACTION SLASH
    "\u{215F}" => "1\u{2044}",

    # 2160 ROMAN NUMERAL ONE
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{2160}" => "I",

    # 2161 ROMAN NUMERAL TWO
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{2161}" => "II",

    # 2162 ROMAN NUMERAL THREE
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{2162}" => "III",

    # 2163 ROMAN NUMERAL FOUR
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{2163}" => "IV",

    # 2164 ROMAN NUMERAL FIVE
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{2164}" => "V",

    # 2165 ROMAN NUMERAL SIX
    # ->   0056 LATIN CAPITAL LETTER V
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{2165}" => "VI",

    # 2166 ROMAN NUMERAL SEVEN
    # ->   0056 LATIN CAPITAL LETTER V
    #    + 0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{2166}" => "VII",

    # 2167 ROMAN NUMERAL EIGHT
    # ->   0056 LATIN CAPITAL LETTER V
    #    + 0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{2167}" => "VIII",

    # 2168 ROMAN NUMERAL NINE
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 0058 LATIN CAPITAL LETTER X
    "\u{2168}" => "IX",

    # 2169 ROMAN NUMERAL TEN
    # ->   0058 LATIN CAPITAL LETTER X
    "\u{2169}" => "X",

    # 216A ROMAN NUMERAL ELEVEN
    # ->   0058 LATIN CAPITAL LETTER X
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{216A}" => "XI",

    # 216B ROMAN NUMERAL TWELVE
    # ->   0058 LATIN CAPITAL LETTER X
    #    + 0049 LATIN CAPITAL LETTER I
    #    + 0049 LATIN CAPITAL LETTER I
    "\u{216B}" => "XII",

    # 216C ROMAN NUMERAL FIFTY
    # ->   004C LATIN CAPITAL LETTER L
    "\u{216C}" => "L",

    # 216D ROMAN NUMERAL ONE HUNDRED
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{216D}" => "C",

    # 216E ROMAN NUMERAL FIVE HUNDRED
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{216E}" => "D",

    # 216F ROMAN NUMERAL ONE THOUSAND
    # ->   004D LATIN CAPITAL LETTER M
    "\u{216F}" => "M",

    # 2170 SMALL ROMAN NUMERAL ONE
    # ->   0069 LATIN SMALL LETTER I
    "\u{2170}" => "i",

    # 2171 SMALL ROMAN NUMERAL TWO
    # ->   0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    "\u{2171}" => "ii",

    # 2172 SMALL ROMAN NUMERAL THREE
    # ->   0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    "\u{2172}" => "iii",

    # 2173 SMALL ROMAN NUMERAL FOUR
    # ->   0069 LATIN SMALL LETTER I
    #    + 0076 LATIN SMALL LETTER V
    "\u{2173}" => "iv",

    # 2174 SMALL ROMAN NUMERAL FIVE
    # ->   0076 LATIN SMALL LETTER V
    "\u{2174}" => "v",

    # 2175 SMALL ROMAN NUMERAL SIX
    # ->   0076 LATIN SMALL LETTER V
    #    + 0069 LATIN SMALL LETTER I
    "\u{2175}" => "vi",

    # 2176 SMALL ROMAN NUMERAL SEVEN
    # ->   0076 LATIN SMALL LETTER V
    #    + 0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    "\u{2176}" => "vii",

    # 2177 SMALL ROMAN NUMERAL EIGHT
    # ->   0076 LATIN SMALL LETTER V
    #    + 0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    "\u{2177}" => "viii",

    # 2178 SMALL ROMAN NUMERAL NINE
    # ->   0069 LATIN SMALL LETTER I
    #    + 0078 LATIN SMALL LETTER X
    "\u{2178}" => "ix",

    # 2179 SMALL ROMAN NUMERAL TEN
    # ->   0078 LATIN SMALL LETTER X
    "\u{2179}" => "x",

    # 217A SMALL ROMAN NUMERAL ELEVEN
    # ->   0078 LATIN SMALL LETTER X
    #    + 0069 LATIN SMALL LETTER I
    "\u{217A}" => "xi",

    # 217B SMALL ROMAN NUMERAL TWELVE
    # ->   0078 LATIN SMALL LETTER X
    #    + 0069 LATIN SMALL LETTER I
    #    + 0069 LATIN SMALL LETTER I
    "\u{217B}" => "xii",

    # 217C SMALL ROMAN NUMERAL FIFTY
    # ->   006C LATIN SMALL LETTER L
    "\u{217C}" => "l",

    # 217D SMALL ROMAN NUMERAL ONE HUNDRED
    # ->   0063 LATIN SMALL LETTER C
    "\u{217D}" => "c",

    # 217E SMALL ROMAN NUMERAL FIVE HUNDRED
    # ->   0064 LATIN SMALL LETTER D
    "\u{217E}" => "d",

    # 217F SMALL ROMAN NUMERAL ONE THOUSAND
    # ->   006D LATIN SMALL LETTER M
    "\u{217F}" => "m",

    # 219A LEFTWARDS ARROW WITH STROKE
    # ->   2190 LEFTWARDS ARROW
    "\u{219A}" => "\u{2190}",

    # 219B RIGHTWARDS ARROW WITH STROKE
    # ->   2192 RIGHTWARDS ARROW
    "\u{219B}" => "\u{2192}",

    # 21A2 LEFTWARDS ARROW WITH TAIL
    # ->   2190 LEFTWARDS ARROW
    "\u{21A2}" => "\u{2190}",

    # 21A3 RIGHTWARDS ARROW WITH TAIL
    # ->   2192 RIGHTWARDS ARROW
    "\u{21A3}" => "\u{2192}",

    # 21A8 UP DOWN ARROW WITH BASE
    # ->   2195 UP DOWN ARROW
    "\u{21A8}" => "\u{2195}",

    # 21A9 LEFTWARDS ARROW WITH HOOK
    # ->   2190 LEFTWARDS ARROW
    "\u{21A9}" => "\u{2190}",

    # 21AA RIGHTWARDS ARROW WITH HOOK
    # ->   2192 RIGHTWARDS ARROW
    "\u{21AA}" => "\u{2192}",

    # 21AB LEFTWARDS ARROW WITH LOOP
    # ->   2190 LEFTWARDS ARROW
    "\u{21AB}" => "\u{2190}",

    # 21AC RIGHTWARDS ARROW WITH LOOP
    # ->   2192 RIGHTWARDS ARROW
    "\u{21AC}" => "\u{2192}",

    # 21AE LEFT RIGHT ARROW WITH STROKE
    # ->   2194 LEFT RIGHT ARROW
    "\u{21AE}" => "\u{2194}",

    # 21B0 UPWARDS ARROW WITH TIP LEFTWARDS
    # ->   2191 UPWARDS ARROW
    "\u{21B0}" => "\u{2191}",

    # 21B1 UPWARDS ARROW WITH TIP RIGHTWARDS
    # ->   2191 UPWARDS ARROW
    "\u{21B1}" => "\u{2191}",

    # 21B2 DOWNWARDS ARROW WITH TIP LEFTWARDS
    # ->   2193 DOWNWARDS ARROW
    "\u{21B2}" => "\u{2193}",

    # 21B3 DOWNWARDS ARROW WITH TIP RIGHTWARDS
    # ->   2193 DOWNWARDS ARROW
    "\u{21B3}" => "\u{2193}",

    # 21B4 RIGHTWARDS ARROW WITH CORNER DOWNWARDS
    # ->   2192 RIGHTWARDS ARROW
    "\u{21B4}" => "\u{2192}",

    # 21B5 DOWNWARDS ARROW WITH CORNER LEFTWARDS
    # ->   2193 DOWNWARDS ARROW
    "\u{21B5}" => "\u{2193}",

    # 21CD LEFTWARDS DOUBLE ARROW WITH STROKE
    # ->   21D0 LEFTWARDS DOUBLE ARROW
    "\u{21CD}" => "\u{21D0}",

    # 21CE LEFT RIGHT DOUBLE ARROW WITH STROKE
    # ->   21D4 LEFT RIGHT DOUBLE ARROW
    "\u{21CE}" => "\u{21D4}",

    # 21CF RIGHTWARDS DOUBLE ARROW WITH STROKE
    # ->   21D2 RIGHTWARDS DOUBLE ARROW
    "\u{21CF}" => "\u{21D2}",

    # 21DE UPWARDS ARROW WITH DOUBLE STROKE
    # ->   2191 UPWARDS ARROW
    "\u{21DE}" => "\u{2191}",

    # 21DF DOWNWARDS ARROW WITH DOUBLE STROKE
    # ->   2193 DOWNWARDS ARROW
    "\u{21DF}" => "\u{2193}",

    # 21EC UPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BAR
    # ->   21EB UPWARDS WHITE ARROW ON PEDESTAL
    "\u{21EC}" => "\u{21EB}",

    # 21ED UPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BAR
    # ->   21EB UPWARDS WHITE ARROW ON PEDESTAL
    "\u{21ED}" => "\u{21EB}",

    # 21F7 LEFTWARDS ARROW WITH VERTICAL STROKE
    # ->   2190 LEFTWARDS ARROW
    "\u{21F7}" => "\u{2190}",

    # 21F8 RIGHTWARDS ARROW WITH VERTICAL STROKE
    # ->   2192 RIGHTWARDS ARROW
    "\u{21F8}" => "\u{2192}",

    # 21F9 LEFT RIGHT ARROW WITH VERTICAL STROKE
    # ->   2194 LEFT RIGHT ARROW
    "\u{21F9}" => "\u{2194}",

    # 21FA LEFTWARDS ARROW WITH DOUBLE VERTICAL STROKE
    # ->   2190 LEFTWARDS ARROW
    "\u{21FA}" => "\u{2190}",

    # 21FB RIGHTWARDS ARROW WITH DOUBLE VERTICAL STROKE
    # ->   2192 RIGHTWARDS ARROW
    "\u{21FB}" => "\u{2192}",

    # 21FC LEFT RIGHT ARROW WITH DOUBLE VERTICAL STROKE
    # ->   2194 LEFT RIGHT ARROW
    "\u{21FC}" => "\u{2194}",

    # 2204 THERE DOES NOT EXIST
    # ->   2203 THERE EXISTS
    "\u{2204}" => "\u{2203}",

    # 2209 NOT AN ELEMENT OF
    # ->   2208 ELEMENT OF
    "\u{2209}" => "\u{2208}",

    # 220C DOES NOT CONTAIN AS MEMBER
    # ->   220B CONTAINS AS MEMBER
    "\u{220C}" => "\u{220B}",

    # 2224 DOES NOT DIVIDE
    # ->   2223 DIVIDES
    "\u{2224}" => "\u{2223}",

    # 2226 NOT PARALLEL TO
    # ->   2225 PARALLEL TO
    "\u{2226}" => "\u{2225}",

    # 222C DOUBLE INTEGRAL
    # ->   222B INTEGRAL
    #    + 222B INTEGRAL
    "\u{222C}" => "\u{222B}\u{222B}",

    # 222D TRIPLE INTEGRAL
    # ->   222B INTEGRAL
    #    + 222B INTEGRAL
    #    + 222B INTEGRAL
    "\u{222D}" => "\u{222B}\u{222B}\u{222B}",

    # 222F SURFACE INTEGRAL
    # ->   222E CONTOUR INTEGRAL
    #    + 222E CONTOUR INTEGRAL
    "\u{222F}" => "\u{222E}\u{222E}",

    # 2230 VOLUME INTEGRAL
    # ->   222E CONTOUR INTEGRAL
    #    + 222E CONTOUR INTEGRAL
    #    + 222E CONTOUR INTEGRAL
    "\u{2230}" => "\u{222E}\u{222E}\u{222E}",

    # 2241 NOT TILDE
    # ->   223C TILDE OPERATOR
    "\u{2241}" => "\u{223C}",

    # 2244 NOT ASYMPTOTICALLY EQUAL TO
    # ->   2243 ASYMPTOTICALLY EQUAL TO
    "\u{2244}" => "\u{2243}",

    # 2247 NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO
    # ->   2245 APPROXIMATELY EQUAL TO
    "\u{2247}" => "\u{2245}",

    # 2249 NOT ALMOST EQUAL TO
    # ->   2248 ALMOST EQUAL TO
    "\u{2249}" => "\u{2248}",

    # 2260 NOT EQUAL TO
    # ->   003D EQUALS SIGN
    "\u{2260}" => "=",

    # 2262 NOT IDENTICAL TO
    # ->   2261 IDENTICAL TO
    "\u{2262}" => "\u{2261}",

    # 226D NOT EQUIVALENT TO
    # ->   224D EQUIVALENT TO
    "\u{226D}" => "\u{224D}",

    # 226E NOT LESS-THAN
    # ->   003C LESS-THAN SIGN
    "\u{226E}" => "<",

    # 226F NOT GREATER-THAN
    # ->   003E GREATER-THAN SIGN
    "\u{226F}" => ">",

    # 2270 NEITHER LESS-THAN NOR EQUAL TO
    # ->   2264 LESS-THAN OR EQUAL TO
    "\u{2270}" => "\u{2264}",

    # 2271 NEITHER GREATER-THAN NOR EQUAL TO
    # ->   2265 GREATER-THAN OR EQUAL TO
    "\u{2271}" => "\u{2265}",

    # 2274 NEITHER LESS-THAN NOR EQUIVALENT TO
    # ->   2272 LESS-THAN OR EQUIVALENT TO
    "\u{2274}" => "\u{2272}",

    # 2275 NEITHER GREATER-THAN NOR EQUIVALENT TO
    # ->   2273 GREATER-THAN OR EQUIVALENT TO
    "\u{2275}" => "\u{2273}",

    # 2278 NEITHER LESS-THAN NOR GREATER-THAN
    # ->   2276 LESS-THAN OR GREATER-THAN
    "\u{2278}" => "\u{2276}",

    # 2279 NEITHER GREATER-THAN NOR LESS-THAN
    # ->   2277 GREATER-THAN OR LESS-THAN
    "\u{2279}" => "\u{2277}",

    # 2280 DOES NOT PRECEDE
    # ->   227A PRECEDES
    "\u{2280}" => "\u{227A}",

    # 2281 DOES NOT SUCCEED
    # ->   227B SUCCEEDS
    "\u{2281}" => "\u{227B}",

    # 2284 NOT A SUBSET OF
    # ->   2282 SUBSET OF
    "\u{2284}" => "\u{2282}",

    # 2285 NOT A SUPERSET OF
    # ->   2283 SUPERSET OF
    "\u{2285}" => "\u{2283}",

    # 2288 NEITHER A SUBSET OF NOR EQUAL TO
    # ->   2286 SUBSET OF OR EQUAL TO
    "\u{2288}" => "\u{2286}",

    # 2289 NEITHER A SUPERSET OF NOR EQUAL TO
    # ->   2287 SUPERSET OF OR EQUAL TO
    "\u{2289}" => "\u{2287}",

    # 228A SUBSET OF WITH NOT EQUAL TO
    # ->   2282 SUBSET OF
    "\u{228A}" => "\u{2282}",

    # 228B SUPERSET OF WITH NOT EQUAL TO
    # ->   2283 SUPERSET OF
    "\u{228B}" => "\u{2283}",

    # 22AC DOES NOT PROVE
    # ->   22A2 RIGHT TACK
    "\u{22AC}" => "\u{22A2}",

    # 22AD NOT TRUE
    # ->   22A8 TRUE
    "\u{22AD}" => "\u{22A8}",

    # 22AE DOES NOT FORCE
    # ->   22A9 FORCES
    "\u{22AE}" => "\u{22A9}",

    # 22AF NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
    # ->   22AB DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
    "\u{22AF}" => "\u{22AB}",

    # 22BE RIGHT ANGLE WITH ARC
    # ->   221F RIGHT ANGLE
    "\u{22BE}" => "\u{221F}",

    # 22E0 DOES NOT PRECEDE OR EQUAL
    # ->   227C PRECEDES OR EQUAL TO
    "\u{22E0}" => "\u{227C}",

    # 22E1 DOES NOT SUCCEED OR EQUAL
    # ->   227D SUCCEEDS OR EQUAL TO
    "\u{22E1}" => "\u{227D}",

    # 22E2 NOT SQUARE IMAGE OF OR EQUAL TO
    # ->   2291 SQUARE IMAGE OF OR EQUAL TO
    "\u{22E2}" => "\u{2291}",

    # 22E3 NOT SQUARE ORIGINAL OF OR EQUAL TO
    # ->   2292 SQUARE ORIGINAL OF OR EQUAL TO
    "\u{22E3}" => "\u{2292}",

    # 22EA NOT NORMAL SUBGROUP OF
    # ->   22B2 NORMAL SUBGROUP OF
    "\u{22EA}" => "\u{22B2}",

    # 22EB DOES NOT CONTAIN AS NORMAL SUBGROUP
    # ->   22B3 CONTAINS AS NORMAL SUBGROUP
    "\u{22EB}" => "\u{22B3}",

    # 22EC NOT NORMAL SUBGROUP OF OR EQUAL TO
    # ->   22B4 NORMAL SUBGROUP OF OR EQUAL TO
    "\u{22EC}" => "\u{22B4}",

    # 22ED DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL
    # ->   22B5 CONTAINS AS NORMAL SUBGROUP OR EQUAL TO
    "\u{22ED}" => "\u{22B5}",

    # 22F2 ELEMENT OF WITH LONG HORIZONTAL STROKE
    # ->   2208 ELEMENT OF
    "\u{22F2}" => "\u{2208}",

    # 22F3 ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
    # ->   2208 ELEMENT OF
    "\u{22F3}" => "\u{2208}",

    # 22F4 SMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
    # ->   220A SMALL ELEMENT OF
    "\u{22F4}" => "\u{220A}",

    # 22F5 ELEMENT OF WITH DOT ABOVE
    # ->   2208 ELEMENT OF
    "\u{22F5}" => "\u{2208}",

    # 22F6 ELEMENT OF WITH OVERBAR
    # ->   2208 ELEMENT OF
    "\u{22F6}" => "\u{2208}",

    # 22F7 SMALL ELEMENT OF WITH OVERBAR
    # ->   220A SMALL ELEMENT OF
    "\u{22F7}" => "\u{220A}",

    # 22F8 ELEMENT OF WITH UNDERBAR
    # ->   2208 ELEMENT OF
    "\u{22F8}" => "\u{2208}",

    # 22F9 ELEMENT OF WITH TWO HORIZONTAL STROKES
    # ->   2208 ELEMENT OF
    "\u{22F9}" => "\u{2208}",

    # 2329 LEFT-POINTING ANGLE BRACKET
    # ->   3008 LEFT ANGLE BRACKET
    "\u{2329}" => "\u{3008}",

    # 232A RIGHT-POINTING ANGLE BRACKET
    # ->   3009 RIGHT ANGLE BRACKET
    "\u{232A}" => "\u{3009}",

    # 237C RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
    # ->   221F RIGHT ANGLE
    "\u{237C}" => "\u{221F}",

    # 237F VERTICAL LINE WITH MIDDLE DOT
    # ->   007C VERTICAL LINE
    "\u{237F}" => "|",

    # 2385 WHITE SQUARE WITH CENTRE VERTICAL LINE
    # ->   25A1 WHITE SQUARE
    "\u{2385}" => "\u{25A1}",

    # 23C1 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH CIRCLE
    # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
    "\u{23C1}" => "\u{23C9}",

    # 23C2 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH CIRCLE
    # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
    "\u{23C2}" => "\u{23CA}",

    # 23C4 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH TRIANGLE
    # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
    "\u{23C4}" => "\u{23C9}",

    # 23C5 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH TRIANGLE
    # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
    "\u{23C5}" => "\u{23CA}",

    # 23C7 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH WAVE
    # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
    "\u{23C7}" => "\u{23C9}",

    # 23C8 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH WAVE
    # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
    "\u{23C8}" => "\u{23CA}",

    # 2460 CIRCLED DIGIT ONE
    # ->   0031 DIGIT ONE
    "\u{2460}" => "1",

    # 2461 CIRCLED DIGIT TWO
    # ->   0032 DIGIT TWO
    "\u{2461}" => "2",

    # 2462 CIRCLED DIGIT THREE
    # ->   0033 DIGIT THREE
    "\u{2462}" => "3",

    # 2463 CIRCLED DIGIT FOUR
    # ->   0034 DIGIT FOUR
    "\u{2463}" => "4",

    # 2464 CIRCLED DIGIT FIVE
    # ->   0035 DIGIT FIVE
    "\u{2464}" => "5",

    # 2465 CIRCLED DIGIT SIX
    # ->   0036 DIGIT SIX
    "\u{2465}" => "6",

    # 2466 CIRCLED DIGIT SEVEN
    # ->   0037 DIGIT SEVEN
    "\u{2466}" => "7",

    # 2467 CIRCLED DIGIT EIGHT
    # ->   0038 DIGIT EIGHT
    "\u{2467}" => "8",

    # 2468 CIRCLED DIGIT NINE
    # ->   0039 DIGIT NINE
    "\u{2468}" => "9",

    # 2469 CIRCLED NUMBER TEN
    # ->   0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    "\u{2469}" => "10",

    # 246A CIRCLED NUMBER ELEVEN
    # ->   0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    "\u{246A}" => "11",

    # 246B CIRCLED NUMBER TWELVE
    # ->   0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    "\u{246B}" => "12",

    # 246C CIRCLED NUMBER THIRTEEN
    # ->   0031 DIGIT ONE
    #    + 0033 DIGIT THREE
    "\u{246C}" => "13",

    # 246D CIRCLED NUMBER FOURTEEN
    # ->   0031 DIGIT ONE
    #    + 0034 DIGIT FOUR
    "\u{246D}" => "14",

    # 246E CIRCLED NUMBER FIFTEEN
    # ->   0031 DIGIT ONE
    #    + 0035 DIGIT FIVE
    "\u{246E}" => "15",

    # 246F CIRCLED NUMBER SIXTEEN
    # ->   0031 DIGIT ONE
    #    + 0036 DIGIT SIX
    "\u{246F}" => "16",

    # 2470 CIRCLED NUMBER SEVENTEEN
    # ->   0031 DIGIT ONE
    #    + 0037 DIGIT SEVEN
    "\u{2470}" => "17",

    # 2471 CIRCLED NUMBER EIGHTEEN
    # ->   0031 DIGIT ONE
    #    + 0038 DIGIT EIGHT
    "\u{2471}" => "18",

    # 2472 CIRCLED NUMBER NINETEEN
    # ->   0031 DIGIT ONE
    #    + 0039 DIGIT NINE
    "\u{2472}" => "19",

    # 2473 CIRCLED NUMBER TWENTY
    # ->   0032 DIGIT TWO
    #    + 0030 DIGIT ZERO
    "\u{2473}" => "20",

    # 2474 PARENTHESIZED DIGIT ONE
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2474}" => "(1)",

    # 2475 PARENTHESIZED DIGIT TWO
    # ->   0028 LEFT PARENTHESIS
    #    + 0032 DIGIT TWO
    #    + 0029 RIGHT PARENTHESIS
    "\u{2475}" => "(2)",

    # 2476 PARENTHESIZED DIGIT THREE
    # ->   0028 LEFT PARENTHESIS
    #    + 0033 DIGIT THREE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2476}" => "(3)",

    # 2477 PARENTHESIZED DIGIT FOUR
    # ->   0028 LEFT PARENTHESIS
    #    + 0034 DIGIT FOUR
    #    + 0029 RIGHT PARENTHESIS
    "\u{2477}" => "(4)",

    # 2478 PARENTHESIZED DIGIT FIVE
    # ->   0028 LEFT PARENTHESIS
    #    + 0035 DIGIT FIVE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2478}" => "(5)",

    # 2479 PARENTHESIZED DIGIT SIX
    # ->   0028 LEFT PARENTHESIS
    #    + 0036 DIGIT SIX
    #    + 0029 RIGHT PARENTHESIS
    "\u{2479}" => "(6)",

    # 247A PARENTHESIZED DIGIT SEVEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0037 DIGIT SEVEN
    #    + 0029 RIGHT PARENTHESIS
    "\u{247A}" => "(7)",

    # 247B PARENTHESIZED DIGIT EIGHT
    # ->   0028 LEFT PARENTHESIS
    #    + 0038 DIGIT EIGHT
    #    + 0029 RIGHT PARENTHESIS
    "\u{247B}" => "(8)",

    # 247C PARENTHESIZED DIGIT NINE
    # ->   0028 LEFT PARENTHESIS
    #    + 0039 DIGIT NINE
    #    + 0029 RIGHT PARENTHESIS
    "\u{247C}" => "(9)",

    # 247D PARENTHESIZED NUMBER TEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    #    + 0029 RIGHT PARENTHESIS
    "\u{247D}" => "(10)",

    # 247E PARENTHESIZED NUMBER ELEVEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    #    + 0029 RIGHT PARENTHESIS
    "\u{247E}" => "(11)",

    # 247F PARENTHESIZED NUMBER TWELVE
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    #    + 0029 RIGHT PARENTHESIS
    "\u{247F}" => "(12)",

    # 2480 PARENTHESIZED NUMBER THIRTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0033 DIGIT THREE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2480}" => "(13)",

    # 2481 PARENTHESIZED NUMBER FOURTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0034 DIGIT FOUR
    #    + 0029 RIGHT PARENTHESIS
    "\u{2481}" => "(14)",

    # 2482 PARENTHESIZED NUMBER FIFTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0035 DIGIT FIVE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2482}" => "(15)",

    # 2483 PARENTHESIZED NUMBER SIXTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0036 DIGIT SIX
    #    + 0029 RIGHT PARENTHESIS
    "\u{2483}" => "(16)",

    # 2484 PARENTHESIZED NUMBER SEVENTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0037 DIGIT SEVEN
    #    + 0029 RIGHT PARENTHESIS
    "\u{2484}" => "(17)",

    # 2485 PARENTHESIZED NUMBER EIGHTEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0038 DIGIT EIGHT
    #    + 0029 RIGHT PARENTHESIS
    "\u{2485}" => "(18)",

    # 2486 PARENTHESIZED NUMBER NINETEEN
    # ->   0028 LEFT PARENTHESIS
    #    + 0031 DIGIT ONE
    #    + 0039 DIGIT NINE
    #    + 0029 RIGHT PARENTHESIS
    "\u{2486}" => "(19)",

    # 2487 PARENTHESIZED NUMBER TWENTY
    # ->   0028 LEFT PARENTHESIS
    #    + 0032 DIGIT TWO
    #    + 0030 DIGIT ZERO
    #    + 0029 RIGHT PARENTHESIS
    "\u{2487}" => "(20)",

    # 2488 DIGIT ONE FULL STOP
    # ->   0031 DIGIT ONE
    #    + 002E FULL STOP
    "\u{2488}" => "1.",

    # 2489 DIGIT TWO FULL STOP
    # ->   0032 DIGIT TWO
    #    + 002E FULL STOP
    "\u{2489}" => "2.",

    # 248A DIGIT THREE FULL STOP
    # ->   0033 DIGIT THREE
    #    + 002E FULL STOP
    "\u{248A}" => "3.",

    # 248B DIGIT FOUR FULL STOP
    # ->   0034 DIGIT FOUR
    #    + 002E FULL STOP
    "\u{248B}" => "4.",

    # 248C DIGIT FIVE FULL STOP
    # ->   0035 DIGIT FIVE
    #    + 002E FULL STOP
    "\u{248C}" => "5.",

    # 248D DIGIT SIX FULL STOP
    # ->   0036 DIGIT SIX
    #    + 002E FULL STOP
    "\u{248D}" => "6.",

    # 248E DIGIT SEVEN FULL STOP
    # ->   0037 DIGIT SEVEN
    #    + 002E FULL STOP
    "\u{248E}" => "7.",

    # 248F DIGIT EIGHT FULL STOP
    # ->   0038 DIGIT EIGHT
    #    + 002E FULL STOP
    "\u{248F}" => "8.",

    # 2490 DIGIT NINE FULL STOP
    # ->   0039 DIGIT NINE
    #    + 002E FULL STOP
    "\u{2490}" => "9.",

    # 2491 NUMBER TEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    #    + 002E FULL STOP
    "\u{2491}" => "10.",

    # 2492 NUMBER ELEVEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    #    + 002E FULL STOP
    "\u{2492}" => "11.",

    # 2493 NUMBER TWELVE FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    #    + 002E FULL STOP
    "\u{2493}" => "12.",

    # 2494 NUMBER THIRTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0033 DIGIT THREE
    #    + 002E FULL STOP
    "\u{2494}" => "13.",

    # 2495 NUMBER FOURTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0034 DIGIT FOUR
    #    + 002E FULL STOP
    "\u{2495}" => "14.",

    # 2496 NUMBER FIFTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0035 DIGIT FIVE
    #    + 002E FULL STOP
    "\u{2496}" => "15.",

    # 2497 NUMBER SIXTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0036 DIGIT SIX
    #    + 002E FULL STOP
    "\u{2497}" => "16.",

    # 2498 NUMBER SEVENTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0037 DIGIT SEVEN
    #    + 002E FULL STOP
    "\u{2498}" => "17.",

    # 2499 NUMBER EIGHTEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0038 DIGIT EIGHT
    #    + 002E FULL STOP
    "\u{2499}" => "18.",

    # 249A NUMBER NINETEEN FULL STOP
    # ->   0031 DIGIT ONE
    #    + 0039 DIGIT NINE
    #    + 002E FULL STOP
    "\u{249A}" => "19.",

    # 249B NUMBER TWENTY FULL STOP
    # ->   0032 DIGIT TWO
    #    + 0030 DIGIT ZERO
    #    + 002E FULL STOP
    "\u{249B}" => "20.",

    # 249C PARENTHESIZED LATIN SMALL LETTER A
    # ->   0028 LEFT PARENTHESIS
    #    + 0061 LATIN SMALL LETTER A
    #    + 0029 RIGHT PARENTHESIS
    "\u{249C}" => "(a)",

    # 249D PARENTHESIZED LATIN SMALL LETTER B
    # ->   0028 LEFT PARENTHESIS
    #    + 0062 LATIN SMALL LETTER B
    #    + 0029 RIGHT PARENTHESIS
    "\u{249D}" => "(b)",

    # 249E PARENTHESIZED LATIN SMALL LETTER C
    # ->   0028 LEFT PARENTHESIS
    #    + 0063 LATIN SMALL LETTER C
    #    + 0029 RIGHT PARENTHESIS
    "\u{249E}" => "(c)",

    # 249F PARENTHESIZED LATIN SMALL LETTER D
    # ->   0028 LEFT PARENTHESIS
    #    + 0064 LATIN SMALL LETTER D
    #    + 0029 RIGHT PARENTHESIS
    "\u{249F}" => "(d)",

    # 24A0 PARENTHESIZED LATIN SMALL LETTER E
    # ->   0028 LEFT PARENTHESIS
    #    + 0065 LATIN SMALL LETTER E
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A0}" => "(e)",

    # 24A1 PARENTHESIZED LATIN SMALL LETTER F
    # ->   0028 LEFT PARENTHESIS
    #    + 0066 LATIN SMALL LETTER F
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A1}" => "(f)",

    # 24A2 PARENTHESIZED LATIN SMALL LETTER G
    # ->   0028 LEFT PARENTHESIS
    #    + 0067 LATIN SMALL LETTER G
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A2}" => "(g)",

    # 24A3 PARENTHESIZED LATIN SMALL LETTER H
    # ->   0028 LEFT PARENTHESIS
    #    + 0068 LATIN SMALL LETTER H
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A3}" => "(h)",

    # 24A4 PARENTHESIZED LATIN SMALL LETTER I
    # ->   0028 LEFT PARENTHESIS
    #    + 0069 LATIN SMALL LETTER I
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A4}" => "(i)",

    # 24A5 PARENTHESIZED LATIN SMALL LETTER J
    # ->   0028 LEFT PARENTHESIS
    #    + 006A LATIN SMALL LETTER J
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A5}" => "(j)",

    # 24A6 PARENTHESIZED LATIN SMALL LETTER K
    # ->   0028 LEFT PARENTHESIS
    #    + 006B LATIN SMALL LETTER K
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A6}" => "(k)",

    # 24A7 PARENTHESIZED LATIN SMALL LETTER L
    # ->   0028 LEFT PARENTHESIS
    #    + 006C LATIN SMALL LETTER L
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A7}" => "(l)",

    # 24A8 PARENTHESIZED LATIN SMALL LETTER M
    # ->   0028 LEFT PARENTHESIS
    #    + 006D LATIN SMALL LETTER M
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A8}" => "(m)",

    # 24A9 PARENTHESIZED LATIN SMALL LETTER N
    # ->   0028 LEFT PARENTHESIS
    #    + 006E LATIN SMALL LETTER N
    #    + 0029 RIGHT PARENTHESIS
    "\u{24A9}" => "(n)",

    # 24AA PARENTHESIZED LATIN SMALL LETTER O
    # ->   0028 LEFT PARENTHESIS
    #    + 006F LATIN SMALL LETTER O
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AA}" => "(o)",

    # 24AB PARENTHESIZED LATIN SMALL LETTER P
    # ->   0028 LEFT PARENTHESIS
    #    + 0070 LATIN SMALL LETTER P
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AB}" => "(p)",

    # 24AC PARENTHESIZED LATIN SMALL LETTER Q
    # ->   0028 LEFT PARENTHESIS
    #    + 0071 LATIN SMALL LETTER Q
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AC}" => "(q)",

    # 24AD PARENTHESIZED LATIN SMALL LETTER R
    # ->   0028 LEFT PARENTHESIS
    #    + 0072 LATIN SMALL LETTER R
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AD}" => "(r)",

    # 24AE PARENTHESIZED LATIN SMALL LETTER S
    # ->   0028 LEFT PARENTHESIS
    #    + 0073 LATIN SMALL LETTER S
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AE}" => "(s)",

    # 24AF PARENTHESIZED LATIN SMALL LETTER T
    # ->   0028 LEFT PARENTHESIS
    #    + 0074 LATIN SMALL LETTER T
    #    + 0029 RIGHT PARENTHESIS
    "\u{24AF}" => "(t)",

    # 24B0 PARENTHESIZED LATIN SMALL LETTER U
    # ->   0028 LEFT PARENTHESIS
    #    + 0075 LATIN SMALL LETTER U
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B0}" => "(u)",

    # 24B1 PARENTHESIZED LATIN SMALL LETTER V
    # ->   0028 LEFT PARENTHESIS
    #    + 0076 LATIN SMALL LETTER V
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B1}" => "(v)",

    # 24B2 PARENTHESIZED LATIN SMALL LETTER W
    # ->   0028 LEFT PARENTHESIS
    #    + 0077 LATIN SMALL LETTER W
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B2}" => "(w)",

    # 24B3 PARENTHESIZED LATIN SMALL LETTER X
    # ->   0028 LEFT PARENTHESIS
    #    + 0078 LATIN SMALL LETTER X
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B3}" => "(x)",

    # 24B4 PARENTHESIZED LATIN SMALL LETTER Y
    # ->   0028 LEFT PARENTHESIS
    #    + 0079 LATIN SMALL LETTER Y
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B4}" => "(y)",

    # 24B5 PARENTHESIZED LATIN SMALL LETTER Z
    # ->   0028 LEFT PARENTHESIS
    #    + 007A LATIN SMALL LETTER Z
    #    + 0029 RIGHT PARENTHESIS
    "\u{24B5}" => "(z)",

    # 24B6 CIRCLED LATIN CAPITAL LETTER A
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{24B6}" => "A",

    # 24B7 CIRCLED LATIN CAPITAL LETTER B
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{24B7}" => "B",

    # 24B8 CIRCLED LATIN CAPITAL LETTER C
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{24B8}" => "C",

    # 24B9 CIRCLED LATIN CAPITAL LETTER D
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{24B9}" => "D",

    # 24BA CIRCLED LATIN CAPITAL LETTER E
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{24BA}" => "E",

    # 24BB CIRCLED LATIN CAPITAL LETTER F
    # ->   0046 LATIN CAPITAL LETTER F
    "\u{24BB}" => "F",

    # 24BC CIRCLED LATIN CAPITAL LETTER G
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{24BC}" => "G",

    # 24BD CIRCLED LATIN CAPITAL LETTER H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{24BD}" => "H",

    # 24BE CIRCLED LATIN CAPITAL LETTER I
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{24BE}" => "I",

    # 24BF CIRCLED LATIN CAPITAL LETTER J
    # ->   004A LATIN CAPITAL LETTER J
    "\u{24BF}" => "J",

    # 24C0 CIRCLED LATIN CAPITAL LETTER K
    # ->   004B LATIN CAPITAL LETTER K
    "\u{24C0}" => "K",

    # 24C1 CIRCLED LATIN CAPITAL LETTER L
    # ->   004C LATIN CAPITAL LETTER L
    "\u{24C1}" => "L",

    # 24C2 CIRCLED LATIN CAPITAL LETTER M
    # ->   004D LATIN CAPITAL LETTER M
    "\u{24C2}" => "M",

    # 24C3 CIRCLED LATIN CAPITAL LETTER N
    # ->   004E LATIN CAPITAL LETTER N
    "\u{24C3}" => "N",

    # 24C4 CIRCLED LATIN CAPITAL LETTER O
    # ->   004F LATIN CAPITAL LETTER O
    "\u{24C4}" => "O",

    # 24C5 CIRCLED LATIN CAPITAL LETTER P
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{24C5}" => "P",

    # 24C6 CIRCLED LATIN CAPITAL LETTER Q
    # ->   0051 LATIN CAPITAL LETTER Q
    "\u{24C6}" => "Q",

    # 24C7 CIRCLED LATIN CAPITAL LETTER R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{24C7}" => "R",

    # 24C8 CIRCLED LATIN CAPITAL LETTER S
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{24C8}" => "S",

    # 24C9 CIRCLED LATIN CAPITAL LETTER T
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{24C9}" => "T",

    # 24CA CIRCLED LATIN CAPITAL LETTER U
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{24CA}" => "U",

    # 24CB CIRCLED LATIN CAPITAL LETTER V
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{24CB}" => "V",

    # 24CC CIRCLED LATIN CAPITAL LETTER W
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{24CC}" => "W",

    # 24CD CIRCLED LATIN CAPITAL LETTER X
    # ->   0058 LATIN CAPITAL LETTER X
    "\u{24CD}" => "X",

    # 24CE CIRCLED LATIN CAPITAL LETTER Y
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{24CE}" => "Y",

    # 24CF CIRCLED LATIN CAPITAL LETTER Z
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{24CF}" => "Z",

    # 24D0 CIRCLED LATIN SMALL LETTER A
    # ->   0061 LATIN SMALL LETTER A
    "\u{24D0}" => "a",

    # 24D1 CIRCLED LATIN SMALL LETTER B
    # ->   0062 LATIN SMALL LETTER B
    "\u{24D1}" => "b",

    # 24D2 CIRCLED LATIN SMALL LETTER C
    # ->   0063 LATIN SMALL LETTER C
    "\u{24D2}" => "c",

    # 24D3 CIRCLED LATIN SMALL LETTER D
    # ->   0064 LATIN SMALL LETTER D
    "\u{24D3}" => "d",

    # 24D4 CIRCLED LATIN SMALL LETTER E
    # ->   0065 LATIN SMALL LETTER E
    "\u{24D4}" => "e",

    # 24D5 CIRCLED LATIN SMALL LETTER F
    # ->   0066 LATIN SMALL LETTER F
    "\u{24D5}" => "f",

    # 24D6 CIRCLED LATIN SMALL LETTER G
    # ->   0067 LATIN SMALL LETTER G
    "\u{24D6}" => "g",

    # 24D7 CIRCLED LATIN SMALL LETTER H
    # ->   0068 LATIN SMALL LETTER H
    "\u{24D7}" => "h",

    # 24D8 CIRCLED LATIN SMALL LETTER I
    # ->   0069 LATIN SMALL LETTER I
    "\u{24D8}" => "i",

    # 24D9 CIRCLED LATIN SMALL LETTER J
    # ->   006A LATIN SMALL LETTER J
    "\u{24D9}" => "j",

    # 24DA CIRCLED LATIN SMALL LETTER K
    # ->   006B LATIN SMALL LETTER K
    "\u{24DA}" => "k",

    # 24DB CIRCLED LATIN SMALL LETTER L
    # ->   006C LATIN SMALL LETTER L
    "\u{24DB}" => "l",

    # 24DC CIRCLED LATIN SMALL LETTER M
    # ->   006D LATIN SMALL LETTER M
    "\u{24DC}" => "m",

    # 24DD CIRCLED LATIN SMALL LETTER N
    # ->   006E LATIN SMALL LETTER N
    "\u{24DD}" => "n",

    # 24DE CIRCLED LATIN SMALL LETTER O
    # ->   006F LATIN SMALL LETTER O
    "\u{24DE}" => "o",

    # 24DF CIRCLED LATIN SMALL LETTER P
    # ->   0070 LATIN SMALL LETTER P
    "\u{24DF}" => "p",

    # 24E0 CIRCLED LATIN SMALL LETTER Q
    # ->   0071 LATIN SMALL LETTER Q
    "\u{24E0}" => "q",

    # 24E1 CIRCLED LATIN SMALL LETTER R
    # ->   0072 LATIN SMALL LETTER R
    "\u{24E1}" => "r",

    # 24E2 CIRCLED LATIN SMALL LETTER S
    # ->   0073 LATIN SMALL LETTER S
    "\u{24E2}" => "s",

    # 24E3 CIRCLED LATIN SMALL LETTER T
    # ->   0074 LATIN SMALL LETTER T
    "\u{24E3}" => "t",

    # 24E4 CIRCLED LATIN SMALL LETTER U
    # ->   0075 LATIN SMALL LETTER U
    "\u{24E4}" => "u",

    # 24E5 CIRCLED LATIN SMALL LETTER V
    # ->   0076 LATIN SMALL LETTER V
    "\u{24E5}" => "v",

    # 24E6 CIRCLED LATIN SMALL LETTER W
    # ->   0077 LATIN SMALL LETTER W
    "\u{24E6}" => "w",

    # 24E7 CIRCLED LATIN SMALL LETTER X
    # ->   0078 LATIN SMALL LETTER X
    "\u{24E7}" => "x",

    # 24E8 CIRCLED LATIN SMALL LETTER Y
    # ->   0079 LATIN SMALL LETTER Y
    "\u{24E8}" => "y",

    # 24E9 CIRCLED LATIN SMALL LETTER Z
    # ->   007A LATIN SMALL LETTER Z
    "\u{24E9}" => "z",

    # 24EA CIRCLED DIGIT ZERO
    # ->   0030 DIGIT ZERO
    "\u{24EA}" => "0",

    # 25A2 WHITE SQUARE WITH ROUNDED CORNERS
    # ->   25A1 WHITE SQUARE
    "\u{25A2}" => "\u{25A1}",

    # 25EB WHITE SQUARE WITH VERTICAL BISECTING LINE
    # ->   25A1 WHITE SQUARE
    "\u{25EB}" => "\u{25A1}",

    # 25EC WHITE UP-POINTING TRIANGLE WITH DOT
    # ->   25B3 WHITE UP-POINTING TRIANGLE
    "\u{25EC}" => "\u{25B3}",

    # 25F0 WHITE SQUARE WITH UPPER LEFT QUADRANT
    # ->   25A1 WHITE SQUARE
    "\u{25F0}" => "\u{25A1}",

    # 25F1 WHITE SQUARE WITH LOWER LEFT QUADRANT
    # ->   25A1 WHITE SQUARE
    "\u{25F1}" => "\u{25A1}",

    # 25F2 WHITE SQUARE WITH LOWER RIGHT QUADRANT
    # ->   25A1 WHITE SQUARE
    "\u{25F2}" => "\u{25A1}",

    # 25F3 WHITE SQUARE WITH UPPER RIGHT QUADRANT
    # ->   25A1 WHITE SQUARE
    "\u{25F3}" => "\u{25A1}",

    # 25F4 WHITE CIRCLE WITH UPPER LEFT QUADRANT
    # ->   25CB WHITE CIRCLE
    "\u{25F4}" => "\u{25CB}",

    # 25F5 WHITE CIRCLE WITH LOWER LEFT QUADRANT
    # ->   25CB WHITE CIRCLE
    "\u{25F5}" => "\u{25CB}",

    # 25F6 WHITE CIRCLE WITH LOWER RIGHT QUADRANT
    # ->   25CB WHITE CIRCLE
    "\u{25F6}" => "\u{25CB}",

    # 25F7 WHITE CIRCLE WITH UPPER RIGHT QUADRANT
    # ->   25CB WHITE CIRCLE
    "\u{25F7}" => "\u{25CB}",

    # 2611 BALLOT BOX WITH CHECK
    # ->   2610 BALLOT BOX
    "\u{2611}" => "\u{2610}",

    # 2612 BALLOT BOX WITH X
    # ->   2610 BALLOT BOX
    "\u{2612}" => "\u{2610}",

    # 2614 UMBRELLA WITH RAIN DROPS
    # ->   2602 UMBRELLA
    "\u{2614}" => "\u{2602}",

    # 2686 WHITE CIRCLE WITH DOT RIGHT
    # ->   25CB WHITE CIRCLE
    "\u{2686}" => "\u{25CB}",

    # 2687 WHITE CIRCLE WITH TWO DOTS
    # ->   25CB WHITE CIRCLE
    "\u{2687}" => "\u{25CB}",

    # 2688 BLACK CIRCLE WITH WHITE DOT RIGHT
    # ->   25CF BLACK CIRCLE
    "\u{2688}" => "\u{25CF}",

    # 2689 BLACK CIRCLE WITH TWO WHITE DOTS
    # ->   25CF BLACK CIRCLE
    "\u{2689}" => "\u{25CF}",

    # 27D0 WHITE DIAMOND WITH CENTRED DOT
    # ->   25C7 WHITE DIAMOND
    "\u{27D0}" => "\u{25C7}",

    # 27DF UP TACK WITH CIRCLE ABOVE
    # ->   22A5 UP TACK
    "\u{27DF}" => "\u{22A5}",

    # 27E2 WHITE CONCAVE-SIDED DIAMOND WITH LEFTWARDS TICK
    # ->   27E1 WHITE CONCAVE-SIDED DIAMOND
    "\u{27E2}" => "\u{27E1}",

    # 27E3 WHITE CONCAVE-SIDED DIAMOND WITH RIGHTWARDS TICK
    # ->   27E1 WHITE CONCAVE-SIDED DIAMOND
    "\u{27E3}" => "\u{27E1}",

    # 27E4 WHITE SQUARE WITH LEFTWARDS TICK
    # ->   25A1 WHITE SQUARE
    "\u{27E4}" => "\u{25A1}",

    # 27E5 WHITE SQUARE WITH RIGHTWARDS TICK
    # ->   25A1 WHITE SQUARE
    "\u{27E5}" => "\u{25A1}",

    # 2902 LEFTWARDS DOUBLE ARROW WITH VERTICAL STROKE
    # ->   21D0 LEFTWARDS DOUBLE ARROW
    "\u{2902}" => "\u{21D0}",

    # 2903 RIGHTWARDS DOUBLE ARROW WITH VERTICAL STROKE
    # ->   21D2 RIGHTWARDS DOUBLE ARROW
    "\u{2903}" => "\u{21D2}",

    # 2904 LEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKE
    # ->   21D4 LEFT RIGHT DOUBLE ARROW
    "\u{2904}" => "\u{21D4}",

    # 2908 DOWNWARDS ARROW WITH HORIZONTAL STROKE
    # ->   2193 DOWNWARDS ARROW
    "\u{2908}" => "\u{2193}",

    # 2909 UPWARDS ARROW WITH HORIZONTAL STROKE
    # ->   2191 UPWARDS ARROW
    "\u{2909}" => "\u{2191}",

    # 2911 RIGHTWARDS ARROW WITH DOTTED STEM
    # ->   2192 RIGHTWARDS ARROW
    "\u{2911}" => "\u{2192}",

    # 2914 RIGHTWARDS ARROW WITH TAIL WITH VERTICAL STROKE
    # ->   2192 RIGHTWARDS ARROW
    "\u{2914}" => "\u{2192}",

    # 2915 RIGHTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
    # ->   2192 RIGHTWARDS ARROW
    "\u{2915}" => "\u{2192}",

    # 2923 NORTH WEST ARROW WITH HOOK
    # ->   2196 NORTH WEST ARROW
    "\u{2923}" => "\u{2196}",

    # 2924 NORTH EAST ARROW WITH HOOK
    # ->   2197 NORTH EAST ARROW
    "\u{2924}" => "\u{2197}",

    # 2925 SOUTH EAST ARROW WITH HOOK
    # ->   2198 SOUTH EAST ARROW
    "\u{2925}" => "\u{2198}",

    # 2926 SOUTH WEST ARROW WITH HOOK
    # ->   2199 SOUTH WEST ARROW
    "\u{2926}" => "\u{2199}",

    # 293D TOP ARC ANTICLOCKWISE ARROW WITH PLUS
    # ->   293A TOP ARC ANTICLOCKWISE ARROW
    "\u{293D}" => "\u{293A}",

    # 2945 RIGHTWARDS ARROW WITH PLUS BELOW
    # ->   2192 RIGHTWARDS ARROW
    "\u{2945}" => "\u{2192}",

    # 2946 LEFTWARDS ARROW WITH PLUS BELOW
    # ->   2190 LEFTWARDS ARROW
    "\u{2946}" => "\u{2190}",

    # 298B LEFT SQUARE BRACKET WITH UNDERBAR
    # ->   005B LEFT SQUARE BRACKET
    "\u{298B}" => "[",

    # 298C RIGHT SQUARE BRACKET WITH UNDERBAR
    # ->   005D RIGHT SQUARE BRACKET
    "\u{298C}" => "]",

    # 298D LEFT SQUARE BRACKET WITH TICK IN TOP CORNER
    # ->   005B LEFT SQUARE BRACKET
    "\u{298D}" => "[",

    # 298E RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    # ->   005D RIGHT SQUARE BRACKET
    "\u{298E}" => "]",

    # 298F LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    # ->   005B LEFT SQUARE BRACKET
    "\u{298F}" => "[",

    # 2990 RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER
    # ->   005D RIGHT SQUARE BRACKET
    "\u{2990}" => "]",

    # 2991 LEFT ANGLE BRACKET WITH DOT
    # ->   3008 LEFT ANGLE BRACKET
    "\u{2991}" => "\u{3008}",

    # 2992 RIGHT ANGLE BRACKET WITH DOT
    # ->   3009 RIGHT ANGLE BRACKET
    "\u{2992}" => "\u{3009}",

    # 299E ANGLE WITH S INSIDE
    # ->   2220 ANGLE
    "\u{299E}" => "\u{2220}",

    # 29A4 ANGLE WITH UNDERBAR
    # ->   2220 ANGLE
    "\u{29A4}" => "\u{2220}",

    # 29A5 REVERSED ANGLE WITH UNDERBAR
    # ->   29A3 REVERSED ANGLE
    "\u{29A5}" => "\u{29A3}",

    # 29A8 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHT
    # ->   2221 MEASURED ANGLE
    "\u{29A8}" => "\u{2221}",

    # 29A9 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFT
    # ->   2221 MEASURED ANGLE
    "\u{29A9}" => "\u{2221}",

    # 29AA MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHT
    # ->   2221 MEASURED ANGLE
    "\u{29AA}" => "\u{2221}",

    # 29AB MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND LEFT
    # ->   2221 MEASURED ANGLE
    "\u{29AB}" => "\u{2221}",

    # 29AC MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UP
    # ->   2221 MEASURED ANGLE
    "\u{29AC}" => "\u{2221}",

    # 29AD MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UP
    # ->   2221 MEASURED ANGLE
    "\u{29AD}" => "\u{2221}",

    # 29AE MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWN
    # ->   2221 MEASURED ANGLE
    "\u{29AE}" => "\u{2221}",

    # 29AF MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND DOWN
    # ->   2221 MEASURED ANGLE
    "\u{29AF}" => "\u{2221}",

    # 29B1 EMPTY SET WITH OVERBAR
    # ->   2205 EMPTY SET
    "\u{29B1}" => "\u{2205}",

    # 29B2 EMPTY SET WITH SMALL CIRCLE ABOVE
    # ->   2205 EMPTY SET
    "\u{29B2}" => "\u{2205}",

    # 29B3 EMPTY SET WITH RIGHT ARROW ABOVE
    # ->   2205 EMPTY SET
    "\u{29B3}" => "\u{2205}",

    # 29B4 EMPTY SET WITH LEFT ARROW ABOVE
    # ->   2205 EMPTY SET
    "\u{29B4}" => "\u{2205}",

    # 29D1 BOWTIE WITH LEFT HALF BLACK
    # ->   22C8 BOWTIE
    "\u{29D1}" => "\u{22C8}",

    # 29D2 BOWTIE WITH RIGHT HALF BLACK
    # ->   22C8 BOWTIE
    "\u{29D2}" => "\u{22C8}",

    # 29E4 EQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVE
    # ->   29E3 EQUALS SIGN AND SLANTED PARALLEL
    "\u{29E4}" => "\u{29E3}",

    # 29EA BLACK DIAMOND WITH DOWN ARROW
    # ->   25C6 BLACK DIAMOND
    "\u{29EA}" => "\u{25C6}",

    # 29EC WHITE CIRCLE WITH DOWN ARROW
    # ->   25CB WHITE CIRCLE
    "\u{29EC}" => "\u{25CB}",

    # 29ED BLACK CIRCLE WITH DOWN ARROW
    # ->   25CF BLACK CIRCLE
    "\u{29ED}" => "\u{25CF}",

    # 29F6 SOLIDUS WITH OVERBAR
    # ->   002F SOLIDUS
    "\u{29F6}" => "/",

    # 29F7 REVERSE SOLIDUS WITH HORIZONTAL STROKE
    # ->   005C REVERSE SOLIDUS
    "\u{29F7}" => "\\",

    # 2A0C QUADRUPLE INTEGRAL OPERATOR
    # ->   222B INTEGRAL
    #    + 222B INTEGRAL
    #    + 222B INTEGRAL
    #    + 222B INTEGRAL
    "\u{2A0C}" => "\u{222B}\u{222B}\u{222B}\u{222B}",

    # 2A0E INTEGRAL WITH DOUBLE STROKE
    # ->   222B INTEGRAL
    "\u{2A0E}" => "\u{222B}",

    # 2A17 INTEGRAL WITH LEFTWARDS ARROW WITH HOOK
    # ->   222B INTEGRAL
    "\u{2A17}" => "\u{222B}",

    # 2A18 INTEGRAL WITH TIMES SIGN
    # ->   222B INTEGRAL
    "\u{2A18}" => "\u{222B}",

    # 2A19 INTEGRAL WITH INTERSECTION
    # ->   222B INTEGRAL
    "\u{2A19}" => "\u{222B}",

    # 2A1A INTEGRAL WITH UNION
    # ->   222B INTEGRAL
    "\u{2A1A}" => "\u{222B}",

    # 2A1B INTEGRAL WITH OVERBAR
    # ->   222B INTEGRAL
    "\u{2A1B}" => "\u{222B}",

    # 2A1C INTEGRAL WITH UNDERBAR
    # ->   222B INTEGRAL
    "\u{2A1C}" => "\u{222B}",

    # 2A22 PLUS SIGN WITH SMALL CIRCLE ABOVE
    # ->   002B PLUS SIGN
    "\u{2A22}" => "+",

    # 2A23 PLUS SIGN WITH CIRCUMFLEX ACCENT ABOVE
    # ->   002B PLUS SIGN
    "\u{2A23}" => "+",

    # 2A24 PLUS SIGN WITH TILDE ABOVE
    # ->   002B PLUS SIGN
    "\u{2A24}" => "+",

    # 2A25 PLUS SIGN WITH DOT BELOW
    # ->   002B PLUS SIGN
    "\u{2A25}" => "+",

    # 2A26 PLUS SIGN WITH TILDE BELOW
    # ->   002B PLUS SIGN
    "\u{2A26}" => "+",

    # 2A27 PLUS SIGN WITH SUBSCRIPT TWO
    # ->   002B PLUS SIGN
    "\u{2A27}" => "+",

    # 2A28 PLUS SIGN WITH BLACK TRIANGLE
    # ->   002B PLUS SIGN
    "\u{2A28}" => "+",

    # 2A29 MINUS SIGN WITH COMMA ABOVE
    # ->   2212 MINUS SIGN
    "\u{2A29}" => "\u{2212}",

    # 2A2A MINUS SIGN WITH DOT BELOW
    # ->   2212 MINUS SIGN
    "\u{2A2A}" => "\u{2212}",

    # 2A2B MINUS SIGN WITH FALLING DOTS
    # ->   2212 MINUS SIGN
    "\u{2A2B}" => "\u{2212}",

    # 2A2C MINUS SIGN WITH RISING DOTS
    # ->   2212 MINUS SIGN
    "\u{2A2C}" => "\u{2212}",

    # 2A30 MULTIPLICATION SIGN WITH DOT ABOVE
    # ->   00D7 MULTIPLICATION SIGN
    "\u{2A30}" => "\u{D7}",

    # 2A31 MULTIPLICATION SIGN WITH UNDERBAR
    # ->   00D7 MULTIPLICATION SIGN
    "\u{2A31}" => "\u{D7}",

    # 2A40 INTERSECTION WITH DOT
    # ->   2229 INTERSECTION
    "\u{2A40}" => "\u{2229}",

    # 2A41 UNION WITH MINUS SIGN
    # ->   222A UNION
    "\u{2A41}" => "\u{222A}",

    # 2A42 UNION WITH OVERBAR
    # ->   222A UNION
    "\u{2A42}" => "\u{222A}",

    # 2A43 INTERSECTION WITH OVERBAR
    # ->   2229 INTERSECTION
    "\u{2A43}" => "\u{2229}",

    # 2A44 INTERSECTION WITH LOGICAL AND
    # ->   2229 INTERSECTION
    "\u{2A44}" => "\u{2229}",

    # 2A45 UNION WITH LOGICAL OR
    # ->   222A UNION
    "\u{2A45}" => "\u{222A}",

    # 2A51 LOGICAL AND WITH DOT ABOVE
    # ->   2227 LOGICAL AND
    "\u{2A51}" => "\u{2227}",

    # 2A52 LOGICAL OR WITH DOT ABOVE
    # ->   2228 LOGICAL OR
    "\u{2A52}" => "\u{2228}",

    # 2A5A LOGICAL AND WITH MIDDLE STEM
    # ->   2227 LOGICAL AND
    "\u{2A5A}" => "\u{2227}",

    # 2A5B LOGICAL OR WITH MIDDLE STEM
    # ->   2228 LOGICAL OR
    "\u{2A5B}" => "\u{2228}",

    # 2A5C LOGICAL AND WITH HORIZONTAL DASH
    # ->   2227 LOGICAL AND
    "\u{2A5C}" => "\u{2227}",

    # 2A5D LOGICAL OR WITH HORIZONTAL DASH
    # ->   2228 LOGICAL OR
    "\u{2A5D}" => "\u{2228}",

    # 2A5E LOGICAL AND WITH DOUBLE OVERBAR
    # ->   2227 LOGICAL AND
    "\u{2A5E}" => "\u{2227}",

    # 2A5F LOGICAL AND WITH UNDERBAR
    # ->   2227 LOGICAL AND
    "\u{2A5F}" => "\u{2227}",

    # 2A60 LOGICAL AND WITH DOUBLE UNDERBAR
    # ->   2227 LOGICAL AND
    "\u{2A60}" => "\u{2227}",

    # 2A62 LOGICAL OR WITH DOUBLE OVERBAR
    # ->   2228 LOGICAL OR
    "\u{2A62}" => "\u{2228}",

    # 2A63 LOGICAL OR WITH DOUBLE UNDERBAR
    # ->   2228 LOGICAL OR
    "\u{2A63}" => "\u{2228}",

    # 2A66 EQUALS SIGN WITH DOT BELOW
    # ->   003D EQUALS SIGN
    "\u{2A66}" => "=",

    # 2A6A TILDE OPERATOR WITH DOT ABOVE
    # ->   223C TILDE OPERATOR
    "\u{2A6A}" => "\u{223C}",

    # 2A6B TILDE OPERATOR WITH RISING DOTS
    # ->   223C TILDE OPERATOR
    "\u{2A6B}" => "\u{223C}",

    # 2A6F ALMOST EQUAL TO WITH CIRCUMFLEX ACCENT
    # ->   2248 ALMOST EQUAL TO
    "\u{2A6F}" => "\u{2248}",

    # 2A74 DOUBLE COLON EQUAL
    # ->   003A COLON
    #    + 003A COLON
    #    + 003D EQUALS SIGN
    "\u{2A74}" => "::=",

    # 2A75 TWO CONSECUTIVE EQUALS SIGNS
    # ->   003D EQUALS SIGN
    #    + 003D EQUALS SIGN
    "\u{2A75}" => "==",

    # 2A76 THREE CONSECUTIVE EQUALS SIGNS
    # ->   003D EQUALS SIGN
    #    + 003D EQUALS SIGN
    #    + 003D EQUALS SIGN
    "\u{2A76}" => "===",

    # 2A77 EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW
    # ->   003D EQUALS SIGN
    "\u{2A77}" => "=",

    # 2A7F LESS-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
    # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
    "\u{2A7F}" => "\u{2A7D}",

    # 2A80 GREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
    # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
    "\u{2A80}" => "\u{2A7E}",

    # 2A81 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
    # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
    "\u{2A81}" => "\u{2A7D}",

    # 2A82 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
    # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
    "\u{2A82}" => "\u{2A7E}",

    # 2A83 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHT
    # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
    "\u{2A83}" => "\u{2A7D}",

    # 2A84 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE LEFT
    # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
    "\u{2A84}" => "\u{2A7E}",

    # 2A97 SLANTED EQUAL TO OR LESS-THAN WITH DOT INSIDE
    # ->   2A95 SLANTED EQUAL TO OR LESS-THAN
    "\u{2A97}" => "\u{2A95}",

    # 2A98 SLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDE
    # ->   2A96 SLANTED EQUAL TO OR GREATER-THAN
    "\u{2A98}" => "\u{2A96}",

    # 2AA3 DOUBLE NESTED LESS-THAN WITH UNDERBAR
    # ->   2AA1 DOUBLE NESTED LESS-THAN
    "\u{2AA3}" => "\u{2AA1}",

    # 2AAE EQUALS SIGN WITH BUMPY ABOVE
    # ->   003D EQUALS SIGN
    "\u{2AAE}" => "=",

    # 2AC3 SUBSET OF OR EQUAL TO WITH DOT ABOVE
    # ->   2286 SUBSET OF OR EQUAL TO
    "\u{2AC3}" => "\u{2286}",

    # 2AC4 SUPERSET OF OR EQUAL TO WITH DOT ABOVE
    # ->   2287 SUPERSET OF OR EQUAL TO
    "\u{2AC4}" => "\u{2287}",

    # 2ADA PITCHFORK WITH TEE TOP
    # ->   22D4 PITCHFORK
    "\u{2ADA}" => "\u{22D4}",

    # 2ADC FORKING
    # ->   2ADD NONFORKING
    "\u{2ADC}" => "\u{2ADD}",

    # 2AE7 SHORT DOWN TACK WITH OVERBAR
    # ->   2ADF SHORT DOWN TACK
    "\u{2AE7}" => "\u{2ADF}",

    # 2AE8 SHORT UP TACK WITH UNDERBAR
    # ->   2AE0 SHORT UP TACK
    "\u{2AE8}" => "\u{2AE0}",

    # 2AEE DOES NOT DIVIDE WITH REVERSED NEGATION SLASH
    # ->   2223 DIVIDES
    "\u{2AEE}" => "\u{2223}",

    # 2AEF VERTICAL LINE WITH CIRCLE ABOVE
    # ->   007C VERTICAL LINE
    "\u{2AEF}" => "|",

    # 2AF0 VERTICAL LINE WITH CIRCLE BELOW
    # ->   007C VERTICAL LINE
    "\u{2AF0}" => "|",

    # 2AF1 DOWN TACK WITH CIRCLE BELOW
    # ->   22A4 DOWN TACK
    "\u{2AF1}" => "\u{22A4}",

    # 2E9F CJK RADICAL MOTHER
    # ->   6BCD CJK UNIFIED IDEOGRAPH 6BCD
    "\u{2E9F}" => "\u{6BCD}",

    # 2EF3 CJK RADICAL C-SIMPLIFIED TURTLE
    # ->   9F9F CJK UNIFIED IDEOGRAPH 9F9F
    "\u{2EF3}" => "\u{9F9F}",

    # 2F00 KANGXI RADICAL ONE
    # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
    "\u{2F00}" => "\u{4E00}",

    # 2F01 KANGXI RADICAL LINE
    # ->   4E28 CJK UNIFIED IDEOGRAPH 4E28
    "\u{2F01}" => "\u{4E28}",

    # 2F02 KANGXI RADICAL DOT
    # ->   4E36 CJK UNIFIED IDEOGRAPH 4E36
    "\u{2F02}" => "\u{4E36}",

    # 2F03 KANGXI RADICAL SLASH
    # ->   4E3F CJK UNIFIED IDEOGRAPH 4E3F
    "\u{2F03}" => "\u{4E3F}",

    # 2F04 KANGXI RADICAL SECOND
    # ->   4E59 CJK UNIFIED IDEOGRAPH 4E59
    "\u{2F04}" => "\u{4E59}",

    # 2F05 KANGXI RADICAL HOOK
    # ->   4E85 CJK UNIFIED IDEOGRAPH 4E85
    "\u{2F05}" => "\u{4E85}",

    # 2F06 KANGXI RADICAL TWO
    # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
    "\u{2F06}" => "\u{4E8C}",

    # 2F07 KANGXI RADICAL LID
    # ->   4EA0 CJK UNIFIED IDEOGRAPH 4EA0
    "\u{2F07}" => "\u{4EA0}",

    # 2F08 KANGXI RADICAL MAN
    # ->   4EBA CJK UNIFIED IDEOGRAPH 4EBA
    "\u{2F08}" => "\u{4EBA}",

    # 2F09 KANGXI RADICAL LEGS
    # ->   513F CJK UNIFIED IDEOGRAPH 513F
    "\u{2F09}" => "\u{513F}",

    # 2F0A KANGXI RADICAL ENTER
    # ->   5165 CJK UNIFIED IDEOGRAPH 5165
    "\u{2F0A}" => "\u{5165}",

    # 2F0B KANGXI RADICAL EIGHT
    # ->   516B CJK UNIFIED IDEOGRAPH 516B
    "\u{2F0B}" => "\u{516B}",

    # 2F0C KANGXI RADICAL DOWN BOX
    # ->   5182 CJK UNIFIED IDEOGRAPH 5182
    "\u{2F0C}" => "\u{5182}",

    # 2F0D KANGXI RADICAL COVER
    # ->   5196 CJK UNIFIED IDEOGRAPH 5196
    "\u{2F0D}" => "\u{5196}",

    # 2F0E KANGXI RADICAL ICE
    # ->   51AB CJK UNIFIED IDEOGRAPH 51AB
    "\u{2F0E}" => "\u{51AB}",

    # 2F0F KANGXI RADICAL TABLE
    # ->   51E0 CJK UNIFIED IDEOGRAPH 51E0
    "\u{2F0F}" => "\u{51E0}",

    # 2F10 KANGXI RADICAL OPEN BOX
    # ->   51F5 CJK UNIFIED IDEOGRAPH 51F5
    "\u{2F10}" => "\u{51F5}",

    # 2F11 KANGXI RADICAL KNIFE
    # ->   5200 CJK UNIFIED IDEOGRAPH 5200
    "\u{2F11}" => "\u{5200}",

    # 2F12 KANGXI RADICAL POWER
    # ->   529B CJK UNIFIED IDEOGRAPH 529B
    "\u{2F12}" => "\u{529B}",

    # 2F13 KANGXI RADICAL WRAP
    # ->   52F9 CJK UNIFIED IDEOGRAPH 52F9
    "\u{2F13}" => "\u{52F9}",

    # 2F14 KANGXI RADICAL SPOON
    # ->   5315 CJK UNIFIED IDEOGRAPH 5315
    "\u{2F14}" => "\u{5315}",

    # 2F15 KANGXI RADICAL RIGHT OPEN BOX
    # ->   531A CJK UNIFIED IDEOGRAPH 531A
    "\u{2F15}" => "\u{531A}",

    # 2F16 KANGXI RADICAL HIDING ENCLOSURE
    # ->   5338 CJK UNIFIED IDEOGRAPH 5338
    "\u{2F16}" => "\u{5338}",

    # 2F17 KANGXI RADICAL TEN
    # ->   5341 CJK UNIFIED IDEOGRAPH 5341
    "\u{2F17}" => "\u{5341}",

    # 2F18 KANGXI RADICAL DIVINATION
    # ->   535C CJK UNIFIED IDEOGRAPH 535C
    "\u{2F18}" => "\u{535C}",

    # 2F19 KANGXI RADICAL SEAL
    # ->   5369 CJK UNIFIED IDEOGRAPH 5369
    "\u{2F19}" => "\u{5369}",

    # 2F1A KANGXI RADICAL CLIFF
    # ->   5382 CJK UNIFIED IDEOGRAPH 5382
    "\u{2F1A}" => "\u{5382}",

    # 2F1B KANGXI RADICAL PRIVATE
    # ->   53B6 CJK UNIFIED IDEOGRAPH 53B6
    "\u{2F1B}" => "\u{53B6}",

    # 2F1C KANGXI RADICAL AGAIN
    # ->   53C8 CJK UNIFIED IDEOGRAPH 53C8
    "\u{2F1C}" => "\u{53C8}",

    # 2F1D KANGXI RADICAL MOUTH
    # ->   53E3 CJK UNIFIED IDEOGRAPH 53E3
    "\u{2F1D}" => "\u{53E3}",

    # 2F1E KANGXI RADICAL ENCLOSURE
    # ->   56D7 CJK UNIFIED IDEOGRAPH 56D7
    "\u{2F1E}" => "\u{56D7}",

    # 2F1F KANGXI RADICAL EARTH
    # ->   571F CJK UNIFIED IDEOGRAPH 571F
    "\u{2F1F}" => "\u{571F}",

    # 2F20 KANGXI RADICAL SCHOLAR
    # ->   58EB CJK UNIFIED IDEOGRAPH 58EB
    "\u{2F20}" => "\u{58EB}",

    # 2F21 KANGXI RADICAL GO
    # ->   5902 CJK UNIFIED IDEOGRAPH 5902
    "\u{2F21}" => "\u{5902}",

    # 2F22 KANGXI RADICAL GO SLOWLY
    # ->   590A CJK UNIFIED IDEOGRAPH 590A
    "\u{2F22}" => "\u{590A}",

    # 2F23 KANGXI RADICAL EVENING
    # ->   5915 CJK UNIFIED IDEOGRAPH 5915
    "\u{2F23}" => "\u{5915}",

    # 2F24 KANGXI RADICAL BIG
    # ->   5927 CJK UNIFIED IDEOGRAPH 5927
    "\u{2F24}" => "\u{5927}",

    # 2F25 KANGXI RADICAL WOMAN
    # ->   5973 CJK UNIFIED IDEOGRAPH 5973
    "\u{2F25}" => "\u{5973}",

    # 2F26 KANGXI RADICAL CHILD
    # ->   5B50 CJK UNIFIED IDEOGRAPH 5B50
    "\u{2F26}" => "\u{5B50}",

    # 2F27 KANGXI RADICAL ROOF
    # ->   5B80 CJK UNIFIED IDEOGRAPH 5B80
    "\u{2F27}" => "\u{5B80}",

    # 2F28 KANGXI RADICAL INCH
    # ->   5BF8 CJK UNIFIED IDEOGRAPH 5BF8
    "\u{2F28}" => "\u{5BF8}",

    # 2F29 KANGXI RADICAL SMALL
    # ->   5C0F CJK UNIFIED IDEOGRAPH 5C0F
    "\u{2F29}" => "\u{5C0F}",

    # 2F2A KANGXI RADICAL LAME
    # ->   5C22 CJK UNIFIED IDEOGRAPH 5C22
    "\u{2F2A}" => "\u{5C22}",

    # 2F2B KANGXI RADICAL CORPSE
    # ->   5C38 CJK UNIFIED IDEOGRAPH 5C38
    "\u{2F2B}" => "\u{5C38}",

    # 2F2C KANGXI RADICAL SPROUT
    # ->   5C6E CJK UNIFIED IDEOGRAPH 5C6E
    "\u{2F2C}" => "\u{5C6E}",

    # 2F2D KANGXI RADICAL MOUNTAIN
    # ->   5C71 CJK UNIFIED IDEOGRAPH 5C71
    "\u{2F2D}" => "\u{5C71}",

    # 2F2E KANGXI RADICAL RIVER
    # ->   5DDB CJK UNIFIED IDEOGRAPH 5DDB
    "\u{2F2E}" => "\u{5DDB}",

    # 2F2F KANGXI RADICAL WORK
    # ->   5DE5 CJK UNIFIED IDEOGRAPH 5DE5
    "\u{2F2F}" => "\u{5DE5}",

    # 2F30 KANGXI RADICAL ONESELF
    # ->   5DF1 CJK UNIFIED IDEOGRAPH 5DF1
    "\u{2F30}" => "\u{5DF1}",

    # 2F31 KANGXI RADICAL TURBAN
    # ->   5DFE CJK UNIFIED IDEOGRAPH 5DFE
    "\u{2F31}" => "\u{5DFE}",

    # 2F32 KANGXI RADICAL DRY
    # ->   5E72 CJK UNIFIED IDEOGRAPH 5E72
    "\u{2F32}" => "\u{5E72}",

    # 2F33 KANGXI RADICAL SHORT THREAD
    # ->   5E7A CJK UNIFIED IDEOGRAPH 5E7A
    "\u{2F33}" => "\u{5E7A}",

    # 2F34 KANGXI RADICAL DOTTED CLIFF
    # ->   5E7F CJK UNIFIED IDEOGRAPH 5E7F
    "\u{2F34}" => "\u{5E7F}",

    # 2F35 KANGXI RADICAL LONG STRIDE
    # ->   5EF4 CJK UNIFIED IDEOGRAPH 5EF4
    "\u{2F35}" => "\u{5EF4}",

    # 2F36 KANGXI RADICAL TWO HANDS
    # ->   5EFE CJK UNIFIED IDEOGRAPH 5EFE
    "\u{2F36}" => "\u{5EFE}",

    # 2F37 KANGXI RADICAL SHOOT
    # ->   5F0B CJK UNIFIED IDEOGRAPH 5F0B
    "\u{2F37}" => "\u{5F0B}",

    # 2F38 KANGXI RADICAL BOW
    # ->   5F13 CJK UNIFIED IDEOGRAPH 5F13
    "\u{2F38}" => "\u{5F13}",

    # 2F39 KANGXI RADICAL SNOUT
    # ->   5F50 CJK UNIFIED IDEOGRAPH 5F50
    "\u{2F39}" => "\u{5F50}",

    # 2F3A KANGXI RADICAL BRISTLE
    # ->   5F61 CJK UNIFIED IDEOGRAPH 5F61
    "\u{2F3A}" => "\u{5F61}",

    # 2F3B KANGXI RADICAL STEP
    # ->   5F73 CJK UNIFIED IDEOGRAPH 5F73
    "\u{2F3B}" => "\u{5F73}",

    # 2F3C KANGXI RADICAL HEART
    # ->   5FC3 CJK UNIFIED IDEOGRAPH 5FC3
    "\u{2F3C}" => "\u{5FC3}",

    # 2F3D KANGXI RADICAL HALBERD
    # ->   6208 CJK UNIFIED IDEOGRAPH 6208
    "\u{2F3D}" => "\u{6208}",

    # 2F3E KANGXI RADICAL DOOR
    # ->   6236 CJK UNIFIED IDEOGRAPH 6236
    "\u{2F3E}" => "\u{6236}",

    # 2F3F KANGXI RADICAL HAND
    # ->   624B CJK UNIFIED IDEOGRAPH 624B
    "\u{2F3F}" => "\u{624B}",

    # 2F40 KANGXI RADICAL BRANCH
    # ->   652F CJK UNIFIED IDEOGRAPH 652F
    "\u{2F40}" => "\u{652F}",

    # 2F41 KANGXI RADICAL RAP
    # ->   6534 CJK UNIFIED IDEOGRAPH 6534
    "\u{2F41}" => "\u{6534}",

    # 2F42 KANGXI RADICAL SCRIPT
    # ->   6587 CJK UNIFIED IDEOGRAPH 6587
    "\u{2F42}" => "\u{6587}",

    # 2F43 KANGXI RADICAL DIPPER
    # ->   6597 CJK UNIFIED IDEOGRAPH 6597
    "\u{2F43}" => "\u{6597}",

    # 2F44 KANGXI RADICAL AXE
    # ->   65A4 CJK UNIFIED IDEOGRAPH 65A4
    "\u{2F44}" => "\u{65A4}",

    # 2F45 KANGXI RADICAL SQUARE
    # ->   65B9 CJK UNIFIED IDEOGRAPH 65B9
    "\u{2F45}" => "\u{65B9}",

    # 2F46 KANGXI RADICAL NOT
    # ->   65E0 CJK UNIFIED IDEOGRAPH 65E0
    "\u{2F46}" => "\u{65E0}",

    # 2F47 KANGXI RADICAL SUN
    # ->   65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{2F47}" => "\u{65E5}",

    # 2F48 KANGXI RADICAL SAY
    # ->   66F0 CJK UNIFIED IDEOGRAPH 66F0
    "\u{2F48}" => "\u{66F0}",

    # 2F49 KANGXI RADICAL MOON
    # ->   6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{2F49}" => "\u{6708}",

    # 2F4A KANGXI RADICAL TREE
    # ->   6728 CJK UNIFIED IDEOGRAPH 6728
    "\u{2F4A}" => "\u{6728}",

    # 2F4B KANGXI RADICAL LACK
    # ->   6B20 CJK UNIFIED IDEOGRAPH 6B20
    "\u{2F4B}" => "\u{6B20}",

    # 2F4C KANGXI RADICAL STOP
    # ->   6B62 CJK UNIFIED IDEOGRAPH 6B62
    "\u{2F4C}" => "\u{6B62}",

    # 2F4D KANGXI RADICAL DEATH
    # ->   6B79 CJK UNIFIED IDEOGRAPH 6B79
    "\u{2F4D}" => "\u{6B79}",

    # 2F4E KANGXI RADICAL WEAPON
    # ->   6BB3 CJK UNIFIED IDEOGRAPH 6BB3
    "\u{2F4E}" => "\u{6BB3}",

    # 2F4F KANGXI RADICAL DO NOT
    # ->   6BCB CJK UNIFIED IDEOGRAPH 6BCB
    "\u{2F4F}" => "\u{6BCB}",

    # 2F50 KANGXI RADICAL COMPARE
    # ->   6BD4 CJK UNIFIED IDEOGRAPH 6BD4
    "\u{2F50}" => "\u{6BD4}",

    # 2F51 KANGXI RADICAL FUR
    # ->   6BDB CJK UNIFIED IDEOGRAPH 6BDB
    "\u{2F51}" => "\u{6BDB}",

    # 2F52 KANGXI RADICAL CLAN
    # ->   6C0F CJK UNIFIED IDEOGRAPH 6C0F
    "\u{2F52}" => "\u{6C0F}",

    # 2F53 KANGXI RADICAL STEAM
    # ->   6C14 CJK UNIFIED IDEOGRAPH 6C14
    "\u{2F53}" => "\u{6C14}",

    # 2F54 KANGXI RADICAL WATER
    # ->   6C34 CJK UNIFIED IDEOGRAPH 6C34
    "\u{2F54}" => "\u{6C34}",

    # 2F55 KANGXI RADICAL FIRE
    # ->   706B CJK UNIFIED IDEOGRAPH 706B
    "\u{2F55}" => "\u{706B}",

    # 2F56 KANGXI RADICAL CLAW
    # ->   722A CJK UNIFIED IDEOGRAPH 722A
    "\u{2F56}" => "\u{722A}",

    # 2F57 KANGXI RADICAL FATHER
    # ->   7236 CJK UNIFIED IDEOGRAPH 7236
    "\u{2F57}" => "\u{7236}",

    # 2F58 KANGXI RADICAL DOUBLE X
    # ->   723B CJK UNIFIED IDEOGRAPH 723B
    "\u{2F58}" => "\u{723B}",

    # 2F59 KANGXI RADICAL HALF TREE TRUNK
    # ->   723F CJK UNIFIED IDEOGRAPH 723F
    "\u{2F59}" => "\u{723F}",

    # 2F5A KANGXI RADICAL SLICE
    # ->   7247 CJK UNIFIED IDEOGRAPH 7247
    "\u{2F5A}" => "\u{7247}",

    # 2F5B KANGXI RADICAL FANG
    # ->   7259 CJK UNIFIED IDEOGRAPH 7259
    "\u{2F5B}" => "\u{7259}",

    # 2F5C KANGXI RADICAL COW
    # ->   725B CJK UNIFIED IDEOGRAPH 725B
    "\u{2F5C}" => "\u{725B}",

    # 2F5D KANGXI RADICAL DOG
    # ->   72AC CJK UNIFIED IDEOGRAPH 72AC
    "\u{2F5D}" => "\u{72AC}",

    # 2F5E KANGXI RADICAL PROFOUND
    # ->   7384 CJK UNIFIED IDEOGRAPH 7384
    "\u{2F5E}" => "\u{7384}",

    # 2F5F KANGXI RADICAL JADE
    # ->   7389 CJK UNIFIED IDEOGRAPH 7389
    "\u{2F5F}" => "\u{7389}",

    # 2F60 KANGXI RADICAL MELON
    # ->   74DC CJK UNIFIED IDEOGRAPH 74DC
    "\u{2F60}" => "\u{74DC}",

    # 2F61 KANGXI RADICAL TILE
    # ->   74E6 CJK UNIFIED IDEOGRAPH 74E6
    "\u{2F61}" => "\u{74E6}",

    # 2F62 KANGXI RADICAL SWEET
    # ->   7518 CJK UNIFIED IDEOGRAPH 7518
    "\u{2F62}" => "\u{7518}",

    # 2F63 KANGXI RADICAL LIFE
    # ->   751F CJK UNIFIED IDEOGRAPH 751F
    "\u{2F63}" => "\u{751F}",

    # 2F64 KANGXI RADICAL USE
    # ->   7528 CJK UNIFIED IDEOGRAPH 7528
    "\u{2F64}" => "\u{7528}",

    # 2F65 KANGXI RADICAL FIELD
    # ->   7530 CJK UNIFIED IDEOGRAPH 7530
    "\u{2F65}" => "\u{7530}",

    # 2F66 KANGXI RADICAL BOLT OF CLOTH
    # ->   758B CJK UNIFIED IDEOGRAPH 758B
    "\u{2F66}" => "\u{758B}",

    # 2F67 KANGXI RADICAL SICKNESS
    # ->   7592 CJK UNIFIED IDEOGRAPH 7592
    "\u{2F67}" => "\u{7592}",

    # 2F68 KANGXI RADICAL DOTTED TENT
    # ->   7676 CJK UNIFIED IDEOGRAPH 7676
    "\u{2F68}" => "\u{7676}",

    # 2F69 KANGXI RADICAL WHITE
    # ->   767D CJK UNIFIED IDEOGRAPH 767D
    "\u{2F69}" => "\u{767D}",

    # 2F6A KANGXI RADICAL SKIN
    # ->   76AE CJK UNIFIED IDEOGRAPH 76AE
    "\u{2F6A}" => "\u{76AE}",

    # 2F6B KANGXI RADICAL DISH
    # ->   76BF CJK UNIFIED IDEOGRAPH 76BF
    "\u{2F6B}" => "\u{76BF}",

    # 2F6C KANGXI RADICAL EYE
    # ->   76EE CJK UNIFIED IDEOGRAPH 76EE
    "\u{2F6C}" => "\u{76EE}",

    # 2F6D KANGXI RADICAL SPEAR
    # ->   77DB CJK UNIFIED IDEOGRAPH 77DB
    "\u{2F6D}" => "\u{77DB}",

    # 2F6E KANGXI RADICAL ARROW
    # ->   77E2 CJK UNIFIED IDEOGRAPH 77E2
    "\u{2F6E}" => "\u{77E2}",

    # 2F6F KANGXI RADICAL STONE
    # ->   77F3 CJK UNIFIED IDEOGRAPH 77F3
    "\u{2F6F}" => "\u{77F3}",

    # 2F70 KANGXI RADICAL SPIRIT
    # ->   793A CJK UNIFIED IDEOGRAPH 793A
    "\u{2F70}" => "\u{793A}",

    # 2F71 KANGXI RADICAL TRACK
    # ->   79B8 CJK UNIFIED IDEOGRAPH 79B8
    "\u{2F71}" => "\u{79B8}",

    # 2F72 KANGXI RADICAL GRAIN
    # ->   79BE CJK UNIFIED IDEOGRAPH 79BE
    "\u{2F72}" => "\u{79BE}",

    # 2F73 KANGXI RADICAL CAVE
    # ->   7A74 CJK UNIFIED IDEOGRAPH 7A74
    "\u{2F73}" => "\u{7A74}",

    # 2F74 KANGXI RADICAL STAND
    # ->   7ACB CJK UNIFIED IDEOGRAPH 7ACB
    "\u{2F74}" => "\u{7ACB}",

    # 2F75 KANGXI RADICAL BAMBOO
    # ->   7AF9 CJK UNIFIED IDEOGRAPH 7AF9
    "\u{2F75}" => "\u{7AF9}",

    # 2F76 KANGXI RADICAL RICE
    # ->   7C73 CJK UNIFIED IDEOGRAPH 7C73
    "\u{2F76}" => "\u{7C73}",

    # 2F77 KANGXI RADICAL SILK
    # ->   7CF8 CJK UNIFIED IDEOGRAPH 7CF8
    "\u{2F77}" => "\u{7CF8}",

    # 2F78 KANGXI RADICAL JAR
    # ->   7F36 CJK UNIFIED IDEOGRAPH 7F36
    "\u{2F78}" => "\u{7F36}",

    # 2F79 KANGXI RADICAL NET
    # ->   7F51 CJK UNIFIED IDEOGRAPH 7F51
    "\u{2F79}" => "\u{7F51}",

    # 2F7A KANGXI RADICAL SHEEP
    # ->   7F8A CJK UNIFIED IDEOGRAPH 7F8A
    "\u{2F7A}" => "\u{7F8A}",

    # 2F7B KANGXI RADICAL FEATHER
    # ->   7FBD CJK UNIFIED IDEOGRAPH 7FBD
    "\u{2F7B}" => "\u{7FBD}",

    # 2F7C KANGXI RADICAL OLD
    # ->   8001 CJK UNIFIED IDEOGRAPH 8001
    "\u{2F7C}" => "\u{8001}",

    # 2F7D KANGXI RADICAL AND
    # ->   800C CJK UNIFIED IDEOGRAPH 800C
    "\u{2F7D}" => "\u{800C}",

    # 2F7E KANGXI RADICAL PLOW
    # ->   8012 CJK UNIFIED IDEOGRAPH 8012
    "\u{2F7E}" => "\u{8012}",

    # 2F7F KANGXI RADICAL EAR
    # ->   8033 CJK UNIFIED IDEOGRAPH 8033
    "\u{2F7F}" => "\u{8033}",

    # 2F80 KANGXI RADICAL BRUSH
    # ->   807F CJK UNIFIED IDEOGRAPH 807F
    "\u{2F80}" => "\u{807F}",

    # 2F81 KANGXI RADICAL MEAT
    # ->   8089 CJK UNIFIED IDEOGRAPH 8089
    "\u{2F81}" => "\u{8089}",

    # 2F82 KANGXI RADICAL MINISTER
    # ->   81E3 CJK UNIFIED IDEOGRAPH 81E3
    "\u{2F82}" => "\u{81E3}",

    # 2F83 KANGXI RADICAL SELF
    # ->   81EA CJK UNIFIED IDEOGRAPH 81EA
    "\u{2F83}" => "\u{81EA}",

    # 2F84 KANGXI RADICAL ARRIVE
    # ->   81F3 CJK UNIFIED IDEOGRAPH 81F3
    "\u{2F84}" => "\u{81F3}",

    # 2F85 KANGXI RADICAL MORTAR
    # ->   81FC CJK UNIFIED IDEOGRAPH 81FC
    "\u{2F85}" => "\u{81FC}",

    # 2F86 KANGXI RADICAL TONGUE
    # ->   820C CJK UNIFIED IDEOGRAPH 820C
    "\u{2F86}" => "\u{820C}",

    # 2F87 KANGXI RADICAL OPPOSE
    # ->   821B CJK UNIFIED IDEOGRAPH 821B
    "\u{2F87}" => "\u{821B}",

    # 2F88 KANGXI RADICAL BOAT
    # ->   821F CJK UNIFIED IDEOGRAPH 821F
    "\u{2F88}" => "\u{821F}",

    # 2F89 KANGXI RADICAL STOPPING
    # ->   826E CJK UNIFIED IDEOGRAPH 826E
    "\u{2F89}" => "\u{826E}",

    # 2F8A KANGXI RADICAL COLOR
    # ->   8272 CJK UNIFIED IDEOGRAPH 8272
    "\u{2F8A}" => "\u{8272}",

    # 2F8B KANGXI RADICAL GRASS
    # ->   8278 CJK UNIFIED IDEOGRAPH 8278
    "\u{2F8B}" => "\u{8278}",

    # 2F8C KANGXI RADICAL TIGER
    # ->   864D CJK UNIFIED IDEOGRAPH 864D
    "\u{2F8C}" => "\u{864D}",

    # 2F8D KANGXI RADICAL INSECT
    # ->   866B CJK UNIFIED IDEOGRAPH 866B
    "\u{2F8D}" => "\u{866B}",

    # 2F8E KANGXI RADICAL BLOOD
    # ->   8840 CJK UNIFIED IDEOGRAPH 8840
    "\u{2F8E}" => "\u{8840}",

    # 2F8F KANGXI RADICAL WALK ENCLOSURE
    # ->   884C CJK UNIFIED IDEOGRAPH 884C
    "\u{2F8F}" => "\u{884C}",

    # 2F90 KANGXI RADICAL CLOTHES
    # ->   8863 CJK UNIFIED IDEOGRAPH 8863
    "\u{2F90}" => "\u{8863}",

    # 2F91 KANGXI RADICAL WEST
    # ->   897E CJK UNIFIED IDEOGRAPH 897E
    "\u{2F91}" => "\u{897E}",

    # 2F92 KANGXI RADICAL SEE
    # ->   898B CJK UNIFIED IDEOGRAPH 898B
    "\u{2F92}" => "\u{898B}",

    # 2F93 KANGXI RADICAL HORN
    # ->   89D2 CJK UNIFIED IDEOGRAPH 89D2
    "\u{2F93}" => "\u{89D2}",

    # 2F94 KANGXI RADICAL SPEECH
    # ->   8A00 CJK UNIFIED IDEOGRAPH 8A00
    "\u{2F94}" => "\u{8A00}",

    # 2F95 KANGXI RADICAL VALLEY
    # ->   8C37 CJK UNIFIED IDEOGRAPH 8C37
    "\u{2F95}" => "\u{8C37}",

    # 2F96 KANGXI RADICAL BEAN
    # ->   8C46 CJK UNIFIED IDEOGRAPH 8C46
    "\u{2F96}" => "\u{8C46}",

    # 2F97 KANGXI RADICAL PIG
    # ->   8C55 CJK UNIFIED IDEOGRAPH 8C55
    "\u{2F97}" => "\u{8C55}",

    # 2F98 KANGXI RADICAL BADGER
    # ->   8C78 CJK UNIFIED IDEOGRAPH 8C78
    "\u{2F98}" => "\u{8C78}",

    # 2F99 KANGXI RADICAL SHELL
    # ->   8C9D CJK UNIFIED IDEOGRAPH 8C9D
    "\u{2F99}" => "\u{8C9D}",

    # 2F9A KANGXI RADICAL RED
    # ->   8D64 CJK UNIFIED IDEOGRAPH 8D64
    "\u{2F9A}" => "\u{8D64}",

    # 2F9B KANGXI RADICAL RUN
    # ->   8D70 CJK UNIFIED IDEOGRAPH 8D70
    "\u{2F9B}" => "\u{8D70}",

    # 2F9C KANGXI RADICAL FOOT
    # ->   8DB3 CJK UNIFIED IDEOGRAPH 8DB3
    "\u{2F9C}" => "\u{8DB3}",

    # 2F9D KANGXI RADICAL BODY
    # ->   8EAB CJK UNIFIED IDEOGRAPH 8EAB
    "\u{2F9D}" => "\u{8EAB}",

    # 2F9E KANGXI RADICAL CART
    # ->   8ECA CJK UNIFIED IDEOGRAPH 8ECA
    "\u{2F9E}" => "\u{8ECA}",

    # 2F9F KANGXI RADICAL BITTER
    # ->   8F9B CJK UNIFIED IDEOGRAPH 8F9B
    "\u{2F9F}" => "\u{8F9B}",

    # 2FA0 KANGXI RADICAL MORNING
    # ->   8FB0 CJK UNIFIED IDEOGRAPH 8FB0
    "\u{2FA0}" => "\u{8FB0}",

    # 2FA1 KANGXI RADICAL WALK
    # ->   8FB5 CJK UNIFIED IDEOGRAPH 8FB5
    "\u{2FA1}" => "\u{8FB5}",

    # 2FA2 KANGXI RADICAL CITY
    # ->   9091 CJK UNIFIED IDEOGRAPH 9091
    "\u{2FA2}" => "\u{9091}",

    # 2FA3 KANGXI RADICAL WINE
    # ->   9149 CJK UNIFIED IDEOGRAPH 9149
    "\u{2FA3}" => "\u{9149}",

    # 2FA4 KANGXI RADICAL DISTINGUISH
    # ->   91C6 CJK UNIFIED IDEOGRAPH 91C6
    "\u{2FA4}" => "\u{91C6}",

    # 2FA5 KANGXI RADICAL VILLAGE
    # ->   91CC CJK UNIFIED IDEOGRAPH 91CC
    "\u{2FA5}" => "\u{91CC}",

    # 2FA6 KANGXI RADICAL GOLD
    # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
    "\u{2FA6}" => "\u{91D1}",

    # 2FA7 KANGXI RADICAL LONG
    # ->   9577 CJK UNIFIED IDEOGRAPH 9577
    "\u{2FA7}" => "\u{9577}",

    # 2FA8 KANGXI RADICAL GATE
    # ->   9580 CJK UNIFIED IDEOGRAPH 9580
    "\u{2FA8}" => "\u{9580}",

    # 2FA9 KANGXI RADICAL MOUND
    # ->   961C CJK UNIFIED IDEOGRAPH 961C
    "\u{2FA9}" => "\u{961C}",

    # 2FAA KANGXI RADICAL SLAVE
    # ->   96B6 CJK UNIFIED IDEOGRAPH 96B6
    "\u{2FAA}" => "\u{96B6}",

    # 2FAB KANGXI RADICAL SHORT TAILED BIRD
    # ->   96B9 CJK UNIFIED IDEOGRAPH 96B9
    "\u{2FAB}" => "\u{96B9}",

    # 2FAC KANGXI RADICAL RAIN
    # ->   96E8 CJK UNIFIED IDEOGRAPH 96E8
    "\u{2FAC}" => "\u{96E8}",

    # 2FAD KANGXI RADICAL BLUE
    # ->   9751 CJK UNIFIED IDEOGRAPH 9751
    "\u{2FAD}" => "\u{9751}",

    # 2FAE KANGXI RADICAL WRONG
    # ->   975E CJK UNIFIED IDEOGRAPH 975E
    "\u{2FAE}" => "\u{975E}",

    # 2FAF KANGXI RADICAL FACE
    # ->   9762 CJK UNIFIED IDEOGRAPH 9762
    "\u{2FAF}" => "\u{9762}",

    # 2FB0 KANGXI RADICAL LEATHER
    # ->   9769 CJK UNIFIED IDEOGRAPH 9769
    "\u{2FB0}" => "\u{9769}",

    # 2FB1 KANGXI RADICAL TANNED LEATHER
    # ->   97CB CJK UNIFIED IDEOGRAPH 97CB
    "\u{2FB1}" => "\u{97CB}",

    # 2FB2 KANGXI RADICAL LEEK
    # ->   97ED CJK UNIFIED IDEOGRAPH 97ED
    "\u{2FB2}" => "\u{97ED}",

    # 2FB3 KANGXI RADICAL SOUND
    # ->   97F3 CJK UNIFIED IDEOGRAPH 97F3
    "\u{2FB3}" => "\u{97F3}",

    # 2FB4 KANGXI RADICAL LEAF
    # ->   9801 CJK UNIFIED IDEOGRAPH 9801
    "\u{2FB4}" => "\u{9801}",

    # 2FB5 KANGXI RADICAL WIND
    # ->   98A8 CJK UNIFIED IDEOGRAPH 98A8
    "\u{2FB5}" => "\u{98A8}",

    # 2FB6 KANGXI RADICAL FLY
    # ->   98DB CJK UNIFIED IDEOGRAPH 98DB
    "\u{2FB6}" => "\u{98DB}",

    # 2FB7 KANGXI RADICAL EAT
    # ->   98DF CJK UNIFIED IDEOGRAPH 98DF
    "\u{2FB7}" => "\u{98DF}",

    # 2FB8 KANGXI RADICAL HEAD
    # ->   9996 CJK UNIFIED IDEOGRAPH 9996
    "\u{2FB8}" => "\u{9996}",

    # 2FB9 KANGXI RADICAL FRAGRANT
    # ->   9999 CJK UNIFIED IDEOGRAPH 9999
    "\u{2FB9}" => "\u{9999}",

    # 2FBA KANGXI RADICAL HORSE
    # ->   99AC CJK UNIFIED IDEOGRAPH 99AC
    "\u{2FBA}" => "\u{99AC}",

    # 2FBB KANGXI RADICAL BONE
    # ->   9AA8 CJK UNIFIED IDEOGRAPH 9AA8
    "\u{2FBB}" => "\u{9AA8}",

    # 2FBC KANGXI RADICAL TALL
    # ->   9AD8 CJK UNIFIED IDEOGRAPH 9AD8
    "\u{2FBC}" => "\u{9AD8}",

    # 2FBD KANGXI RADICAL HAIR
    # ->   9ADF CJK UNIFIED IDEOGRAPH 9ADF
    "\u{2FBD}" => "\u{9ADF}",

    # 2FBE KANGXI RADICAL FIGHT
    # ->   9B25 CJK UNIFIED IDEOGRAPH 9B25
    "\u{2FBE}" => "\u{9B25}",

    # 2FBF KANGXI RADICAL SACRIFICIAL WINE
    # ->   9B2F CJK UNIFIED IDEOGRAPH 9B2F
    "\u{2FBF}" => "\u{9B2F}",

    # 2FC0 KANGXI RADICAL CAULDRON
    # ->   9B32 CJK UNIFIED IDEOGRAPH 9B32
    "\u{2FC0}" => "\u{9B32}",

    # 2FC1 KANGXI RADICAL GHOST
    # ->   9B3C CJK UNIFIED IDEOGRAPH 9B3C
    "\u{2FC1}" => "\u{9B3C}",

    # 2FC2 KANGXI RADICAL FISH
    # ->   9B5A CJK UNIFIED IDEOGRAPH 9B5A
    "\u{2FC2}" => "\u{9B5A}",

    # 2FC3 KANGXI RADICAL BIRD
    # ->   9CE5 CJK UNIFIED IDEOGRAPH 9CE5
    "\u{2FC3}" => "\u{9CE5}",

    # 2FC4 KANGXI RADICAL SALT
    # ->   9E75 CJK UNIFIED IDEOGRAPH 9E75
    "\u{2FC4}" => "\u{9E75}",

    # 2FC5 KANGXI RADICAL DEER
    # ->   9E7F CJK UNIFIED IDEOGRAPH 9E7F
    "\u{2FC5}" => "\u{9E7F}",

    # 2FC6 KANGXI RADICAL WHEAT
    # ->   9EA5 CJK UNIFIED IDEOGRAPH 9EA5
    "\u{2FC6}" => "\u{9EA5}",

    # 2FC7 KANGXI RADICAL HEMP
    # ->   9EBB CJK UNIFIED IDEOGRAPH 9EBB
    "\u{2FC7}" => "\u{9EBB}",

    # 2FC8 KANGXI RADICAL YELLOW
    # ->   9EC3 CJK UNIFIED IDEOGRAPH 9EC3
    "\u{2FC8}" => "\u{9EC3}",

    # 2FC9 KANGXI RADICAL MILLET
    # ->   9ECD CJK UNIFIED IDEOGRAPH 9ECD
    "\u{2FC9}" => "\u{9ECD}",

    # 2FCA KANGXI RADICAL BLACK
    # ->   9ED1 CJK UNIFIED IDEOGRAPH 9ED1
    "\u{2FCA}" => "\u{9ED1}",

    # 2FCB KANGXI RADICAL EMBROIDERY
    # ->   9EF9 CJK UNIFIED IDEOGRAPH 9EF9
    "\u{2FCB}" => "\u{9EF9}",

    # 2FCC KANGXI RADICAL FROG
    # ->   9EFD CJK UNIFIED IDEOGRAPH 9EFD
    "\u{2FCC}" => "\u{9EFD}",

    # 2FCD KANGXI RADICAL TRIPOD
    # ->   9F0E CJK UNIFIED IDEOGRAPH 9F0E
    "\u{2FCD}" => "\u{9F0E}",

    # 2FCE KANGXI RADICAL DRUM
    # ->   9F13 CJK UNIFIED IDEOGRAPH 9F13
    "\u{2FCE}" => "\u{9F13}",

    # 2FCF KANGXI RADICAL RAT
    # ->   9F20 CJK UNIFIED IDEOGRAPH 9F20
    "\u{2FCF}" => "\u{9F20}",

    # 2FD0 KANGXI RADICAL NOSE
    # ->   9F3B CJK UNIFIED IDEOGRAPH 9F3B
    "\u{2FD0}" => "\u{9F3B}",

    # 2FD1 KANGXI RADICAL EVEN
    # ->   9F4A CJK UNIFIED IDEOGRAPH 9F4A
    "\u{2FD1}" => "\u{9F4A}",

    # 2FD2 KANGXI RADICAL TOOTH
    # ->   9F52 CJK UNIFIED IDEOGRAPH 9F52
    "\u{2FD2}" => "\u{9F52}",

    # 2FD3 KANGXI RADICAL DRAGON
    # ->   9F8D CJK UNIFIED IDEOGRAPH 9F8D
    "\u{2FD3}" => "\u{9F8D}",

    # 2FD4 KANGXI RADICAL TURTLE
    # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
    "\u{2FD4}" => "\u{9F9C}",

    # 2FD5 KANGXI RADICAL FLUTE
    # ->   9FA0 CJK UNIFIED IDEOGRAPH 9FA0
    "\u{2FD5}" => "\u{9FA0}",

    # 3000 IDEOGRAPHIC SPACE
    # ->   0020 SPACE
    "\u{3000}" => " ",

    # 3036 CIRCLED POSTAL MARK
    # ->   3012 POSTAL MARK
    "\u{3036}" => "\u{3012}",

    # 3038 HANGZHOU NUMERAL TEN
    # ->   5341 CJK UNIFIED IDEOGRAPH 5341
    "\u{3038}" => "\u{5341}",

    # 3039 HANGZHOU NUMERAL TWENTY
    # ->   5344 CJK UNIFIED IDEOGRAPH 5344
    "\u{3039}" => "\u{5344}",

    # 303A HANGZHOU NUMERAL THIRTY
    # ->   5345 CJK UNIFIED IDEOGRAPH 5345
    "\u{303A}" => "\u{5345}",

    # 304C HIRAGANA LETTER GA
    # ->   304B HIRAGANA LETTER KA
    "\u{304C}" => "\u{304B}",

    # 304E HIRAGANA LETTER GI
    # ->   304D HIRAGANA LETTER KI
    "\u{304E}" => "\u{304D}",

    # 3050 HIRAGANA LETTER GU
    # ->   304F HIRAGANA LETTER KU
    "\u{3050}" => "\u{304F}",

    # 3052 HIRAGANA LETTER GE
    # ->   3051 HIRAGANA LETTER KE
    "\u{3052}" => "\u{3051}",

    # 3054 HIRAGANA LETTER GO
    # ->   3053 HIRAGANA LETTER KO
    "\u{3054}" => "\u{3053}",

    # 3056 HIRAGANA LETTER ZA
    # ->   3055 HIRAGANA LETTER SA
    "\u{3056}" => "\u{3055}",

    # 3058 HIRAGANA LETTER ZI
    # ->   3057 HIRAGANA LETTER SI
    "\u{3058}" => "\u{3057}",

    # 305A HIRAGANA LETTER ZU
    # ->   3059 HIRAGANA LETTER SU
    "\u{305A}" => "\u{3059}",

    # 305C HIRAGANA LETTER ZE
    # ->   305B HIRAGANA LETTER SE
    "\u{305C}" => "\u{305B}",

    # 305E HIRAGANA LETTER ZO
    # ->   305D HIRAGANA LETTER SO
    "\u{305E}" => "\u{305D}",

    # 3060 HIRAGANA LETTER DA
    # ->   305F HIRAGANA LETTER TA
    "\u{3060}" => "\u{305F}",

    # 3062 HIRAGANA LETTER DI
    # ->   3061 HIRAGANA LETTER TI
    "\u{3062}" => "\u{3061}",

    # 3065 HIRAGANA LETTER DU
    # ->   3064 HIRAGANA LETTER TU
    "\u{3065}" => "\u{3064}",

    # 3067 HIRAGANA LETTER DE
    # ->   3066 HIRAGANA LETTER TE
    "\u{3067}" => "\u{3066}",

    # 3069 HIRAGANA LETTER DO
    # ->   3068 HIRAGANA LETTER TO
    "\u{3069}" => "\u{3068}",

    # 3070 HIRAGANA LETTER BA
    # ->   306F HIRAGANA LETTER HA
    "\u{3070}" => "\u{306F}",

    # 3071 HIRAGANA LETTER PA
    # ->   306F HIRAGANA LETTER HA
    "\u{3071}" => "\u{306F}",

    # 3073 HIRAGANA LETTER BI
    # ->   3072 HIRAGANA LETTER HI
    "\u{3073}" => "\u{3072}",

    # 3074 HIRAGANA LETTER PI
    # ->   3072 HIRAGANA LETTER HI
    "\u{3074}" => "\u{3072}",

    # 3076 HIRAGANA LETTER BU
    # ->   3075 HIRAGANA LETTER HU
    "\u{3076}" => "\u{3075}",

    # 3077 HIRAGANA LETTER PU
    # ->   3075 HIRAGANA LETTER HU
    "\u{3077}" => "\u{3075}",

    # 3079 HIRAGANA LETTER BE
    # ->   3078 HIRAGANA LETTER HE
    "\u{3079}" => "\u{3078}",

    # 307A HIRAGANA LETTER PE
    # ->   3078 HIRAGANA LETTER HE
    "\u{307A}" => "\u{3078}",

    # 307C HIRAGANA LETTER BO
    # ->   307B HIRAGANA LETTER HO
    "\u{307C}" => "\u{307B}",

    # 307D HIRAGANA LETTER PO
    # ->   307B HIRAGANA LETTER HO
    "\u{307D}" => "\u{307B}",

    # 3094 HIRAGANA LETTER VU
    # ->   3046 HIRAGANA LETTER U
    "\u{3094}" => "\u{3046}",

    # 309B KATAKANA-HIRAGANA VOICED SOUND MARK
    # ->   0020 SPACE
    "\u{309B}" => " ",

    # 309C KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    # ->   0020 SPACE
    "\u{309C}" => " ",

    # 309E HIRAGANA VOICED ITERATION MARK
    # ->   309D HIRAGANA ITERATION MARK
    "\u{309E}" => "\u{309D}",

    # 309F HIRAGANA DIGRAPH YORI
    # ->   3088 HIRAGANA LETTER YO
    #    + 308A HIRAGANA LETTER RI
    "\u{309F}" => "\u{3088}\u{308A}",

    # 30AC KATAKANA LETTER GA
    # ->   30AB KATAKANA LETTER KA
    "\u{30AC}" => "\u{30AB}",

    # 30AE KATAKANA LETTER GI
    # ->   30AD KATAKANA LETTER KI
    "\u{30AE}" => "\u{30AD}",

    # 30B0 KATAKANA LETTER GU
    # ->   30AF KATAKANA LETTER KU
    "\u{30B0}" => "\u{30AF}",

    # 30B2 KATAKANA LETTER GE
    # ->   30B1 KATAKANA LETTER KE
    "\u{30B2}" => "\u{30B1}",

    # 30B4 KATAKANA LETTER GO
    # ->   30B3 KATAKANA LETTER KO
    "\u{30B4}" => "\u{30B3}",

    # 30B6 KATAKANA LETTER ZA
    # ->   30B5 KATAKANA LETTER SA
    "\u{30B6}" => "\u{30B5}",

    # 30B8 KATAKANA LETTER ZI
    # ->   30B7 KATAKANA LETTER SI
    "\u{30B8}" => "\u{30B7}",

    # 30BA KATAKANA LETTER ZU
    # ->   30B9 KATAKANA LETTER SU
    "\u{30BA}" => "\u{30B9}",

    # 30BC KATAKANA LETTER ZE
    # ->   30BB KATAKANA LETTER SE
    "\u{30BC}" => "\u{30BB}",

    # 30BE KATAKANA LETTER ZO
    # ->   30BD KATAKANA LETTER SO
    "\u{30BE}" => "\u{30BD}",

    # 30C0 KATAKANA LETTER DA
    # ->   30BF KATAKANA LETTER TA
    "\u{30C0}" => "\u{30BF}",

    # 30C2 KATAKANA LETTER DI
    # ->   30C1 KATAKANA LETTER TI
    "\u{30C2}" => "\u{30C1}",

    # 30C5 KATAKANA LETTER DU
    # ->   30C4 KATAKANA LETTER TU
    "\u{30C5}" => "\u{30C4}",

    # 30C7 KATAKANA LETTER DE
    # ->   30C6 KATAKANA LETTER TE
    "\u{30C7}" => "\u{30C6}",

    # 30C9 KATAKANA LETTER DO
    # ->   30C8 KATAKANA LETTER TO
    "\u{30C9}" => "\u{30C8}",

    # 30D0 KATAKANA LETTER BA
    # ->   30CF KATAKANA LETTER HA
    "\u{30D0}" => "\u{30CF}",

    # 30D1 KATAKANA LETTER PA
    # ->   30CF KATAKANA LETTER HA
    "\u{30D1}" => "\u{30CF}",

    # 30D3 KATAKANA LETTER BI
    # ->   30D2 KATAKANA LETTER HI
    "\u{30D3}" => "\u{30D2}",

    # 30D4 KATAKANA LETTER PI
    # ->   30D2 KATAKANA LETTER HI
    "\u{30D4}" => "\u{30D2}",

    # 30D6 KATAKANA LETTER BU
    # ->   30D5 KATAKANA LETTER HU
    "\u{30D6}" => "\u{30D5}",

    # 30D7 KATAKANA LETTER PU
    # ->   30D5 KATAKANA LETTER HU
    "\u{30D7}" => "\u{30D5}",

    # 30D9 KATAKANA LETTER BE
    # ->   30D8 KATAKANA LETTER HE
    "\u{30D9}" => "\u{30D8}",

    # 30DA KATAKANA LETTER PE
    # ->   30D8 KATAKANA LETTER HE
    "\u{30DA}" => "\u{30D8}",

    # 30DC KATAKANA LETTER BO
    # ->   30DB KATAKANA LETTER HO
    "\u{30DC}" => "\u{30DB}",

    # 30DD KATAKANA LETTER PO
    # ->   30DB KATAKANA LETTER HO
    "\u{30DD}" => "\u{30DB}",

    # 30F4 KATAKANA LETTER VU
    # ->   30A6 KATAKANA LETTER U
    "\u{30F4}" => "\u{30A6}",

    # 30F7 KATAKANA LETTER VA
    # ->   30EF KATAKANA LETTER WA
    "\u{30F7}" => "\u{30EF}",

    # 30F8 KATAKANA LETTER VI
    # ->   30F0 KATAKANA LETTER WI
    "\u{30F8}" => "\u{30F0}",

    # 30F9 KATAKANA LETTER VE
    # ->   30F1 KATAKANA LETTER WE
    "\u{30F9}" => "\u{30F1}",

    # 30FA KATAKANA LETTER VO
    # ->   30F2 KATAKANA LETTER WO
    "\u{30FA}" => "\u{30F2}",

    # 30FE KATAKANA VOICED ITERATION MARK
    # ->   30FD KATAKANA ITERATION MARK
    "\u{30FE}" => "\u{30FD}",

    # 30FF KATAKANA DIGRAPH KOTO
    # ->   30B3 KATAKANA LETTER KO
    #    + 30C8 KATAKANA LETTER TO
    "\u{30FF}" => "\u{30B3}\u{30C8}",

    # 3131 HANGUL LETTER KIYEOK
    # ->   1100 HANGUL CHOSEONG KIYEOK
    "\u{3131}" => "\u{1100}",

    # 3132 HANGUL LETTER SSANGKIYEOK
    # ->   1101 HANGUL CHOSEONG SSANGKIYEOK
    "\u{3132}" => "\u{1101}",

    # 3133 HANGUL LETTER KIYEOK-SIOS
    # ->   11AA HANGUL JONGSEONG KIYEOK-SIOS
    "\u{3133}" => "\u{11AA}",

    # 3134 HANGUL LETTER NIEUN
    # ->   1102 HANGUL CHOSEONG NIEUN
    "\u{3134}" => "\u{1102}",

    # 3135 HANGUL LETTER NIEUN-CIEUC
    # ->   11AC HANGUL JONGSEONG NIEUN-CIEUC
    "\u{3135}" => "\u{11AC}",

    # 3136 HANGUL LETTER NIEUN-HIEUH
    # ->   11AD HANGUL JONGSEONG NIEUN-HIEUH
    "\u{3136}" => "\u{11AD}",

    # 3137 HANGUL LETTER TIKEUT
    # ->   1103 HANGUL CHOSEONG TIKEUT
    "\u{3137}" => "\u{1103}",

    # 3138 HANGUL LETTER SSANGTIKEUT
    # ->   1104 HANGUL CHOSEONG SSANGTIKEUT
    "\u{3138}" => "\u{1104}",

    # 3139 HANGUL LETTER RIEUL
    # ->   1105 HANGUL CHOSEONG RIEUL
    "\u{3139}" => "\u{1105}",

    # 313A HANGUL LETTER RIEUL-KIYEOK
    # ->   11B0 HANGUL JONGSEONG RIEUL-KIYEOK
    "\u{313A}" => "\u{11B0}",

    # 313B HANGUL LETTER RIEUL-MIEUM
    # ->   11B1 HANGUL JONGSEONG RIEUL-MIEUM
    "\u{313B}" => "\u{11B1}",

    # 313C HANGUL LETTER RIEUL-PIEUP
    # ->   11B2 HANGUL JONGSEONG RIEUL-PIEUP
    "\u{313C}" => "\u{11B2}",

    # 313D HANGUL LETTER RIEUL-SIOS
    # ->   11B3 HANGUL JONGSEONG RIEUL-SIOS
    "\u{313D}" => "\u{11B3}",

    # 313E HANGUL LETTER RIEUL-THIEUTH
    # ->   11B4 HANGUL JONGSEONG RIEUL-THIEUTH
    "\u{313E}" => "\u{11B4}",

    # 313F HANGUL LETTER RIEUL-PHIEUPH
    # ->   11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
    "\u{313F}" => "\u{11B5}",

    # 3140 HANGUL LETTER RIEUL-HIEUH
    # ->   111A HANGUL CHOSEONG RIEUL-HIEUH
    "\u{3140}" => "\u{111A}",

    # 3141 HANGUL LETTER MIEUM
    # ->   1106 HANGUL CHOSEONG MIEUM
    "\u{3141}" => "\u{1106}",

    # 3142 HANGUL LETTER PIEUP
    # ->   1107 HANGUL CHOSEONG PIEUP
    "\u{3142}" => "\u{1107}",

    # 3143 HANGUL LETTER SSANGPIEUP
    # ->   1108 HANGUL CHOSEONG SSANGPIEUP
    "\u{3143}" => "\u{1108}",

    # 3144 HANGUL LETTER PIEUP-SIOS
    # ->   1121 HANGUL CHOSEONG PIEUP-SIOS
    "\u{3144}" => "\u{1121}",

    # 3145 HANGUL LETTER SIOS
    # ->   1109 HANGUL CHOSEONG SIOS
    "\u{3145}" => "\u{1109}",

    # 3146 HANGUL LETTER SSANGSIOS
    # ->   110A HANGUL CHOSEONG SSANGSIOS
    "\u{3146}" => "\u{110A}",

    # 3147 HANGUL LETTER IEUNG
    # ->   110B HANGUL CHOSEONG IEUNG
    "\u{3147}" => "\u{110B}",

    # 3148 HANGUL LETTER CIEUC
    # ->   110C HANGUL CHOSEONG CIEUC
    "\u{3148}" => "\u{110C}",

    # 3149 HANGUL LETTER SSANGCIEUC
    # ->   110D HANGUL CHOSEONG SSANGCIEUC
    "\u{3149}" => "\u{110D}",

    # 314A HANGUL LETTER CHIEUCH
    # ->   110E HANGUL CHOSEONG CHIEUCH
    "\u{314A}" => "\u{110E}",

    # 314B HANGUL LETTER KHIEUKH
    # ->   110F HANGUL CHOSEONG KHIEUKH
    "\u{314B}" => "\u{110F}",

    # 314C HANGUL LETTER THIEUTH
    # ->   1110 HANGUL CHOSEONG THIEUTH
    "\u{314C}" => "\u{1110}",

    # 314D HANGUL LETTER PHIEUPH
    # ->   1111 HANGUL CHOSEONG PHIEUPH
    "\u{314D}" => "\u{1111}",

    # 314E HANGUL LETTER HIEUH
    # ->   1112 HANGUL CHOSEONG HIEUH
    "\u{314E}" => "\u{1112}",

    # 314F HANGUL LETTER A
    # ->   1161 HANGUL JUNGSEONG A
    "\u{314F}" => "\u{1161}",

    # 3150 HANGUL LETTER AE
    # ->   1162 HANGUL JUNGSEONG AE
    "\u{3150}" => "\u{1162}",

    # 3151 HANGUL LETTER YA
    # ->   1163 HANGUL JUNGSEONG YA
    "\u{3151}" => "\u{1163}",

    # 3152 HANGUL LETTER YAE
    # ->   1164 HANGUL JUNGSEONG YAE
    "\u{3152}" => "\u{1164}",

    # 3153 HANGUL LETTER EO
    # ->   1165 HANGUL JUNGSEONG EO
    "\u{3153}" => "\u{1165}",

    # 3154 HANGUL LETTER E
    # ->   1166 HANGUL JUNGSEONG E
    "\u{3154}" => "\u{1166}",

    # 3155 HANGUL LETTER YEO
    # ->   1167 HANGUL JUNGSEONG YEO
    "\u{3155}" => "\u{1167}",

    # 3156 HANGUL LETTER YE
    # ->   1168 HANGUL JUNGSEONG YE
    "\u{3156}" => "\u{1168}",

    # 3157 HANGUL LETTER O
    # ->   1169 HANGUL JUNGSEONG O
    "\u{3157}" => "\u{1169}",

    # 3158 HANGUL LETTER WA
    # ->   116A HANGUL JUNGSEONG WA
    "\u{3158}" => "\u{116A}",

    # 3159 HANGUL LETTER WAE
    # ->   116B HANGUL JUNGSEONG WAE
    "\u{3159}" => "\u{116B}",

    # 315A HANGUL LETTER OE
    # ->   116C HANGUL JUNGSEONG OE
    "\u{315A}" => "\u{116C}",

    # 315B HANGUL LETTER YO
    # ->   116D HANGUL JUNGSEONG YO
    "\u{315B}" => "\u{116D}",

    # 315C HANGUL LETTER U
    # ->   116E HANGUL JUNGSEONG U
    "\u{315C}" => "\u{116E}",

    # 315D HANGUL LETTER WEO
    # ->   116F HANGUL JUNGSEONG WEO
    "\u{315D}" => "\u{116F}",

    # 315E HANGUL LETTER WE
    # ->   1170 HANGUL JUNGSEONG WE
    "\u{315E}" => "\u{1170}",

    # 315F HANGUL LETTER WI
    # ->   1171 HANGUL JUNGSEONG WI
    "\u{315F}" => "\u{1171}",

    # 3160 HANGUL LETTER YU
    # ->   1172 HANGUL JUNGSEONG YU
    "\u{3160}" => "\u{1172}",

    # 3161 HANGUL LETTER EU
    # ->   1173 HANGUL JUNGSEONG EU
    "\u{3161}" => "\u{1173}",

    # 3162 HANGUL LETTER YI
    # ->   1174 HANGUL JUNGSEONG YI
    "\u{3162}" => "\u{1174}",

    # 3163 HANGUL LETTER I
    # ->   1175 HANGUL JUNGSEONG I
    "\u{3163}" => "\u{1175}",

    # 3164 HANGUL FILLER
    # ->   1160 HANGUL JUNGSEONG FILLER
    "\u{3164}" => "\u{1160}",

    # 3165 HANGUL LETTER SSANGNIEUN
    # ->   1114 HANGUL CHOSEONG SSANGNIEUN
    "\u{3165}" => "\u{1114}",

    # 3166 HANGUL LETTER NIEUN-TIKEUT
    # ->   1115 HANGUL CHOSEONG NIEUN-TIKEUT
    "\u{3166}" => "\u{1115}",

    # 3167 HANGUL LETTER NIEUN-SIOS
    # ->   11C7 HANGUL JONGSEONG NIEUN-SIOS
    "\u{3167}" => "\u{11C7}",

    # 3168 HANGUL LETTER NIEUN-PANSIOS
    # ->   11C8 HANGUL JONGSEONG NIEUN-PANSIOS
    "\u{3168}" => "\u{11C8}",

    # 3169 HANGUL LETTER RIEUL-KIYEOK-SIOS
    # ->   11CC HANGUL JONGSEONG RIEUL-KIYEOK-SIOS
    "\u{3169}" => "\u{11CC}",

    # 316A HANGUL LETTER RIEUL-TIKEUT
    # ->   11CE HANGUL JONGSEONG RIEUL-TIKEUT
    "\u{316A}" => "\u{11CE}",

    # 316B HANGUL LETTER RIEUL-PIEUP-SIOS
    # ->   11D3 HANGUL JONGSEONG RIEUL-PIEUP-SIOS
    "\u{316B}" => "\u{11D3}",

    # 316C HANGUL LETTER RIEUL-PANSIOS
    # ->   11D7 HANGUL JONGSEONG RIEUL-PANSIOS
    "\u{316C}" => "\u{11D7}",

    # 316D HANGUL LETTER RIEUL-YEORINHIEUH
    # ->   11D9 HANGUL JONGSEONG RIEUL-YEORINHIEUH
    "\u{316D}" => "\u{11D9}",

    # 316E HANGUL LETTER MIEUM-PIEUP
    # ->   111C HANGUL CHOSEONG MIEUM-PIEUP
    "\u{316E}" => "\u{111C}",

    # 316F HANGUL LETTER MIEUM-SIOS
    # ->   11DD HANGUL JONGSEONG MIEUM-SIOS
    "\u{316F}" => "\u{11DD}",

    # 3170 HANGUL LETTER MIEUM-PANSIOS
    # ->   11DF HANGUL JONGSEONG MIEUM-PANSIOS
    "\u{3170}" => "\u{11DF}",

    # 3171 HANGUL LETTER KAPYEOUNMIEUM
    # ->   111D HANGUL CHOSEONG KAPYEOUNMIEUM
    "\u{3171}" => "\u{111D}",

    # 3172 HANGUL LETTER PIEUP-KIYEOK
    # ->   111E HANGUL CHOSEONG PIEUP-KIYEOK
    "\u{3172}" => "\u{111E}",

    # 3173 HANGUL LETTER PIEUP-TIKEUT
    # ->   1120 HANGUL CHOSEONG PIEUP-TIKEUT
    "\u{3173}" => "\u{1120}",

    # 3174 HANGUL LETTER PIEUP-SIOS-KIYEOK
    # ->   1122 HANGUL CHOSEONG PIEUP-SIOS-KIYEOK
    "\u{3174}" => "\u{1122}",

    # 3175 HANGUL LETTER PIEUP-SIOS-TIKEUT
    # ->   1123 HANGUL CHOSEONG PIEUP-SIOS-TIKEUT
    "\u{3175}" => "\u{1123}",

    # 3176 HANGUL LETTER PIEUP-CIEUC
    # ->   1127 HANGUL CHOSEONG PIEUP-CIEUC
    "\u{3176}" => "\u{1127}",

    # 3177 HANGUL LETTER PIEUP-THIEUTH
    # ->   1129 HANGUL CHOSEONG PIEUP-THIEUTH
    "\u{3177}" => "\u{1129}",

    # 3178 HANGUL LETTER KAPYEOUNPIEUP
    # ->   112B HANGUL CHOSEONG KAPYEOUNPIEUP
    "\u{3178}" => "\u{112B}",

    # 3179 HANGUL LETTER KAPYEOUNSSANGPIEUP
    # ->   112C HANGUL CHOSEONG KAPYEOUNSSANGPIEUP
    "\u{3179}" => "\u{112C}",

    # 317A HANGUL LETTER SIOS-KIYEOK
    # ->   112D HANGUL CHOSEONG SIOS-KIYEOK
    "\u{317A}" => "\u{112D}",

    # 317B HANGUL LETTER SIOS-NIEUN
    # ->   112E HANGUL CHOSEONG SIOS-NIEUN
    "\u{317B}" => "\u{112E}",

    # 317C HANGUL LETTER SIOS-TIKEUT
    # ->   112F HANGUL CHOSEONG SIOS-TIKEUT
    "\u{317C}" => "\u{112F}",

    # 317D HANGUL LETTER SIOS-PIEUP
    # ->   1132 HANGUL CHOSEONG SIOS-PIEUP
    "\u{317D}" => "\u{1132}",

    # 317E HANGUL LETTER SIOS-CIEUC
    # ->   1136 HANGUL CHOSEONG SIOS-CIEUC
    "\u{317E}" => "\u{1136}",

    # 317F HANGUL LETTER PANSIOS
    # ->   1140 HANGUL CHOSEONG PANSIOS
    "\u{317F}" => "\u{1140}",

    # 3180 HANGUL LETTER SSANGIEUNG
    # ->   1147 HANGUL CHOSEONG SSANGIEUNG
    "\u{3180}" => "\u{1147}",

    # 3181 HANGUL LETTER YESIEUNG
    # ->   114C HANGUL CHOSEONG YESIEUNG
    "\u{3181}" => "\u{114C}",

    # 3182 HANGUL LETTER YESIEUNG-SIOS
    # ->   11F1 HANGUL JONGSEONG YESIEUNG-SIOS
    "\u{3182}" => "\u{11F1}",

    # 3183 HANGUL LETTER YESIEUNG-PANSIOS
    # ->   11F2 HANGUL JONGSEONG YESIEUNG-PANSIOS
    "\u{3183}" => "\u{11F2}",

    # 3184 HANGUL LETTER KAPYEOUNPHIEUPH
    # ->   1157 HANGUL CHOSEONG KAPYEOUNPHIEUPH
    "\u{3184}" => "\u{1157}",

    # 3185 HANGUL LETTER SSANGHIEUH
    # ->   1158 HANGUL CHOSEONG SSANGHIEUH
    "\u{3185}" => "\u{1158}",

    # 3186 HANGUL LETTER YEORINHIEUH
    # ->   1159 HANGUL CHOSEONG YEORINHIEUH
    "\u{3186}" => "\u{1159}",

    # 3187 HANGUL LETTER YO-YA
    # ->   1184 HANGUL JUNGSEONG YO-YA
    "\u{3187}" => "\u{1184}",

    # 3188 HANGUL LETTER YO-YAE
    # ->   1185 HANGUL JUNGSEONG YO-YAE
    "\u{3188}" => "\u{1185}",

    # 3189 HANGUL LETTER YO-I
    # ->   1188 HANGUL JUNGSEONG YO-I
    "\u{3189}" => "\u{1188}",

    # 318A HANGUL LETTER YU-YEO
    # ->   1191 HANGUL JUNGSEONG YU-YEO
    "\u{318A}" => "\u{1191}",

    # 318B HANGUL LETTER YU-YE
    # ->   1192 HANGUL JUNGSEONG YU-YE
    "\u{318B}" => "\u{1192}",

    # 318C HANGUL LETTER YU-I
    # ->   1194 HANGUL JUNGSEONG YU-I
    "\u{318C}" => "\u{1194}",

    # 318D HANGUL LETTER ARAEA
    # ->   119E HANGUL JUNGSEONG ARAEA
    "\u{318D}" => "\u{119E}",

    # 318E HANGUL LETTER ARAEAE
    # ->   11A1 HANGUL JUNGSEONG ARAEA-I
    "\u{318E}" => "\u{11A1}",

    # 3192 IDEOGRAPHIC ANNOTATION ONE MARK
    # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
    "\u{3192}" => "\u{4E00}",

    # 3193 IDEOGRAPHIC ANNOTATION TWO MARK
    # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
    "\u{3193}" => "\u{4E8C}",

    # 3194 IDEOGRAPHIC ANNOTATION THREE MARK
    # ->   4E09 CJK UNIFIED IDEOGRAPH 4E09
    "\u{3194}" => "\u{4E09}",

    # 3195 IDEOGRAPHIC ANNOTATION FOUR MARK
    # ->   56DB CJK UNIFIED IDEOGRAPH 56DB
    "\u{3195}" => "\u{56DB}",

    # 3196 IDEOGRAPHIC ANNOTATION TOP MARK
    # ->   4E0A CJK UNIFIED IDEOGRAPH 4E0A
    "\u{3196}" => "\u{4E0A}",

    # 3197 IDEOGRAPHIC ANNOTATION MIDDLE MARK
    # ->   4E2D CJK UNIFIED IDEOGRAPH 4E2D
    "\u{3197}" => "\u{4E2D}",

    # 3198 IDEOGRAPHIC ANNOTATION BOTTOM MARK
    # ->   4E0B CJK UNIFIED IDEOGRAPH 4E0B
    "\u{3198}" => "\u{4E0B}",

    # 3199 IDEOGRAPHIC ANNOTATION FIRST MARK
    # ->   7532 CJK UNIFIED IDEOGRAPH 7532
    "\u{3199}" => "\u{7532}",

    # 319A IDEOGRAPHIC ANNOTATION SECOND MARK
    # ->   4E59 CJK UNIFIED IDEOGRAPH 4E59
    "\u{319A}" => "\u{4E59}",

    # 319B IDEOGRAPHIC ANNOTATION THIRD MARK
    # ->   4E19 CJK UNIFIED IDEOGRAPH 4E19
    "\u{319B}" => "\u{4E19}",

    # 319C IDEOGRAPHIC ANNOTATION FOURTH MARK
    # ->   4E01 CJK UNIFIED IDEOGRAPH 4E01
    "\u{319C}" => "\u{4E01}",

    # 319D IDEOGRAPHIC ANNOTATION HEAVEN MARK
    # ->   5929 CJK UNIFIED IDEOGRAPH 5929
    "\u{319D}" => "\u{5929}",

    # 319E IDEOGRAPHIC ANNOTATION EARTH MARK
    # ->   5730 CJK UNIFIED IDEOGRAPH 5730
    "\u{319E}" => "\u{5730}",

    # 319F IDEOGRAPHIC ANNOTATION MAN MARK
    # ->   4EBA CJK UNIFIED IDEOGRAPH 4EBA
    "\u{319F}" => "\u{4EBA}",

    # 3200 PARENTHESIZED HANGUL KIYEOK
    # ->   0028 LEFT PARENTHESIS
    #    + 1100 HANGUL CHOSEONG KIYEOK
    #    + 0029 RIGHT PARENTHESIS
    "\u{3200}" => "(\u{1100})",

    # 3201 PARENTHESIZED HANGUL NIEUN
    # ->   0028 LEFT PARENTHESIS
    #    + 1102 HANGUL CHOSEONG NIEUN
    #    + 0029 RIGHT PARENTHESIS
    "\u{3201}" => "(\u{1102})",

    # 3202 PARENTHESIZED HANGUL TIKEUT
    # ->   0028 LEFT PARENTHESIS
    #    + 1103 HANGUL CHOSEONG TIKEUT
    #    + 0029 RIGHT PARENTHESIS
    "\u{3202}" => "(\u{1103})",

    # 3203 PARENTHESIZED HANGUL RIEUL
    # ->   0028 LEFT PARENTHESIS
    #    + 1105 HANGUL CHOSEONG RIEUL
    #    + 0029 RIGHT PARENTHESIS
    "\u{3203}" => "(\u{1105})",

    # 3204 PARENTHESIZED HANGUL MIEUM
    # ->   0028 LEFT PARENTHESIS
    #    + 1106 HANGUL CHOSEONG MIEUM
    #    + 0029 RIGHT PARENTHESIS
    "\u{3204}" => "(\u{1106})",

    # 3205 PARENTHESIZED HANGUL PIEUP
    # ->   0028 LEFT PARENTHESIS
    #    + 1107 HANGUL CHOSEONG PIEUP
    #    + 0029 RIGHT PARENTHESIS
    "\u{3205}" => "(\u{1107})",

    # 3206 PARENTHESIZED HANGUL SIOS
    # ->   0028 LEFT PARENTHESIS
    #    + 1109 HANGUL CHOSEONG SIOS
    #    + 0029 RIGHT PARENTHESIS
    "\u{3206}" => "(\u{1109})",

    # 3207 PARENTHESIZED HANGUL IEUNG
    # ->   0028 LEFT PARENTHESIS
    #    + 110B HANGUL CHOSEONG IEUNG
    #    + 0029 RIGHT PARENTHESIS
    "\u{3207}" => "(\u{110B})",

    # 3208 PARENTHESIZED HANGUL CIEUC
    # ->   0028 LEFT PARENTHESIS
    #    + 110C HANGUL CHOSEONG CIEUC
    #    + 0029 RIGHT PARENTHESIS
    "\u{3208}" => "(\u{110C})",

    # 3209 PARENTHESIZED HANGUL CHIEUCH
    # ->   0028 LEFT PARENTHESIS
    #    + 110E HANGUL CHOSEONG CHIEUCH
    #    + 0029 RIGHT PARENTHESIS
    "\u{3209}" => "(\u{110E})",

    # 320A PARENTHESIZED HANGUL KHIEUKH
    # ->   0028 LEFT PARENTHESIS
    #    + 110F HANGUL CHOSEONG KHIEUKH
    #    + 0029 RIGHT PARENTHESIS
    "\u{320A}" => "(\u{110F})",

    # 320B PARENTHESIZED HANGUL THIEUTH
    # ->   0028 LEFT PARENTHESIS
    #    + 1110 HANGUL CHOSEONG THIEUTH
    #    + 0029 RIGHT PARENTHESIS
    "\u{320B}" => "(\u{1110})",

    # 320C PARENTHESIZED HANGUL PHIEUPH
    # ->   0028 LEFT PARENTHESIS
    #    + 1111 HANGUL CHOSEONG PHIEUPH
    #    + 0029 RIGHT PARENTHESIS
    "\u{320C}" => "(\u{1111})",

    # 320D PARENTHESIZED HANGUL HIEUH
    # ->   0028 LEFT PARENTHESIS
    #    + 1112 HANGUL CHOSEONG HIEUH
    #    + 0029 RIGHT PARENTHESIS
    "\u{320D}" => "(\u{1112})",

    # 320E PARENTHESIZED HANGUL KIYEOK A
    # ->   0028 LEFT PARENTHESIS
    #    + 1100 HANGUL CHOSEONG KIYEOK
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{320E}" => "(\u{1100}\u{1161})",

    # 320F PARENTHESIZED HANGUL NIEUN A
    # ->   0028 LEFT PARENTHESIS
    #    + 1102 HANGUL CHOSEONG NIEUN
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{320F}" => "(\u{1102}\u{1161})",

    # 3210 PARENTHESIZED HANGUL TIKEUT A
    # ->   0028 LEFT PARENTHESIS
    #    + 1103 HANGUL CHOSEONG TIKEUT
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3210}" => "(\u{1103}\u{1161})",

    # 3211 PARENTHESIZED HANGUL RIEUL A
    # ->   0028 LEFT PARENTHESIS
    #    + 1105 HANGUL CHOSEONG RIEUL
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3211}" => "(\u{1105}\u{1161})",

    # 3212 PARENTHESIZED HANGUL MIEUM A
    # ->   0028 LEFT PARENTHESIS
    #    + 1106 HANGUL CHOSEONG MIEUM
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3212}" => "(\u{1106}\u{1161})",

    # 3213 PARENTHESIZED HANGUL PIEUP A
    # ->   0028 LEFT PARENTHESIS
    #    + 1107 HANGUL CHOSEONG PIEUP
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3213}" => "(\u{1107}\u{1161})",

    # 3214 PARENTHESIZED HANGUL SIOS A
    # ->   0028 LEFT PARENTHESIS
    #    + 1109 HANGUL CHOSEONG SIOS
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3214}" => "(\u{1109}\u{1161})",

    # 3215 PARENTHESIZED HANGUL IEUNG A
    # ->   0028 LEFT PARENTHESIS
    #    + 110B HANGUL CHOSEONG IEUNG
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3215}" => "(\u{110B}\u{1161})",

    # 3216 PARENTHESIZED HANGUL CIEUC A
    # ->   0028 LEFT PARENTHESIS
    #    + 110C HANGUL CHOSEONG CIEUC
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3216}" => "(\u{110C}\u{1161})",

    # 3217 PARENTHESIZED HANGUL CHIEUCH A
    # ->   0028 LEFT PARENTHESIS
    #    + 110E HANGUL CHOSEONG CHIEUCH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3217}" => "(\u{110E}\u{1161})",

    # 3218 PARENTHESIZED HANGUL KHIEUKH A
    # ->   0028 LEFT PARENTHESIS
    #    + 110F HANGUL CHOSEONG KHIEUKH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3218}" => "(\u{110F}\u{1161})",

    # 3219 PARENTHESIZED HANGUL THIEUTH A
    # ->   0028 LEFT PARENTHESIS
    #    + 1110 HANGUL CHOSEONG THIEUTH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3219}" => "(\u{1110}\u{1161})",

    # 321A PARENTHESIZED HANGUL PHIEUPH A
    # ->   0028 LEFT PARENTHESIS
    #    + 1111 HANGUL CHOSEONG PHIEUPH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{321A}" => "(\u{1111}\u{1161})",

    # 321B PARENTHESIZED HANGUL HIEUH A
    # ->   0028 LEFT PARENTHESIS
    #    + 1112 HANGUL CHOSEONG HIEUH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 0029 RIGHT PARENTHESIS
    "\u{321B}" => "(\u{1112}\u{1161})",

    # 321C PARENTHESIZED HANGUL CIEUC U
    # ->   0028 LEFT PARENTHESIS
    #    + 110C HANGUL CHOSEONG CIEUC
    #    + 116E HANGUL JUNGSEONG U
    #    + 0029 RIGHT PARENTHESIS
    "\u{321C}" => "(\u{110C}\u{116E})",

    # 321D PARENTHESIZED KOREAN CHARACTER OJEON
    # ->   0028 LEFT PARENTHESIS
    #    + 110B HANGUL CHOSEONG IEUNG
    #    + 1169 HANGUL JUNGSEONG O
    #    + 110C HANGUL CHOSEONG CIEUC
    #    + 1165 HANGUL JUNGSEONG EO
    #    + 11AB HANGUL JONGSEONG NIEUN
    #    + 0029 RIGHT PARENTHESIS
    "\u{321D}" => "(\u{110B}\u{1169}\u{110C}\u{1165}\u{11AB})",

    # 321E PARENTHESIZED KOREAN CHARACTER O HU
    # ->   0028 LEFT PARENTHESIS
    #    + 110B HANGUL CHOSEONG IEUNG
    #    + 1169 HANGUL JUNGSEONG O
    #    + 1112 HANGUL CHOSEONG HIEUH
    #    + 116E HANGUL JUNGSEONG U
    #    + 0029 RIGHT PARENTHESIS
    "\u{321E}" => "(\u{110B}\u{1169}\u{1112}\u{116E})",

    # 3220 PARENTHESIZED IDEOGRAPH ONE
    # ->   0028 LEFT PARENTHESIS
    #    + 4E00 CJK UNIFIED IDEOGRAPH 4E00
    #    + 0029 RIGHT PARENTHESIS
    "\u{3220}" => "(\u{4E00})",

    # 3221 PARENTHESIZED IDEOGRAPH TWO
    # ->   0028 LEFT PARENTHESIS
    #    + 4E8C CJK UNIFIED IDEOGRAPH 4E8C
    #    + 0029 RIGHT PARENTHESIS
    "\u{3221}" => "(\u{4E8C})",

    # 3222 PARENTHESIZED IDEOGRAPH THREE
    # ->   0028 LEFT PARENTHESIS
    #    + 4E09 CJK UNIFIED IDEOGRAPH 4E09
    #    + 0029 RIGHT PARENTHESIS
    "\u{3222}" => "(\u{4E09})",

    # 3223 PARENTHESIZED IDEOGRAPH FOUR
    # ->   0028 LEFT PARENTHESIS
    #    + 56DB CJK UNIFIED IDEOGRAPH 56DB
    #    + 0029 RIGHT PARENTHESIS
    "\u{3223}" => "(\u{56DB})",

    # 3224 PARENTHESIZED IDEOGRAPH FIVE
    # ->   0028 LEFT PARENTHESIS
    #    + 4E94 CJK UNIFIED IDEOGRAPH 4E94
    #    + 0029 RIGHT PARENTHESIS
    "\u{3224}" => "(\u{4E94})",

    # 3225 PARENTHESIZED IDEOGRAPH SIX
    # ->   0028 LEFT PARENTHESIS
    #    + 516D CJK UNIFIED IDEOGRAPH 516D
    #    + 0029 RIGHT PARENTHESIS
    "\u{3225}" => "(\u{516D})",

    # 3226 PARENTHESIZED IDEOGRAPH SEVEN
    # ->   0028 LEFT PARENTHESIS
    #    + 4E03 CJK UNIFIED IDEOGRAPH 4E03
    #    + 0029 RIGHT PARENTHESIS
    "\u{3226}" => "(\u{4E03})",

    # 3227 PARENTHESIZED IDEOGRAPH EIGHT
    # ->   0028 LEFT PARENTHESIS
    #    + 516B CJK UNIFIED IDEOGRAPH 516B
    #    + 0029 RIGHT PARENTHESIS
    "\u{3227}" => "(\u{516B})",

    # 3228 PARENTHESIZED IDEOGRAPH NINE
    # ->   0028 LEFT PARENTHESIS
    #    + 4E5D CJK UNIFIED IDEOGRAPH 4E5D
    #    + 0029 RIGHT PARENTHESIS
    "\u{3228}" => "(\u{4E5D})",

    # 3229 PARENTHESIZED IDEOGRAPH TEN
    # ->   0028 LEFT PARENTHESIS
    #    + 5341 CJK UNIFIED IDEOGRAPH 5341
    #    + 0029 RIGHT PARENTHESIS
    "\u{3229}" => "(\u{5341})",

    # 322A PARENTHESIZED IDEOGRAPH MOON
    # ->   0028 LEFT PARENTHESIS
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    #    + 0029 RIGHT PARENTHESIS
    "\u{322A}" => "(\u{6708})",

    # 322B PARENTHESIZED IDEOGRAPH FIRE
    # ->   0028 LEFT PARENTHESIS
    #    + 706B CJK UNIFIED IDEOGRAPH 706B
    #    + 0029 RIGHT PARENTHESIS
    "\u{322B}" => "(\u{706B})",

    # 322C PARENTHESIZED IDEOGRAPH WATER
    # ->   0028 LEFT PARENTHESIS
    #    + 6C34 CJK UNIFIED IDEOGRAPH 6C34
    #    + 0029 RIGHT PARENTHESIS
    "\u{322C}" => "(\u{6C34})",

    # 322D PARENTHESIZED IDEOGRAPH WOOD
    # ->   0028 LEFT PARENTHESIS
    #    + 6728 CJK UNIFIED IDEOGRAPH 6728
    #    + 0029 RIGHT PARENTHESIS
    "\u{322D}" => "(\u{6728})",

    # 322E PARENTHESIZED IDEOGRAPH METAL
    # ->   0028 LEFT PARENTHESIS
    #    + 91D1 CJK UNIFIED IDEOGRAPH 91D1
    #    + 0029 RIGHT PARENTHESIS
    "\u{322E}" => "(\u{91D1})",

    # 322F PARENTHESIZED IDEOGRAPH EARTH
    # ->   0028 LEFT PARENTHESIS
    #    + 571F CJK UNIFIED IDEOGRAPH 571F
    #    + 0029 RIGHT PARENTHESIS
    "\u{322F}" => "(\u{571F})",

    # 3230 PARENTHESIZED IDEOGRAPH SUN
    # ->   0028 LEFT PARENTHESIS
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    #    + 0029 RIGHT PARENTHESIS
    "\u{3230}" => "(\u{65E5})",

    # 3231 PARENTHESIZED IDEOGRAPH STOCK
    # ->   0028 LEFT PARENTHESIS
    #    + 682A CJK UNIFIED IDEOGRAPH 682A
    #    + 0029 RIGHT PARENTHESIS
    "\u{3231}" => "(\u{682A})",

    # 3232 PARENTHESIZED IDEOGRAPH HAVE
    # ->   0028 LEFT PARENTHESIS
    #    + 6709 CJK UNIFIED IDEOGRAPH 6709
    #    + 0029 RIGHT PARENTHESIS
    "\u{3232}" => "(\u{6709})",

    # 3233 PARENTHESIZED IDEOGRAPH SOCIETY
    # ->   0028 LEFT PARENTHESIS
    #    + 793E CJK UNIFIED IDEOGRAPH 793E
    #    + 0029 RIGHT PARENTHESIS
    "\u{3233}" => "(\u{793E})",

    # 3234 PARENTHESIZED IDEOGRAPH NAME
    # ->   0028 LEFT PARENTHESIS
    #    + 540D CJK UNIFIED IDEOGRAPH 540D
    #    + 0029 RIGHT PARENTHESIS
    "\u{3234}" => "(\u{540D})",

    # 3235 PARENTHESIZED IDEOGRAPH SPECIAL
    # ->   0028 LEFT PARENTHESIS
    #    + 7279 CJK UNIFIED IDEOGRAPH 7279
    #    + 0029 RIGHT PARENTHESIS
    "\u{3235}" => "(\u{7279})",

    # 3236 PARENTHESIZED IDEOGRAPH FINANCIAL
    # ->   0028 LEFT PARENTHESIS
    #    + 8CA1 CJK UNIFIED IDEOGRAPH 8CA1
    #    + 0029 RIGHT PARENTHESIS
    "\u{3236}" => "(\u{8CA1})",

    # 3237 PARENTHESIZED IDEOGRAPH CONGRATULATION
    # ->   0028 LEFT PARENTHESIS
    #    + 795D CJK UNIFIED IDEOGRAPH 795D
    #    + 0029 RIGHT PARENTHESIS
    "\u{3237}" => "(\u{795D})",

    # 3238 PARENTHESIZED IDEOGRAPH LABOR
    # ->   0028 LEFT PARENTHESIS
    #    + 52B4 CJK UNIFIED IDEOGRAPH 52B4
    #    + 0029 RIGHT PARENTHESIS
    "\u{3238}" => "(\u{52B4})",

    # 3239 PARENTHESIZED IDEOGRAPH REPRESENT
    # ->   0028 LEFT PARENTHESIS
    #    + 4EE3 CJK UNIFIED IDEOGRAPH 4EE3
    #    + 0029 RIGHT PARENTHESIS
    "\u{3239}" => "(\u{4EE3})",

    # 323A PARENTHESIZED IDEOGRAPH CALL
    # ->   0028 LEFT PARENTHESIS
    #    + 547C CJK UNIFIED IDEOGRAPH 547C
    #    + 0029 RIGHT PARENTHESIS
    "\u{323A}" => "(\u{547C})",

    # 323B PARENTHESIZED IDEOGRAPH STUDY
    # ->   0028 LEFT PARENTHESIS
    #    + 5B66 CJK UNIFIED IDEOGRAPH 5B66
    #    + 0029 RIGHT PARENTHESIS
    "\u{323B}" => "(\u{5B66})",

    # 323C PARENTHESIZED IDEOGRAPH SUPERVISE
    # ->   0028 LEFT PARENTHESIS
    #    + 76E3 CJK UNIFIED IDEOGRAPH 76E3
    #    + 0029 RIGHT PARENTHESIS
    "\u{323C}" => "(\u{76E3})",

    # 323D PARENTHESIZED IDEOGRAPH ENTERPRISE
    # ->   0028 LEFT PARENTHESIS
    #    + 4F01 CJK UNIFIED IDEOGRAPH 4F01
    #    + 0029 RIGHT PARENTHESIS
    "\u{323D}" => "(\u{4F01})",

    # 323E PARENTHESIZED IDEOGRAPH RESOURCE
    # ->   0028 LEFT PARENTHESIS
    #    + 8CC7 CJK UNIFIED IDEOGRAPH 8CC7
    #    + 0029 RIGHT PARENTHESIS
    "\u{323E}" => "(\u{8CC7})",

    # 323F PARENTHESIZED IDEOGRAPH ALLIANCE
    # ->   0028 LEFT PARENTHESIS
    #    + 5354 CJK UNIFIED IDEOGRAPH 5354
    #    + 0029 RIGHT PARENTHESIS
    "\u{323F}" => "(\u{5354})",

    # 3240 PARENTHESIZED IDEOGRAPH FESTIVAL
    # ->   0028 LEFT PARENTHESIS
    #    + 796D CJK UNIFIED IDEOGRAPH 796D
    #    + 0029 RIGHT PARENTHESIS
    "\u{3240}" => "(\u{796D})",

    # 3241 PARENTHESIZED IDEOGRAPH REST
    # ->   0028 LEFT PARENTHESIS
    #    + 4F11 CJK UNIFIED IDEOGRAPH 4F11
    #    + 0029 RIGHT PARENTHESIS
    "\u{3241}" => "(\u{4F11})",

    # 3242 PARENTHESIZED IDEOGRAPH SELF
    # ->   0028 LEFT PARENTHESIS
    #    + 81EA CJK UNIFIED IDEOGRAPH 81EA
    #    + 0029 RIGHT PARENTHESIS
    "\u{3242}" => "(\u{81EA})",

    # 3243 PARENTHESIZED IDEOGRAPH REACH
    # ->   0028 LEFT PARENTHESIS
    #    + 81F3 CJK UNIFIED IDEOGRAPH 81F3
    #    + 0029 RIGHT PARENTHESIS
    "\u{3243}" => "(\u{81F3})",

    # 3250 PARTNERSHIP SIGN
    # ->   0050 LATIN CAPITAL LETTER P
    #    + 0054 LATIN CAPITAL LETTER T
    #    + 0045 LATIN CAPITAL LETTER E
    "\u{3250}" => "PTE",

    # 3251 CIRCLED NUMBER TWENTY ONE
    # ->   0032 DIGIT TWO
    #    + 0031 DIGIT ONE
    "\u{3251}" => "21",

    # 3252 CIRCLED NUMBER TWENTY TWO
    # ->   0032 DIGIT TWO
    #    + 0032 DIGIT TWO
    "\u{3252}" => "22",

    # 3253 CIRCLED NUMBER TWENTY THREE
    # ->   0032 DIGIT TWO
    #    + 0033 DIGIT THREE
    "\u{3253}" => "23",

    # 3254 CIRCLED NUMBER TWENTY FOUR
    # ->   0032 DIGIT TWO
    #    + 0034 DIGIT FOUR
    "\u{3254}" => "24",

    # 3255 CIRCLED NUMBER TWENTY FIVE
    # ->   0032 DIGIT TWO
    #    + 0035 DIGIT FIVE
    "\u{3255}" => "25",

    # 3256 CIRCLED NUMBER TWENTY SIX
    # ->   0032 DIGIT TWO
    #    + 0036 DIGIT SIX
    "\u{3256}" => "26",

    # 3257 CIRCLED NUMBER TWENTY SEVEN
    # ->   0032 DIGIT TWO
    #    + 0037 DIGIT SEVEN
    "\u{3257}" => "27",

    # 3258 CIRCLED NUMBER TWENTY EIGHT
    # ->   0032 DIGIT TWO
    #    + 0038 DIGIT EIGHT
    "\u{3258}" => "28",

    # 3259 CIRCLED NUMBER TWENTY NINE
    # ->   0032 DIGIT TWO
    #    + 0039 DIGIT NINE
    "\u{3259}" => "29",

    # 325A CIRCLED NUMBER THIRTY
    # ->   0033 DIGIT THREE
    #    + 0030 DIGIT ZERO
    "\u{325A}" => "30",

    # 325B CIRCLED NUMBER THIRTY ONE
    # ->   0033 DIGIT THREE
    #    + 0031 DIGIT ONE
    "\u{325B}" => "31",

    # 325C CIRCLED NUMBER THIRTY TWO
    # ->   0033 DIGIT THREE
    #    + 0032 DIGIT TWO
    "\u{325C}" => "32",

    # 325D CIRCLED NUMBER THIRTY THREE
    # ->   0033 DIGIT THREE
    #    + 0033 DIGIT THREE
    "\u{325D}" => "33",

    # 325E CIRCLED NUMBER THIRTY FOUR
    # ->   0033 DIGIT THREE
    #    + 0034 DIGIT FOUR
    "\u{325E}" => "34",

    # 325F CIRCLED NUMBER THIRTY FIVE
    # ->   0033 DIGIT THREE
    #    + 0035 DIGIT FIVE
    "\u{325F}" => "35",

    # 3260 CIRCLED HANGUL KIYEOK
    # ->   1100 HANGUL CHOSEONG KIYEOK
    "\u{3260}" => "\u{1100}",

    # 3261 CIRCLED HANGUL NIEUN
    # ->   1102 HANGUL CHOSEONG NIEUN
    "\u{3261}" => "\u{1102}",

    # 3262 CIRCLED HANGUL TIKEUT
    # ->   1103 HANGUL CHOSEONG TIKEUT
    "\u{3262}" => "\u{1103}",

    # 3263 CIRCLED HANGUL RIEUL
    # ->   1105 HANGUL CHOSEONG RIEUL
    "\u{3263}" => "\u{1105}",

    # 3264 CIRCLED HANGUL MIEUM
    # ->   1106 HANGUL CHOSEONG MIEUM
    "\u{3264}" => "\u{1106}",

    # 3265 CIRCLED HANGUL PIEUP
    # ->   1107 HANGUL CHOSEONG PIEUP
    "\u{3265}" => "\u{1107}",

    # 3266 CIRCLED HANGUL SIOS
    # ->   1109 HANGUL CHOSEONG SIOS
    "\u{3266}" => "\u{1109}",

    # 3267 CIRCLED HANGUL IEUNG
    # ->   110B HANGUL CHOSEONG IEUNG
    "\u{3267}" => "\u{110B}",

    # 3268 CIRCLED HANGUL CIEUC
    # ->   110C HANGUL CHOSEONG CIEUC
    "\u{3268}" => "\u{110C}",

    # 3269 CIRCLED HANGUL CHIEUCH
    # ->   110E HANGUL CHOSEONG CHIEUCH
    "\u{3269}" => "\u{110E}",

    # 326A CIRCLED HANGUL KHIEUKH
    # ->   110F HANGUL CHOSEONG KHIEUKH
    "\u{326A}" => "\u{110F}",

    # 326B CIRCLED HANGUL THIEUTH
    # ->   1110 HANGUL CHOSEONG THIEUTH
    "\u{326B}" => "\u{1110}",

    # 326C CIRCLED HANGUL PHIEUPH
    # ->   1111 HANGUL CHOSEONG PHIEUPH
    "\u{326C}" => "\u{1111}",

    # 326D CIRCLED HANGUL HIEUH
    # ->   1112 HANGUL CHOSEONG HIEUH
    "\u{326D}" => "\u{1112}",

    # 326E CIRCLED HANGUL KIYEOK A
    # ->   1100 HANGUL CHOSEONG KIYEOK
    #    + 1161 HANGUL JUNGSEONG A
    "\u{326E}" => "\u{1100}\u{1161}",

    # 326F CIRCLED HANGUL NIEUN A
    # ->   1102 HANGUL CHOSEONG NIEUN
    #    + 1161 HANGUL JUNGSEONG A
    "\u{326F}" => "\u{1102}\u{1161}",

    # 3270 CIRCLED HANGUL TIKEUT A
    # ->   1103 HANGUL CHOSEONG TIKEUT
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3270}" => "\u{1103}\u{1161}",

    # 3271 CIRCLED HANGUL RIEUL A
    # ->   1105 HANGUL CHOSEONG RIEUL
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3271}" => "\u{1105}\u{1161}",

    # 3272 CIRCLED HANGUL MIEUM A
    # ->   1106 HANGUL CHOSEONG MIEUM
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3272}" => "\u{1106}\u{1161}",

    # 3273 CIRCLED HANGUL PIEUP A
    # ->   1107 HANGUL CHOSEONG PIEUP
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3273}" => "\u{1107}\u{1161}",

    # 3274 CIRCLED HANGUL SIOS A
    # ->   1109 HANGUL CHOSEONG SIOS
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3274}" => "\u{1109}\u{1161}",

    # 3275 CIRCLED HANGUL IEUNG A
    # ->   110B HANGUL CHOSEONG IEUNG
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3275}" => "\u{110B}\u{1161}",

    # 3276 CIRCLED HANGUL CIEUC A
    # ->   110C HANGUL CHOSEONG CIEUC
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3276}" => "\u{110C}\u{1161}",

    # 3277 CIRCLED HANGUL CHIEUCH A
    # ->   110E HANGUL CHOSEONG CHIEUCH
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3277}" => "\u{110E}\u{1161}",

    # 3278 CIRCLED HANGUL KHIEUKH A
    # ->   110F HANGUL CHOSEONG KHIEUKH
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3278}" => "\u{110F}\u{1161}",

    # 3279 CIRCLED HANGUL THIEUTH A
    # ->   1110 HANGUL CHOSEONG THIEUTH
    #    + 1161 HANGUL JUNGSEONG A
    "\u{3279}" => "\u{1110}\u{1161}",

    # 327A CIRCLED HANGUL PHIEUPH A
    # ->   1111 HANGUL CHOSEONG PHIEUPH
    #    + 1161 HANGUL JUNGSEONG A
    "\u{327A}" => "\u{1111}\u{1161}",

    # 327B CIRCLED HANGUL HIEUH A
    # ->   1112 HANGUL CHOSEONG HIEUH
    #    + 1161 HANGUL JUNGSEONG A
    "\u{327B}" => "\u{1112}\u{1161}",

    # 327C CIRCLED KOREAN CHARACTER CHAMKO
    # ->   110E HANGUL CHOSEONG CHIEUCH
    #    + 1161 HANGUL JUNGSEONG A
    #    + 11B7 HANGUL JONGSEONG MIEUM
    #    + 1100 HANGUL CHOSEONG KIYEOK
    #    + 1169 HANGUL JUNGSEONG O
    "\u{327C}" => "\u{110E}\u{1161}\u{11B7}\u{1100}\u{1169}",

    # 327D CIRCLED KOREAN CHARACTER JUEUI
    # ->   110C HANGUL CHOSEONG CIEUC
    #    + 116E HANGUL JUNGSEONG U
    #    + 110B HANGUL CHOSEONG IEUNG
    #    + 1174 HANGUL JUNGSEONG YI
    "\u{327D}" => "\u{110C}\u{116E}\u{110B}\u{1174}",

    # 3280 CIRCLED IDEOGRAPH ONE
    # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
    "\u{3280}" => "\u{4E00}",

    # 3281 CIRCLED IDEOGRAPH TWO
    # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
    "\u{3281}" => "\u{4E8C}",

    # 3282 CIRCLED IDEOGRAPH THREE
    # ->   4E09 CJK UNIFIED IDEOGRAPH 4E09
    "\u{3282}" => "\u{4E09}",

    # 3283 CIRCLED IDEOGRAPH FOUR
    # ->   56DB CJK UNIFIED IDEOGRAPH 56DB
    "\u{3283}" => "\u{56DB}",

    # 3284 CIRCLED IDEOGRAPH FIVE
    # ->   4E94 CJK UNIFIED IDEOGRAPH 4E94
    "\u{3284}" => "\u{4E94}",

    # 3285 CIRCLED IDEOGRAPH SIX
    # ->   516D CJK UNIFIED IDEOGRAPH 516D
    "\u{3285}" => "\u{516D}",

    # 3286 CIRCLED IDEOGRAPH SEVEN
    # ->   4E03 CJK UNIFIED IDEOGRAPH 4E03
    "\u{3286}" => "\u{4E03}",

    # 3287 CIRCLED IDEOGRAPH EIGHT
    # ->   516B CJK UNIFIED IDEOGRAPH 516B
    "\u{3287}" => "\u{516B}",

    # 3288 CIRCLED IDEOGRAPH NINE
    # ->   4E5D CJK UNIFIED IDEOGRAPH 4E5D
    "\u{3288}" => "\u{4E5D}",

    # 3289 CIRCLED IDEOGRAPH TEN
    # ->   5341 CJK UNIFIED IDEOGRAPH 5341
    "\u{3289}" => "\u{5341}",

    # 328A CIRCLED IDEOGRAPH MOON
    # ->   6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{328A}" => "\u{6708}",

    # 328B CIRCLED IDEOGRAPH FIRE
    # ->   706B CJK UNIFIED IDEOGRAPH 706B
    "\u{328B}" => "\u{706B}",

    # 328C CIRCLED IDEOGRAPH WATER
    # ->   6C34 CJK UNIFIED IDEOGRAPH 6C34
    "\u{328C}" => "\u{6C34}",

    # 328D CIRCLED IDEOGRAPH WOOD
    # ->   6728 CJK UNIFIED IDEOGRAPH 6728
    "\u{328D}" => "\u{6728}",

    # 328E CIRCLED IDEOGRAPH METAL
    # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
    "\u{328E}" => "\u{91D1}",

    # 328F CIRCLED IDEOGRAPH EARTH
    # ->   571F CJK UNIFIED IDEOGRAPH 571F
    "\u{328F}" => "\u{571F}",

    # 3290 CIRCLED IDEOGRAPH SUN
    # ->   65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{3290}" => "\u{65E5}",

    # 3291 CIRCLED IDEOGRAPH STOCK
    # ->   682A CJK UNIFIED IDEOGRAPH 682A
    "\u{3291}" => "\u{682A}",

    # 3292 CIRCLED IDEOGRAPH HAVE
    # ->   6709 CJK UNIFIED IDEOGRAPH 6709
    "\u{3292}" => "\u{6709}",

    # 3293 CIRCLED IDEOGRAPH SOCIETY
    # ->   793E CJK UNIFIED IDEOGRAPH 793E
    "\u{3293}" => "\u{793E}",

    # 3294 CIRCLED IDEOGRAPH NAME
    # ->   540D CJK UNIFIED IDEOGRAPH 540D
    "\u{3294}" => "\u{540D}",

    # 3295 CIRCLED IDEOGRAPH SPECIAL
    # ->   7279 CJK UNIFIED IDEOGRAPH 7279
    "\u{3295}" => "\u{7279}",

    # 3296 CIRCLED IDEOGRAPH FINANCIAL
    # ->   8CA1 CJK UNIFIED IDEOGRAPH 8CA1
    "\u{3296}" => "\u{8CA1}",

    # 3297 CIRCLED IDEOGRAPH CONGRATULATION
    # ->   795D CJK UNIFIED IDEOGRAPH 795D
    "\u{3297}" => "\u{795D}",

    # 3298 CIRCLED IDEOGRAPH LABOR
    # ->   52B4 CJK UNIFIED IDEOGRAPH 52B4
    "\u{3298}" => "\u{52B4}",

    # 3299 CIRCLED IDEOGRAPH SECRET
    # ->   79D8 CJK UNIFIED IDEOGRAPH 79D8
    "\u{3299}" => "\u{79D8}",

    # 329A CIRCLED IDEOGRAPH MALE
    # ->   7537 CJK UNIFIED IDEOGRAPH 7537
    "\u{329A}" => "\u{7537}",

    # 329B CIRCLED IDEOGRAPH FEMALE
    # ->   5973 CJK UNIFIED IDEOGRAPH 5973
    "\u{329B}" => "\u{5973}",

    # 329C CIRCLED IDEOGRAPH SUITABLE
    # ->   9069 CJK UNIFIED IDEOGRAPH 9069
    "\u{329C}" => "\u{9069}",

    # 329D CIRCLED IDEOGRAPH EXCELLENT
    # ->   512A CJK UNIFIED IDEOGRAPH 512A
    "\u{329D}" => "\u{512A}",

    # 329E CIRCLED IDEOGRAPH PRINT
    # ->   5370 CJK UNIFIED IDEOGRAPH 5370
    "\u{329E}" => "\u{5370}",

    # 329F CIRCLED IDEOGRAPH ATTENTION
    # ->   6CE8 CJK UNIFIED IDEOGRAPH 6CE8
    "\u{329F}" => "\u{6CE8}",

    # 32A0 CIRCLED IDEOGRAPH ITEM
    # ->   9805 CJK UNIFIED IDEOGRAPH 9805
    "\u{32A0}" => "\u{9805}",

    # 32A1 CIRCLED IDEOGRAPH REST
    # ->   4F11 CJK UNIFIED IDEOGRAPH 4F11
    "\u{32A1}" => "\u{4F11}",

    # 32A2 CIRCLED IDEOGRAPH COPY
    # ->   5199 CJK UNIFIED IDEOGRAPH 5199
    "\u{32A2}" => "\u{5199}",

    # 32A3 CIRCLED IDEOGRAPH CORRECT
    # ->   6B63 CJK UNIFIED IDEOGRAPH 6B63
    "\u{32A3}" => "\u{6B63}",

    # 32A4 CIRCLED IDEOGRAPH HIGH
    # ->   4E0A CJK UNIFIED IDEOGRAPH 4E0A
    "\u{32A4}" => "\u{4E0A}",

    # 32A5 CIRCLED IDEOGRAPH CENTRE
    # ->   4E2D CJK UNIFIED IDEOGRAPH 4E2D
    "\u{32A5}" => "\u{4E2D}",

    # 32A6 CIRCLED IDEOGRAPH LOW
    # ->   4E0B CJK UNIFIED IDEOGRAPH 4E0B
    "\u{32A6}" => "\u{4E0B}",

    # 32A7 CIRCLED IDEOGRAPH LEFT
    # ->   5DE6 CJK UNIFIED IDEOGRAPH 5DE6
    "\u{32A7}" => "\u{5DE6}",

    # 32A8 CIRCLED IDEOGRAPH RIGHT
    # ->   53F3 CJK UNIFIED IDEOGRAPH 53F3
    "\u{32A8}" => "\u{53F3}",

    # 32A9 CIRCLED IDEOGRAPH MEDICINE
    # ->   533B CJK UNIFIED IDEOGRAPH 533B
    "\u{32A9}" => "\u{533B}",

    # 32AA CIRCLED IDEOGRAPH RELIGION
    # ->   5B97 CJK UNIFIED IDEOGRAPH 5B97
    "\u{32AA}" => "\u{5B97}",

    # 32AB CIRCLED IDEOGRAPH STUDY
    # ->   5B66 CJK UNIFIED IDEOGRAPH 5B66
    "\u{32AB}" => "\u{5B66}",

    # 32AC CIRCLED IDEOGRAPH SUPERVISE
    # ->   76E3 CJK UNIFIED IDEOGRAPH 76E3
    "\u{32AC}" => "\u{76E3}",

    # 32AD CIRCLED IDEOGRAPH ENTERPRISE
    # ->   4F01 CJK UNIFIED IDEOGRAPH 4F01
    "\u{32AD}" => "\u{4F01}",

    # 32AE CIRCLED IDEOGRAPH RESOURCE
    # ->   8CC7 CJK UNIFIED IDEOGRAPH 8CC7
    "\u{32AE}" => "\u{8CC7}",

    # 32AF CIRCLED IDEOGRAPH ALLIANCE
    # ->   5354 CJK UNIFIED IDEOGRAPH 5354
    "\u{32AF}" => "\u{5354}",

    # 32B0 CIRCLED IDEOGRAPH NIGHT
    # ->   591C CJK UNIFIED IDEOGRAPH 591C
    "\u{32B0}" => "\u{591C}",

    # 32B1 CIRCLED NUMBER THIRTY SIX
    # ->   0033 DIGIT THREE
    #    + 0036 DIGIT SIX
    "\u{32B1}" => "36",

    # 32B2 CIRCLED NUMBER THIRTY SEVEN
    # ->   0033 DIGIT THREE
    #    + 0037 DIGIT SEVEN
    "\u{32B2}" => "37",

    # 32B3 CIRCLED NUMBER THIRTY EIGHT
    # ->   0033 DIGIT THREE
    #    + 0038 DIGIT EIGHT
    "\u{32B3}" => "38",

    # 32B4 CIRCLED NUMBER THIRTY NINE
    # ->   0033 DIGIT THREE
    #    + 0039 DIGIT NINE
    "\u{32B4}" => "39",

    # 32B5 CIRCLED NUMBER FORTY
    # ->   0034 DIGIT FOUR
    #    + 0030 DIGIT ZERO
    "\u{32B5}" => "40",

    # 32B6 CIRCLED NUMBER FORTY ONE
    # ->   0034 DIGIT FOUR
    #    + 0031 DIGIT ONE
    "\u{32B6}" => "41",

    # 32B7 CIRCLED NUMBER FORTY TWO
    # ->   0034 DIGIT FOUR
    #    + 0032 DIGIT TWO
    "\u{32B7}" => "42",

    # 32B8 CIRCLED NUMBER FORTY THREE
    # ->   0034 DIGIT FOUR
    #    + 0033 DIGIT THREE
    "\u{32B8}" => "43",

    # 32B9 CIRCLED NUMBER FORTY FOUR
    # ->   0034 DIGIT FOUR
    #    + 0034 DIGIT FOUR
    "\u{32B9}" => "44",

    # 32BA CIRCLED NUMBER FORTY FIVE
    # ->   0034 DIGIT FOUR
    #    + 0035 DIGIT FIVE
    "\u{32BA}" => "45",

    # 32BB CIRCLED NUMBER FORTY SIX
    # ->   0034 DIGIT FOUR
    #    + 0036 DIGIT SIX
    "\u{32BB}" => "46",

    # 32BC CIRCLED NUMBER FORTY SEVEN
    # ->   0034 DIGIT FOUR
    #    + 0037 DIGIT SEVEN
    "\u{32BC}" => "47",

    # 32BD CIRCLED NUMBER FORTY EIGHT
    # ->   0034 DIGIT FOUR
    #    + 0038 DIGIT EIGHT
    "\u{32BD}" => "48",

    # 32BE CIRCLED NUMBER FORTY NINE
    # ->   0034 DIGIT FOUR
    #    + 0039 DIGIT NINE
    "\u{32BE}" => "49",

    # 32BF CIRCLED NUMBER FIFTY
    # ->   0035 DIGIT FIVE
    #    + 0030 DIGIT ZERO
    "\u{32BF}" => "50",

    # 32C0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY
    # ->   0031 DIGIT ONE
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C0}" => "1\u{6708}",

    # 32C1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY
    # ->   0032 DIGIT TWO
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C1}" => "2\u{6708}",

    # 32C2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH
    # ->   0033 DIGIT THREE
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C2}" => "3\u{6708}",

    # 32C3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL
    # ->   0034 DIGIT FOUR
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C3}" => "4\u{6708}",

    # 32C4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY
    # ->   0035 DIGIT FIVE
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C4}" => "5\u{6708}",

    # 32C5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE
    # ->   0036 DIGIT SIX
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C5}" => "6\u{6708}",

    # 32C6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY
    # ->   0037 DIGIT SEVEN
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C6}" => "7\u{6708}",

    # 32C7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST
    # ->   0038 DIGIT EIGHT
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C7}" => "8\u{6708}",

    # 32C8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER
    # ->   0039 DIGIT NINE
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C8}" => "9\u{6708}",

    # 32C9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER
    # ->   0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32C9}" => "10\u{6708}",

    # 32CA IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER
    # ->   0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32CA}" => "11\u{6708}",

    # 32CB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER
    # ->   0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    #    + 6708 CJK UNIFIED IDEOGRAPH 6708
    "\u{32CB}" => "12\u{6708}",

    # 32CC SQUARE HG
    # ->   0048 LATIN CAPITAL LETTER H
    #    + 0067 LATIN SMALL LETTER G
    "\u{32CC}" => "Hg",

    # 32CD SQUARE ERG
    # ->   0065 LATIN SMALL LETTER E
    #    + 0072 LATIN SMALL LETTER R
    #    + 0067 LATIN SMALL LETTER G
    "\u{32CD}" => "erg",

    # 32CE SQUARE EV
    # ->   0065 LATIN SMALL LETTER E
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{32CE}" => "eV",

    # 32CF LIMITED LIABILITY SIGN
    # ->   004C LATIN CAPITAL LETTER L
    #    + 0054 LATIN CAPITAL LETTER T
    #    + 0044 LATIN CAPITAL LETTER D
    "\u{32CF}" => "LTD",

    # 32D0 CIRCLED KATAKANA A
    # ->   30A2 KATAKANA LETTER A
    "\u{32D0}" => "\u{30A2}",

    # 32D1 CIRCLED KATAKANA I
    # ->   30A4 KATAKANA LETTER I
    "\u{32D1}" => "\u{30A4}",

    # 32D2 CIRCLED KATAKANA U
    # ->   30A6 KATAKANA LETTER U
    "\u{32D2}" => "\u{30A6}",

    # 32D3 CIRCLED KATAKANA E
    # ->   30A8 KATAKANA LETTER E
    "\u{32D3}" => "\u{30A8}",

    # 32D4 CIRCLED KATAKANA O
    # ->   30AA KATAKANA LETTER O
    "\u{32D4}" => "\u{30AA}",

    # 32D5 CIRCLED KATAKANA KA
    # ->   30AB KATAKANA LETTER KA
    "\u{32D5}" => "\u{30AB}",

    # 32D6 CIRCLED KATAKANA KI
    # ->   30AD KATAKANA LETTER KI
    "\u{32D6}" => "\u{30AD}",

    # 32D7 CIRCLED KATAKANA KU
    # ->   30AF KATAKANA LETTER KU
    "\u{32D7}" => "\u{30AF}",

    # 32D8 CIRCLED KATAKANA KE
    # ->   30B1 KATAKANA LETTER KE
    "\u{32D8}" => "\u{30B1}",

    # 32D9 CIRCLED KATAKANA KO
    # ->   30B3 KATAKANA LETTER KO
    "\u{32D9}" => "\u{30B3}",

    # 32DA CIRCLED KATAKANA SA
    # ->   30B5 KATAKANA LETTER SA
    "\u{32DA}" => "\u{30B5}",

    # 32DB CIRCLED KATAKANA SI
    # ->   30B7 KATAKANA LETTER SI
    "\u{32DB}" => "\u{30B7}",

    # 32DC CIRCLED KATAKANA SU
    # ->   30B9 KATAKANA LETTER SU
    "\u{32DC}" => "\u{30B9}",

    # 32DD CIRCLED KATAKANA SE
    # ->   30BB KATAKANA LETTER SE
    "\u{32DD}" => "\u{30BB}",

    # 32DE CIRCLED KATAKANA SO
    # ->   30BD KATAKANA LETTER SO
    "\u{32DE}" => "\u{30BD}",

    # 32DF CIRCLED KATAKANA TA
    # ->   30BF KATAKANA LETTER TA
    "\u{32DF}" => "\u{30BF}",

    # 32E0 CIRCLED KATAKANA TI
    # ->   30C1 KATAKANA LETTER TI
    "\u{32E0}" => "\u{30C1}",

    # 32E1 CIRCLED KATAKANA TU
    # ->   30C4 KATAKANA LETTER TU
    "\u{32E1}" => "\u{30C4}",

    # 32E2 CIRCLED KATAKANA TE
    # ->   30C6 KATAKANA LETTER TE
    "\u{32E2}" => "\u{30C6}",

    # 32E3 CIRCLED KATAKANA TO
    # ->   30C8 KATAKANA LETTER TO
    "\u{32E3}" => "\u{30C8}",

    # 32E4 CIRCLED KATAKANA NA
    # ->   30CA KATAKANA LETTER NA
    "\u{32E4}" => "\u{30CA}",

    # 32E5 CIRCLED KATAKANA NI
    # ->   30CB KATAKANA LETTER NI
    "\u{32E5}" => "\u{30CB}",

    # 32E6 CIRCLED KATAKANA NU
    # ->   30CC KATAKANA LETTER NU
    "\u{32E6}" => "\u{30CC}",

    # 32E7 CIRCLED KATAKANA NE
    # ->   30CD KATAKANA LETTER NE
    "\u{32E7}" => "\u{30CD}",

    # 32E8 CIRCLED KATAKANA NO
    # ->   30CE KATAKANA LETTER NO
    "\u{32E8}" => "\u{30CE}",

    # 32E9 CIRCLED KATAKANA HA
    # ->   30CF KATAKANA LETTER HA
    "\u{32E9}" => "\u{30CF}",

    # 32EA CIRCLED KATAKANA HI
    # ->   30D2 KATAKANA LETTER HI
    "\u{32EA}" => "\u{30D2}",

    # 32EB CIRCLED KATAKANA HU
    # ->   30D5 KATAKANA LETTER HU
    "\u{32EB}" => "\u{30D5}",

    # 32EC CIRCLED KATAKANA HE
    # ->   30D8 KATAKANA LETTER HE
    "\u{32EC}" => "\u{30D8}",

    # 32ED CIRCLED KATAKANA HO
    # ->   30DB KATAKANA LETTER HO
    "\u{32ED}" => "\u{30DB}",

    # 32EE CIRCLED KATAKANA MA
    # ->   30DE KATAKANA LETTER MA
    "\u{32EE}" => "\u{30DE}",

    # 32EF CIRCLED KATAKANA MI
    # ->   30DF KATAKANA LETTER MI
    "\u{32EF}" => "\u{30DF}",

    # 32F0 CIRCLED KATAKANA MU
    # ->   30E0 KATAKANA LETTER MU
    "\u{32F0}" => "\u{30E0}",

    # 32F1 CIRCLED KATAKANA ME
    # ->   30E1 KATAKANA LETTER ME
    "\u{32F1}" => "\u{30E1}",

    # 32F2 CIRCLED KATAKANA MO
    # ->   30E2 KATAKANA LETTER MO
    "\u{32F2}" => "\u{30E2}",

    # 32F3 CIRCLED KATAKANA YA
    # ->   30E4 KATAKANA LETTER YA
    "\u{32F3}" => "\u{30E4}",

    # 32F4 CIRCLED KATAKANA YU
    # ->   30E6 KATAKANA LETTER YU
    "\u{32F4}" => "\u{30E6}",

    # 32F5 CIRCLED KATAKANA YO
    # ->   30E8 KATAKANA LETTER YO
    "\u{32F5}" => "\u{30E8}",

    # 32F6 CIRCLED KATAKANA RA
    # ->   30E9 KATAKANA LETTER RA
    "\u{32F6}" => "\u{30E9}",

    # 32F7 CIRCLED KATAKANA RI
    # ->   30EA KATAKANA LETTER RI
    "\u{32F7}" => "\u{30EA}",

    # 32F8 CIRCLED KATAKANA RU
    # ->   30EB KATAKANA LETTER RU
    "\u{32F8}" => "\u{30EB}",

    # 32F9 CIRCLED KATAKANA RE
    # ->   30EC KATAKANA LETTER RE
    "\u{32F9}" => "\u{30EC}",

    # 32FA CIRCLED KATAKANA RO
    # ->   30ED KATAKANA LETTER RO
    "\u{32FA}" => "\u{30ED}",

    # 32FB CIRCLED KATAKANA WA
    # ->   30EF KATAKANA LETTER WA
    "\u{32FB}" => "\u{30EF}",

    # 32FC CIRCLED KATAKANA WI
    # ->   30F0 KATAKANA LETTER WI
    "\u{32FC}" => "\u{30F0}",

    # 32FD CIRCLED KATAKANA WE
    # ->   30F1 KATAKANA LETTER WE
    "\u{32FD}" => "\u{30F1}",

    # 32FE CIRCLED KATAKANA WO
    # ->   30F2 KATAKANA LETTER WO
    "\u{32FE}" => "\u{30F2}",

    # 3300 SQUARE APAATO
    # ->   30A2 KATAKANA LETTER A
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    #    + 30CF KATAKANA LETTER HA
    "\u{3300}" => "\u{30A2}\u{30FC}\u{30C8}\u{30CF}",

    # 3301 SQUARE ARUHUA
    # ->   30A2 KATAKANA LETTER A
    #    + 30EB KATAKANA LETTER RU
    #    + 30D5 KATAKANA LETTER HU
    #    + 30A1 KATAKANA LETTER SMALL A
    "\u{3301}" => "\u{30A2}\u{30EB}\u{30D5}\u{30A1}",

    # 3302 SQUARE ANPEA
    # ->   30A2 KATAKANA LETTER A
    #    + 30F3 KATAKANA LETTER N
    #    + 30A2 KATAKANA LETTER A
    #    + 30D8 KATAKANA LETTER HE
    "\u{3302}" => "\u{30A2}\u{30F3}\u{30A2}\u{30D8}",

    # 3303 SQUARE AARU
    # ->   30A2 KATAKANA LETTER A
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    "\u{3303}" => "\u{30A2}\u{30FC}\u{30EB}",

    # 3304 SQUARE ININGU
    # ->   30A4 KATAKANA LETTER I
    #    + 30CB KATAKANA LETTER NI
    #    + 30F3 KATAKANA LETTER N
    #    + 30AF KATAKANA LETTER KU
    "\u{3304}" => "\u{30A4}\u{30CB}\u{30F3}\u{30AF}",

    # 3305 SQUARE INTI
    # ->   30A4 KATAKANA LETTER I
    #    + 30F3 KATAKANA LETTER N
    #    + 30C1 KATAKANA LETTER TI
    "\u{3305}" => "\u{30A4}\u{30F3}\u{30C1}",

    # 3306 SQUARE UON
    # ->   30A6 KATAKANA LETTER U
    #    + 30A9 KATAKANA LETTER SMALL O
    #    + 30F3 KATAKANA LETTER N
    "\u{3306}" => "\u{30A6}\u{30A9}\u{30F3}",

    # 3307 SQUARE ESUKUUDO
    # ->   30A8 KATAKANA LETTER E
    #    + 30B9 KATAKANA LETTER SU
    #    + 30AF KATAKANA LETTER KU
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    "\u{3307}" => "\u{30A8}\u{30B9}\u{30AF}\u{30FC}\u{30C8}",

    # 3308 SQUARE EEKAA
    # ->   30A8 KATAKANA LETTER E
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30AB KATAKANA LETTER KA
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    "\u{3308}" => "\u{30A8}\u{30FC}\u{30AB}\u{30FC}",

    # 3309 SQUARE ONSU
    # ->   30AA KATAKANA LETTER O
    #    + 30F3 KATAKANA LETTER N
    #    + 30B9 KATAKANA LETTER SU
    "\u{3309}" => "\u{30AA}\u{30F3}\u{30B9}",

    # 330A SQUARE OOMU
    # ->   30AA KATAKANA LETTER O
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30E0 KATAKANA LETTER MU
    "\u{330A}" => "\u{30AA}\u{30FC}\u{30E0}",

    # 330B SQUARE KAIRI
    # ->   30AB KATAKANA LETTER KA
    #    + 30A4 KATAKANA LETTER I
    #    + 30EA KATAKANA LETTER RI
    "\u{330B}" => "\u{30AB}\u{30A4}\u{30EA}",

    # 330C SQUARE KARATTO
    # ->   30AB KATAKANA LETTER KA
    #    + 30E9 KATAKANA LETTER RA
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    "\u{330C}" => "\u{30AB}\u{30E9}\u{30C3}\u{30C8}",

    # 330D SQUARE KARORII
    # ->   30AB KATAKANA LETTER KA
    #    + 30ED KATAKANA LETTER RO
    #    + 30EA KATAKANA LETTER RI
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    "\u{330D}" => "\u{30AB}\u{30ED}\u{30EA}\u{30FC}",

    # 330E SQUARE GARON
    # ->   30ED KATAKANA LETTER RO
    #    + 30F3 KATAKANA LETTER N
    #    + 30AB KATAKANA LETTER KA
    "\u{330E}" => "\u{30ED}\u{30F3}\u{30AB}",

    # 330F SQUARE GANMA
    # ->   30F3 KATAKANA LETTER N
    #    + 30DE KATAKANA LETTER MA
    #    + 30AB KATAKANA LETTER KA
    "\u{330F}" => "\u{30F3}\u{30DE}\u{30AB}",

    # 3310 SQUARE GIGA
    # ->   30AD KATAKANA LETTER KI
    #    + 30AB KATAKANA LETTER KA
    "\u{3310}" => "\u{30AD}\u{30AB}",

    # 3311 SQUARE GINII
    # ->   30CB KATAKANA LETTER NI
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30AD KATAKANA LETTER KI
    "\u{3311}" => "\u{30CB}\u{30FC}\u{30AD}",

    # 3312 SQUARE KYURII
    # ->   30AD KATAKANA LETTER KI
    #    + 30E5 KATAKANA LETTER SMALL YU
    #    + 30EA KATAKANA LETTER RI
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    "\u{3312}" => "\u{30AD}\u{30E5}\u{30EA}\u{30FC}",

    # 3313 SQUARE GIRUDAA
    # ->   30EB KATAKANA LETTER RU
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30AD KATAKANA LETTER KI
    #    + 30BF KATAKANA LETTER TA
    "\u{3313}" => "\u{30EB}\u{30FC}\u{30AD}\u{30BF}",

    # 3314 SQUARE KIRO
    # ->   30AD KATAKANA LETTER KI
    #    + 30ED KATAKANA LETTER RO
    "\u{3314}" => "\u{30AD}\u{30ED}",

    # 3315 SQUARE KIROGURAMU
    # ->   30AD KATAKANA LETTER KI
    #    + 30ED KATAKANA LETTER RO
    #    + 30E9 KATAKANA LETTER RA
    #    + 30E0 KATAKANA LETTER MU
    #    + 30AF KATAKANA LETTER KU
    "\u{3315}" => "\u{30AD}\u{30ED}\u{30E9}\u{30E0}\u{30AF}",

    # 3316 SQUARE KIROMEETORU
    # ->   30AD KATAKANA LETTER KI
    #    + 30ED KATAKANA LETTER RO
    #    + 30E1 KATAKANA LETTER ME
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    #    + 30EB KATAKANA LETTER RU
    "\u{3316}" => "\u{30AD}\u{30ED}\u{30E1}\u{30FC}\u{30C8}\u{30EB}",

    # 3317 SQUARE KIROWATTO
    # ->   30AD KATAKANA LETTER KI
    #    + 30ED KATAKANA LETTER RO
    #    + 30EF KATAKANA LETTER WA
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    "\u{3317}" => "\u{30AD}\u{30ED}\u{30EF}\u{30C3}\u{30C8}",

    # 3318 SQUARE GURAMU
    # ->   30E9 KATAKANA LETTER RA
    #    + 30E0 KATAKANA LETTER MU
    #    + 30AF KATAKANA LETTER KU
    "\u{3318}" => "\u{30E9}\u{30E0}\u{30AF}",

    # 3319 SQUARE GURAMUTON
    # ->   30E9 KATAKANA LETTER RA
    #    + 30E0 KATAKANA LETTER MU
    #    + 30C8 KATAKANA LETTER TO
    #    + 30F3 KATAKANA LETTER N
    #    + 30AF KATAKANA LETTER KU
    "\u{3319}" => "\u{30E9}\u{30E0}\u{30C8}\u{30F3}\u{30AF}",

    # 331A SQUARE KURUZEIRO
    # ->   30AF KATAKANA LETTER KU
    #    + 30EB KATAKANA LETTER RU
    #    + 30A4 KATAKANA LETTER I
    #    + 30ED KATAKANA LETTER RO
    #    + 30BB KATAKANA LETTER SE
    "\u{331A}" => "\u{30AF}\u{30EB}\u{30A4}\u{30ED}\u{30BB}",

    # 331B SQUARE KUROONE
    # ->   30AF KATAKANA LETTER KU
    #    + 30ED KATAKANA LETTER RO
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30CD KATAKANA LETTER NE
    "\u{331B}" => "\u{30AF}\u{30ED}\u{30FC}\u{30CD}",

    # 331C SQUARE KEESU
    # ->   30B1 KATAKANA LETTER KE
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30B9 KATAKANA LETTER SU
    "\u{331C}" => "\u{30B1}\u{30FC}\u{30B9}",

    # 331D SQUARE KORUNA
    # ->   30B3 KATAKANA LETTER KO
    #    + 30EB KATAKANA LETTER RU
    #    + 30CA KATAKANA LETTER NA
    "\u{331D}" => "\u{30B3}\u{30EB}\u{30CA}",

    # 331E SQUARE KOOPO
    # ->   30B3 KATAKANA LETTER KO
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30DB KATAKANA LETTER HO
    "\u{331E}" => "\u{30B3}\u{30FC}\u{30DB}",

    # 331F SQUARE SAIKURU
    # ->   30B5 KATAKANA LETTER SA
    #    + 30A4 KATAKANA LETTER I
    #    + 30AF KATAKANA LETTER KU
    #    + 30EB KATAKANA LETTER RU
    "\u{331F}" => "\u{30B5}\u{30A4}\u{30AF}\u{30EB}",

    # 3320 SQUARE SANTIIMU
    # ->   30B5 KATAKANA LETTER SA
    #    + 30F3 KATAKANA LETTER N
    #    + 30C1 KATAKANA LETTER TI
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30E0 KATAKANA LETTER MU
    "\u{3320}" => "\u{30B5}\u{30F3}\u{30C1}\u{30FC}\u{30E0}",

    # 3321 SQUARE SIRINGU
    # ->   30B7 KATAKANA LETTER SI
    #    + 30EA KATAKANA LETTER RI
    #    + 30F3 KATAKANA LETTER N
    #    + 30AF KATAKANA LETTER KU
    "\u{3321}" => "\u{30B7}\u{30EA}\u{30F3}\u{30AF}",

    # 3322 SQUARE SENTI
    # ->   30BB KATAKANA LETTER SE
    #    + 30F3 KATAKANA LETTER N
    #    + 30C1 KATAKANA LETTER TI
    "\u{3322}" => "\u{30BB}\u{30F3}\u{30C1}",

    # 3323 SQUARE SENTO
    # ->   30BB KATAKANA LETTER SE
    #    + 30F3 KATAKANA LETTER N
    #    + 30C8 KATAKANA LETTER TO
    "\u{3323}" => "\u{30BB}\u{30F3}\u{30C8}",

    # 3324 SQUARE DAASU
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30B9 KATAKANA LETTER SU
    #    + 30BF KATAKANA LETTER TA
    "\u{3324}" => "\u{30FC}\u{30B9}\u{30BF}",

    # 3325 SQUARE DESI
    # ->   30B7 KATAKANA LETTER SI
    #    + 30C6 KATAKANA LETTER TE
    "\u{3325}" => "\u{30B7}\u{30C6}",

    # 3326 SQUARE DORU
    # ->   30EB KATAKANA LETTER RU
    #    + 30C8 KATAKANA LETTER TO
    "\u{3326}" => "\u{30EB}\u{30C8}",

    # 3327 SQUARE TON
    # ->   30C8 KATAKANA LETTER TO
    #    + 30F3 KATAKANA LETTER N
    "\u{3327}" => "\u{30C8}\u{30F3}",

    # 3328 SQUARE NANO
    # ->   30CA KATAKANA LETTER NA
    #    + 30CE KATAKANA LETTER NO
    "\u{3328}" => "\u{30CA}\u{30CE}",

    # 3329 SQUARE NOTTO
    # ->   30CE KATAKANA LETTER NO
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    "\u{3329}" => "\u{30CE}\u{30C3}\u{30C8}",

    # 332A SQUARE HAITU
    # ->   30CF KATAKANA LETTER HA
    #    + 30A4 KATAKANA LETTER I
    #    + 30C4 KATAKANA LETTER TU
    "\u{332A}" => "\u{30CF}\u{30A4}\u{30C4}",

    # 332B SQUARE PAASENTO
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30BB KATAKANA LETTER SE
    #    + 30F3 KATAKANA LETTER N
    #    + 30C8 KATAKANA LETTER TO
    #    + 30CF KATAKANA LETTER HA
    "\u{332B}" => "\u{30FC}\u{30BB}\u{30F3}\u{30C8}\u{30CF}",

    # 332C SQUARE PAATU
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C4 KATAKANA LETTER TU
    #    + 30CF KATAKANA LETTER HA
    "\u{332C}" => "\u{30FC}\u{30C4}\u{30CF}",

    # 332D SQUARE BAARERU
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EC KATAKANA LETTER RE
    #    + 30EB KATAKANA LETTER RU
    #    + 30CF KATAKANA LETTER HA
    "\u{332D}" => "\u{30FC}\u{30EC}\u{30EB}\u{30CF}",

    # 332E SQUARE PIASUTORU
    # ->   30A2 KATAKANA LETTER A
    #    + 30B9 KATAKANA LETTER SU
    #    + 30C8 KATAKANA LETTER TO
    #    + 30EB KATAKANA LETTER RU
    #    + 30D2 KATAKANA LETTER HI
    "\u{332E}" => "\u{30A2}\u{30B9}\u{30C8}\u{30EB}\u{30D2}",

    # 332F SQUARE PIKURU
    # ->   30AF KATAKANA LETTER KU
    #    + 30EB KATAKANA LETTER RU
    #    + 30D2 KATAKANA LETTER HI
    "\u{332F}" => "\u{30AF}\u{30EB}\u{30D2}",

    # 3330 SQUARE PIKO
    # ->   30B3 KATAKANA LETTER KO
    #    + 30D2 KATAKANA LETTER HI
    "\u{3330}" => "\u{30B3}\u{30D2}",

    # 3331 SQUARE BIRU
    # ->   30EB KATAKANA LETTER RU
    #    + 30D2 KATAKANA LETTER HI
    "\u{3331}" => "\u{30EB}\u{30D2}",

    # 3332 SQUARE HUARADDO
    # ->   30D5 KATAKANA LETTER HU
    #    + 30A1 KATAKANA LETTER SMALL A
    #    + 30E9 KATAKANA LETTER RA
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    "\u{3332}" => "\u{30D5}\u{30A1}\u{30E9}\u{30C3}\u{30C8}",

    # 3333 SQUARE HUIITO
    # ->   30D5 KATAKANA LETTER HU
    #    + 30A3 KATAKANA LETTER SMALL I
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    "\u{3333}" => "\u{30D5}\u{30A3}\u{30FC}\u{30C8}",

    # 3334 SQUARE BUSSYERU
    # ->   30C3 KATAKANA LETTER SMALL TU
    #    + 30B7 KATAKANA LETTER SI
    #    + 30A7 KATAKANA LETTER SMALL E
    #    + 30EB KATAKANA LETTER RU
    #    + 30D5 KATAKANA LETTER HU
    "\u{3334}" => "\u{30C3}\u{30B7}\u{30A7}\u{30EB}\u{30D5}",

    # 3335 SQUARE HURAN
    # ->   30D5 KATAKANA LETTER HU
    #    + 30E9 KATAKANA LETTER RA
    #    + 30F3 KATAKANA LETTER N
    "\u{3335}" => "\u{30D5}\u{30E9}\u{30F3}",

    # 3336 SQUARE HEKUTAARU
    # ->   30D8 KATAKANA LETTER HE
    #    + 30AF KATAKANA LETTER KU
    #    + 30BF KATAKANA LETTER TA
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    "\u{3336}" => "\u{30D8}\u{30AF}\u{30BF}\u{30FC}\u{30EB}",

    # 3337 SQUARE PESO
    # ->   30BD KATAKANA LETTER SO
    #    + 30D8 KATAKANA LETTER HE
    "\u{3337}" => "\u{30BD}\u{30D8}",

    # 3338 SQUARE PENIHI
    # ->   30CB KATAKANA LETTER NI
    #    + 30D2 KATAKANA LETTER HI
    #    + 30D8 KATAKANA LETTER HE
    "\u{3338}" => "\u{30CB}\u{30D2}\u{30D8}",

    # 3339 SQUARE HERUTU
    # ->   30D8 KATAKANA LETTER HE
    #    + 30EB KATAKANA LETTER RU
    #    + 30C4 KATAKANA LETTER TU
    "\u{3339}" => "\u{30D8}\u{30EB}\u{30C4}",

    # 333A SQUARE PENSU
    # ->   30F3 KATAKANA LETTER N
    #    + 30B9 KATAKANA LETTER SU
    #    + 30D8 KATAKANA LETTER HE
    "\u{333A}" => "\u{30F3}\u{30B9}\u{30D8}",

    # 333B SQUARE PEEZI
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30D8 KATAKANA LETTER HE
    #    + 30B7 KATAKANA LETTER SI
    "\u{333B}" => "\u{30FC}\u{30D8}\u{30B7}",

    # 333C SQUARE BEETA
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30BF KATAKANA LETTER TA
    #    + 30D8 KATAKANA LETTER HE
    "\u{333C}" => "\u{30FC}\u{30BF}\u{30D8}",

    # 333D SQUARE POINTO
    # ->   30A4 KATAKANA LETTER I
    #    + 30F3 KATAKANA LETTER N
    #    + 30C8 KATAKANA LETTER TO
    #    + 30DB KATAKANA LETTER HO
    "\u{333D}" => "\u{30A4}\u{30F3}\u{30C8}\u{30DB}",

    # 333E SQUARE BORUTO
    # ->   30EB KATAKANA LETTER RU
    #    + 30C8 KATAKANA LETTER TO
    #    + 30DB KATAKANA LETTER HO
    "\u{333E}" => "\u{30EB}\u{30C8}\u{30DB}",

    # 333F SQUARE HON
    # ->   30DB KATAKANA LETTER HO
    #    + 30F3 KATAKANA LETTER N
    "\u{333F}" => "\u{30DB}\u{30F3}",

    # 3340 SQUARE PONDO
    # ->   30F3 KATAKANA LETTER N
    #    + 30DB KATAKANA LETTER HO
    #    + 30C8 KATAKANA LETTER TO
    "\u{3340}" => "\u{30F3}\u{30DB}\u{30C8}",

    # 3341 SQUARE HOORU
    # ->   30DB KATAKANA LETTER HO
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    "\u{3341}" => "\u{30DB}\u{30FC}\u{30EB}",

    # 3342 SQUARE HOON
    # ->   30DB KATAKANA LETTER HO
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30F3 KATAKANA LETTER N
    "\u{3342}" => "\u{30DB}\u{30FC}\u{30F3}",

    # 3343 SQUARE MAIKURO
    # ->   30DE KATAKANA LETTER MA
    #    + 30A4 KATAKANA LETTER I
    #    + 30AF KATAKANA LETTER KU
    #    + 30ED KATAKANA LETTER RO
    "\u{3343}" => "\u{30DE}\u{30A4}\u{30AF}\u{30ED}",

    # 3344 SQUARE MAIRU
    # ->   30DE KATAKANA LETTER MA
    #    + 30A4 KATAKANA LETTER I
    #    + 30EB KATAKANA LETTER RU
    "\u{3344}" => "\u{30DE}\u{30A4}\u{30EB}",

    # 3345 SQUARE MAHHA
    # ->   30DE KATAKANA LETTER MA
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30CF KATAKANA LETTER HA
    "\u{3345}" => "\u{30DE}\u{30C3}\u{30CF}",

    # 3346 SQUARE MARUKU
    # ->   30DE KATAKANA LETTER MA
    #    + 30EB KATAKANA LETTER RU
    #    + 30AF KATAKANA LETTER KU
    "\u{3346}" => "\u{30DE}\u{30EB}\u{30AF}",

    # 3347 SQUARE MANSYON
    # ->   30DE KATAKANA LETTER MA
    #    + 30F3 KATAKANA LETTER N
    #    + 30B7 KATAKANA LETTER SI
    #    + 30E7 KATAKANA LETTER SMALL YO
    #    + 30F3 KATAKANA LETTER N
    "\u{3347}" => "\u{30DE}\u{30F3}\u{30B7}\u{30E7}\u{30F3}",

    # 3348 SQUARE MIKURON
    # ->   30DF KATAKANA LETTER MI
    #    + 30AF KATAKANA LETTER KU
    #    + 30ED KATAKANA LETTER RO
    #    + 30F3 KATAKANA LETTER N
    "\u{3348}" => "\u{30DF}\u{30AF}\u{30ED}\u{30F3}",

    # 3349 SQUARE MIRI
    # ->   30DF KATAKANA LETTER MI
    #    + 30EA KATAKANA LETTER RI
    "\u{3349}" => "\u{30DF}\u{30EA}",

    # 334A SQUARE MIRIBAARU
    # ->   30DF KATAKANA LETTER MI
    #    + 30EA KATAKANA LETTER RI
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    #    + 30CF KATAKANA LETTER HA
    "\u{334A}" => "\u{30DF}\u{30EA}\u{30FC}\u{30EB}\u{30CF}",

    # 334B SQUARE MEGA
    # ->   30E1 KATAKANA LETTER ME
    #    + 30AB KATAKANA LETTER KA
    "\u{334B}" => "\u{30E1}\u{30AB}",

    # 334C SQUARE MEGATON
    # ->   30E1 KATAKANA LETTER ME
    #    + 30C8 KATAKANA LETTER TO
    #    + 30F3 KATAKANA LETTER N
    #    + 30AB KATAKANA LETTER KA
    "\u{334C}" => "\u{30E1}\u{30C8}\u{30F3}\u{30AB}",

    # 334D SQUARE MEETORU
    # ->   30E1 KATAKANA LETTER ME
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    #    + 30EB KATAKANA LETTER RU
    "\u{334D}" => "\u{30E1}\u{30FC}\u{30C8}\u{30EB}",

    # 334E SQUARE YAADO
    # ->   30E4 KATAKANA LETTER YA
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30C8 KATAKANA LETTER TO
    "\u{334E}" => "\u{30E4}\u{30FC}\u{30C8}",

    # 334F SQUARE YAARU
    # ->   30E4 KATAKANA LETTER YA
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    "\u{334F}" => "\u{30E4}\u{30FC}\u{30EB}",

    # 3350 SQUARE YUAN
    # ->   30E6 KATAKANA LETTER YU
    #    + 30A2 KATAKANA LETTER A
    #    + 30F3 KATAKANA LETTER N
    "\u{3350}" => "\u{30E6}\u{30A2}\u{30F3}",

    # 3351 SQUARE RITTORU
    # ->   30EA KATAKANA LETTER RI
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    #    + 30EB KATAKANA LETTER RU
    "\u{3351}" => "\u{30EA}\u{30C3}\u{30C8}\u{30EB}",

    # 3352 SQUARE RIRA
    # ->   30EA KATAKANA LETTER RI
    #    + 30E9 KATAKANA LETTER RA
    "\u{3352}" => "\u{30EA}\u{30E9}",

    # 3353 SQUARE RUPII
    # ->   30EB KATAKANA LETTER RU
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30D2 KATAKANA LETTER HI
    "\u{3353}" => "\u{30EB}\u{30FC}\u{30D2}",

    # 3354 SQUARE RUUBURU
    # ->   30EB KATAKANA LETTER RU
    #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    #    + 30EB KATAKANA LETTER RU
    #    + 30D5 KATAKANA LETTER HU
    "\u{3354}" => "\u{30EB}\u{30FC}\u{30EB}\u{30D5}",

    # 3355 SQUARE REMU
    # ->   30EC KATAKANA LETTER RE
    #    + 30E0 KATAKANA LETTER MU
    "\u{3355}" => "\u{30EC}\u{30E0}",

    # 3356 SQUARE RENTOGEN
    # ->   30EC KATAKANA LETTER RE
    #    + 30F3 KATAKANA LETTER N
    #    + 30C8 KATAKANA LETTER TO
    #    + 30F3 KATAKANA LETTER N
    #    + 30B1 KATAKANA LETTER KE
    "\u{3356}" => "\u{30EC}\u{30F3}\u{30C8}\u{30F3}\u{30B1}",

    # 3357 SQUARE WATTO
    # ->   30EF KATAKANA LETTER WA
    #    + 30C3 KATAKANA LETTER SMALL TU
    #    + 30C8 KATAKANA LETTER TO
    "\u{3357}" => "\u{30EF}\u{30C3}\u{30C8}",

    # 3358 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO
    # ->   0030 DIGIT ZERO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3358}" => "0\u{70B9}",

    # 3359 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE
    # ->   0031 DIGIT ONE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3359}" => "1\u{70B9}",

    # 335A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO
    # ->   0032 DIGIT TWO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335A}" => "2\u{70B9}",

    # 335B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE
    # ->   0033 DIGIT THREE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335B}" => "3\u{70B9}",

    # 335C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR
    # ->   0034 DIGIT FOUR
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335C}" => "4\u{70B9}",

    # 335D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE
    # ->   0035 DIGIT FIVE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335D}" => "5\u{70B9}",

    # 335E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX
    # ->   0036 DIGIT SIX
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335E}" => "6\u{70B9}",

    # 335F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN
    # ->   0037 DIGIT SEVEN
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{335F}" => "7\u{70B9}",

    # 3360 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT
    # ->   0038 DIGIT EIGHT
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3360}" => "8\u{70B9}",

    # 3361 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE
    # ->   0039 DIGIT NINE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3361}" => "9\u{70B9}",

    # 3362 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN
    # ->   0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3362}" => "10\u{70B9}",

    # 3363 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN
    # ->   0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3363}" => "11\u{70B9}",

    # 3364 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE
    # ->   0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3364}" => "12\u{70B9}",

    # 3365 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN
    # ->   0031 DIGIT ONE
    #    + 0033 DIGIT THREE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3365}" => "13\u{70B9}",

    # 3366 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN
    # ->   0031 DIGIT ONE
    #    + 0034 DIGIT FOUR
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3366}" => "14\u{70B9}",

    # 3367 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN
    # ->   0031 DIGIT ONE
    #    + 0035 DIGIT FIVE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3367}" => "15\u{70B9}",

    # 3368 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN
    # ->   0031 DIGIT ONE
    #    + 0036 DIGIT SIX
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3368}" => "16\u{70B9}",

    # 3369 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN
    # ->   0031 DIGIT ONE
    #    + 0037 DIGIT SEVEN
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3369}" => "17\u{70B9}",

    # 336A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN
    # ->   0031 DIGIT ONE
    #    + 0038 DIGIT EIGHT
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336A}" => "18\u{70B9}",

    # 336B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN
    # ->   0031 DIGIT ONE
    #    + 0039 DIGIT NINE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336B}" => "19\u{70B9}",

    # 336C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY
    # ->   0032 DIGIT TWO
    #    + 0030 DIGIT ZERO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336C}" => "20\u{70B9}",

    # 336D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE
    # ->   0032 DIGIT TWO
    #    + 0031 DIGIT ONE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336D}" => "21\u{70B9}",

    # 336E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO
    # ->   0032 DIGIT TWO
    #    + 0032 DIGIT TWO
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336E}" => "22\u{70B9}",

    # 336F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE
    # ->   0032 DIGIT TWO
    #    + 0033 DIGIT THREE
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{336F}" => "23\u{70B9}",

    # 3370 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR
    # ->   0032 DIGIT TWO
    #    + 0034 DIGIT FOUR
    #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
    "\u{3370}" => "24\u{70B9}",

    # 3371 SQUARE HPA
    # ->   0068 LATIN SMALL LETTER H
    #    + 0050 LATIN CAPITAL LETTER P
    #    + 0061 LATIN SMALL LETTER A
    "\u{3371}" => "hPa",

    # 3372 SQUARE DA
    # ->   0064 LATIN SMALL LETTER D
    #    + 0061 LATIN SMALL LETTER A
    "\u{3372}" => "da",

    # 3373 SQUARE AU
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 0055 LATIN CAPITAL LETTER U
    "\u{3373}" => "AU",

    # 3374 SQUARE BAR
    # ->   0062 LATIN SMALL LETTER B
    #    + 0061 LATIN SMALL LETTER A
    #    + 0072 LATIN SMALL LETTER R
    "\u{3374}" => "bar",

    # 3375 SQUARE OV
    # ->   006F LATIN SMALL LETTER O
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{3375}" => "oV",

    # 3376 SQUARE PC
    # ->   0070 LATIN SMALL LETTER P
    #    + 0063 LATIN SMALL LETTER C
    "\u{3376}" => "pc",

    # 3377 SQUARE DM
    # ->   0064 LATIN SMALL LETTER D
    #    + 006D LATIN SMALL LETTER M
    "\u{3377}" => "dm",

    # 3378 SQUARE DM SQUARED
    # ->   0064 LATIN SMALL LETTER D
    #    + 006D LATIN SMALL LETTER M
    #    + 0032 DIGIT TWO
    "\u{3378}" => "dm2",

    # 3379 SQUARE DM CUBED
    # ->   0064 LATIN SMALL LETTER D
    #    + 006D LATIN SMALL LETTER M
    #    + 0033 DIGIT THREE
    "\u{3379}" => "dm3",

    # 337A SQUARE IU
    # ->   0049 LATIN CAPITAL LETTER I
    #    + 0055 LATIN CAPITAL LETTER U
    "\u{337A}" => "IU",

    # 337B SQUARE ERA NAME HEISEI
    # ->   5E73 CJK UNIFIED IDEOGRAPH 5E73
    #    + 6210 CJK UNIFIED IDEOGRAPH 6210
    "\u{337B}" => "\u{5E73}\u{6210}",

    # 337C SQUARE ERA NAME SYOUWA
    # ->   662D CJK UNIFIED IDEOGRAPH 662D
    #    + 548C CJK UNIFIED IDEOGRAPH 548C
    "\u{337C}" => "\u{662D}\u{548C}",

    # 337D SQUARE ERA NAME TAISYOU
    # ->   5927 CJK UNIFIED IDEOGRAPH 5927
    #    + 6B63 CJK UNIFIED IDEOGRAPH 6B63
    "\u{337D}" => "\u{5927}\u{6B63}",

    # 337E SQUARE ERA NAME MEIZI
    # ->   660E CJK UNIFIED IDEOGRAPH 660E
    #    + 6CBB CJK UNIFIED IDEOGRAPH 6CBB
    "\u{337E}" => "\u{660E}\u{6CBB}",

    # 337F SQUARE CORPORATION
    # ->   682A CJK UNIFIED IDEOGRAPH 682A
    #    + 5F0F CJK UNIFIED IDEOGRAPH 5F0F
    #    + 4F1A CJK UNIFIED IDEOGRAPH 4F1A
    #    + 793E CJK UNIFIED IDEOGRAPH 793E
    "\u{337F}" => "\u{682A}\u{5F0F}\u{4F1A}\u{793E}",

    # 3380 SQUARE PA AMPS
    # ->   0070 LATIN SMALL LETTER P
    #    + 0041 LATIN CAPITAL LETTER A
    "\u{3380}" => "pA",

    # 3381 SQUARE NA
    # ->   006E LATIN SMALL LETTER N
    #    + 0041 LATIN CAPITAL LETTER A
    "\u{3381}" => "nA",

    # 3382 SQUARE MU A
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0041 LATIN CAPITAL LETTER A
    "\u{3382}" => "\u{03BC}A",

    # 3383 SQUARE MA
    # ->   006D LATIN SMALL LETTER M
    #    + 0041 LATIN CAPITAL LETTER A
    "\u{3383}" => "mA",

    # 3384 SQUARE KA
    # ->   006B LATIN SMALL LETTER K
    #    + 0041 LATIN CAPITAL LETTER A
    "\u{3384}" => "kA",

    # 3385 SQUARE KB
    # ->   004B LATIN CAPITAL LETTER K
    #    + 0042 LATIN CAPITAL LETTER B
    "\u{3385}" => "KB",

    # 3386 SQUARE MB
    # ->   004D LATIN CAPITAL LETTER M
    #    + 0042 LATIN CAPITAL LETTER B
    "\u{3386}" => "MB",

    # 3387 SQUARE GB
    # ->   0047 LATIN CAPITAL LETTER G
    #    + 0042 LATIN CAPITAL LETTER B
    "\u{3387}" => "GB",

    # 3388 SQUARE CAL
    # ->   0063 LATIN SMALL LETTER C
    #    + 0061 LATIN SMALL LETTER A
    #    + 006C LATIN SMALL LETTER L
    "\u{3388}" => "cal",

    # 3389 SQUARE KCAL
    # ->   006B LATIN SMALL LETTER K
    #    + 0063 LATIN SMALL LETTER C
    #    + 0061 LATIN SMALL LETTER A
    #    + 006C LATIN SMALL LETTER L
    "\u{3389}" => "kcal",

    # 338A SQUARE PF
    # ->   0070 LATIN SMALL LETTER P
    #    + 0046 LATIN CAPITAL LETTER F
    "\u{338A}" => "pF",

    # 338B SQUARE NF
    # ->   006E LATIN SMALL LETTER N
    #    + 0046 LATIN CAPITAL LETTER F
    "\u{338B}" => "nF",

    # 338C SQUARE MU F
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0046 LATIN CAPITAL LETTER F
    "\u{338C}" => "\u{03BC}F",

    # 338D SQUARE MU G
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0067 LATIN SMALL LETTER G
    "\u{338D}" => "\u{03BC}g",

    # 338E SQUARE MG
    # ->   006D LATIN SMALL LETTER M
    #    + 0067 LATIN SMALL LETTER G
    "\u{338E}" => "mg",

    # 338F SQUARE KG
    # ->   006B LATIN SMALL LETTER K
    #    + 0067 LATIN SMALL LETTER G
    "\u{338F}" => "kg",

    # 3390 SQUARE HZ
    # ->   0048 LATIN CAPITAL LETTER H
    #    + 007A LATIN SMALL LETTER Z
    "\u{3390}" => "Hz",

    # 3391 SQUARE KHZ
    # ->   006B LATIN SMALL LETTER K
    #    + 0048 LATIN CAPITAL LETTER H
    #    + 007A LATIN SMALL LETTER Z
    "\u{3391}" => "kHz",

    # 3392 SQUARE MHZ
    # ->   004D LATIN CAPITAL LETTER M
    #    + 0048 LATIN CAPITAL LETTER H
    #    + 007A LATIN SMALL LETTER Z
    "\u{3392}" => "MHz",

    # 3393 SQUARE GHZ
    # ->   0047 LATIN CAPITAL LETTER G
    #    + 0048 LATIN CAPITAL LETTER H
    #    + 007A LATIN SMALL LETTER Z
    "\u{3393}" => "GHz",

    # 3394 SQUARE THZ
    # ->   0054 LATIN CAPITAL LETTER T
    #    + 0048 LATIN CAPITAL LETTER H
    #    + 007A LATIN SMALL LETTER Z
    "\u{3394}" => "THz",

    # 3395 SQUARE MU L
    # ->   03BC GREEK SMALL LETTER MU
    #    + 006C LATIN SMALL LETTER L
    "\u{3395}" => "\u{03BC}l",

    # 3396 SQUARE ML
    # ->   006D LATIN SMALL LETTER M
    #    + 006C LATIN SMALL LETTER L
    "\u{3396}" => "ml",

    # 3397 SQUARE DL
    # ->   0064 LATIN SMALL LETTER D
    #    + 006C LATIN SMALL LETTER L
    "\u{3397}" => "dl",

    # 3398 SQUARE KL
    # ->   006B LATIN SMALL LETTER K
    #    + 006C LATIN SMALL LETTER L
    "\u{3398}" => "kl",

    # 3399 SQUARE FM
    # ->   0066 LATIN SMALL LETTER F
    #    + 006D LATIN SMALL LETTER M
    "\u{3399}" => "fm",

    # 339A SQUARE NM
    # ->   006E LATIN SMALL LETTER N
    #    + 006D LATIN SMALL LETTER M
    "\u{339A}" => "nm",

    # 339B SQUARE MU M
    # ->   03BC GREEK SMALL LETTER MU
    #    + 006D LATIN SMALL LETTER M
    "\u{339B}" => "\u{03BC}m",

    # 339C SQUARE MM
    # ->   006D LATIN SMALL LETTER M
    #    + 006D LATIN SMALL LETTER M
    "\u{339C}" => "mm",

    # 339D SQUARE CM
    # ->   0063 LATIN SMALL LETTER C
    #    + 006D LATIN SMALL LETTER M
    "\u{339D}" => "cm",

    # 339E SQUARE KM
    # ->   006B LATIN SMALL LETTER K
    #    + 006D LATIN SMALL LETTER M
    "\u{339E}" => "km",

    # 339F SQUARE MM SQUARED
    # ->   006D LATIN SMALL LETTER M
    #    + 006D LATIN SMALL LETTER M
    #    + 0032 DIGIT TWO
    "\u{339F}" => "mm2",

    # 33A0 SQUARE CM SQUARED
    # ->   0063 LATIN SMALL LETTER C
    #    + 006D LATIN SMALL LETTER M
    #    + 0032 DIGIT TWO
    "\u{33A0}" => "cm2",

    # 33A1 SQUARE M SQUARED
    # ->   006D LATIN SMALL LETTER M
    #    + 0032 DIGIT TWO
    "\u{33A1}" => "m2",

    # 33A2 SQUARE KM SQUARED
    # ->   006B LATIN SMALL LETTER K
    #    + 006D LATIN SMALL LETTER M
    #    + 0032 DIGIT TWO
    "\u{33A2}" => "km2",

    # 33A3 SQUARE MM CUBED
    # ->   006D LATIN SMALL LETTER M
    #    + 006D LATIN SMALL LETTER M
    #    + 0033 DIGIT THREE
    "\u{33A3}" => "mm3",

    # 33A4 SQUARE CM CUBED
    # ->   0063 LATIN SMALL LETTER C
    #    + 006D LATIN SMALL LETTER M
    #    + 0033 DIGIT THREE
    "\u{33A4}" => "cm3",

    # 33A5 SQUARE M CUBED
    # ->   006D LATIN SMALL LETTER M
    #    + 0033 DIGIT THREE
    "\u{33A5}" => "m3",

    # 33A6 SQUARE KM CUBED
    # ->   006B LATIN SMALL LETTER K
    #    + 006D LATIN SMALL LETTER M
    #    + 0033 DIGIT THREE
    "\u{33A6}" => "km3",

    # 33A7 SQUARE M OVER S
    # ->   006D LATIN SMALL LETTER M
    #    + 2215 DIVISION SLASH
    #    + 0073 LATIN SMALL LETTER S
    "\u{33A7}" => "m\u{2215}s",

    # 33A8 SQUARE M OVER S SQUARED
    # ->   006D LATIN SMALL LETTER M
    #    + 2215 DIVISION SLASH
    #    + 0073 LATIN SMALL LETTER S
    #    + 0032 DIGIT TWO
    "\u{33A8}" => "m\u{2215}s2",

    # 33A9 SQUARE PA
    # ->   0050 LATIN CAPITAL LETTER P
    #    + 0061 LATIN SMALL LETTER A
    "\u{33A9}" => "Pa",

    # 33AA SQUARE KPA
    # ->   006B LATIN SMALL LETTER K
    #    + 0050 LATIN CAPITAL LETTER P
    #    + 0061 LATIN SMALL LETTER A
    "\u{33AA}" => "kPa",

    # 33AB SQUARE MPA
    # ->   004D LATIN CAPITAL LETTER M
    #    + 0050 LATIN CAPITAL LETTER P
    #    + 0061 LATIN SMALL LETTER A
    "\u{33AB}" => "MPa",

    # 33AC SQUARE GPA
    # ->   0047 LATIN CAPITAL LETTER G
    #    + 0050 LATIN CAPITAL LETTER P
    #    + 0061 LATIN SMALL LETTER A
    "\u{33AC}" => "GPa",

    # 33AD SQUARE RAD
    # ->   0072 LATIN SMALL LETTER R
    #    + 0061 LATIN SMALL LETTER A
    #    + 0064 LATIN SMALL LETTER D
    "\u{33AD}" => "rad",

    # 33AE SQUARE RAD OVER S
    # ->   0072 LATIN SMALL LETTER R
    #    + 0061 LATIN SMALL LETTER A
    #    + 0064 LATIN SMALL LETTER D
    #    + 2215 DIVISION SLASH
    #    + 0073 LATIN SMALL LETTER S
    "\u{33AE}" => "rad\u{2215}s",

    # 33AF SQUARE RAD OVER S SQUARED
    # ->   0072 LATIN SMALL LETTER R
    #    + 0061 LATIN SMALL LETTER A
    #    + 0064 LATIN SMALL LETTER D
    #    + 2215 DIVISION SLASH
    #    + 0073 LATIN SMALL LETTER S
    #    + 0032 DIGIT TWO
    "\u{33AF}" => "rad\u{2215}s2",

    # 33B0 SQUARE PS
    # ->   0070 LATIN SMALL LETTER P
    #    + 0073 LATIN SMALL LETTER S
    "\u{33B0}" => "ps",

    # 33B1 SQUARE NS
    # ->   006E LATIN SMALL LETTER N
    #    + 0073 LATIN SMALL LETTER S
    "\u{33B1}" => "ns",

    # 33B2 SQUARE MU S
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0073 LATIN SMALL LETTER S
    "\u{33B2}" => "\u{03BC}s",

    # 33B3 SQUARE MS
    # ->   006D LATIN SMALL LETTER M
    #    + 0073 LATIN SMALL LETTER S
    "\u{33B3}" => "ms",

    # 33B4 SQUARE PV
    # ->   0070 LATIN SMALL LETTER P
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B4}" => "pV",

    # 33B5 SQUARE NV
    # ->   006E LATIN SMALL LETTER N
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B5}" => "nV",

    # 33B6 SQUARE MU V
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B6}" => "\u{03BC}V",

    # 33B7 SQUARE MV
    # ->   006D LATIN SMALL LETTER M
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B7}" => "mV",

    # 33B8 SQUARE KV
    # ->   006B LATIN SMALL LETTER K
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B8}" => "kV",

    # 33B9 SQUARE MV MEGA
    # ->   004D LATIN CAPITAL LETTER M
    #    + 0056 LATIN CAPITAL LETTER V
    "\u{33B9}" => "MV",

    # 33BA SQUARE PW
    # ->   0070 LATIN SMALL LETTER P
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BA}" => "pW",

    # 33BB SQUARE NW
    # ->   006E LATIN SMALL LETTER N
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BB}" => "nW",

    # 33BC SQUARE MU W
    # ->   03BC GREEK SMALL LETTER MU
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BC}" => "\u{03BC}W",

    # 33BD SQUARE MW
    # ->   006D LATIN SMALL LETTER M
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BD}" => "mW",

    # 33BE SQUARE KW
    # ->   006B LATIN SMALL LETTER K
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BE}" => "kW",

    # 33BF SQUARE MW MEGA
    # ->   004D LATIN CAPITAL LETTER M
    #    + 0057 LATIN CAPITAL LETTER W
    "\u{33BF}" => "MW",

    # 33C0 SQUARE K OHM
    # ->   006B LATIN SMALL LETTER K
    #    + 03A9 GREEK CAPITAL LETTER OMEGA
    "\u{33C0}" => "k\u{03A9}",

    # 33C1 SQUARE M OHM
    # ->   004D LATIN CAPITAL LETTER M
    #    + 03A9 GREEK CAPITAL LETTER OMEGA
    "\u{33C1}" => "M\u{03A9}",

    # 33C2 SQUARE AM
    # ->   0061 LATIN SMALL LETTER A
    #    + 002E FULL STOP
    #    + 006D LATIN SMALL LETTER M
    #    + 002E FULL STOP
    "\u{33C2}" => "a.m.",

    # 33C3 SQUARE BQ
    # ->   0042 LATIN CAPITAL LETTER B
    #    + 0071 LATIN SMALL LETTER Q
    "\u{33C3}" => "Bq",

    # 33C4 SQUARE CC
    # ->   0063 LATIN SMALL LETTER C
    #    + 0063 LATIN SMALL LETTER C
    "\u{33C4}" => "cc",

    # 33C5 SQUARE CD
    # ->   0063 LATIN SMALL LETTER C
    #    + 0064 LATIN SMALL LETTER D
    "\u{33C5}" => "cd",

    # 33C6 SQUARE C OVER KG
    # ->   0043 LATIN CAPITAL LETTER C
    #    + 2215 DIVISION SLASH
    #    + 006B LATIN SMALL LETTER K
    #    + 0067 LATIN SMALL LETTER G
    "\u{33C6}" => "C\u{2215}kg",

    # 33C7 SQUARE CO
    # ->   0043 LATIN CAPITAL LETTER C
    #    + 006F LATIN SMALL LETTER O
    #    + 002E FULL STOP
    "\u{33C7}" => "Co.",

    # 33C8 SQUARE DB
    # ->   0064 LATIN SMALL LETTER D
    #    + 0042 LATIN CAPITAL LETTER B
    "\u{33C8}" => "dB",

    # 33C9 SQUARE GY
    # ->   0047 LATIN CAPITAL LETTER G
    #    + 0079 LATIN SMALL LETTER Y
    "\u{33C9}" => "Gy",

    # 33CA SQUARE HA
    # ->   0068 LATIN SMALL LETTER H
    #    + 0061 LATIN SMALL LETTER A
    "\u{33CA}" => "ha",

    # 33CB SQUARE HP
    # ->   0048 LATIN CAPITAL LETTER H
    #    + 0050 LATIN CAPITAL LETTER P
    "\u{33CB}" => "HP",

    # 33CC SQUARE IN
    # ->   0069 LATIN SMALL LETTER I
    #    + 006E LATIN SMALL LETTER N
    "\u{33CC}" => "in",

    # 33CD SQUARE KK
    # ->   004B LATIN CAPITAL LETTER K
    #    + 004B LATIN CAPITAL LETTER K
    "\u{33CD}" => "KK",

    # 33CE SQUARE KM CAPITAL
    # ->   004B LATIN CAPITAL LETTER K
    #    + 004D LATIN CAPITAL LETTER M
    "\u{33CE}" => "KM",

    # 33CF SQUARE KT
    # ->   006B LATIN SMALL LETTER K
    #    + 0074 LATIN SMALL LETTER T
    "\u{33CF}" => "kt",

    # 33D0 SQUARE LM
    # ->   006C LATIN SMALL LETTER L
    #    + 006D LATIN SMALL LETTER M
    "\u{33D0}" => "lm",

    # 33D1 SQUARE LN
    # ->   006C LATIN SMALL LETTER L
    #    + 006E LATIN SMALL LETTER N
    "\u{33D1}" => "ln",

    # 33D2 SQUARE LOG
    # ->   006C LATIN SMALL LETTER L
    #    + 006F LATIN SMALL LETTER O
    #    + 0067 LATIN SMALL LETTER G
    "\u{33D2}" => "log",

    # 33D3 SQUARE LX
    # ->   006C LATIN SMALL LETTER L
    #    + 0078 LATIN SMALL LETTER X
    "\u{33D3}" => "lx",

    # 33D4 SQUARE MB SMALL
    # ->   006D LATIN SMALL LETTER M
    #    + 0062 LATIN SMALL LETTER B
    "\u{33D4}" => "mb",

    # 33D5 SQUARE MIL
    # ->   006D LATIN SMALL LETTER M
    #    + 0069 LATIN SMALL LETTER I
    #    + 006C LATIN SMALL LETTER L
    "\u{33D5}" => "mil",

    # 33D6 SQUARE MOL
    # ->   006D LATIN SMALL LETTER M
    #    + 006F LATIN SMALL LETTER O
    #    + 006C LATIN SMALL LETTER L
    "\u{33D6}" => "mol",

    # 33D7 SQUARE PH
    # ->   0050 LATIN CAPITAL LETTER P
    #    + 0048 LATIN CAPITAL LETTER H
    "\u{33D7}" => "PH",

    # 33D8 SQUARE PM
    # ->   0070 LATIN SMALL LETTER P
    #    + 002E FULL STOP
    #    + 006D LATIN SMALL LETTER M
    #    + 002E FULL STOP
    "\u{33D8}" => "p.m.",

    # 33D9 SQUARE PPM
    # ->   0050 LATIN CAPITAL LETTER P
    #    + 0050 LATIN CAPITAL LETTER P
    #    + 004D LATIN CAPITAL LETTER M
    "\u{33D9}" => "PPM",

    # 33DA SQUARE PR
    # ->   0050 LATIN CAPITAL LETTER P
    #    + 0052 LATIN CAPITAL LETTER R
    "\u{33DA}" => "PR",

    # 33DB SQUARE SR
    # ->   0073 LATIN SMALL LETTER S
    #    + 0072 LATIN SMALL LETTER R
    "\u{33DB}" => "sr",

    # 33DC SQUARE SV
    # ->   0053 LATIN CAPITAL LETTER S
    #    + 0076 LATIN SMALL LETTER V
    "\u{33DC}" => "Sv",

    # 33DD SQUARE WB
    # ->   0057 LATIN CAPITAL LETTER W
    #    + 0062 LATIN SMALL LETTER B
    "\u{33DD}" => "Wb",

    # 33DE SQUARE V OVER M
    # ->   0056 LATIN CAPITAL LETTER V
    #    + 2215 DIVISION SLASH
    #    + 006D LATIN SMALL LETTER M
    "\u{33DE}" => "V\u{2215}m",

    # 33DF SQUARE A OVER M
    # ->   0041 LATIN CAPITAL LETTER A
    #    + 2215 DIVISION SLASH
    #    + 006D LATIN SMALL LETTER M
    "\u{33DF}" => "A\u{2215}m",

    # 33E0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE
    # ->   0031 DIGIT ONE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E0}" => "1\u{65E5}",

    # 33E1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO
    # ->   0032 DIGIT TWO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E1}" => "2\u{65E5}",

    # 33E2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE
    # ->   0033 DIGIT THREE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E2}" => "3\u{65E5}",

    # 33E3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR
    # ->   0034 DIGIT FOUR
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E3}" => "4\u{65E5}",

    # 33E4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE
    # ->   0035 DIGIT FIVE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E4}" => "5\u{65E5}",

    # 33E5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX
    # ->   0036 DIGIT SIX
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E5}" => "6\u{65E5}",

    # 33E6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN
    # ->   0037 DIGIT SEVEN
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E6}" => "7\u{65E5}",

    # 33E7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT
    # ->   0038 DIGIT EIGHT
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E7}" => "8\u{65E5}",

    # 33E8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE
    # ->   0039 DIGIT NINE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E8}" => "9\u{65E5}",

    # 33E9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN
    # ->   0031 DIGIT ONE
    #    + 0030 DIGIT ZERO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33E9}" => "10\u{65E5}",

    # 33EA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN
    # ->   0031 DIGIT ONE
    #    + 0031 DIGIT ONE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33EA}" => "11\u{65E5}",

    # 33EB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE
    # ->   0031 DIGIT ONE
    #    + 0032 DIGIT TWO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33EB}" => "12\u{65E5}",

    # 33EC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN
    # ->   0031 DIGIT ONE
    #    + 0033 DIGIT THREE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33EC}" => "13\u{65E5}",

    # 33ED IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN
    # ->   0031 DIGIT ONE
    #    + 0034 DIGIT FOUR
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33ED}" => "14\u{65E5}",

    # 33EE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN
    # ->   0031 DIGIT ONE
    #    + 0035 DIGIT FIVE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33EE}" => "15\u{65E5}",

    # 33EF IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN
    # ->   0031 DIGIT ONE
    #    + 0036 DIGIT SIX
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33EF}" => "16\u{65E5}",

    # 33F0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN
    # ->   0031 DIGIT ONE
    #    + 0037 DIGIT SEVEN
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F0}" => "17\u{65E5}",

    # 33F1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN
    # ->   0031 DIGIT ONE
    #    + 0038 DIGIT EIGHT
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F1}" => "18\u{65E5}",

    # 33F2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN
    # ->   0031 DIGIT ONE
    #    + 0039 DIGIT NINE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F2}" => "19\u{65E5}",

    # 33F3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY
    # ->   0032 DIGIT TWO
    #    + 0030 DIGIT ZERO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F3}" => "20\u{65E5}",

    # 33F4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE
    # ->   0032 DIGIT TWO
    #    + 0031 DIGIT ONE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F4}" => "21\u{65E5}",

    # 33F5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO
    # ->   0032 DIGIT TWO
    #    + 0032 DIGIT TWO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F5}" => "22\u{65E5}",

    # 33F6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE
    # ->   0032 DIGIT TWO
    #    + 0033 DIGIT THREE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F6}" => "23\u{65E5}",

    # 33F7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR
    # ->   0032 DIGIT TWO
    #    + 0034 DIGIT FOUR
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F7}" => "24\u{65E5}",

    # 33F8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE
    # ->   0032 DIGIT TWO
    #    + 0035 DIGIT FIVE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F8}" => "25\u{65E5}",

    # 33F9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX
    # ->   0032 DIGIT TWO
    #    + 0036 DIGIT SIX
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33F9}" => "26\u{65E5}",

    # 33FA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN
    # ->   0032 DIGIT TWO
    #    + 0037 DIGIT SEVEN
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33FA}" => "27\u{65E5}",

    # 33FB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT
    # ->   0032 DIGIT TWO
    #    + 0038 DIGIT EIGHT
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33FB}" => "28\u{65E5}",

    # 33FC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE
    # ->   0032 DIGIT TWO
    #    + 0039 DIGIT NINE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33FC}" => "29\u{65E5}",

    # 33FD IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY
    # ->   0033 DIGIT THREE
    #    + 0030 DIGIT ZERO
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33FD}" => "30\u{65E5}",

    # 33FE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE
    # ->   0033 DIGIT THREE
    #    + 0031 DIGIT ONE
    #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
    "\u{33FE}" => "31\u{65E5}",

    # 33FF SQUARE GAL
    # ->   0067 LATIN SMALL LETTER G
    #    + 0061 LATIN SMALL LETTER A
    #    + 006C LATIN SMALL LETTER L
    "\u{33FF}" => "gal",

    # F900 CJK UNIFIED IDEOGRAPH F900
    # ->   8C48 CJK UNIFIED IDEOGRAPH 8C48
    "\u{F900}" => "\u{8C48}",

    # F901 CJK UNIFIED IDEOGRAPH F901
    # ->   66F4 CJK UNIFIED IDEOGRAPH 66F4
    "\u{F901}" => "\u{66F4}",

    # F902 CJK UNIFIED IDEOGRAPH F902
    # ->   8ECA CJK UNIFIED IDEOGRAPH 8ECA
    "\u{F902}" => "\u{8ECA}",

    # F903 CJK UNIFIED IDEOGRAPH F903
    # ->   8CC8 CJK UNIFIED IDEOGRAPH 8CC8
    "\u{F903}" => "\u{8CC8}",

    # F904 CJK UNIFIED IDEOGRAPH F904
    # ->   6ED1 CJK UNIFIED IDEOGRAPH 6ED1
    "\u{F904}" => "\u{6ED1}",

    # F905 CJK UNIFIED IDEOGRAPH F905
    # ->   4E32 CJK UNIFIED IDEOGRAPH 4E32
    "\u{F905}" => "\u{4E32}",

    # F906 CJK UNIFIED IDEOGRAPH F906
    # ->   53E5 CJK UNIFIED IDEOGRAPH 53E5
    "\u{F906}" => "\u{53E5}",

    # F907 CJK UNIFIED IDEOGRAPH F907
    # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
    "\u{F907}" => "\u{9F9C}",

    # F908 CJK UNIFIED IDEOGRAPH F908
    # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
    "\u{F908}" => "\u{9F9C}",

    # F909 CJK UNIFIED IDEOGRAPH F909
    # ->   5951 CJK UNIFIED IDEOGRAPH 5951
    "\u{F909}" => "\u{5951}",

    # F90A CJK UNIFIED IDEOGRAPH F90A
    # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
    "\u{F90A}" => "\u{91D1}",

    # F90B CJK UNIFIED IDEOGRAPH F90B
    # ->   5587 CJK UNIFIED IDEOGRAPH 5587
    "\u{F90B}" => "\u{5587}",

    # F90C CJK UNIFIED IDEOGRAPH F90C
    # ->   5948 CJK UNIFIED IDEOGRAPH 5948
    "\u{F90C}" => "\u{5948}",

    # F90D CJK UNIFIED IDEOGRAPH F90D
    # ->   61F6 CJK UNIFIED IDEOGRAPH 61F6
    "\u{F90D}" => "\u{61F6}",

    # F90E CJK UNIFIED IDEOGRAPH F90E
    # ->   7669 CJK UNIFIED IDEOGRAPH 7669
    "\u{F90E}" => "\u{7669}",

    # F90F CJK UNIFIED IDEOGRAPH F90F
    # ->   7F85 CJK UNIFIED IDEOGRAPH 7F85
    "\u{F90F}" => "\u{7F85}",

    # F910 CJK UNIFIED IDEOGRAPH F910
    # ->   863F CJK UNIFIED IDEOGRAPH 863F
    "\u{F910}" => "\u{863F}",

    # F911 CJK UNIFIED IDEOGRAPH F911
    # ->   87BA CJK UNIFIED IDEOGRAPH 87BA
    "\u{F911}" => "\u{87BA}",

    # F912 CJK UNIFIED IDEOGRAPH F912
    # ->   88F8 CJK UNIFIED IDEOGRAPH 88F8
    "\u{F912}" => "\u{88F8}",

    # F913 CJK UNIFIED IDEOGRAPH F913
    # ->   908F CJK UNIFIED IDEOGRAPH 908F
    "\u{F913}" => "\u{908F}",

    # F914 CJK UNIFIED IDEOGRAPH F914
    # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
    "\u{F914}" => "\u{6A02}",

    # F915 CJK UNIFIED IDEOGRAPH F915
    # ->   6D1B CJK UNIFIED IDEOGRAPH 6D1B
    "\u{F915}" => "\u{6D1B}",

    # F916 CJK UNIFIED IDEOGRAPH F916
    # ->   70D9 CJK UNIFIED IDEOGRAPH 70D9
    "\u{F916}" => "\u{70D9}",

    # F917 CJK UNIFIED IDEOGRAPH F917
    # ->   73DE CJK UNIFIED IDEOGRAPH 73DE
    "\u{F917}" => "\u{73DE}",

    # F918 CJK UNIFIED IDEOGRAPH F918
    # ->   843D CJK UNIFIED IDEOGRAPH 843D
    "\u{F918}" => "\u{843D}",

    # F919 CJK UNIFIED IDEOGRAPH F919
    # ->   916A CJK UNIFIED IDEOGRAPH 916A
    "\u{F919}" => "\u{916A}",

    # F91A CJK UNIFIED IDEOGRAPH F91A
    # ->   99F1 CJK UNIFIED IDEOGRAPH 99F1
    "\u{F91A}" => "\u{99F1}",

    # F91B CJK UNIFIED IDEOGRAPH F91B
    # ->   4E82 CJK UNIFIED IDEOGRAPH 4E82
    "\u{F91B}" => "\u{4E82}",

    # F91C CJK UNIFIED IDEOGRAPH F91C
    # ->   5375 CJK UNIFIED IDEOGRAPH 5375
    "\u{F91C}" => "\u{5375}",

    # F91D CJK UNIFIED IDEOGRAPH F91D
    # ->   6B04 CJK UNIFIED IDEOGRAPH 6B04
    "\u{F91D}" => "\u{6B04}",

    # F91E CJK UNIFIED IDEOGRAPH F91E
    # ->   721B CJK UNIFIED IDEOGRAPH 721B
    "\u{F91E}" => "\u{721B}",

    # F91F CJK UNIFIED IDEOGRAPH F91F
    # ->   862D CJK UNIFIED IDEOGRAPH 862D
    "\u{F91F}" => "\u{862D}",

    # F920 CJK UNIFIED IDEOGRAPH F920
    # ->   9E1E CJK UNIFIED IDEOGRAPH 9E1E
    "\u{F920}" => "\u{9E1E}",

    # F921 CJK UNIFIED IDEOGRAPH F921
    # ->   5D50 CJK UNIFIED IDEOGRAPH 5D50
    "\u{F921}" => "\u{5D50}",

    # F922 CJK UNIFIED IDEOGRAPH F922
    # ->   6FEB CJK UNIFIED IDEOGRAPH 6FEB
    "\u{F922}" => "\u{6FEB}",

    # F923 CJK UNIFIED IDEOGRAPH F923
    # ->   85CD CJK UNIFIED IDEOGRAPH 85CD
    "\u{F923}" => "\u{85CD}",

    # F924 CJK UNIFIED IDEOGRAPH F924
    # ->   8964 CJK UNIFIED IDEOGRAPH 8964
    "\u{F924}" => "\u{8964}",

    # F925 CJK UNIFIED IDEOGRAPH F925
    # ->   62C9 CJK UNIFIED IDEOGRAPH 62C9
    "\u{F925}" => "\u{62C9}",

    # F926 CJK UNIFIED IDEOGRAPH F926
    # ->   81D8 CJK UNIFIED IDEOGRAPH 81D8
    "\u{F926}" => "\u{81D8}",

    # F927 CJK UNIFIED IDEOGRAPH F927
    # ->   881F CJK UNIFIED IDEOGRAPH 881F
    "\u{F927}" => "\u{881F}",

    # F928 CJK UNIFIED IDEOGRAPH F928
    # ->   5ECA CJK UNIFIED IDEOGRAPH 5ECA
    "\u{F928}" => "\u{5ECA}",

    # F929 CJK UNIFIED IDEOGRAPH F929
    # ->   6717 CJK UNIFIED IDEOGRAPH 6717
    "\u{F929}" => "\u{6717}",

    # F92A CJK UNIFIED IDEOGRAPH F92A
    # ->   6D6A CJK UNIFIED IDEOGRAPH 6D6A
    "\u{F92A}" => "\u{6D6A}",

    # F92B CJK UNIFIED IDEOGRAPH F92B
    # ->   72FC CJK UNIFIED IDEOGRAPH 72FC
    "\u{F92B}" => "\u{72FC}",

    # F92C CJK UNIFIED IDEOGRAPH F92C
    # ->   90CE CJK UNIFIED IDEOGRAPH 90CE
    "\u{F92C}" => "\u{90CE}",

    # F92D CJK UNIFIED IDEOGRAPH F92D
    # ->   4F86 CJK UNIFIED IDEOGRAPH 4F86
    "\u{F92D}" => "\u{4F86}",

    # F92E CJK UNIFIED IDEOGRAPH F92E
    # ->   51B7 CJK UNIFIED IDEOGRAPH 51B7
    "\u{F92E}" => "\u{51B7}",

    # F92F CJK UNIFIED IDEOGRAPH F92F
    # ->   52DE CJK UNIFIED IDEOGRAPH 52DE
    "\u{F92F}" => "\u{52DE}",

    # F930 CJK UNIFIED IDEOGRAPH F930
    # ->   64C4 CJK UNIFIED IDEOGRAPH 64C4
    "\u{F930}" => "\u{64C4}",

    # F931 CJK UNIFIED IDEOGRAPH F931
    # ->   6AD3 CJK UNIFIED IDEOGRAPH 6AD3
    "\u{F931}" => "\u{6AD3}",

    # F932 CJK UNIFIED IDEOGRAPH F932
    # ->   7210 CJK UNIFIED IDEOGRAPH 7210
    "\u{F932}" => "\u{7210}",

    # F933 CJK UNIFIED IDEOGRAPH F933
    # ->   76E7 CJK UNIFIED IDEOGRAPH 76E7
    "\u{F933}" => "\u{76E7}",

    # F934 CJK UNIFIED IDEOGRAPH F934
    # ->   8001 CJK UNIFIED IDEOGRAPH 8001
    "\u{F934}" => "\u{8001}",

    # F935 CJK UNIFIED IDEOGRAPH F935
    # ->   8606 CJK UNIFIED IDEOGRAPH 8606
    "\u{F935}" => "\u{8606}",

    # F936 CJK UNIFIED IDEOGRAPH F936
    # ->   865C CJK UNIFIED IDEOGRAPH 865C
    "\u{F936}" => "\u{865C}",

    # F937 CJK UNIFIED IDEOGRAPH F937
    # ->   8DEF CJK UNIFIED IDEOGRAPH 8DEF
    "\u{F937}" => "\u{8DEF}",

    # F938 CJK UNIFIED IDEOGRAPH F938
    # ->   9732 CJK UNIFIED IDEOGRAPH 9732
    "\u{F938}" => "\u{9732}",

    # F939 CJK UNIFIED IDEOGRAPH F939
    # ->   9B6F CJK UNIFIED IDEOGRAPH 9B6F
    "\u{F939}" => "\u{9B6F}",

    # F93A CJK UNIFIED IDEOGRAPH F93A
    # ->   9DFA CJK UNIFIED IDEOGRAPH 9DFA
    "\u{F93A}" => "\u{9DFA}",

    # F93B CJK UNIFIED IDEOGRAPH F93B
    # ->   788C CJK UNIFIED IDEOGRAPH 788C
    "\u{F93B}" => "\u{788C}",

    # F93C CJK UNIFIED IDEOGRAPH F93C
    # ->   797F CJK UNIFIED IDEOGRAPH 797F
    "\u{F93C}" => "\u{797F}",

    # F93D CJK UNIFIED IDEOGRAPH F93D
    # ->   7DA0 CJK UNIFIED IDEOGRAPH 7DA0
    "\u{F93D}" => "\u{7DA0}",

    # F93E CJK UNIFIED IDEOGRAPH F93E
    # ->   83C9 CJK UNIFIED IDEOGRAPH 83C9
    "\u{F93E}" => "\u{83C9}",

    # F93F CJK UNIFIED IDEOGRAPH F93F
    # ->   9304 CJK UNIFIED IDEOGRAPH 9304
    "\u{F93F}" => "\u{9304}",

    # F940 CJK UNIFIED IDEOGRAPH F940
    # ->   9E7F CJK UNIFIED IDEOGRAPH 9E7F
    "\u{F940}" => "\u{9E7F}",

    # F941 CJK UNIFIED IDEOGRAPH F941
    # ->   8AD6 CJK UNIFIED IDEOGRAPH 8AD6
    "\u{F941}" => "\u{8AD6}",

    # F942 CJK UNIFIED IDEOGRAPH F942
    # ->   58DF CJK UNIFIED IDEOGRAPH 58DF
    "\u{F942}" => "\u{58DF}",

    # F943 CJK UNIFIED IDEOGRAPH F943
    # ->   5F04 CJK UNIFIED IDEOGRAPH 5F04
    "\u{F943}" => "\u{5F04}",

    # F944 CJK UNIFIED IDEOGRAPH F944
    # ->   7C60 CJK UNIFIED IDEOGRAPH 7C60
    "\u{F944}" => "\u{7C60}",

    # F945 CJK UNIFIED IDEOGRAPH F945
    # ->   807E CJK UNIFIED IDEOGRAPH 807E
    "\u{F945}" => "\u{807E}",

    # F946 CJK UNIFIED IDEOGRAPH F946
    # ->   7262 CJK UNIFIED IDEOGRAPH 7262
    "\u{F946}" => "\u{7262}",

    # F947 CJK UNIFIED IDEOGRAPH F947
    # ->   78CA CJK UNIFIED IDEOGRAPH 78CA
    "\u{F947}" => "\u{78CA}",

    # F948 CJK UNIFIED IDEOGRAPH F948
    # ->   8CC2 CJK UNIFIED IDEOGRAPH 8CC2
    "\u{F948}" => "\u{8CC2}",

    # F949 CJK UNIFIED IDEOGRAPH F949
    # ->   96F7 CJK UNIFIED IDEOGRAPH 96F7
    "\u{F949}" => "\u{96F7}",

    # F94A CJK UNIFIED IDEOGRAPH F94A
    # ->   58D8 CJK UNIFIED IDEOGRAPH 58D8
    "\u{F94A}" => "\u{58D8}",

    # F94B CJK UNIFIED IDEOGRAPH F94B
    # ->   5C62 CJK UNIFIED IDEOGRAPH 5C62
    "\u{F94B}" => "\u{5C62}",

    # F94C CJK UNIFIED IDEOGRAPH F94C
    # ->   6A13 CJK UNIFIED IDEOGRAPH 6A13
    "\u{F94C}" => "\u{6A13}",

    # F94D CJK UNIFIED IDEOGRAPH F94D
    # ->   6DDA CJK UNIFIED IDEOGRAPH 6DDA
    "\u{F94D}" => "\u{6DDA}",

    # F94E CJK UNIFIED IDEOGRAPH F94E
    # ->   6F0F CJK UNIFIED IDEOGRAPH 6F0F
    "\u{F94E}" => "\u{6F0F}",

    # F94F CJK UNIFIED IDEOGRAPH F94F
    # ->   7D2F CJK UNIFIED IDEOGRAPH 7D2F
    "\u{F94F}" => "\u{7D2F}",

    # F950 CJK UNIFIED IDEOGRAPH F950
    # ->   7E37 CJK UNIFIED IDEOGRAPH 7E37
    "\u{F950}" => "\u{7E37}",

    # F951 CJK UNIFIED IDEOGRAPH F951
    # ->   964B CJK UNIFIED IDEOGRAPH 964B
    "\u{F951}" => "\u{964B}",

    # F952 CJK UNIFIED IDEOGRAPH F952
    # ->   52D2 CJK UNIFIED IDEOGRAPH 52D2
    "\u{F952}" => "\u{52D2}",

    # F953 CJK UNIFIED IDEOGRAPH F953
    # ->   808B CJK UNIFIED IDEOGRAPH 808B
    "\u{F953}" => "\u{808B}",

    # F954 CJK UNIFIED IDEOGRAPH F954
    # ->   51DC CJK UNIFIED IDEOGRAPH 51DC
    "\u{F954}" => "\u{51DC}",

    # F955 CJK UNIFIED IDEOGRAPH F955
    # ->   51CC CJK UNIFIED IDEOGRAPH 51CC
    "\u{F955}" => "\u{51CC}",

    # F956 CJK UNIFIED IDEOGRAPH F956
    # ->   7A1C CJK UNIFIED IDEOGRAPH 7A1C
    "\u{F956}" => "\u{7A1C}",

    # F957 CJK UNIFIED IDEOGRAPH F957
    # ->   7DBE CJK UNIFIED IDEOGRAPH 7DBE
    "\u{F957}" => "\u{7DBE}",

    # F958 CJK UNIFIED IDEOGRAPH F958
    # ->   83F1 CJK UNIFIED IDEOGRAPH 83F1
    "\u{F958}" => "\u{83F1}",

    # F959 CJK UNIFIED IDEOGRAPH F959
    # ->   9675 CJK UNIFIED IDEOGRAPH 9675
    "\u{F959}" => "\u{9675}",

    # F95A CJK UNIFIED IDEOGRAPH F95A
    # ->   8B80 CJK UNIFIED IDEOGRAPH 8B80
    "\u{F95A}" => "\u{8B80}",

    # F95B CJK UNIFIED IDEOGRAPH F95B
    # ->   62CF CJK UNIFIED IDEOGRAPH 62CF
    "\u{F95B}" => "\u{62CF}",

    # F95C CJK UNIFIED IDEOGRAPH F95C
    # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
    "\u{F95C}" => "\u{6A02}",

    # F95D CJK UNIFIED IDEOGRAPH F95D
    # ->   8AFE CJK UNIFIED IDEOGRAPH 8AFE
    "\u{F95D}" => "\u{8AFE}",

    # F95E CJK UNIFIED IDEOGRAPH F95E
    # ->   4E39 CJK UNIFIED IDEOGRAPH 4E39
    "\u{F95E}" => "\u{4E39}",

    # F95F CJK UNIFIED IDEOGRAPH F95F
    # ->   5BE7 CJK UNIFIED IDEOGRAPH 5BE7
    "\u{F95F}" => "\u{5BE7}",

    # F960 CJK UNIFIED IDEOGRAPH F960
    # ->   6012 CJK UNIFIED IDEOGRAPH 6012
    "\u{F960}" => "\u{6012}",

    # F961 CJK UNIFIED IDEOGRAPH F961
    # ->   7387 CJK UNIFIED IDEOGRAPH 7387
    "\u{F961}" => "\u{7387}",

    # F962 CJK UNIFIED IDEOGRAPH F962
    # ->   7570 CJK UNIFIED IDEOGRAPH 7570
    "\u{F962}" => "\u{7570}",

    # F963 CJK UNIFIED IDEOGRAPH F963
    # ->   5317 CJK UNIFIED IDEOGRAPH 5317
    "\u{F963}" => "\u{5317}",

    # F964 CJK UNIFIED IDEOGRAPH F964
    # ->   78FB CJK UNIFIED IDEOGRAPH 78FB
    "\u{F964}" => "\u{78FB}",

    # F965 CJK UNIFIED IDEOGRAPH F965
    # ->   4FBF CJK UNIFIED IDEOGRAPH 4FBF
    "\u{F965}" => "\u{4FBF}",

    # F966 CJK UNIFIED IDEOGRAPH F966
    # ->   5FA9 CJK UNIFIED IDEOGRAPH 5FA9
    "\u{F966}" => "\u{5FA9}",

    # F967 CJK UNIFIED IDEOGRAPH F967
    # ->   4E0D CJK UNIFIED IDEOGRAPH 4E0D
    "\u{F967}" => "\u{4E0D}",

    # F968 CJK UNIFIED IDEOGRAPH F968
    # ->   6CCC CJK UNIFIED IDEOGRAPH 6CCC
    "\u{F968}" => "\u{6CCC}",

    # F969 CJK UNIFIED IDEOGRAPH F969
    # ->   6578 CJK UNIFIED IDEOGRAPH 6578
    "\u{F969}" => "\u{6578}",

    # F96A CJK UNIFIED IDEOGRAPH F96A
    # ->   7D22 CJK UNIFIED IDEOGRAPH 7D22
    "\u{F96A}" => "\u{7D22}",

    # F96B CJK UNIFIED IDEOGRAPH F96B
    # ->   53C3 CJK UNIFIED IDEOGRAPH 53C3
    "\u{F96B}" => "\u{53C3}",

    # F96C CJK UNIFIED IDEOGRAPH F96C
    # ->   585E CJK UNIFIED IDEOGRAPH 585E
    "\u{F96C}" => "\u{585E}",

    # F96D CJK UNIFIED IDEOGRAPH F96D
    # ->   7701 CJK UNIFIED IDEOGRAPH 7701
    "\u{F96D}" => "\u{7701}",

    # F96E CJK UNIFIED IDEOGRAPH F96E
    # ->   8449 CJK UNIFIED IDEOGRAPH 8449
    "\u{F96E}" => "\u{8449}",

    # F96F CJK UNIFIED IDEOGRAPH F96F
    # ->   8AAA CJK UNIFIED IDEOGRAPH 8AAA
    "\u{F96F}" => "\u{8AAA}",

    # F970 CJK UNIFIED IDEOGRAPH F970
    # ->   6BBA CJK UNIFIED IDEOGRAPH 6BBA
    "\u{F970}" => "\u{6BBA}",

    # F971 CJK UNIFIED IDEOGRAPH F971
    # ->   8FB0 CJK UNIFIED IDEOGRAPH 8FB0
    "\u{F971}" => "\u{8FB0}",

    # F972 CJK UNIFIED IDEOGRAPH F972
    # ->   6C88 CJK UNIFIED IDEOGRAPH 6C88
    "\u{F972}" => "\u{6C88}",

    # F973 CJK UNIFIED IDEOGRAPH F973
    # ->   62FE CJK UNIFIED IDEOGRAPH 62FE
    "\u{F973}" => "\u{62FE}",

    # F974 CJK UNIFIED IDEOGRAPH F974
    # ->   82E5 CJK UNIFIED IDEOGRAPH 82E5
    "\u{F974}" => "\u{82E5}",

    # F975 CJK UNIFIED IDEOGRAPH F975
    # ->   63A0 CJK UNIFIED IDEOGRAPH 63A0
    "\u{F975}" => "\u{63A0}",

    # F976 CJK UNIFIED IDEOGRAPH F976
    # ->   7565 CJK UNIFIED IDEOGRAPH 7565
    "\u{F976}" => "\u{7565}",

    # F977 CJK UNIFIED IDEOGRAPH F977
    # ->   4EAE CJK UNIFIED IDEOGRAPH 4EAE
    "\u{F977}" => "\u{4EAE}",

    # F978 CJK UNIFIED IDEOGRAPH F978
    # ->   5169 CJK UNIFIED IDEOGRAPH 5169
    "\u{F978}" => "\u{5169}",

    # F979 CJK UNIFIED IDEOGRAPH F979
    # ->   51C9 CJK UNIFIED IDEOGRAPH 51C9
    "\u{F979}" => "\u{51C9}",

    # F97A CJK UNIFIED IDEOGRAPH F97A
    # ->   6881 CJK UNIFIED IDEOGRAPH 6881
    "\u{F97A}" => "\u{6881}",

    # F97B CJK UNIFIED IDEOGRAPH F97B
    # ->   7CE7 CJK UNIFIED IDEOGRAPH 7CE7
    "\u{F97B}" => "\u{7CE7}",

    # F97C CJK UNIFIED IDEOGRAPH F97C
    # ->   826F CJK UNIFIED IDEOGRAPH 826F
    "\u{F97C}" => "\u{826F}",

    # F97D CJK UNIFIED IDEOGRAPH F97D
    # ->   8AD2 CJK UNIFIED IDEOGRAPH 8AD2
    "\u{F97D}" => "\u{8AD2}",

    # F97E CJK UNIFIED IDEOGRAPH F97E
    # ->   91CF CJK UNIFIED IDEOGRAPH 91CF
    "\u{F97E}" => "\u{91CF}",

    # F97F CJK UNIFIED IDEOGRAPH F97F
    # ->   52F5 CJK UNIFIED IDEOGRAPH 52F5
    "\u{F97F}" => "\u{52F5}",

    # F980 CJK UNIFIED IDEOGRAPH F980
    # ->   5442 CJK UNIFIED IDEOGRAPH 5442
    "\u{F980}" => "\u{5442}",

    # F981 CJK UNIFIED IDEOGRAPH F981
    # ->   5973 CJK UNIFIED IDEOGRAPH 5973
    "\u{F981}" => "\u{5973}",

    # F982 CJK UNIFIED IDEOGRAPH F982
    # ->   5EEC CJK UNIFIED IDEOGRAPH 5EEC
    "\u{F982}" => "\u{5EEC}",

    # F983 CJK UNIFIED IDEOGRAPH F983
    # ->   65C5 CJK UNIFIED IDEOGRAPH 65C5
    "\u{F983}" => "\u{65C5}",

    # F984 CJK UNIFIED IDEOGRAPH F984
    # ->   6FFE CJK UNIFIED IDEOGRAPH 6FFE
    "\u{F984}" => "\u{6FFE}",

    # F985 CJK UNIFIED IDEOGRAPH F985
    # ->   792A CJK UNIFIED IDEOGRAPH 792A
    "\u{F985}" => "\u{792A}",

    # F986 CJK UNIFIED IDEOGRAPH F986
    # ->   95AD CJK UNIFIED IDEOGRAPH 95AD
    "\u{F986}" => "\u{95AD}",

    # F987 CJK UNIFIED IDEOGRAPH F987
    # ->   9A6A CJK UNIFIED IDEOGRAPH 9A6A
    "\u{F987}" => "\u{9A6A}",

    # F988 CJK UNIFIED IDEOGRAPH F988
    # ->   9E97 CJK UNIFIED IDEOGRAPH 9E97
    "\u{F988}" => "\u{9E97}",

    # F989 CJK UNIFIED IDEOGRAPH F989
    # ->   9ECE CJK UNIFIED IDEOGRAPH 9ECE
    "\u{F989}" => "\u{9ECE}",

    # F98A CJK UNIFIED IDEOGRAPH F98A
    # ->   529B CJK UNIFIED IDEOGRAPH 529B
    "\u{F98A}" => "\u{529B}",

    # F98B CJK UNIFIED IDEOGRAPH F98B
    # ->   66C6 CJK UNIFIED IDEOGRAPH 66C6
    "\u{F98B}" => "\u{66C6}",

    # F98C CJK UNIFIED IDEOGRAPH F98C
    # ->   6B77 CJK UNIFIED IDEOGRAPH 6B77
    "\u{F98C}" => "\u{6B77}",

    # F98D CJK UNIFIED IDEOGRAPH F98D
    # ->   8F62 CJK UNIFIED IDEOGRAPH 8F62
    "\u{F98D}" => "\u{8F62}",

    # F98E CJK UNIFIED IDEOGRAPH F98E
    # ->   5E74 CJK UNIFIED IDEOGRAPH 5E74
    "\u{F98E}" => "\u{5E74}",

    # F98F CJK UNIFIED IDEOGRAPH F98F
    # ->   6190 CJK UNIFIED IDEOGRAPH 6190
    "\u{F98F}" => "\u{6190}",

    # F990 CJK UNIFIED IDEOGRAPH F990
    # ->   6200 CJK UNIFIED IDEOGRAPH 6200
    "\u{F990}" => "\u{6200}",

    # F991 CJK UNIFIED IDEOGRAPH F991
    # ->   649A CJK UNIFIED IDEOGRAPH 649A
    "\u{F991}" => "\u{649A}",

    # F992 CJK UNIFIED IDEOGRAPH F992
    # ->   6F23 CJK UNIFIED IDEOGRAPH 6F23
    "\u{F992}" => "\u{6F23}",

    # F993 CJK UNIFIED IDEOGRAPH F993
    # ->   7149 CJK UNIFIED IDEOGRAPH 7149
    "\u{F993}" => "\u{7149}",

    # F994 CJK UNIFIED IDEOGRAPH F994
    # ->   7489 CJK UNIFIED IDEOGRAPH 7489
    "\u{F994}" => "\u{7489}",

    # F995 CJK UNIFIED IDEOGRAPH F995
    # ->   79CA CJK UNIFIED IDEOGRAPH 79CA
    "\u{F995}" => "\u{79CA}",

    # F996 CJK UNIFIED IDEOGRAPH F996
    # ->   7DF4 CJK UNIFIED IDEOGRAPH 7DF4
    "\u{F996}" => "\u{7DF4}",

    # F997 CJK UNIFIED IDEOGRAPH F997
    # ->   806F CJK UNIFIED IDEOGRAPH 806F
    "\u{F997}" => "\u{806F}",

    # F998 CJK UNIFIED IDEOGRAPH F998
    # ->   8F26 CJK UNIFIED IDEOGRAPH 8F26
    "\u{F998}" => "\u{8F26}",

    # F999 CJK UNIFIED IDEOGRAPH F999
    # ->   84EE CJK UNIFIED IDEOGRAPH 84EE
    "\u{F999}" => "\u{84EE}",

    # F99A CJK UNIFIED IDEOGRAPH F99A
    # ->   9023 CJK UNIFIED IDEOGRAPH 9023
    "\u{F99A}" => "\u{9023}",

    # F99B CJK UNIFIED IDEOGRAPH F99B
    # ->   934A CJK UNIFIED IDEOGRAPH 934A
    "\u{F99B}" => "\u{934A}",

    # F99C CJK UNIFIED IDEOGRAPH F99C
    # ->   5217 CJK UNIFIED IDEOGRAPH 5217
    "\u{F99C}" => "\u{5217}",

    # F99D CJK UNIFIED IDEOGRAPH F99D
    # ->   52A3 CJK UNIFIED IDEOGRAPH 52A3
    "\u{F99D}" => "\u{52A3}",

    # F99E CJK UNIFIED IDEOGRAPH F99E
    # ->   54BD CJK UNIFIED IDEOGRAPH 54BD
    "\u{F99E}" => "\u{54BD}",

    # F99F CJK UNIFIED IDEOGRAPH F99F
    # ->   70C8 CJK UNIFIED IDEOGRAPH 70C8
    "\u{F99F}" => "\u{70C8}",

    # F9A0 CJK UNIFIED IDEOGRAPH F9A0
    # ->   88C2 CJK UNIFIED IDEOGRAPH 88C2
    "\u{F9A0}" => "\u{88C2}",

    # F9A1 CJK UNIFIED IDEOGRAPH F9A1
    # ->   8AAA CJK UNIFIED IDEOGRAPH 8AAA
    "\u{F9A1}" => "\u{8AAA}",

    # F9A2 CJK UNIFIED IDEOGRAPH F9A2
    # ->   5EC9 CJK UNIFIED IDEOGRAPH 5EC9
    "\u{F9A2}" => "\u{5EC9}",

    # F9A3 CJK UNIFIED IDEOGRAPH F9A3
    # ->   5FF5 CJK UNIFIED IDEOGRAPH 5FF5
    "\u{F9A3}" => "\u{5FF5}",

    # F9A4 CJK UNIFIED IDEOGRAPH F9A4
    # ->   637B CJK UNIFIED IDEOGRAPH 637B
    "\u{F9A4}" => "\u{637B}",

    # F9A5 CJK UNIFIED IDEOGRAPH F9A5
    # ->   6BAE CJK UNIFIED IDEOGRAPH 6BAE
    "\u{F9A5}" => "\u{6BAE}",

    # F9A6 CJK UNIFIED IDEOGRAPH F9A6
    # ->   7C3E CJK UNIFIED IDEOGRAPH 7C3E
    "\u{F9A6}" => "\u{7C3E}",

    # F9A7 CJK UNIFIED IDEOGRAPH F9A7
    # ->   7375 CJK UNIFIED IDEOGRAPH 7375
    "\u{F9A7}" => "\u{7375}",

    # F9A8 CJK UNIFIED IDEOGRAPH F9A8
    # ->   4EE4 CJK UNIFIED IDEOGRAPH 4EE4
    "\u{F9A8}" => "\u{4EE4}",

    # F9A9 CJK UNIFIED IDEOGRAPH F9A9
    # ->   56F9 CJK UNIFIED IDEOGRAPH 56F9
    "\u{F9A9}" => "\u{56F9}",

    # F9AA CJK UNIFIED IDEOGRAPH F9AA
    # ->   5BE7 CJK UNIFIED IDEOGRAPH 5BE7
    "\u{F9AA}" => "\u{5BE7}",

    # F9AB CJK UNIFIED IDEOGRAPH F9AB
    # ->   5DBA CJK UNIFIED IDEOGRAPH 5DBA
    "\u{F9AB}" => "\u{5DBA}",

    # F9AC CJK UNIFIED IDEOGRAPH F9AC
    # ->   601C CJK UNIFIED IDEOGRAPH 601C
    "\u{F9AC}" => "\u{601C}",

    # F9AD CJK UNIFIED IDEOGRAPH F9AD
    # ->   73B2 CJK UNIFIED IDEOGRAPH 73B2
    "\u{F9AD}" => "\u{73B2}",

    # F9AE CJK UNIFIED IDEOGRAPH F9AE
    # ->   7469 CJK UNIFIED IDEOGRAPH 7469
    "\u{F9AE}" => "\u{7469}",

    # F9AF CJK UNIFIED IDEOGRAPH F9AF
    # ->   7F9A CJK UNIFIED IDEOGRAPH 7F9A
    "\u{F9AF}" => "\u{7F9A}",

    # F9B0 CJK UNIFIED IDEOGRAPH F9B0
    # ->   8046 CJK UNIFIED IDEOGRAPH 8046
    "\u{F9B0}" => "\u{8046}",

    # F9B1 CJK UNIFIED IDEOGRAPH F9B1
    # ->   9234 CJK UNIFIED IDEOGRAPH 9234
    "\u{F9B1}" => "\u{9234}",

    # F9B2 CJK UNIFIED IDEOGRAPH F9B2
    # ->   96F6 CJK UNIFIED IDEOGRAPH 96F6
    "\u{F9B2}" => "\u{96F6}",

    # F9B3 CJK UNIFIED IDEOGRAPH F9B3
    # ->   9748 CJK UNIFIED IDEOGRAPH 9748
    "\u{F9B3}" => "\u{9748}",

    # F9B4 CJK UNIFIED IDEOGRAPH F9B4
    # ->   9818 CJK UNIFIED IDEOGRAPH 9818
    "\u{F9B4}" => "\u{9818}",

    # F9B5 CJK UNIFIED IDEOGRAPH F9B5
    # ->   4F8B CJK UNIFIED IDEOGRAPH 4F8B
    "\u{F9B5}" => "\u{4F8B}",

    # F9B6 CJK UNIFIED IDEOGRAPH F9B6
    # ->   79AE CJK UNIFIED IDEOGRAPH 79AE
    "\u{F9B6}" => "\u{79AE}",

    # F9B7 CJK UNIFIED IDEOGRAPH F9B7
    # ->   91B4 CJK UNIFIED IDEOGRAPH 91B4
    "\u{F9B7}" => "\u{91B4}",

    # F9B8 CJK UNIFIED IDEOGRAPH F9B8
    # ->   96B8 CJK UNIFIED IDEOGRAPH 96B8
    "\u{F9B8}" => "\u{96B8}",

    # F9B9 CJK UNIFIED IDEOGRAPH F9B9
    # ->   60E1 CJK UNIFIED IDEOGRAPH 60E1
    "\u{F9B9}" => "\u{60E1}",

    # F9BA CJK UNIFIED IDEOGRAPH F9BA
    # ->   4E86 CJK UNIFIED IDEOGRAPH 4E86
    "\u{F9BA}" => "\u{4E86}",

    # F9BB CJK UNIFIED IDEOGRAPH F9BB
    # ->   50DA CJK UNIFIED IDEOGRAPH 50DA
    "\u{F9BB}" => "\u{50DA}",

    # F9BC CJK UNIFIED IDEOGRAPH F9BC
    # ->   5BEE CJK UNIFIED IDEOGRAPH 5BEE
    "\u{F9BC}" => "\u{5BEE}",

    # F9BD CJK UNIFIED IDEOGRAPH F9BD
    # ->   5C3F CJK UNIFIED IDEOGRAPH 5C3F
    "\u{F9BD}" => "\u{5C3F}",

    # F9BE CJK UNIFIED IDEOGRAPH F9BE
    # ->   6599 CJK UNIFIED IDEOGRAPH 6599
    "\u{F9BE}" => "\u{6599}",

    # F9BF CJK UNIFIED IDEOGRAPH F9BF
    # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
    "\u{F9BF}" => "\u{6A02}",

    # F9C0 CJK UNIFIED IDEOGRAPH F9C0
    # ->   71CE CJK UNIFIED IDEOGRAPH 71CE
    "\u{F9C0}" => "\u{71CE}",

    # F9C1 CJK UNIFIED IDEOGRAPH F9C1
    # ->   7642 CJK UNIFIED IDEOGRAPH 7642
    "\u{F9C1}" => "\u{7642}",

    # F9C2 CJK UNIFIED IDEOGRAPH F9C2
    # ->   84FC CJK UNIFIED IDEOGRAPH 84FC
    "\u{F9C2}" => "\u{84FC}",

    # F9C3 CJK UNIFIED IDEOGRAPH F9C3
    # ->   907C CJK UNIFIED IDEOGRAPH 907C
    "\u{F9C3}" => "\u{907C}",

    # F9C4 CJK UNIFIED IDEOGRAPH F9C4
    # ->   9F8D CJK UNIFIED IDEOGRAPH 9F8D
    "\u{F9C4}" => "\u{9F8D}",

    # F9C5 CJK UNIFIED IDEOGRAPH F9C5
    # ->   6688 CJK UNIFIED IDEOGRAPH 6688
    "\u{F9C5}" => "\u{6688}",

    # F9C6 CJK UNIFIED IDEOGRAPH F9C6
    # ->   962E CJK UNIFIED IDEOGRAPH 962E
    "\u{F9C6}" => "\u{962E}",

    # F9C7 CJK UNIFIED IDEOGRAPH F9C7
    # ->   5289 CJK UNIFIED IDEOGRAPH 5289
    "\u{F9C7}" => "\u{5289}",

    # F9C8 CJK UNIFIED IDEOGRAPH F9C8
    # ->   677B CJK UNIFIED IDEOGRAPH 677B
    "\u{F9C8}" => "\u{677B}",

    # F9C9 CJK UNIFIED IDEOGRAPH F9C9
    # ->   67F3 CJK UNIFIED IDEOGRAPH 67F3
    "\u{F9C9}" => "\u{67F3}",

    # F9CA CJK UNIFIED IDEOGRAPH F9CA
    # ->   6D41 CJK UNIFIED IDEOGRAPH 6D41
    "\u{F9CA}" => "\u{6D41}",

    # F9CB CJK UNIFIED IDEOGRAPH F9CB
    # ->   6E9C CJK UNIFIED IDEOGRAPH 6E9C
    "\u{F9CB}" => "\u{6E9C}",

    # F9CC CJK UNIFIED IDEOGRAPH F9CC
    # ->   7409 CJK UNIFIED IDEOGRAPH 7409
    "\u{F9CC}" => "\u{7409}",

    # F9CD CJK UNIFIED IDEOGRAPH F9CD
    # ->   7559 CJK UNIFIED IDEOGRAPH 7559
    "\u{F9CD}" => "\u{7559}",

    # F9CE CJK UNIFIED IDEOGRAPH F9CE
    # ->   786B CJK UNIFIED IDEOGRAPH 786B
    "\u{F9CE}" => "\u{786B}",

    # F9CF CJK UNIFIED IDEOGRAPH F9CF
    # ->   7D10 CJK UNIFIED IDEOGRAPH 7D10
    "\u{F9CF}" => "\u{7D10}",

    # F9D0 CJK UNIFIED IDEOGRAPH F9D0
    # ->   985E CJK UNIFIED IDEOGRAPH 985E
    "\u{F9D0}" => "\u{985E}",

    # F9D1 CJK UNIFIED IDEOGRAPH F9D1
    # ->   516D CJK UNIFIED IDEOGRAPH 516D
    "\u{F9D1}" => "\u{516D}",

    # F9D2 CJK UNIFIED IDEOGRAPH F9D2
    # ->   622E CJK UNIFIED IDEOGRAPH 622E
    "\u{F9D2}" => "\u{622E}",

    # F9D3 CJK UNIFIED IDEOGRAPH F9D3
    # ->   9678 CJK UNIFIED IDEOGRAPH 9678
    "\u{F9D3}" => "\u{9678}",

    # F9D4 CJK UNIFIED IDEOGRAPH F9D4
    # ->   502B CJK UNIFIED IDEOGRAPH 502B
    "\u{F9D4}" => "\u{502B}",

    # F9D5 CJK UNIFIED IDEOGRAPH F9D5
    # ->   5D19 CJK UNIFIED IDEOGRAPH 5D19
    "\u{F9D5}" => "\u{5D19}",

    # F9D6 CJK UNIFIED IDEOGRAPH F9D6
    # ->   6DEA CJK UNIFIED IDEOGRAPH 6DEA
    "\u{F9D6}" => "\u{6DEA}",

    # F9D7 CJK UNIFIED IDEOGRAPH F9D7
    # ->   8F2A CJK UNIFIED IDEOGRAPH 8F2A
    "\u{F9D7}" => "\u{8F2A}",

    # F9D8 CJK UNIFIED IDEOGRAPH F9D8
    # ->   5F8B CJK UNIFIED IDEOGRAPH 5F8B
    "\u{F9D8}" => "\u{5F8B}",

    # F9D9 CJK UNIFIED IDEOGRAPH F9D9
    # ->   6144 CJK UNIFIED IDEOGRAPH 6144
    "\u{F9D9}" => "\u{6144}",

    # F9DA CJK UNIFIED IDEOGRAPH F9DA
    # ->   6817 CJK UNIFIED IDEOGRAPH 6817
    "\u{F9DA}" => "\u{6817}",

    # F9DB CJK UNIFIED IDEOGRAPH F9DB
    # ->   7387 CJK UNIFIED IDEOGRAPH 7387
    "\u{F9DB}" => "\u{7387}",

    # F9DC CJK UNIFIED IDEOGRAPH F9DC
    # ->   9686 CJK UNIFIED IDEOGRAPH 9686
    "\u{F9DC}" => "\u{9686}",

    # F9DD CJK UNIFIED IDEOGRAPH F9DD
    # ->   5229 CJK UNIFIED IDEOGRAPH 5229
    "\u{F9DD}" => "\u{5229}",

    # F9DE CJK UNIFIED IDEOGRAPH F9DE
    # ->   540F CJK UNIFIED IDEOGRAPH 540F
    "\u{F9DE}" => "\u{540F}",

    # F9DF CJK UNIFIED IDEOGRAPH F9DF
    # ->   5C65 CJK UNIFIED IDEOGRAPH 5C65
    "\u{F9DF}" => "\u{5C65}",

    # F9E0 CJK UNIFIED IDEOGRAPH F9E0
    # ->   6613 CJK UNIFIED IDEOGRAPH 6613
    "\u{F9E0}" => "\u{6613}",

    # F9E1 CJK UNIFIED IDEOGRAPH F9E1
    # ->   674E CJK UNIFIED IDEOGRAPH 674E
    "\u{F9E1}" => "\u{674E}",

    # F9E2 CJK UNIFIED IDEOGRAPH F9E2
    # ->   68A8 CJK UNIFIED IDEOGRAPH 68A8
    "\u{F9E2}" => "\u{68A8}",

    # F9E3 CJK UNIFIED IDEOGRAPH F9E3
    # ->   6CE5 CJK UNIFIED IDEOGRAPH 6CE5
    "\u{F9E3}" => "\u{6CE5}",

    # F9E4 CJK UNIFIED IDEOGRAPH F9E4
    # ->   7406 CJK UNIFIED IDEOGRAPH 7406
    "\u{F9E4}" => "\u{7406}",

    # F9E5 CJK UNIFIED IDEOGRAPH F9E5
    # ->   75E2 CJK UNIFIED IDEOGRAPH 75E2
    "\u{F9E5}" => "\u{75E2}",

    # F9E6 CJK UNIFIED IDEOGRAPH F9E6
    # ->   7F79 CJK UNIFIED IDEOGRAPH 7F79
    "\u{F9E6}" => "\u{7F79}",

    # F9E7 CJK UNIFIED IDEOGRAPH F9E7
    # ->   88CF CJK UNIFIED IDEOGRAPH 88CF
    "\u{F9E7}" => "\u{88CF}",

    # F9E8 CJK UNIFIED IDEOGRAPH F9E8
    # ->   88E1 CJK UNIFIED IDEOGRAPH 88E1
    "\u{F9E8}" => "\u{88E1}",

    # F9E9 CJK UNIFIED IDEOGRAPH F9E9
    # ->   91CC CJK UNIFIED IDEOGRAPH 91CC
    "\u{F9E9}" => "\u{91CC}",

    # F9EA CJK UNIFIED IDEOGRAPH F9EA
    # ->   96E2 CJK UNIFIED IDEOGRAPH 96E2
    "\u{F9EA}" => "\u{96E2}",

    # F9EB CJK UNIFIED IDEOGRAPH F9EB
    # ->   533F CJK UNIFIED IDEOGRAPH 533F
    "\u{F9EB}" => "\u{533F}",

    # F9EC CJK UNIFIED IDEOGRAPH F9EC
    # ->   6EBA CJK UNIFIED IDEOGRAPH 6EBA
    "\u{F9EC}" => "\u{6EBA}",

    # F9ED CJK UNIFIED IDEOGRAPH F9ED
    # ->   541D CJK UNIFIED IDEOGRAPH 541D
    "\u{F9ED}" => "\u{541D}",

    # F9EE CJK UNIFIED IDEOGRAPH F9EE
    # ->   71D0 CJK UNIFIED IDEOGRAPH 71D0
    "\u{F9EE}" => "\u{71D0}",

    # F9EF CJK UNIFIED IDEOGRAPH F9EF
    # ->   7498 CJK UNIFIED IDEOGRAPH 7498
    "\u{F9EF}" => "\u{7498}",

    # F9F0 CJK UNIFIED IDEOGRAPH F9F0
    # ->   85FA CJK UNIFIED IDEOGRAPH 85FA
    "\u{F9F0}" => "\u{85FA}",

    # F9F1 CJK UNIFIED IDEOGRAPH F9F1
    # ->   96A3 CJK UNIFIED IDEOGRAPH 96A3
    "\u{F9F1}" => "\u{96A3}",

    # F9F2 CJK UNIFIED IDEOGRAPH F9F2
    # ->   9C57 CJK UNIFIED IDEOGRAPH 9C57
    "\u{F9F2}" => "\u{9C57}",

    # F9F3 CJK UNIFIED IDEOGRAPH F9F3
    # ->   9E9F CJK UNIFIED IDEOGRAPH 9E9F
    "\u{F9F3}" => "\u{9E9F}",

    # F9F4 CJK UNIFIED IDEOGRAPH F9F4
    # ->   6797 CJK UNIFIED IDEOGRAPH 6797
    "\u{F9F4}" => "\u{6797}",

    # F9F5 CJK UNIFIED IDEOGRAPH F9F5
    # ->   6DCB CJK UNIFIED IDEOGRAPH 6DCB
    "\u{F9F5}" => "\u{6DCB}",

    # F9F6 CJK UNIFIED IDEOGRAPH F9F6
    # ->   81E8 CJK UNIFIED IDEOGRAPH 81E8
    "\u{F9F6}" => "\u{81E8}",

    # F9F7 CJK UNIFIED IDEOGRAPH F9F7
    # ->   7ACB CJK UNIFIED IDEOGRAPH 7ACB
    "\u{F9F7}" => "\u{7ACB}",

    # F9F8 CJK UNIFIED IDEOGRAPH F9F8
    # ->   7B20 CJK UNIFIED IDEOGRAPH 7B20
    "\u{F9F8}" => "\u{7B20}",

    # F9F9 CJK UNIFIED IDEOGRAPH F9F9
    # ->   7C92 CJK UNIFIED IDEOGRAPH 7C92
    "\u{F9F9}" => "\u{7C92}",

    # F9FA CJK UNIFIED IDEOGRAPH F9FA
    # ->   72C0 CJK UNIFIED IDEOGRAPH 72C0
    "\u{F9FA}" => "\u{72C0}",

    # F9FB CJK UNIFIED IDEOGRAPH F9FB
    # ->   7099 CJK UNIFIED IDEOGRAPH 7099
    "\u{F9FB}" => "\u{7099}",

    # F9FC CJK UNIFIED IDEOGRAPH F9FC
    # ->   8B58 CJK UNIFIED IDEOGRAPH 8B58
    "\u{F9FC}" => "\u{8B58}",

    # F9FD CJK UNIFIED IDEOGRAPH F9FD
    # ->   4EC0 CJK UNIFIED IDEOGRAPH 4EC0
    "\u{F9FD}" => "\u{4EC0}",

    # F9FE CJK UNIFIED IDEOGRAPH F9FE
    # ->   8336 CJK UNIFIED IDEOGRAPH 8336
    "\u{F9FE}" => "\u{8336}",

    # F9FF CJK UNIFIED IDEOGRAPH F9FF
    # ->   523A CJK UNIFIED IDEOGRAPH 523A
    "\u{F9FF}" => "\u{523A}",

    # FA00 CJK UNIFIED IDEOGRAPH FA00
    # ->   5207 CJK UNIFIED IDEOGRAPH 5207
    "\u{FA00}" => "\u{5207}",

    # FA01 CJK UNIFIED IDEOGRAPH FA01
    # ->   5EA6 CJK UNIFIED IDEOGRAPH 5EA6
    "\u{FA01}" => "\u{5EA6}",

    # FA02 CJK UNIFIED IDEOGRAPH FA02
    # ->   62D3 CJK UNIFIED IDEOGRAPH 62D3
    "\u{FA02}" => "\u{62D3}",

    # FA03 CJK UNIFIED IDEOGRAPH FA03
    # ->   7CD6 CJK UNIFIED IDEOGRAPH 7CD6
    "\u{FA03}" => "\u{7CD6}",

    # FA04 CJK UNIFIED IDEOGRAPH FA04
    # ->   5B85 CJK UNIFIED IDEOGRAPH 5B85
    "\u{FA04}" => "\u{5B85}",

    # FA05 CJK UNIFIED IDEOGRAPH FA05
    # ->   6D1E CJK UNIFIED IDEOGRAPH 6D1E
    "\u{FA05}" => "\u{6D1E}",

    # FA06 CJK UNIFIED IDEOGRAPH FA06
    # ->   66B4 CJK UNIFIED IDEOGRAPH 66B4
    "\u{FA06}" => "\u{66B4}",

    # FA07 CJK UNIFIED IDEOGRAPH FA07
    # ->   8F3B CJK UNIFIED IDEOGRAPH 8F3B
    "\u{FA07}" => "\u{8F3B}",

    # FA08 CJK UNIFIED IDEOGRAPH FA08
    # ->   884C CJK UNIFIED IDEOGRAPH 884C
    "\u{FA08}" => "\u{884C}",

    # FA09 CJK UNIFIED IDEOGRAPH FA09
    # ->   964D CJK UNIFIED IDEOGRAPH 964D
    "\u{FA09}" => "\u{964D}",

    # FA0A CJK UNIFIED IDEOGRAPH FA0A
    # ->   898B CJK UNIFIED IDEOGRAPH 898B
    "\u{FA0A}" => "\u{898B}",

    # FA0B CJK UNIFIED IDEOGRAPH FA0B
    # ->   5ED3 CJK UNIFIED IDEOGRAPH 5ED3
    "\u{FA0B}" => "\u{5ED3}",

    # FA0C CJK UNIFIED IDEOGRAPH FA0C
    # ->   5140 CJK UNIFIED IDEOGRAPH 5140
    "\u{FA0C}" => "\u{5140}",

    # FA0D CJK UNIFIED IDEOGRAPH FA0D
    # ->   55C0 CJK UNIFIED IDEOGRAPH 55C0
    "\u{FA0D}" => "\u{55C0}",

    # FA10 CJK UNIFIED IDEOGRAPH FA10
    # ->   585A CJK UNIFIED IDEOGRAPH 585A
    "\u{FA10}" => "\u{585A}",

    # FA12 CJK UNIFIED IDEOGRAPH FA12
    # ->   6674 CJK UNIFIED IDEOGRAPH 6674
    "\u{FA12}" => "\u{6674}",

    # FA15 CJK UNIFIED IDEOGRAPH FA15
    # ->   51DE CJK UNIFIED IDEOGRAPH 51DE
    "\u{FA15}" => "\u{51DE}",

    # FA16 CJK UNIFIED IDEOGRAPH FA16
    # ->   732A CJK UNIFIED IDEOGRAPH 732A
    "\u{FA16}" => "\u{732A}",

    # FA17 CJK UNIFIED IDEOGRAPH FA17
    # ->   76CA CJK UNIFIED IDEOGRAPH 76CA
    "\u{FA17}" => "\u{76CA}",

    # FA18 CJK UNIFIED IDEOGRAPH FA18
    # ->   793C CJK UNIFIED IDEOGRAPH 793C
    "\u{FA18}" => "\u{793C}",

    # FA19 CJK UNIFIED IDEOGRAPH FA19
    # ->   795E CJK UNIFIED IDEOGRAPH 795E
    "\u{FA19}" => "\u{795E}",

    # FA1A CJK UNIFIED IDEOGRAPH FA1A
    # ->   7965 CJK UNIFIED IDEOGRAPH 7965
    "\u{FA1A}" => "\u{7965}",

    # FA1B CJK UNIFIED IDEOGRAPH FA1B
    # ->   798F CJK UNIFIED IDEOGRAPH 798F
    "\u{FA1B}" => "\u{798F}",

    # FA1C CJK UNIFIED IDEOGRAPH FA1C
    # ->   9756 CJK UNIFIED IDEOGRAPH 9756
    "\u{FA1C}" => "\u{9756}",

    # FA1D CJK UNIFIED IDEOGRAPH FA1D
    # ->   7CBE CJK UNIFIED IDEOGRAPH 7CBE
    "\u{FA1D}" => "\u{7CBE}",

    # FA1E CJK UNIFIED IDEOGRAPH FA1E
    # ->   7FBD CJK UNIFIED IDEOGRAPH 7FBD
    "\u{FA1E}" => "\u{7FBD}",

    # FA20 CJK UNIFIED IDEOGRAPH FA20
    # ->   8612 CJK UNIFIED IDEOGRAPH 8612
    "\u{FA20}" => "\u{8612}",

    # FA22 CJK UNIFIED IDEOGRAPH FA22
    # ->   8AF8 CJK UNIFIED IDEOGRAPH 8AF8
    "\u{FA22}" => "\u{8AF8}",

    # FA25 CJK UNIFIED IDEOGRAPH FA25
    # ->   9038 CJK UNIFIED IDEOGRAPH 9038
    "\u{FA25}" => "\u{9038}",

    # FA26 CJK UNIFIED IDEOGRAPH FA26
    # ->   90FD CJK UNIFIED IDEOGRAPH 90FD
    "\u{FA26}" => "\u{90FD}",

    # FA2A CJK UNIFIED IDEOGRAPH FA2A
    # ->   98EF CJK UNIFIED IDEOGRAPH 98EF
    "\u{FA2A}" => "\u{98EF}",

    # FA2B CJK UNIFIED IDEOGRAPH FA2B
    # ->   98FC CJK UNIFIED IDEOGRAPH 98FC
    "\u{FA2B}" => "\u{98FC}",

    # FA2C CJK UNIFIED IDEOGRAPH FA2C
    # ->   9928 CJK UNIFIED IDEOGRAPH 9928
    "\u{FA2C}" => "\u{9928}",

    # FA2D CJK UNIFIED IDEOGRAPH FA2D
    # ->   9DB4 CJK UNIFIED IDEOGRAPH 9DB4
    "\u{FA2D}" => "\u{9DB4}",

    # FA30 CJK UNIFIED IDEOGRAPH FA30
    # ->   4FAE CJK UNIFIED IDEOGRAPH 4FAE
    "\u{FA30}" => "\u{4FAE}",

    # FA31 CJK UNIFIED IDEOGRAPH FA31
    # ->   50E7 CJK UNIFIED IDEOGRAPH 50E7
    "\u{FA31}" => "\u{50E7}",

    # FA32 CJK UNIFIED IDEOGRAPH FA32
    # ->   514D CJK UNIFIED IDEOGRAPH 514D
    "\u{FA32}" => "\u{514D}",

    # FA33 CJK UNIFIED IDEOGRAPH FA33
    # ->   52C9 CJK UNIFIED IDEOGRAPH 52C9
    "\u{FA33}" => "\u{52C9}",

    # FA34 CJK UNIFIED IDEOGRAPH FA34
    # ->   52E4 CJK UNIFIED IDEOGRAPH 52E4
    "\u{FA34}" => "\u{52E4}",

    # FA35 CJK UNIFIED IDEOGRAPH FA35
    # ->   5351 CJK UNIFIED IDEOGRAPH 5351
    "\u{FA35}" => "\u{5351}",

    # FA36 CJK UNIFIED IDEOGRAPH FA36
    # ->   559D CJK UNIFIED IDEOGRAPH 559D
    "\u{FA36}" => "\u{559D}",

    # FA37 CJK UNIFIED IDEOGRAPH FA37
    # ->   5606 CJK UNIFIED IDEOGRAPH 5606
    "\u{FA37}" => "\u{5606}",

    # FA38 CJK UNIFIED IDEOGRAPH FA38
    # ->   5668 CJK UNIFIED IDEOGRAPH 5668
    "\u{FA38}" => "\u{5668}",

    # FA39 CJK UNIFIED IDEOGRAPH FA39
    # ->   5840 CJK UNIFIED IDEOGRAPH 5840
    "\u{FA39}" => "\u{5840}",

    # FA3A CJK UNIFIED IDEOGRAPH FA3A
    # ->   58A8 CJK UNIFIED IDEOGRAPH 58A8
    "\u{FA3A}" => "\u{58A8}",

    # FA3B CJK UNIFIED IDEOGRAPH FA3B
    # ->   5C64 CJK UNIFIED IDEOGRAPH 5C64
    "\u{FA3B}" => "\u{5C64}",

    # FA3C CJK UNIFIED IDEOGRAPH FA3C
    # ->   5C6E CJK UNIFIED IDEOGRAPH 5C6E
    "\u{FA3C}" => "\u{5C6E}",

    # FA3D CJK UNIFIED IDEOGRAPH FA3D
    # ->   6094 CJK UNIFIED IDEOGRAPH 6094
    "\u{FA3D}" => "\u{6094}",

    # FA3E CJK UNIFIED IDEOGRAPH FA3E
    # ->   6168 CJK UNIFIED IDEOGRAPH 6168
    "\u{FA3E}" => "\u{6168}",

    # FA3F CJK UNIFIED IDEOGRAPH FA3F
    # ->   618E CJK UNIFIED IDEOGRAPH 618E
    "\u{FA3F}" => "\u{618E}",

    # FA40 CJK UNIFIED IDEOGRAPH FA40
    # ->   61F2 CJK UNIFIED IDEOGRAPH 61F2
    "\u{FA40}" => "\u{61F2}",

    # FA41 CJK UNIFIED IDEOGRAPH FA41
    # ->   654F CJK UNIFIED IDEOGRAPH 654F
    "\u{FA41}" => "\u{654F}",

    # FA42 CJK UNIFIED IDEOGRAPH FA42
    # ->   65E2 CJK UNIFIED IDEOGRAPH 65E2
    "\u{FA42}" => "\u{65E2}",

    # FA43 CJK UNIFIED IDEOGRAPH FA43
    # ->   6691 CJK UNIFIED IDEOGRAPH 6691
    "\u{FA43}" => "\u{6691}",

    # FA44 CJK UNIFIED IDEOGRAPH FA44
    # ->   6885 CJK UNIFIED IDEOGRAPH 6885
    "\u{FA44}" => "\u{6885}",

    # FA45 CJK UNIFIED IDEOGRAPH FA45
    # ->   6D77 CJK UNIFIED IDEOGRAPH 6D77
    "\u{FA45}" => "\u{6D77}",

    # FA46 CJK UNIFIED IDEOGRAPH FA46
    # ->   6E1A CJK UNIFIED IDEOGRAPH 6E1A
    "\u{FA46}" => "\u{6E1A}",

    # FA47 CJK UNIFIED IDEOGRAPH FA47
    # ->   6F22 CJK UNIFIED IDEOGRAPH 6F22
    "\u{FA47}" => "\u{6F22}",

    # FA48 CJK UNIFIED IDEOGRAPH FA48
    # ->   716E CJK UNIFIED IDEOGRAPH 716E
    "\u{FA48}" => "\u{716E}",

    # FA49 CJK UNIFIED IDEOGRAPH FA49
    # ->   722B CJK UNIFIED IDEOGRAPH 722B
    "\u{FA49}" => "\u{722B}",

    # FA4A CJK UNIFIED IDEOGRAPH FA4A
    # ->   7422 CJK UNIFIED IDEOGRAPH 7422
    "\u{FA4A}" => "\u{7422}",

    # FA4B CJK UNIFIED IDEOGRAPH FA4B
    # ->   7891 CJK UNIFIED IDEOGRAPH 7891
    "\u{FA4B}" => "\u{7891}",

    # FA4C CJK UNIFIED IDEOGRAPH FA4C
    # ->   793E CJK UNIFIED IDEOGRAPH 793E
    "\u{FA4C}" => "\u{793E}",

    # FA4D CJK UNIFIED IDEOGRAPH FA4D
    # ->   7949 CJK UNIFIED IDEOGRAPH 7949
    "\u{FA4D}" => "\u{7949}",

    # FA4E CJK UNIFIED IDEOGRAPH FA4E
    # ->   7948 CJK UNIFIED IDEOGRAPH 7948
    "\u{FA4E}" => "\u{7948}",

    # FA4F CJK UNIFIED IDEOGRAPH FA4F
    # ->   7950 CJK UNIFIED IDEOGRAPH 7950
    "\u{FA4F}" => "\u{7950}",

    # FA50 CJK UNIFIED IDEOGRAPH FA50
    # ->   7956 CJK UNIFIED IDEOGRAPH 7956
    "\u{FA50}" => "\u{7956}",

    # FA51 CJK UNIFIED IDEOGRAPH FA51
    # ->   795D CJK UNIFIED IDEOGRAPH 795D
    "\u{FA51}" => "\u{795D}",

    # FA52 CJK UNIFIED IDEOGRAPH FA52
    # ->   798D CJK UNIFIED IDEOGRAPH 798D
    "\u{FA52}" => "\u{798D}",

    # FA53 CJK UNIFIED IDEOGRAPH FA53
    # ->   798E CJK UNIFIED IDEOGRAPH 798E
    "\u{FA53}" => "\u{798E}",

    # FA54 CJK UNIFIED IDEOGRAPH FA54
    # ->   7A40 CJK UNIFIED IDEOGRAPH 7A40
    "\u{FA54}" => "\u{7A40}",

    # FA55 CJK UNIFIED IDEOGRAPH FA55
    # ->   7A81 CJK UNIFIED IDEOGRAPH 7A81
    "\u{FA55}" => "\u{7A81}",

    # FA56 CJK UNIFIED IDEOGRAPH FA56
    # ->   7BC0 CJK UNIFIED IDEOGRAPH 7BC0
    "\u{FA56}" => "\u{7BC0}",

    # FA57 CJK UNIFIED IDEOGRAPH FA57
    # ->   7DF4 CJK UNIFIED IDEOGRAPH 7DF4
    "\u{FA57}" => "\u{7DF4}",

    # FA58 CJK UNIFIED IDEOGRAPH FA58
    # ->   7E09 CJK UNIFIED IDEOGRAPH 7E09
    "\u{FA58}" => "\u{7E09}",

    # FA59 CJK UNIFIED IDEOGRAPH FA59
    # ->   7E41 CJK UNIFIED IDEOGRAPH 7E41
    "\u{FA59}" => "\u{7E41}",

    # FA5A CJK UNIFIED IDEOGRAPH FA5A
    # ->   7F72 CJK UNIFIED IDEOGRAPH 7F72
    "\u{FA5A}" => "\u{7F72}",

    # FA5B CJK UNIFIED IDEOGRAPH FA5B
    # ->   8005 CJK UNIFIED IDEOGRAPH 8005
    "\u{FA5B}" => "\u{8005}",

    # FA5C CJK UNIFIED IDEOGRAPH FA5C
    # ->   81ED CJK UNIFIED IDEOGRAPH 81ED
    "\u{FA5C}" => "\u{81ED}",

    # FA5D CJK UNIFIED IDEOGRAPH FA5D
    # ->   8279 CJK UNIFIED IDEOGRAPH 8279
    "\u{FA5D}" => "\u{8279}",

    # FA5E CJK UNIFIED IDEOGRAPH FA5E
    # ->   8279 CJK UNIFIED IDEOGRAPH 8279
    "\u{FA5E}" => "\u{8279}",

    # FA5F CJK UNIFIED IDEOGRAPH FA5F
    # ->   8457 CJK UNIFIED IDEOGRAPH 8457
    "\u{FA5F}" => "\u{8457}",

    # FA60 CJK UNIFIED IDEOGRAPH FA60
    # ->   8910 CJK UNIFIED IDEOGRAPH 8910
    "\u{FA60}" => "\u{8910}",

    # FA61 CJK UNIFIED IDEOGRAPH FA61
    # ->   8996 CJK UNIFIED IDEOGRAPH 8996
    "\u{FA61}" => "\u{8996}",

    # FA62 CJK UNIFIED IDEOGRAPH FA62
    # ->   8B01 CJK UNIFIED IDEOGRAPH 8B01
    "\u{FA62}" => "\u{8B01}",

    # FA63 CJK UNIFIED IDEOGRAPH FA63
    # ->   8B39 CJK UNIFIED IDEOGRAPH 8B39
    "\u{FA63}" => "\u{8B39}",

    # FA64 CJK UNIFIED IDEOGRAPH FA64
    # ->   8CD3 CJK UNIFIED IDEOGRAPH 8CD3
    "\u{FA64}" => "\u{8CD3}",

    # FA65 CJK UNIFIED IDEOGRAPH FA65
    # ->   8D08 CJK UNIFIED IDEOGRAPH 8D08
    "\u{FA65}" => "\u{8D08}",

    # FA66 CJK UNIFIED IDEOGRAPH FA66
    # ->   8FB6 CJK UNIFIED IDEOGRAPH 8FB6
    "\u{FA66}" => "\u{8FB6}",

    # FA67 CJK UNIFIED IDEOGRAPH FA67
    # ->   9038 CJK UNIFIED IDEOGRAPH 9038
    "\u{FA67}" => "\u{9038}",

    # FA68 CJK UNIFIED IDEOGRAPH FA68
    # ->   96E3 CJK UNIFIED IDEOGRAPH 96E3
    "\u{FA68}" => "\u{96E3}",

    # FA69 CJK UNIFIED IDEOGRAPH FA69
    # ->   97FF CJK UNIFIED IDEOGRAPH 97FF
    "\u{FA69}" => "\u{97FF}",

    # FA6A CJK UNIFIED IDEOGRAPH FA6A
    # ->   983B CJK UNIFIED IDEOGRAPH 983B
    "\u{FA6A}" => "\u{983B}",

    # FB00 LATIN SMALL LIGATURE FF
    # ->   0066 LATIN SMALL LETTER F
    #    + 0066 LATIN SMALL LETTER F
    "\u{FB00}" => "ff",

    # FB01 LATIN SMALL LIGATURE FI
    # ->   0066 LATIN SMALL LETTER F
    #    + 0069 LATIN SMALL LETTER I
    "\u{FB01}" => "fi",

    # FB02 LATIN SMALL LIGATURE FL
    # ->   0066 LATIN SMALL LETTER F
    #    + 006C LATIN SMALL LETTER L
    "\u{FB02}" => "fl",

    # FB03 LATIN SMALL LIGATURE FFI
    # ->   0066 LATIN SMALL LETTER F
    #    + 0066 LATIN SMALL LETTER F
    #    + 0069 LATIN SMALL LETTER I
    "\u{FB03}" => "ffi",

    # FB04 LATIN SMALL LIGATURE FFL
    # ->   0066 LATIN SMALL LETTER F
    #    + 0066 LATIN SMALL LETTER F
    #    + 006C LATIN SMALL LETTER L
    "\u{FB04}" => "ffl",

    # FB05 LATIN SMALL LIGATURE LONG S T
    # ->   0074 LATIN SMALL LETTER T
    #    + 0073 LATIN SMALL LETTER S
    "\u{FB05}" => "ts",

    # FB06 LATIN SMALL LIGATURE ST
    # ->   0073 LATIN SMALL LETTER S
    #    + 0074 LATIN SMALL LETTER T
    "\u{FB06}" => "st",

    # FB13 ARMENIAN SMALL LIGATURE MEN NOW
    # ->   0574 ARMENIAN SMALL LETTER MEN
    #    + 0576 ARMENIAN SMALL LETTER NOW
    "\u{FB13}" => "\u{0574}\u{0576}",

    # FB14 ARMENIAN SMALL LIGATURE MEN ECH
    # ->   0574 ARMENIAN SMALL LETTER MEN
    #    + 0565 ARMENIAN SMALL LETTER ECH
    "\u{FB14}" => "\u{0574}\u{0565}",

    # FB15 ARMENIAN SMALL LIGATURE MEN INI
    # ->   0574 ARMENIAN SMALL LETTER MEN
    #    + 056B ARMENIAN SMALL LETTER INI
    "\u{FB15}" => "\u{0574}\u{056B}",

    # FB16 ARMENIAN SMALL LIGATURE VEW NOW
    # ->   057E ARMENIAN SMALL LETTER VEW
    #    + 0576 ARMENIAN SMALL LETTER NOW
    "\u{FB16}" => "\u{057E}\u{0576}",

    # FB17 ARMENIAN SMALL LIGATURE MEN XEH
    # ->   0574 ARMENIAN SMALL LETTER MEN
    #    + 056D ARMENIAN SMALL LETTER XEH
    "\u{FB17}" => "\u{0574}\u{056D}",

    # FB1D HEBREW LETTER YOD WITH HIRIQ
    # ->   05D9 HEBREW LETTER YOD
    "\u{FB1D}" => "\u{05D9}",

    # FB1F HEBREW LIGATURE YIDDISH YOD YOD PATAH
    # ->   05F2 HEBREW LIGATURE YIDDISH DOUBLE YOD
    "\u{FB1F}" => "\u{05F2}",

    # FB20 HEBREW LETTER ALTERNATIVE AYIN
    # ->   05E2 HEBREW LETTER AYIN
    "\u{FB20}" => "\u{05E2}",

    # FB21 HEBREW LETTER WIDE ALEF
    # ->   05D0 HEBREW LETTER ALEF
    "\u{FB21}" => "\u{05D0}",

    # FB22 HEBREW LETTER WIDE DALET
    # ->   05D3 HEBREW LETTER DALET
    "\u{FB22}" => "\u{05D3}",

    # FB23 HEBREW LETTER WIDE HE
    # ->   05D4 HEBREW LETTER HE
    "\u{FB23}" => "\u{05D4}",

    # FB24 HEBREW LETTER WIDE KAF
    # ->   05DB HEBREW LETTER KAF
    "\u{FB24}" => "\u{05DB}",

    # FB25 HEBREW LETTER WIDE LAMED
    # ->   05DC HEBREW LETTER LAMED
    "\u{FB25}" => "\u{05DC}",

    # FB26 HEBREW LETTER WIDE FINAL MEM
    # ->   05DD HEBREW LETTER FINAL MEM
    "\u{FB26}" => "\u{05DD}",

    # FB27 HEBREW LETTER WIDE RESH
    # ->   05E8 HEBREW LETTER RESH
    "\u{FB27}" => "\u{05E8}",

    # FB28 HEBREW LETTER WIDE TAV
    # ->   05EA HEBREW LETTER TAV
    "\u{FB28}" => "\u{05EA}",

    # FB29 HEBREW LETTER ALTERNATIVE PLUS SIGN
    # ->   002B PLUS SIGN
    "\u{FB29}" => "+",

    # FB2A HEBREW LETTER SHIN WITH SHIN DOT
    # ->   05E9 HEBREW LETTER SHIN
    "\u{FB2A}" => "\u{05E9}",

    # FB2B HEBREW LETTER SHIN WITH SIN DOT
    # ->   05E9 HEBREW LETTER SHIN
    "\u{FB2B}" => "\u{05E9}",

    # FB2C HEBREW LETTER SHIN WITH DAGESH AND SHIN DOT
    # ->   05E9 HEBREW LETTER SHIN
    "\u{FB2C}" => "\u{05E9}",

    # FB2D HEBREW LETTER SHIN WITH DAGESH AND SIN DOT
    # ->   05E9 HEBREW LETTER SHIN
    "\u{FB2D}" => "\u{05E9}",

    # FB2E HEBREW LETTER ALEF WITH PATAH
    # ->   05D0 HEBREW LETTER ALEF
    "\u{FB2E}" => "\u{05D0}",

    # FB2F HEBREW LETTER ALEF WITH QAMATS
    # ->   05D0 HEBREW LETTER ALEF
    "\u{FB2F}" => "\u{05D0}",

    # FB30 HEBREW LETTER ALEF WITH MAPIQ
    # ->   05D0 HEBREW LETTER ALEF
    "\u{FB30}" => "\u{05D0}",

    # FB31 HEBREW LETTER BET WITH DAGESH
    # ->   05D1 HEBREW LETTER BET
    "\u{FB31}" => "\u{05D1}",

    # FB32 HEBREW LETTER GIMEL WITH DAGESH
    # ->   05D2 HEBREW LETTER GIMEL
    "\u{FB32}" => "\u{05D2}",

    # FB33 HEBREW LETTER DALET WITH DAGESH
    # ->   05D3 HEBREW LETTER DALET
    "\u{FB33}" => "\u{05D3}",

    # FB34 HEBREW LETTER HE WITH MAPIQ
    # ->   05D4 HEBREW LETTER HE
    "\u{FB34}" => "\u{05D4}",

    # FB35 HEBREW LETTER VAV WITH DAGESH
    # ->   05D5 HEBREW LETTER VAV
    "\u{FB35}" => "\u{05D5}",

    # FB36 HEBREW LETTER ZAYIN WITH DAGESH
    # ->   05D6 HEBREW LETTER ZAYIN
    "\u{FB36}" => "\u{05D6}",

    # FB38 HEBREW LETTER TET WITH DAGESH
    # ->   05D8 HEBREW LETTER TET
    "\u{FB38}" => "\u{05D8}",

    # FB39 HEBREW LETTER YOD WITH DAGESH
    # ->   05D9 HEBREW LETTER YOD
    "\u{FB39}" => "\u{05D9}",

    # FB3A HEBREW LETTER FINAL KAF WITH DAGESH
    # ->   05DA HEBREW LETTER FINAL KAF
    "\u{FB3A}" => "\u{05DA}",

    # FB3B HEBREW LETTER KAF WITH DAGESH
    # ->   05DB HEBREW LETTER KAF
    "\u{FB3B}" => "\u{05DB}",

    # FB3C HEBREW LETTER LAMED WITH DAGESH
    # ->   05DC HEBREW LETTER LAMED
    "\u{FB3C}" => "\u{05DC}",

    # FB3E HEBREW LETTER MEM WITH DAGESH
    # ->   05DE HEBREW LETTER MEM
    "\u{FB3E}" => "\u{05DE}",

    # FB40 HEBREW LETTER NUN WITH DAGESH
    # ->   05E0 HEBREW LETTER NUN
    "\u{FB40}" => "\u{05E0}",

    # FB41 HEBREW LETTER SAMEKH WITH DAGESH
    # ->   05E1 HEBREW LETTER SAMEKH
    "\u{FB41}" => "\u{05E1}",

    # FB43 HEBREW LETTER FINAL PE WITH DAGESH
    # ->   05E3 HEBREW LETTER FINAL PE
    "\u{FB43}" => "\u{05E3}",

    # FB44 HEBREW LETTER PE WITH DAGESH
    # ->   05E4 HEBREW LETTER PE
    "\u{FB44}" => "\u{05E4}",

    # FB46 HEBREW LETTER TSADI WITH DAGESH
    # ->   05E6 HEBREW LETTER TSADI
    "\u{FB46}" => "\u{05E6}",

    # FB47 HEBREW LETTER QOF WITH DAGESH
    # ->   05E7 HEBREW LETTER QOF
    "\u{FB47}" => "\u{05E7}",

    # FB48 HEBREW LETTER RESH WITH DAGESH
    # ->   05E8 HEBREW LETTER RESH
    "\u{FB48}" => "\u{05E8}",

    # FB49 HEBREW LETTER SHIN WITH DAGESH
    # ->   05E9 HEBREW LETTER SHIN
    "\u{FB49}" => "\u{05E9}",

    # FB4A HEBREW LETTER TAV WITH DAGESH
    # ->   05EA HEBREW LETTER TAV
    "\u{FB4A}" => "\u{05EA}",

    # FB4B HEBREW LETTER VAV WITH HOLAM
    # ->   05D5 HEBREW LETTER VAV
    "\u{FB4B}" => "\u{05D5}",

    # FB4C HEBREW LETTER BET WITH RAFE
    # ->   05D1 HEBREW LETTER BET
    "\u{FB4C}" => "\u{05D1}",

    # FB4D HEBREW LETTER KAF WITH RAFE
    # ->   05DB HEBREW LETTER KAF
    "\u{FB4D}" => "\u{05DB}",

    # FB4E HEBREW LETTER PE WITH RAFE
    # ->   05E4 HEBREW LETTER PE
    "\u{FB4E}" => "\u{05E4}",

    # FB4F HEBREW LIGATURE ALEF LAMED
    # ->   05D0 HEBREW LETTER ALEF
    #    + 05DC HEBREW LETTER LAMED
    "\u{FB4F}" => "\u{05D0}\u{05DC}",

    # FB50 ARABIC LETTER ALEF WASLA ISOLATED FORM
    # ->   0671 ARABIC LETTER ALEF WASLA
    "\u{FB50}" => "\u{0671}",

    # FB51 ARABIC LETTER ALEF WASLA FINAL FORM
    # ->   0671 ARABIC LETTER ALEF WASLA
    "\u{FB51}" => "\u{0671}",

    # FB52 ARABIC LETTER BEEH ISOLATED FORM
    # ->   067B ARABIC LETTER BEEH
    "\u{FB52}" => "\u{067B}",

    # FB53 ARABIC LETTER BEEH FINAL FORM
    # ->   067B ARABIC LETTER BEEH
    "\u{FB53}" => "\u{067B}",

    # FB54 ARABIC LETTER BEEH INITIAL FORM
    # ->   067B ARABIC LETTER BEEH
    "\u{FB54}" => "\u{067B}",

    # FB55 ARABIC LETTER BEEH MEDIAL FORM
    # ->   067B ARABIC LETTER BEEH
    "\u{FB55}" => "\u{067B}",

    # FB56 ARABIC LETTER PEH ISOLATED FORM
    # ->   067E ARABIC LETTER PEH
    "\u{FB56}" => "\u{067E}",

    # FB57 ARABIC LETTER PEH FINAL FORM
    # ->   067E ARABIC LETTER PEH
    "\u{FB57}" => "\u{067E}",

    # FB58 ARABIC LETTER PEH INITIAL FORM
    # ->   067E ARABIC LETTER PEH
    "\u{FB58}" => "\u{067E}",

    # FB59 ARABIC LETTER PEH MEDIAL FORM
    # ->   067E ARABIC LETTER PEH
    "\u{FB59}" => "\u{067E}",

    # FB5A ARABIC LETTER BEHEH ISOLATED FORM
    # ->   0680 ARABIC LETTER BEHEH
    "\u{FB5A}" => "\u{0680}",

    # FB5B ARABIC LETTER BEHEH FINAL FORM
    # ->   0680 ARABIC LETTER BEHEH
    "\u{FB5B}" => "\u{0680}",

    # FB5C ARABIC LETTER BEHEH INITIAL FORM
    # ->   0680 ARABIC LETTER BEHEH
    "\u{FB5C}" => "\u{0680}",

    # FB5D ARABIC LETTER BEHEH MEDIAL FORM
    # ->   0680 ARABIC LETTER BEHEH
    "\u{FB5D}" => "\u{0680}",

    # FB5E ARABIC LETTER TTEHEH ISOLATED FORM
    # ->   067A ARABIC LETTER TTEHEH
    "\u{FB5E}" => "\u{067A}",

    # FB5F ARABIC LETTER TTEHEH FINAL FORM
    # ->   067A ARABIC LETTER TTEHEH
    "\u{FB5F}" => "\u{067A}",

    # FB60 ARABIC LETTER TTEHEH INITIAL FORM
    # ->   067A ARABIC LETTER TTEHEH
    "\u{FB60}" => "\u{067A}",

    # FB61 ARABIC LETTER TTEHEH MEDIAL FORM
    # ->   067A ARABIC LETTER TTEHEH
    "\u{FB61}" => "\u{067A}",

    # FB62 ARABIC LETTER TEHEH ISOLATED FORM
    # ->   067F ARABIC LETTER TEHEH
    "\u{FB62}" => "\u{067F}",

    # FB63 ARABIC LETTER TEHEH FINAL FORM
    # ->   067F ARABIC LETTER TEHEH
    "\u{FB63}" => "\u{067F}",

    # FB64 ARABIC LETTER TEHEH INITIAL FORM
    # ->   067F ARABIC LETTER TEHEH
    "\u{FB64}" => "\u{067F}",

    # FB65 ARABIC LETTER TEHEH MEDIAL FORM
    # ->   067F ARABIC LETTER TEHEH
    "\u{FB65}" => "\u{067F}",

    # FB66 ARABIC LETTER TTEH ISOLATED FORM
    # ->   0679 ARABIC LETTER TTEH
    "\u{FB66}" => "\u{0679}",

    # FB67 ARABIC LETTER TTEH FINAL FORM
    # ->   0679 ARABIC LETTER TTEH
    "\u{FB67}" => "\u{0679}",

    # FB68 ARABIC LETTER TTEH INITIAL FORM
    # ->   0679 ARABIC LETTER TTEH
    "\u{FB68}" => "\u{0679}",

    # FB69 ARABIC LETTER TTEH MEDIAL FORM
    # ->   0679 ARABIC LETTER TTEH
    "\u{FB69}" => "\u{0679}",

    # FB6A ARABIC LETTER VEH ISOLATED FORM
    # ->   06A4 ARABIC LETTER VEH
    "\u{FB6A}" => "\u{06A4}",

    # FB6B ARABIC LETTER VEH FINAL FORM
    # ->   06A4 ARABIC LETTER VEH
    "\u{FB6B}" => "\u{06A4}",

    # FB6C ARABIC LETTER VEH INITIAL FORM
    # ->   06A4 ARABIC LETTER VEH
    "\u{FB6C}" => "\u{06A4}",

    # FB6D ARABIC LETTER VEH MEDIAL FORM
    # ->   06A4 ARABIC LETTER VEH
    "\u{FB6D}" => "\u{06A4}",

    # FB6E ARABIC LETTER PEHEH ISOLATED FORM
    # ->   06A6 ARABIC LETTER PEHEH
    "\u{FB6E}" => "\u{06A6}",

    # FB6F ARABIC LETTER PEHEH FINAL FORM
    # ->   06A6 ARABIC LETTER PEHEH
    "\u{FB6F}" => "\u{06A6}",

    # FB70 ARABIC LETTER PEHEH INITIAL FORM
    # ->   06A6 ARABIC LETTER PEHEH
    "\u{FB70}" => "\u{06A6}",

    # FB71 ARABIC LETTER PEHEH MEDIAL FORM
    # ->   06A6 ARABIC LETTER PEHEH
    "\u{FB71}" => "\u{06A6}",

    # FB72 ARABIC LETTER DYEH ISOLATED FORM
    # ->   0684 ARABIC LETTER DYEH
    "\u{FB72}" => "\u{0684}",

    # FB73 ARABIC LETTER DYEH FINAL FORM
    # ->   0684 ARABIC LETTER DYEH
    "\u{FB73}" => "\u{0684}",

    # FB74 ARABIC LETTER DYEH INITIAL FORM
    # ->   0684 ARABIC LETTER DYEH
    "\u{FB74}" => "\u{0684}",

    # FB75 ARABIC LETTER DYEH MEDIAL FORM
    # ->   0684 ARABIC LETTER DYEH
    "\u{FB75}" => "\u{0684}",

    # FB76 ARABIC LETTER NYEH ISOLATED FORM
    # ->   0683 ARABIC LETTER NYEH
    "\u{FB76}" => "\u{0683}",

    # FB77 ARABIC LETTER NYEH FINAL FORM
    # ->   0683 ARABIC LETTER NYEH
    "\u{FB77}" => "\u{0683}",

    # FB78 ARABIC LETTER NYEH INITIAL FORM
    # ->   0683 ARABIC LETTER NYEH
    "\u{FB78}" => "\u{0683}",

    # FB79 ARABIC LETTER NYEH MEDIAL FORM
    # ->   0683 ARABIC LETTER NYEH
    "\u{FB79}" => "\u{0683}",

    # FB7A ARABIC LETTER TCHEH ISOLATED FORM
    # ->   0686 ARABIC LETTER TCHEH
    "\u{FB7A}" => "\u{0686}",

    # FB7B ARABIC LETTER TCHEH FINAL FORM
    # ->   0686 ARABIC LETTER TCHEH
    "\u{FB7B}" => "\u{0686}",

    # FB7C ARABIC LETTER TCHEH INITIAL FORM
    # ->   0686 ARABIC LETTER TCHEH
    "\u{FB7C}" => "\u{0686}",

    # FB7D ARABIC LETTER TCHEH MEDIAL FORM
    # ->   0686 ARABIC LETTER TCHEH
    "\u{FB7D}" => "\u{0686}",

    # FB7E ARABIC LETTER TCHEHEH ISOLATED FORM
    # ->   0687 ARABIC LETTER TCHEHEH
    "\u{FB7E}" => "\u{0687}",

    # FB7F ARABIC LETTER TCHEHEH FINAL FORM
    # ->   0687 ARABIC LETTER TCHEHEH
    "\u{FB7F}" => "\u{0687}",

    # FB80 ARABIC LETTER TCHEHEH INITIAL FORM
    # ->   0687 ARABIC LETTER TCHEHEH
    "\u{FB80}" => "\u{0687}",

    # FB81 ARABIC LETTER TCHEHEH MEDIAL FORM
    # ->   0687 ARABIC LETTER TCHEHEH
    "\u{FB81}" => "\u{0687}",

    # FB82 ARABIC LETTER DDAHAL ISOLATED FORM
    # ->   068D ARABIC LETTER DDAHAL
    "\u{FB82}" => "\u{068D}",

    # FB83 ARABIC LETTER DDAHAL FINAL FORM
    # ->   068D ARABIC LETTER DDAHAL
    "\u{FB83}" => "\u{068D}",

    # FB84 ARABIC LETTER DAHAL ISOLATED FORM
    # ->   068C ARABIC LETTER DAHAL
    "\u{FB84}" => "\u{068C}",

    # FB85 ARABIC LETTER DAHAL FINAL FORM
    # ->   068C ARABIC LETTER DAHAL
    "\u{FB85}" => "\u{068C}",

    # FB86 ARABIC LETTER DUL ISOLATED FORM
    # ->   068E ARABIC LETTER DUL
    "\u{FB86}" => "\u{068E}",

    # FB87 ARABIC LETTER DUL FINAL FORM
    # ->   068E ARABIC LETTER DUL
    "\u{FB87}" => "\u{068E}",

    # FB88 ARABIC LETTER DDAL ISOLATED FORM
    # ->   0688 ARABIC LETTER DDAL
    "\u{FB88}" => "\u{0688}",

    # FB89 ARABIC LETTER DDAL FINAL FORM
    # ->   0688 ARABIC LETTER DDAL
    "\u{FB89}" => "\u{0688}",

    # FB8A ARABIC LETTER JEH ISOLATED FORM
    # ->   0698 ARABIC LETTER JEH
    "\u{FB8A}" => "\u{0698}",

    # FB8B ARABIC LETTER JEH FINAL FORM
    # ->   0698 ARABIC LETTER JEH
    "\u{FB8B}" => "\u{0698}",

    # FB8C ARABIC LETTER RREH ISOLATED FORM
    # ->   0691 ARABIC LETTER RREH
    "\u{FB8C}" => "\u{0691}",

    # FB8D ARABIC LETTER RREH FINAL FORM
    # ->   0691 ARABIC LETTER RREH
    "\u{FB8D}" => "\u{0691}",

    # FB8E ARABIC LETTER KEHEH ISOLATED FORM
    # ->   06A9 ARABIC LETTER KEHEH
    "\u{FB8E}" => "\u{06A9}",

    # FB8F ARABIC LETTER KEHEH FINAL FORM
    # ->   06A9 ARABIC LETTER KEHEH
    "\u{FB8F}" => "\u{06A9}",

    # FB90 ARABIC LETTER KEHEH INITIAL FORM
    # ->   06A9 ARABIC LETTER KEHEH
    "\u{FB90}" => "\u{06A9}",

    # FB91 ARABIC LETTER KEHEH MEDIAL FORM
    # ->   06A9 ARABIC LETTER KEHEH
    "\u{FB91}" => "\u{06A9}",

    # FB92 ARABIC LETTER GAF ISOLATED FORM
    # ->   06AF ARABIC LETTER GAF
    "\u{FB92}" => "\u{06AF}",

    # FB93 ARABIC LETTER GAF FINAL FORM
    # ->   06AF ARABIC LETTER GAF
    "\u{FB93}" => "\u{06AF}",

    # FB94 ARABIC LETTER GAF INITIAL FORM
    # ->   06AF ARABIC LETTER GAF
    "\u{FB94}" => "\u{06AF}",

    # FB95 ARABIC LETTER GAF MEDIAL FORM
    # ->   06AF ARABIC LETTER GAF
    "\u{FB95}" => "\u{06AF}",

    # FB96 ARABIC LETTER GUEH ISOLATED FORM
    # ->   06B3 ARABIC LETTER GUEH
    "\u{FB96}" => "\u{06B3}",

    # FB97 ARABIC LETTER GUEH FINAL FORM
    # ->   06B3 ARABIC LETTER GUEH
    "\u{FB97}" => "\u{06B3}",

    # FB98 ARABIC LETTER GUEH INITIAL FORM
    # ->   06B3 ARABIC LETTER GUEH
    "\u{FB98}" => "\u{06B3}",

    # FB99 ARABIC LETTER GUEH MEDIAL FORM
    # ->   06B3 ARABIC LETTER GUEH
    "\u{FB99}" => "\u{06B3}",

    # FB9A ARABIC LETTER NGOEH ISOLATED FORM
    # ->   06B1 ARABIC LETTER NGOEH
    "\u{FB9A}" => "\u{06B1}",

    # FB9B ARABIC LETTER NGOEH FINAL FORM
    # ->   06B1 ARABIC LETTER NGOEH
    "\u{FB9B}" => "\u{06B1}",

    # FB9C ARABIC LETTER NGOEH INITIAL FORM
    # ->   06B1 ARABIC LETTER NGOEH
    "\u{FB9C}" => "\u{06B1}",

    # FB9D ARABIC LETTER NGOEH MEDIAL FORM
    # ->   06B1 ARABIC LETTER NGOEH
    "\u{FB9D}" => "\u{06B1}",

    # FB9E ARABIC LETTER NOON GHUNNA ISOLATED FORM
    # ->   06BA ARABIC LETTER NOON GHUNNA
    "\u{FB9E}" => "\u{06BA}",

    # FB9F ARABIC LETTER NOON GHUNNA FINAL FORM
    # ->   06BA ARABIC LETTER NOON GHUNNA
    "\u{FB9F}" => "\u{06BA}",

    # FBA0 ARABIC LETTER RNOON ISOLATED FORM
    # ->   06BB ARABIC LETTER RNOON
    "\u{FBA0}" => "\u{06BB}",

    # FBA1 ARABIC LETTER RNOON FINAL FORM
    # ->   06BB ARABIC LETTER RNOON
    "\u{FBA1}" => "\u{06BB}",

    # FBA2 ARABIC LETTER RNOON INITIAL FORM
    # ->   06BB ARABIC LETTER RNOON
    "\u{FBA2}" => "\u{06BB}",

    # FBA3 ARABIC LETTER RNOON MEDIAL FORM
    # ->   06BB ARABIC LETTER RNOON
    "\u{FBA3}" => "\u{06BB}",

    # FBA4 ARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORM
    # ->   06D5 ARABIC LETTER AE
    "\u{FBA4}" => "\u{06D5}",

    # FBA5 ARABIC LETTER HEH WITH YEH ABOVE FINAL FORM
    # ->   06D5 ARABIC LETTER AE
    "\u{FBA5}" => "\u{06D5}",

    # FBA6 ARABIC LETTER HEH GOAL ISOLATED FORM
    # ->   06C1 ARABIC LETTER HEH GOAL
    "\u{FBA6}" => "\u{06C1}",

    # FBA7 ARABIC LETTER HEH GOAL FINAL FORM
    # ->   06C1 ARABIC LETTER HEH GOAL
    "\u{FBA7}" => "\u{06C1}",

    # FBA8 ARABIC LETTER HEH GOAL INITIAL FORM
    # ->   06C1 ARABIC LETTER HEH GOAL
    "\u{FBA8}" => "\u{06C1}",

    # FBA9 ARABIC LETTER HEH GOAL MEDIAL FORM
    # ->   06C1 ARABIC LETTER HEH GOAL
    "\u{FBA9}" => "\u{06C1}",

    # FBAA ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM
    # ->   06BE ARABIC LETTER HEH DOACHASHMEE
    "\u{FBAA}" => "\u{06BE}",

    # FBAB ARABIC LETTER HEH DOACHASHMEE FINAL FORM
    # ->   06BE ARABIC LETTER HEH DOACHASHMEE
    "\u{FBAB}" => "\u{06BE}",

    # FBAC ARABIC LETTER HEH DOACHASHMEE INITIAL FORM
    # ->   06BE ARABIC LETTER HEH DOACHASHMEE
    "\u{FBAC}" => "\u{06BE}",

    # FBAD ARABIC LETTER HEH DOACHASHMEE MEDIAL FORM
    # ->   06BE ARABIC LETTER HEH DOACHASHMEE
    "\u{FBAD}" => "\u{06BE}",

    # FBAE ARABIC LETTER YEH BARREE ISOLATED FORM
    # ->   06D2 ARABIC LETTER YEH BARREE
    "\u{FBAE}" => "\u{06D2}",

    # FBAF ARABIC LETTER YEH BARREE FINAL FORM
    # ->   06D2 ARABIC LETTER YEH BARREE
    "\u{FBAF}" => "\u{06D2}",

    # FBB0 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM
    # ->   06D2 ARABIC LETTER YEH BARREE
    "\u{FBB0}" => "\u{06D2}",

    # FBB1 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    # ->   06D2 ARABIC LETTER YEH BARREE
    "\u{FBB1}" => "\u{06D2}",

    # FBD3 ARABIC LETTER NG ISOLATED FORM
    # ->   06AD ARABIC LETTER NG
    "\u{FBD3}" => "\u{06AD}",

    # FBD4 ARABIC LETTER NG FINAL FORM
    # ->   06AD ARABIC LETTER NG
    "\u{FBD4}" => "\u{06AD}",

    # FBD5 ARABIC LETTER NG INITIAL FORM
    # ->   06AD ARABIC LETTER NG
    "\u{FBD5}" => "\u{06AD}",

    # FBD6 ARABIC LETTER NG MEDIAL FORM
    # ->   06AD ARABIC LETTER NG
    "\u{FBD6}" => "\u{06AD}",

    # FBD7 ARABIC LETTER U ISOLATED FORM
    # ->   06C7 ARABIC LETTER U
    "\u{FBD7}" => "\u{06C7}",

    # FBD8 ARABIC LETTER U FINAL FORM
    # ->   06C7 ARABIC LETTER U
    "\u{FBD8}" => "\u{06C7}",

    # FBD9 ARABIC LETTER OE ISOLATED FORM
    # ->   06C6 ARABIC LETTER OE
    "\u{FBD9}" => "\u{06C6}",

    # FBDA ARABIC LETTER OE FINAL FORM
    # ->   06C6 ARABIC LETTER OE
    "\u{FBDA}" => "\u{06C6}",

    # FBDB ARABIC LETTER YU ISOLATED FORM
    # ->   06C8 ARABIC LETTER YU
    "\u{FBDB}" => "\u{06C8}",

    # FBDC ARABIC LETTER YU FINAL FORM
    # ->   06C8 ARABIC LETTER YU
    "\u{FBDC}" => "\u{06C8}",

    # FBDD ARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORM
    # ->   06C7 ARABIC LETTER U
    #    + 0674 ARABIC LETTER HIGH HAMZA
    "\u{FBDD}" => "\u{06C7}\u{0674}",

    # FBDE ARABIC LETTER VE ISOLATED FORM
    # ->   06CB ARABIC LETTER VE
    "\u{FBDE}" => "\u{06CB}",

    # FBDF ARABIC LETTER VE FINAL FORM
    # ->   06CB ARABIC LETTER VE
    "\u{FBDF}" => "\u{06CB}",

    # FBE0 ARABIC LETTER KIRGHIZ OE ISOLATED FORM
    # ->   06C5 ARABIC LETTER KIRGHIZ OE
    "\u{FBE0}" => "\u{06C5}",

    # FBE1 ARABIC LETTER KIRGHIZ OE FINAL FORM
    # ->   06C5 ARABIC LETTER KIRGHIZ OE
    "\u{FBE1}" => "\u{06C5}",

    # FBE2 ARABIC LETTER KIRGHIZ YU ISOLATED FORM
    # ->   06C9 ARABIC LETTER KIRGHIZ YU
    "\u{FBE2}" => "\u{06C9}",

    # FBE3 ARABIC LETTER KIRGHIZ YU FINAL FORM
    # ->   06C9 ARABIC LETTER KIRGHIZ YU
    "\u{FBE3}" => "\u{06C9}",

    # FBE4 ARABIC LETTER E ISOLATED FORM
    # ->   06D0 ARABIC LETTER E
    "\u{FBE4}" => "\u{06D0}",

    # FBE5 ARABIC LETTER E FINAL FORM
    # ->   06D0 ARABIC LETTER E
    "\u{FBE5}" => "\u{06D0}",

    # FBE6 ARABIC LETTER E INITIAL FORM
    # ->   06D0 ARABIC LETTER E
    "\u{FBE6}" => "\u{06D0}",

    # FBE7 ARABIC LETTER E MEDIAL FORM
    # ->   06D0 ARABIC LETTER E
    "\u{FBE7}" => "\u{06D0}",

    # FBE8 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FBE8}" => "\u{0649}",

    # FBE9 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA MEDIAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FBE9}" => "\u{0649}",

    # FBEA ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    #    + 064A ARABIC LETTER YEH
    "\u{FBEA}" => "\u{0627}\u{064A}",

    # FBEB ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    #    + 064A ARABIC LETTER YEH
    "\u{FBEB}" => "\u{0627}\u{064A}",

    # FBEC ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORM
    # ->   06D5 ARABIC LETTER AE
    #    + 064A ARABIC LETTER YEH
    "\u{FBEC}" => "\u{06D5}\u{064A}",

    # FBED ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORM
    # ->   06D5 ARABIC LETTER AE
    #    + 064A ARABIC LETTER YEH
    "\u{FBED}" => "\u{06D5}\u{064A}",

    # FBEE ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORM
    # ->   0648 ARABIC LETTER WAW
    #    + 064A ARABIC LETTER YEH
    "\u{FBEE}" => "\u{0648}\u{064A}",

    # FBEF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORM
    # ->   0648 ARABIC LETTER WAW
    #    + 064A ARABIC LETTER YEH
    "\u{FBEF}" => "\u{0648}\u{064A}",

    # FBF0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORM
    # ->   06C7 ARABIC LETTER U
    #    + 064A ARABIC LETTER YEH
    "\u{FBF0}" => "\u{06C7}\u{064A}",

    # FBF1 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORM
    # ->   06C7 ARABIC LETTER U
    #    + 064A ARABIC LETTER YEH
    "\u{FBF1}" => "\u{06C7}\u{064A}",

    # FBF2 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE ISOLATED FORM
    # ->   06C6 ARABIC LETTER OE
    #    + 064A ARABIC LETTER YEH
    "\u{FBF2}" => "\u{06C6}\u{064A}",

    # FBF3 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORM
    # ->   06C6 ARABIC LETTER OE
    #    + 064A ARABIC LETTER YEH
    "\u{FBF3}" => "\u{06C6}\u{064A}",

    # FBF4 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORM
    # ->   06C8 ARABIC LETTER YU
    #    + 064A ARABIC LETTER YEH
    "\u{FBF4}" => "\u{06C8}\u{064A}",

    # FBF5 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORM
    # ->   06C8 ARABIC LETTER YU
    #    + 064A ARABIC LETTER YEH
    "\u{FBF5}" => "\u{06C8}\u{064A}",

    # FBF6 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORM
    # ->   06D0 ARABIC LETTER E
    #    + 064A ARABIC LETTER YEH
    "\u{FBF6}" => "\u{06D0}\u{064A}",

    # FBF7 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E FINAL FORM
    # ->   06D0 ARABIC LETTER E
    #    + 064A ARABIC LETTER YEH
    "\u{FBF7}" => "\u{06D0}\u{064A}",

    # FBF8 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORM
    # ->   06D0 ARABIC LETTER E
    #    + 064A ARABIC LETTER YEH
    "\u{FBF8}" => "\u{06D0}\u{064A}",

    # FBF9 ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    #    + 064A ARABIC LETTER YEH
    "\u{FBF9}" => "\u{0649}\u{064A}",

    # FBFA ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    #    + 064A ARABIC LETTER YEH
    "\u{FBFA}" => "\u{0649}\u{064A}",

    # FBFB ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    #    + 064A ARABIC LETTER YEH
    "\u{FBFB}" => "\u{0649}\u{064A}",

    # FBFC ARABIC LETTER FARSI YEH ISOLATED FORM
    # ->   06CC ARABIC LETTER FARSI YEH
    "\u{FBFC}" => "\u{06CC}",

    # FBFD ARABIC LETTER FARSI YEH FINAL FORM
    # ->   06CC ARABIC LETTER FARSI YEH
    "\u{FBFD}" => "\u{06CC}",

    # FBFE ARABIC LETTER FARSI YEH INITIAL FORM
    # ->   06CC ARABIC LETTER FARSI YEH
    "\u{FBFE}" => "\u{06CC}",

    # FBFF ARABIC LETTER FARSI YEH MEDIAL FORM
    # ->   06CC ARABIC LETTER FARSI YEH
    "\u{FBFF}" => "\u{06CC}",

    # FC00 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC00}" => "\u{062C}\u{064A}",

    # FC01 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FC01}" => "\u{062D}\u{064A}",

    # FC02 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC02}" => "\u{0645}\u{064A}",

    # FC03 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    #    + 064A ARABIC LETTER YEH
    "\u{FC03}" => "\u{0649}\u{064A}",

    # FC04 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC04}" => "\u{064A}\u{064A}",

    # FC05 ARABIC LIGATURE BEH WITH JEEM ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC05}" => "\u{0628}\u{062C}",

    # FC06 ARABIC LIGATURE BEH WITH HAH ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062D ARABIC LETTER HAH
    "\u{FC06}" => "\u{0628}\u{062D}",

    # FC07 ARABIC LIGATURE BEH WITH KHAH ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FC07}" => "\u{0628}\u{062E}",

    # FC08 ARABIC LIGATURE BEH WITH MEEM ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC08}" => "\u{0628}\u{0645}",

    # FC09 ARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC09}" => "\u{0628}\u{0649}",

    # FC0A ARABIC LIGATURE BEH WITH YEH ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC0A}" => "\u{0628}\u{064A}",

    # FC0B ARABIC LIGATURE TEH WITH JEEM ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC0B}" => "\u{062A}\u{062C}",

    # FC0C ARABIC LIGATURE TEH WITH HAH ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062D ARABIC LETTER HAH
    "\u{FC0C}" => "\u{062A}\u{062D}",

    # FC0D ARABIC LIGATURE TEH WITH KHAH ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FC0D}" => "\u{062A}\u{062E}",

    # FC0E ARABIC LIGATURE TEH WITH MEEM ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC0E}" => "\u{062A}\u{0645}",

    # FC0F ARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC0F}" => "\u{062A}\u{0649}",

    # FC10 ARABIC LIGATURE TEH WITH YEH ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC10}" => "\u{062A}\u{064A}",

    # FC11 ARABIC LIGATURE THEH WITH JEEM ISOLATED FORM
    # ->   062B ARABIC LETTER THEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC11}" => "\u{062B}\u{062C}",

    # FC12 ARABIC LIGATURE THEH WITH MEEM ISOLATED FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC12}" => "\u{062B}\u{0645}",

    # FC13 ARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC13}" => "\u{062B}\u{0649}",

    # FC14 ARABIC LIGATURE THEH WITH YEH ISOLATED FORM
    # ->   062B ARABIC LETTER THEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC14}" => "\u{062B}\u{064A}",

    # FC15 ARABIC LIGATURE JEEM WITH HAH ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FC15}" => "\u{062C}\u{062D}",

    # FC16 ARABIC LIGATURE JEEM WITH MEEM ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC16}" => "\u{062C}\u{0645}",

    # FC17 ARABIC LIGATURE HAH WITH JEEM ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC17}" => "\u{062D}\u{062C}",

    # FC18 ARABIC LIGATURE HAH WITH MEEM ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC18}" => "\u{062D}\u{0645}",

    # FC19 ARABIC LIGATURE KHAH WITH JEEM ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC19}" => "\u{062E}\u{062C}",

    # FC1A ARABIC LIGATURE KHAH WITH HAH ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 062D ARABIC LETTER HAH
    "\u{FC1A}" => "\u{062E}\u{062D}",

    # FC1B ARABIC LIGATURE KHAH WITH MEEM ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC1B}" => "\u{062E}\u{0645}",

    # FC1C ARABIC LIGATURE SEEN WITH JEEM ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FC1C}" => "\u{0633}\u{062C}",

    # FC1D ARABIC LIGATURE SEEN WITH HAH ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FC1D}" => "\u{0633}\u{062D}",

    # FC1E ARABIC LIGATURE SEEN WITH KHAH ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FC1E}" => "\u{0633}\u{062E}",

    # FC1F ARABIC LIGATURE SEEN WITH MEEM ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC1F}" => "\u{0633}\u{0645}",

    # FC20 ARABIC LIGATURE SAD WITH HAH ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062D ARABIC LETTER HAH
    "\u{FC20}" => "\u{0635}\u{062D}",

    # FC21 ARABIC LIGATURE SAD WITH MEEM ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC21}" => "\u{0635}\u{0645}",

    # FC22 ARABIC LIGATURE DAD WITH JEEM ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062C ARABIC LETTER JEEM
    "\u{FC22}" => "\u{0636}\u{062C}",

    # FC23 ARABIC LIGATURE DAD WITH HAH ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062D ARABIC LETTER HAH
    "\u{FC23}" => "\u{0636}\u{062D}",

    # FC24 ARABIC LIGATURE DAD WITH KHAH ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062E ARABIC LETTER KHAH
    "\u{FC24}" => "\u{0636}\u{062E}",

    # FC25 ARABIC LIGATURE DAD WITH MEEM ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC25}" => "\u{0636}\u{0645}",

    # FC26 ARABIC LIGATURE TAH WITH HAH ISOLATED FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 062D ARABIC LETTER HAH
    "\u{FC26}" => "\u{0637}\u{062D}",

    # FC27 ARABIC LIGATURE TAH WITH MEEM ISOLATED FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC27}" => "\u{0637}\u{0645}",

    # FC28 ARABIC LIGATURE ZAH WITH MEEM ISOLATED FORM
    # ->   0638 ARABIC LETTER ZAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC28}" => "\u{0638}\u{0645}",

    # FC29 ARABIC LIGATURE AIN WITH JEEM ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 062C ARABIC LETTER JEEM
    "\u{FC29}" => "\u{0639}\u{062C}",

    # FC2A ARABIC LIGATURE AIN WITH MEEM ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC2A}" => "\u{0639}\u{0645}",

    # FC2B ARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 062C ARABIC LETTER JEEM
    "\u{FC2B}" => "\u{063A}\u{062C}",

    # FC2C ARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC2C}" => "\u{063A}\u{0645}",

    # FC2D ARABIC LIGATURE FEH WITH JEEM ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC2D}" => "\u{0641}\u{062C}",

    # FC2E ARABIC LIGATURE FEH WITH HAH ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062D ARABIC LETTER HAH
    "\u{FC2E}" => "\u{0641}\u{062D}",

    # FC2F ARABIC LIGATURE FEH WITH KHAH ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FC2F}" => "\u{0641}\u{062E}",

    # FC30 ARABIC LIGATURE FEH WITH MEEM ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC30}" => "\u{0641}\u{0645}",

    # FC31 ARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC31}" => "\u{0641}\u{0649}",

    # FC32 ARABIC LIGATURE FEH WITH YEH ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC32}" => "\u{0641}\u{064A}",

    # FC33 ARABIC LIGATURE QAF WITH HAH ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 062D ARABIC LETTER HAH
    "\u{FC33}" => "\u{0642}\u{062D}",

    # FC34 ARABIC LIGATURE QAF WITH MEEM ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC34}" => "\u{0642}\u{0645}",

    # FC35 ARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC35}" => "\u{0642}\u{0649}",

    # FC36 ARABIC LIGATURE QAF WITH YEH ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 064A ARABIC LETTER YEH
    "\u{FC36}" => "\u{0642}\u{064A}",

    # FC37 ARABIC LIGATURE KAF WITH ALEF ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0627 ARABIC LETTER ALEF
    "\u{FC37}" => "\u{0643}\u{0627}",

    # FC38 ARABIC LIGATURE KAF WITH JEEM ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062C ARABIC LETTER JEEM
    "\u{FC38}" => "\u{0643}\u{062C}",

    # FC39 ARABIC LIGATURE KAF WITH HAH ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062D ARABIC LETTER HAH
    "\u{FC39}" => "\u{0643}\u{062D}",

    # FC3A ARABIC LIGATURE KAF WITH KHAH ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062E ARABIC LETTER KHAH
    "\u{FC3A}" => "\u{0643}\u{062E}",

    # FC3B ARABIC LIGATURE KAF WITH LAM ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0644 ARABIC LETTER LAM
    "\u{FC3B}" => "\u{0643}\u{0644}",

    # FC3C ARABIC LIGATURE KAF WITH MEEM ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC3C}" => "\u{0643}\u{0645}",

    # FC3D ARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC3D}" => "\u{0643}\u{0649}",

    # FC3E ARABIC LIGATURE KAF WITH YEH ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 064A ARABIC LETTER YEH
    "\u{FC3E}" => "\u{0643}\u{064A}",

    # FC3F ARABIC LIGATURE LAM WITH JEEM ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    "\u{FC3F}" => "\u{0644}\u{062C}",

    # FC40 ARABIC LIGATURE LAM WITH HAH ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    "\u{FC40}" => "\u{0644}\u{062D}",

    # FC41 ARABIC LIGATURE LAM WITH KHAH ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062E ARABIC LETTER KHAH
    "\u{FC41}" => "\u{0644}\u{062E}",

    # FC42 ARABIC LIGATURE LAM WITH MEEM ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC42}" => "\u{0644}\u{0645}",

    # FC43 ARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC43}" => "\u{0644}\u{0649}",

    # FC44 ARABIC LIGATURE LAM WITH YEH ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 064A ARABIC LETTER YEH
    "\u{FC44}" => "\u{0644}\u{064A}",

    # FC45 ARABIC LIGATURE MEEM WITH JEEM ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FC45}" => "\u{0645}\u{062C}",

    # FC46 ARABIC LIGATURE MEEM WITH HAH ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FC46}" => "\u{0645}\u{062D}",

    # FC47 ARABIC LIGATURE MEEM WITH KHAH ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FC47}" => "\u{0645}\u{062E}",

    # FC48 ARABIC LIGATURE MEEM WITH MEEM ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC48}" => "\u{0645}\u{0645}",

    # FC49 ARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC49}" => "\u{0645}\u{0649}",

    # FC4A ARABIC LIGATURE MEEM WITH YEH ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC4A}" => "\u{0645}\u{064A}",

    # FC4B ARABIC LIGATURE NOON WITH JEEM ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    "\u{FC4B}" => "\u{0646}\u{062C}",

    # FC4C ARABIC LIGATURE NOON WITH HAH ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062D ARABIC LETTER HAH
    "\u{FC4C}" => "\u{0646}\u{062D}",

    # FC4D ARABIC LIGATURE NOON WITH KHAH ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062E ARABIC LETTER KHAH
    "\u{FC4D}" => "\u{0646}\u{062E}",

    # FC4E ARABIC LIGATURE NOON WITH MEEM ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC4E}" => "\u{0646}\u{0645}",

    # FC4F ARABIC LIGATURE NOON WITH ALEF MAKSURA ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC4F}" => "\u{0646}\u{0649}",

    # FC50 ARABIC LIGATURE NOON WITH YEH ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 064A ARABIC LETTER YEH
    "\u{FC50}" => "\u{0646}\u{064A}",

    # FC51 ARABIC LIGATURE HEH WITH JEEM ISOLATED FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC51}" => "\u{0647}\u{062C}",

    # FC52 ARABIC LIGATURE HEH WITH MEEM ISOLATED FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC52}" => "\u{0647}\u{0645}",

    # FC53 ARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC53}" => "\u{0647}\u{0649}",

    # FC54 ARABIC LIGATURE HEH WITH YEH ISOLATED FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC54}" => "\u{0647}\u{064A}",

    # FC55 ARABIC LIGATURE YEH WITH JEEM ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC55}" => "\u{064A}\u{062C}",

    # FC56 ARABIC LIGATURE YEH WITH HAH ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062D ARABIC LETTER HAH
    "\u{FC56}" => "\u{064A}\u{062D}",

    # FC57 ARABIC LIGATURE YEH WITH KHAH ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FC57}" => "\u{064A}\u{062E}",

    # FC58 ARABIC LIGATURE YEH WITH MEEM ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC58}" => "\u{064A}\u{0645}",

    # FC59 ARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC59}" => "\u{064A}\u{0649}",

    # FC5A ARABIC LIGATURE YEH WITH YEH ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC5A}" => "\u{064A}\u{064A}",

    # FC5B ARABIC LIGATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORM
    # ->   0630 ARABIC LETTER THAL
    "\u{FC5B}" => "\u{0630}",

    # FC5C ARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORM
    # ->   0631 ARABIC LETTER REH
    "\u{FC5C}" => "\u{0631}",

    # FC5D ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC5D}" => "\u{0649}",

    # FC5E ARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC5E}" => " ",

    # FC5F ARABIC LIGATURE SHADDA WITH KASRATAN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC5F}" => " ",

    # FC60 ARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC60}" => " ",

    # FC61 ARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC61}" => " ",

    # FC62 ARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC62}" => " ",

    # FC63 ARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORM
    # ->   0020 SPACE
    "\u{FC63}" => " ",

    # FC64 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM
    # ->   0631 ARABIC LETTER REH
    #    + 064A ARABIC LETTER YEH
    "\u{FC64}" => "\u{0631}\u{064A}",

    # FC65 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORM
    # ->   0632 ARABIC LETTER ZAIN
    #    + 064A ARABIC LETTER YEH
    "\u{FC65}" => "\u{0632}\u{064A}",

    # FC66 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC66}" => "\u{0645}\u{064A}",

    # FC67 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 064A ARABIC LETTER YEH
    "\u{FC67}" => "\u{0646}\u{064A}",

    # FC68 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    #    + 064A ARABIC LETTER YEH
    "\u{FC68}" => "\u{0649}\u{064A}",

    # FC69 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC69}" => "\u{064A}\u{064A}",

    # FC6A ARABIC LIGATURE BEH WITH REH FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0631 ARABIC LETTER REH
    "\u{FC6A}" => "\u{0628}\u{0631}",

    # FC6B ARABIC LIGATURE BEH WITH ZAIN FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0632 ARABIC LETTER ZAIN
    "\u{FC6B}" => "\u{0628}\u{0632}",

    # FC6C ARABIC LIGATURE BEH WITH MEEM FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC6C}" => "\u{0628}\u{0645}",

    # FC6D ARABIC LIGATURE BEH WITH NOON FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0646 ARABIC LETTER NOON
    "\u{FC6D}" => "\u{0628}\u{0646}",

    # FC6E ARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC6E}" => "\u{0628}\u{0649}",

    # FC6F ARABIC LIGATURE BEH WITH YEH FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC6F}" => "\u{0628}\u{064A}",

    # FC70 ARABIC LIGATURE TEH WITH REH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0631 ARABIC LETTER REH
    "\u{FC70}" => "\u{062A}\u{0631}",

    # FC71 ARABIC LIGATURE TEH WITH ZAIN FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0632 ARABIC LETTER ZAIN
    "\u{FC71}" => "\u{062A}\u{0632}",

    # FC72 ARABIC LIGATURE TEH WITH MEEM FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC72}" => "\u{062A}\u{0645}",

    # FC73 ARABIC LIGATURE TEH WITH NOON FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0646 ARABIC LETTER NOON
    "\u{FC73}" => "\u{062A}\u{0646}",

    # FC74 ARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC74}" => "\u{062A}\u{0649}",

    # FC75 ARABIC LIGATURE TEH WITH YEH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC75}" => "\u{062A}\u{064A}",

    # FC76 ARABIC LIGATURE THEH WITH REH FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0631 ARABIC LETTER REH
    "\u{FC76}" => "\u{062B}\u{0631}",

    # FC77 ARABIC LIGATURE THEH WITH ZAIN FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0632 ARABIC LETTER ZAIN
    "\u{FC77}" => "\u{062B}\u{0632}",

    # FC78 ARABIC LIGATURE THEH WITH MEEM FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC78}" => "\u{062B}\u{0645}",

    # FC79 ARABIC LIGATURE THEH WITH NOON FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0646 ARABIC LETTER NOON
    "\u{FC79}" => "\u{062B}\u{0646}",

    # FC7A ARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC7A}" => "\u{062B}\u{0649}",

    # FC7B ARABIC LIGATURE THEH WITH YEH FINAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC7B}" => "\u{062B}\u{064A}",

    # FC7C ARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC7C}" => "\u{0641}\u{0649}",

    # FC7D ARABIC LIGATURE FEH WITH YEH FINAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC7D}" => "\u{0641}\u{064A}",

    # FC7E ARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC7E}" => "\u{0642}\u{0649}",

    # FC7F ARABIC LIGATURE QAF WITH YEH FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 064A ARABIC LETTER YEH
    "\u{FC7F}" => "\u{0642}\u{064A}",

    # FC80 ARABIC LIGATURE KAF WITH ALEF FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0627 ARABIC LETTER ALEF
    "\u{FC80}" => "\u{0643}\u{0627}",

    # FC81 ARABIC LIGATURE KAF WITH LAM FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0644 ARABIC LETTER LAM
    "\u{FC81}" => "\u{0643}\u{0644}",

    # FC82 ARABIC LIGATURE KAF WITH MEEM FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC82}" => "\u{0643}\u{0645}",

    # FC83 ARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC83}" => "\u{0643}\u{0649}",

    # FC84 ARABIC LIGATURE KAF WITH YEH FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 064A ARABIC LETTER YEH
    "\u{FC84}" => "\u{0643}\u{064A}",

    # FC85 ARABIC LIGATURE LAM WITH MEEM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC85}" => "\u{0644}\u{0645}",

    # FC86 ARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC86}" => "\u{0644}\u{0649}",

    # FC87 ARABIC LIGATURE LAM WITH YEH FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 064A ARABIC LETTER YEH
    "\u{FC87}" => "\u{0644}\u{064A}",

    # FC88 ARABIC LIGATURE MEEM WITH ALEF FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FC88}" => "\u{0645}\u{0627}",

    # FC89 ARABIC LIGATURE MEEM WITH MEEM FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC89}" => "\u{0645}\u{0645}",

    # FC8A ARABIC LIGATURE NOON WITH REH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0631 ARABIC LETTER REH
    "\u{FC8A}" => "\u{0646}\u{0631}",

    # FC8B ARABIC LIGATURE NOON WITH ZAIN FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0632 ARABIC LETTER ZAIN
    "\u{FC8B}" => "\u{0646}\u{0632}",

    # FC8C ARABIC LIGATURE NOON WITH MEEM FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC8C}" => "\u{0646}\u{0645}",

    # FC8D ARABIC LIGATURE NOON WITH NOON FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0646 ARABIC LETTER NOON
    "\u{FC8D}" => "\u{0646}\u{0646}",

    # FC8E ARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC8E}" => "\u{0646}\u{0649}",

    # FC8F ARABIC LIGATURE NOON WITH YEH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 064A ARABIC LETTER YEH
    "\u{FC8F}" => "\u{0646}\u{064A}",

    # FC90 ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC90}" => "\u{0649}",

    # FC91 ARABIC LIGATURE YEH WITH REH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0631 ARABIC LETTER REH
    "\u{FC91}" => "\u{064A}\u{0631}",

    # FC92 ARABIC LIGATURE YEH WITH ZAIN FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0632 ARABIC LETTER ZAIN
    "\u{FC92}" => "\u{064A}\u{0632}",

    # FC93 ARABIC LIGATURE YEH WITH MEEM FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC93}" => "\u{064A}\u{0645}",

    # FC94 ARABIC LIGATURE YEH WITH NOON FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0646 ARABIC LETTER NOON
    "\u{FC94}" => "\u{064A}\u{0646}",

    # FC95 ARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FC95}" => "\u{064A}\u{0649}",

    # FC96 ARABIC LIGATURE YEH WITH YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC96}" => "\u{064A}\u{064A}",

    # FC97 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC97}" => "\u{062C}\u{064A}",

    # FC98 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH INITIAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FC98}" => "\u{062D}\u{064A}",

    # FC99 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FC99}" => "\u{062E}\u{064A}",

    # FC9A ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FC9A}" => "\u{0645}\u{064A}",

    # FC9B ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 064A ARABIC LETTER YEH
    "\u{FC9B}" => "\u{0647}\u{064A}",

    # FC9C ARABIC LIGATURE BEH WITH JEEM INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FC9C}" => "\u{0628}\u{062C}",

    # FC9D ARABIC LIGATURE BEH WITH HAH INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062D ARABIC LETTER HAH
    "\u{FC9D}" => "\u{0628}\u{062D}",

    # FC9E ARABIC LIGATURE BEH WITH KHAH INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FC9E}" => "\u{0628}\u{062E}",

    # FC9F ARABIC LIGATURE BEH WITH MEEM INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FC9F}" => "\u{0628}\u{0645}",

    # FCA0 ARABIC LIGATURE BEH WITH HEH INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCA0}" => "\u{0628}\u{0647}",

    # FCA1 ARABIC LIGATURE TEH WITH JEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCA1}" => "\u{062A}\u{062C}",

    # FCA2 ARABIC LIGATURE TEH WITH HAH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062D ARABIC LETTER HAH
    "\u{FCA2}" => "\u{062A}\u{062D}",

    # FCA3 ARABIC LIGATURE TEH WITH KHAH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FCA3}" => "\u{062A}\u{062E}",

    # FCA4 ARABIC LIGATURE TEH WITH MEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCA4}" => "\u{062A}\u{0645}",

    # FCA5 ARABIC LIGATURE TEH WITH HEH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCA5}" => "\u{062A}\u{0647}",

    # FCA6 ARABIC LIGATURE THEH WITH MEEM INITIAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCA6}" => "\u{062B}\u{0645}",

    # FCA7 ARABIC LIGATURE JEEM WITH HAH INITIAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FCA7}" => "\u{062C}\u{062D}",

    # FCA8 ARABIC LIGATURE JEEM WITH MEEM INITIAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCA8}" => "\u{062C}\u{0645}",

    # FCA9 ARABIC LIGATURE HAH WITH JEEM INITIAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCA9}" => "\u{062D}\u{062C}",

    # FCAA ARABIC LIGATURE HAH WITH MEEM INITIAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCAA}" => "\u{062D}\u{0645}",

    # FCAB ARABIC LIGATURE KHAH WITH JEEM INITIAL FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCAB}" => "\u{062E}\u{062C}",

    # FCAC ARABIC LIGATURE KHAH WITH MEEM INITIAL FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCAC}" => "\u{062E}\u{0645}",

    # FCAD ARABIC LIGATURE SEEN WITH JEEM INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FCAD}" => "\u{0633}\u{062C}",

    # FCAE ARABIC LIGATURE SEEN WITH HAH INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FCAE}" => "\u{0633}\u{062D}",

    # FCAF ARABIC LIGATURE SEEN WITH KHAH INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FCAF}" => "\u{0633}\u{062E}",

    # FCB0 ARABIC LIGATURE SEEN WITH MEEM INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCB0}" => "\u{0633}\u{0645}",

    # FCB1 ARABIC LIGATURE SAD WITH HAH INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062D ARABIC LETTER HAH
    "\u{FCB1}" => "\u{0635}\u{062D}",

    # FCB2 ARABIC LIGATURE SAD WITH KHAH INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062E ARABIC LETTER KHAH
    "\u{FCB2}" => "\u{0635}\u{062E}",

    # FCB3 ARABIC LIGATURE SAD WITH MEEM INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCB3}" => "\u{0635}\u{0645}",

    # FCB4 ARABIC LIGATURE DAD WITH JEEM INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062C ARABIC LETTER JEEM
    "\u{FCB4}" => "\u{0636}\u{062C}",

    # FCB5 ARABIC LIGATURE DAD WITH HAH INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062D ARABIC LETTER HAH
    "\u{FCB5}" => "\u{0636}\u{062D}",

    # FCB6 ARABIC LIGATURE DAD WITH KHAH INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062E ARABIC LETTER KHAH
    "\u{FCB6}" => "\u{0636}\u{062E}",

    # FCB7 ARABIC LIGATURE DAD WITH MEEM INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCB7}" => "\u{0636}\u{0645}",

    # FCB8 ARABIC LIGATURE TAH WITH HAH INITIAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 062D ARABIC LETTER HAH
    "\u{FCB8}" => "\u{0637}\u{062D}",

    # FCB9 ARABIC LIGATURE ZAH WITH MEEM INITIAL FORM
    # ->   0638 ARABIC LETTER ZAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCB9}" => "\u{0638}\u{0645}",

    # FCBA ARABIC LIGATURE AIN WITH JEEM INITIAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 062C ARABIC LETTER JEEM
    "\u{FCBA}" => "\u{0639}\u{062C}",

    # FCBB ARABIC LIGATURE AIN WITH MEEM INITIAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCBB}" => "\u{0639}\u{0645}",

    # FCBC ARABIC LIGATURE GHAIN WITH JEEM INITIAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 062C ARABIC LETTER JEEM
    "\u{FCBC}" => "\u{063A}\u{062C}",

    # FCBD ARABIC LIGATURE GHAIN WITH MEEM INITIAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCBD}" => "\u{063A}\u{0645}",

    # FCBE ARABIC LIGATURE FEH WITH JEEM INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCBE}" => "\u{0641}\u{062C}",

    # FCBF ARABIC LIGATURE FEH WITH HAH INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062D ARABIC LETTER HAH
    "\u{FCBF}" => "\u{0641}\u{062D}",

    # FCC0 ARABIC LIGATURE FEH WITH KHAH INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FCC0}" => "\u{0641}\u{062E}",

    # FCC1 ARABIC LIGATURE FEH WITH MEEM INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCC1}" => "\u{0641}\u{0645}",

    # FCC2 ARABIC LIGATURE QAF WITH HAH INITIAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 062D ARABIC LETTER HAH
    "\u{FCC2}" => "\u{0642}\u{062D}",

    # FCC3 ARABIC LIGATURE QAF WITH MEEM INITIAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCC3}" => "\u{0642}\u{0645}",

    # FCC4 ARABIC LIGATURE KAF WITH JEEM INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062C ARABIC LETTER JEEM
    "\u{FCC4}" => "\u{0643}\u{062C}",

    # FCC5 ARABIC LIGATURE KAF WITH HAH INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062D ARABIC LETTER HAH
    "\u{FCC5}" => "\u{0643}\u{062D}",

    # FCC6 ARABIC LIGATURE KAF WITH KHAH INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 062E ARABIC LETTER KHAH
    "\u{FCC6}" => "\u{0643}\u{062E}",

    # FCC7 ARABIC LIGATURE KAF WITH LAM INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0644 ARABIC LETTER LAM
    "\u{FCC7}" => "\u{0643}\u{0644}",

    # FCC8 ARABIC LIGATURE KAF WITH MEEM INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCC8}" => "\u{0643}\u{0645}",

    # FCC9 ARABIC LIGATURE LAM WITH JEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    "\u{FCC9}" => "\u{0644}\u{062C}",

    # FCCA ARABIC LIGATURE LAM WITH HAH INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    "\u{FCCA}" => "\u{0644}\u{062D}",

    # FCCB ARABIC LIGATURE LAM WITH KHAH INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062E ARABIC LETTER KHAH
    "\u{FCCB}" => "\u{0644}\u{062E}",

    # FCCC ARABIC LIGATURE LAM WITH MEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCCC}" => "\u{0644}\u{0645}",

    # FCCD ARABIC LIGATURE LAM WITH HEH INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0647 ARABIC LETTER HEH
    "\u{FCCD}" => "\u{0644}\u{0647}",

    # FCCE ARABIC LIGATURE MEEM WITH JEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FCCE}" => "\u{0645}\u{062C}",

    # FCCF ARABIC LIGATURE MEEM WITH HAH INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FCCF}" => "\u{0645}\u{062D}",

    # FCD0 ARABIC LIGATURE MEEM WITH KHAH INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FCD0}" => "\u{0645}\u{062E}",

    # FCD1 ARABIC LIGATURE MEEM WITH MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCD1}" => "\u{0645}\u{0645}",

    # FCD2 ARABIC LIGATURE NOON WITH JEEM INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    "\u{FCD2}" => "\u{0646}\u{062C}",

    # FCD3 ARABIC LIGATURE NOON WITH HAH INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062D ARABIC LETTER HAH
    "\u{FCD3}" => "\u{0646}\u{062D}",

    # FCD4 ARABIC LIGATURE NOON WITH KHAH INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062E ARABIC LETTER KHAH
    "\u{FCD4}" => "\u{0646}\u{062E}",

    # FCD5 ARABIC LIGATURE NOON WITH MEEM INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCD5}" => "\u{0646}\u{0645}",

    # FCD6 ARABIC LIGATURE NOON WITH HEH INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0647 ARABIC LETTER HEH
    "\u{FCD6}" => "\u{0646}\u{0647}",

    # FCD7 ARABIC LIGATURE HEH WITH JEEM INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCD7}" => "\u{0647}\u{062C}",

    # FCD8 ARABIC LIGATURE HEH WITH MEEM INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCD8}" => "\u{0647}\u{0645}",

    # FCD9 ARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    "\u{FCD9}" => "\u{0647}",

    # FCDA ARABIC LIGATURE YEH WITH JEEM INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062C ARABIC LETTER JEEM
    "\u{FCDA}" => "\u{064A}\u{062C}",

    # FCDB ARABIC LIGATURE YEH WITH HAH INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062D ARABIC LETTER HAH
    "\u{FCDB}" => "\u{064A}\u{062D}",

    # FCDC ARABIC LIGATURE YEH WITH KHAH INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062E ARABIC LETTER KHAH
    "\u{FCDC}" => "\u{064A}\u{062E}",

    # FCDD ARABIC LIGATURE YEH WITH MEEM INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCDD}" => "\u{064A}\u{0645}",

    # FCDE ARABIC LIGATURE YEH WITH HEH INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCDE}" => "\u{064A}\u{0647}",

    # FCDF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM MEDIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FCDF}" => "\u{0645}\u{064A}",

    # FCE0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 064A ARABIC LETTER YEH
    "\u{FCE0}" => "\u{0647}\u{064A}",

    # FCE1 ARABIC LIGATURE BEH WITH MEEM MEDIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCE1}" => "\u{0628}\u{0645}",

    # FCE2 ARABIC LIGATURE BEH WITH HEH MEDIAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCE2}" => "\u{0628}\u{0647}",

    # FCE3 ARABIC LIGATURE TEH WITH MEEM MEDIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCE3}" => "\u{062A}\u{0645}",

    # FCE4 ARABIC LIGATURE TEH WITH HEH MEDIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCE4}" => "\u{062A}\u{0647}",

    # FCE5 ARABIC LIGATURE THEH WITH MEEM MEDIAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCE5}" => "\u{062B}\u{0645}",

    # FCE6 ARABIC LIGATURE THEH WITH HEH MEDIAL FORM
    # ->   062B ARABIC LETTER THEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCE6}" => "\u{062B}\u{0647}",

    # FCE7 ARABIC LIGATURE SEEN WITH MEEM MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCE7}" => "\u{0633}\u{0645}",

    # FCE8 ARABIC LIGATURE SEEN WITH HEH MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0647 ARABIC LETTER HEH
    "\u{FCE8}" => "\u{0633}\u{0647}",

    # FCE9 ARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCE9}" => "\u{0634}\u{0645}",

    # FCEA ARABIC LIGATURE SHEEN WITH HEH MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0647 ARABIC LETTER HEH
    "\u{FCEA}" => "\u{0634}\u{0647}",

    # FCEB ARABIC LIGATURE KAF WITH LAM MEDIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0644 ARABIC LETTER LAM
    "\u{FCEB}" => "\u{0643}\u{0644}",

    # FCEC ARABIC LIGATURE KAF WITH MEEM MEDIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCEC}" => "\u{0643}\u{0645}",

    # FCED ARABIC LIGATURE LAM WITH MEEM MEDIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCED}" => "\u{0644}\u{0645}",

    # FCEE ARABIC LIGATURE NOON WITH MEEM MEDIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCEE}" => "\u{0646}\u{0645}",

    # FCEF ARABIC LIGATURE NOON WITH HEH MEDIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0647 ARABIC LETTER HEH
    "\u{FCEF}" => "\u{0646}\u{0647}",

    # FCF0 ARABIC LIGATURE YEH WITH MEEM MEDIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FCF0}" => "\u{064A}\u{0645}",

    # FCF1 ARABIC LIGATURE YEH WITH HEH MEDIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FCF1}" => "\u{064A}\u{0647}",

    # FCF2 ARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FCF2}" => "\u{0640}",

    # FCF3 ARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FCF3}" => "\u{0640}",

    # FCF4 ARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FCF4}" => "\u{0640}",

    # FCF5 ARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCF5}" => "\u{0637}\u{0649}",

    # FCF6 ARABIC LIGATURE TAH WITH YEH ISOLATED FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 064A ARABIC LETTER YEH
    "\u{FCF6}" => "\u{0637}\u{064A}",

    # FCF7 ARABIC LIGATURE AIN WITH ALEF MAKSURA ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCF7}" => "\u{0639}\u{0649}",

    # FCF8 ARABIC LIGATURE AIN WITH YEH ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 064A ARABIC LETTER YEH
    "\u{FCF8}" => "\u{0639}\u{064A}",

    # FCF9 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCF9}" => "\u{063A}\u{0649}",

    # FCFA ARABIC LIGATURE GHAIN WITH YEH ISOLATED FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 064A ARABIC LETTER YEH
    "\u{FCFA}" => "\u{063A}\u{064A}",

    # FCFB ARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCFB}" => "\u{0633}\u{0649}",

    # FCFC ARABIC LIGATURE SEEN WITH YEH ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 064A ARABIC LETTER YEH
    "\u{FCFC}" => "\u{0633}\u{064A}",

    # FCFD ARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCFD}" => "\u{0634}\u{0649}",

    # FCFE ARABIC LIGATURE SHEEN WITH YEH ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 064A ARABIC LETTER YEH
    "\u{FCFE}" => "\u{0634}\u{064A}",

    # FCFF ARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FCFF}" => "\u{062D}\u{0649}",

    # FD00 ARABIC LIGATURE HAH WITH YEH ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD00}" => "\u{062D}\u{064A}",

    # FD01 ARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD01}" => "\u{062C}\u{0649}",

    # FD02 ARABIC LIGATURE JEEM WITH YEH ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD02}" => "\u{062C}\u{064A}",

    # FD03 ARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD03}" => "\u{062E}\u{0649}",

    # FD04 ARABIC LIGATURE KHAH WITH YEH ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD04}" => "\u{062E}\u{064A}",

    # FD05 ARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD05}" => "\u{0635}\u{0649}",

    # FD06 ARABIC LIGATURE SAD WITH YEH ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 064A ARABIC LETTER YEH
    "\u{FD06}" => "\u{0635}\u{064A}",

    # FD07 ARABIC LIGATURE DAD WITH ALEF MAKSURA ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD07}" => "\u{0636}\u{0649}",

    # FD08 ARABIC LIGATURE DAD WITH YEH ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 064A ARABIC LETTER YEH
    "\u{FD08}" => "\u{0636}\u{064A}",

    # FD09 ARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FD09}" => "\u{0634}\u{062C}",

    # FD0A ARABIC LIGATURE SHEEN WITH HAH ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FD0A}" => "\u{0634}\u{062D}",

    # FD0B ARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FD0B}" => "\u{0634}\u{062E}",

    # FD0C ARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD0C}" => "\u{0634}\u{0645}",

    # FD0D ARABIC LIGATURE SHEEN WITH REH ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0631 ARABIC LETTER REH
    "\u{FD0D}" => "\u{0634}\u{0631}",

    # FD0E ARABIC LIGATURE SEEN WITH REH ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0631 ARABIC LETTER REH
    "\u{FD0E}" => "\u{0633}\u{0631}",

    # FD0F ARABIC LIGATURE SAD WITH REH ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0631 ARABIC LETTER REH
    "\u{FD0F}" => "\u{0635}\u{0631}",

    # FD10 ARABIC LIGATURE DAD WITH REH ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0631 ARABIC LETTER REH
    "\u{FD10}" => "\u{0636}\u{0631}",

    # FD11 ARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD11}" => "\u{0637}\u{0649}",

    # FD12 ARABIC LIGATURE TAH WITH YEH FINAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD12}" => "\u{0637}\u{064A}",

    # FD13 ARABIC LIGATURE AIN WITH ALEF MAKSURA FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD13}" => "\u{0639}\u{0649}",

    # FD14 ARABIC LIGATURE AIN WITH YEH FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 064A ARABIC LETTER YEH
    "\u{FD14}" => "\u{0639}\u{064A}",

    # FD15 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD15}" => "\u{063A}\u{0649}",

    # FD16 ARABIC LIGATURE GHAIN WITH YEH FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 064A ARABIC LETTER YEH
    "\u{FD16}" => "\u{063A}\u{064A}",

    # FD17 ARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD17}" => "\u{0633}\u{0649}",

    # FD18 ARABIC LIGATURE SEEN WITH YEH FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 064A ARABIC LETTER YEH
    "\u{FD18}" => "\u{0633}\u{064A}",

    # FD19 ARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD19}" => "\u{0634}\u{0649}",

    # FD1A ARABIC LIGATURE SHEEN WITH YEH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 064A ARABIC LETTER YEH
    "\u{FD1A}" => "\u{0634}\u{064A}",

    # FD1B ARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD1B}" => "\u{062D}\u{0649}",

    # FD1C ARABIC LIGATURE HAH WITH YEH FINAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD1C}" => "\u{062D}\u{064A}",

    # FD1D ARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD1D}" => "\u{062C}\u{0649}",

    # FD1E ARABIC LIGATURE JEEM WITH YEH FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD1E}" => "\u{062C}\u{064A}",

    # FD1F ARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD1F}" => "\u{062E}\u{0649}",

    # FD20 ARABIC LIGATURE KHAH WITH YEH FINAL FORM
    # ->   062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD20}" => "\u{062E}\u{064A}",

    # FD21 ARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD21}" => "\u{0635}\u{0649}",

    # FD22 ARABIC LIGATURE SAD WITH YEH FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 064A ARABIC LETTER YEH
    "\u{FD22}" => "\u{0635}\u{064A}",

    # FD23 ARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD23}" => "\u{0636}\u{0649}",

    # FD24 ARABIC LIGATURE DAD WITH YEH FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 064A ARABIC LETTER YEH
    "\u{FD24}" => "\u{0636}\u{064A}",

    # FD25 ARABIC LIGATURE SHEEN WITH JEEM FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FD25}" => "\u{0634}\u{062C}",

    # FD26 ARABIC LIGATURE SHEEN WITH HAH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FD26}" => "\u{0634}\u{062D}",

    # FD27 ARABIC LIGATURE SHEEN WITH KHAH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FD27}" => "\u{0634}\u{062E}",

    # FD28 ARABIC LIGATURE SHEEN WITH MEEM FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD28}" => "\u{0634}\u{0645}",

    # FD29 ARABIC LIGATURE SHEEN WITH REH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0631 ARABIC LETTER REH
    "\u{FD29}" => "\u{0634}\u{0631}",

    # FD2A ARABIC LIGATURE SEEN WITH REH FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0631 ARABIC LETTER REH
    "\u{FD2A}" => "\u{0633}\u{0631}",

    # FD2B ARABIC LIGATURE SAD WITH REH FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0631 ARABIC LETTER REH
    "\u{FD2B}" => "\u{0635}\u{0631}",

    # FD2C ARABIC LIGATURE DAD WITH REH FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 0631 ARABIC LETTER REH
    "\u{FD2C}" => "\u{0636}\u{0631}",

    # FD2D ARABIC LIGATURE SHEEN WITH JEEM INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FD2D}" => "\u{0634}\u{062C}",

    # FD2E ARABIC LIGATURE SHEEN WITH HAH INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FD2E}" => "\u{0634}\u{062D}",

    # FD2F ARABIC LIGATURE SHEEN WITH KHAH INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FD2F}" => "\u{0634}\u{062E}",

    # FD30 ARABIC LIGATURE SHEEN WITH MEEM INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD30}" => "\u{0634}\u{0645}",

    # FD31 ARABIC LIGATURE SEEN WITH HEH INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0647 ARABIC LETTER HEH
    "\u{FD31}" => "\u{0633}\u{0647}",

    # FD32 ARABIC LIGATURE SHEEN WITH HEH INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0647 ARABIC LETTER HEH
    "\u{FD32}" => "\u{0634}\u{0647}",

    # FD33 ARABIC LIGATURE TAH WITH MEEM INITIAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD33}" => "\u{0637}\u{0645}",

    # FD34 ARABIC LIGATURE SEEN WITH JEEM MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FD34}" => "\u{0633}\u{062C}",

    # FD35 ARABIC LIGATURE SEEN WITH HAH MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FD35}" => "\u{0633}\u{062D}",

    # FD36 ARABIC LIGATURE SEEN WITH KHAH MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FD36}" => "\u{0633}\u{062E}",

    # FD37 ARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062C ARABIC LETTER JEEM
    "\u{FD37}" => "\u{0634}\u{062C}",

    # FD38 ARABIC LIGATURE SHEEN WITH HAH MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    "\u{FD38}" => "\u{0634}\u{062D}",

    # FD39 ARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062E ARABIC LETTER KHAH
    "\u{FD39}" => "\u{0634}\u{062E}",

    # FD3A ARABIC LIGATURE TAH WITH MEEM MEDIAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD3A}" => "\u{0637}\u{0645}",

    # FD3B ARABIC LIGATURE ZAH WITH MEEM MEDIAL FORM
    # ->   0638 ARABIC LETTER ZAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD3B}" => "\u{0638}\u{0645}",

    # FD3C ARABIC LIGATURE ALEF WITH FATHATAN FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FD3C}" => "\u{0627}",

    # FD3D ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FD3D}" => "\u{0627}",

    # FD50 ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD50}" => "\u{062A}\u{062C}\u{0645}",

    # FD51 ARABIC LIGATURE TEH WITH HAH WITH JEEM FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FD51}" => "\u{062A}\u{062D}\u{062C}",

    # FD52 ARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FD52}" => "\u{062A}\u{062D}\u{062C}",

    # FD53 ARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD53}" => "\u{062A}\u{062D}\u{0645}",

    # FD54 ARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD54}" => "\u{062A}\u{062E}\u{0645}",

    # FD55 ARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FD55}" => "\u{062A}\u{0645}\u{062C}",

    # FD56 ARABIC LIGATURE TEH WITH MEEM WITH HAH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD56}" => "\u{062A}\u{0645}\u{062D}",

    # FD57 ARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FD57}" => "\u{062A}\u{0645}\u{062E}",

    # FD58 ARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD58}" => "\u{062C}\u{0645}\u{062D}",

    # FD59 ARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD59}" => "\u{062C}\u{0645}\u{062D}",

    # FD5A ARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD5A}" => "\u{062D}\u{0645}\u{064A}",

    # FD5B ARABIC LIGATURE HAH WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD5B}" => "\u{062D}\u{0645}\u{0649}",

    # FD5C ARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FD5C}" => "\u{0633}\u{062D}\u{062C}",

    # FD5D ARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD5D}" => "\u{0633}\u{062C}\u{062D}",

    # FD5E ARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062C ARABIC LETTER JEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD5E}" => "\u{0633}\u{062C}\u{0649}",

    # FD5F ARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD5F}" => "\u{0633}\u{0645}\u{062D}",

    # FD60 ARABIC LIGATURE SEEN WITH MEEM WITH HAH INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD60}" => "\u{0633}\u{0645}\u{062D}",

    # FD61 ARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FD61}" => "\u{0633}\u{0645}\u{062C}",

    # FD62 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD62}" => "\u{0633}\u{0645}\u{0645}",

    # FD63 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD63}" => "\u{0633}\u{0645}\u{0645}",

    # FD64 ARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062D ARABIC LETTER HAH
    #    + 062D ARABIC LETTER HAH
    "\u{FD64}" => "\u{0635}\u{062D}\u{062D}",

    # FD65 ARABIC LIGATURE SAD WITH HAH WITH HAH INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062D ARABIC LETTER HAH
    #    + 062D ARABIC LETTER HAH
    "\u{FD65}" => "\u{0635}\u{062D}\u{062D}",

    # FD66 ARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD66}" => "\u{0635}\u{0645}\u{0645}",

    # FD67 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD67}" => "\u{0634}\u{062D}\u{0645}",

    # FD68 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD68}" => "\u{0634}\u{062D}\u{0645}",

    # FD69 ARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD69}" => "\u{0634}\u{062C}\u{064A}",

    # FD6A ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FD6A}" => "\u{0634}\u{0645}\u{062E}",

    # FD6B ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FD6B}" => "\u{0634}\u{0645}\u{062E}",

    # FD6C ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD6C}" => "\u{0634}\u{0645}\u{0645}",

    # FD6D ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD6D}" => "\u{0634}\u{0645}\u{0645}",

    # FD6E ARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD6E}" => "\u{0636}\u{062D}\u{0649}",

    # FD6F ARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD6F}" => "\u{0636}\u{062E}\u{0645}",

    # FD70 ARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD70}" => "\u{0636}\u{062E}\u{0645}",

    # FD71 ARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD71}" => "\u{0637}\u{0645}\u{062D}",

    # FD72 ARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD72}" => "\u{0637}\u{0645}\u{062D}",

    # FD73 ARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD73}" => "\u{0637}\u{0645}\u{0645}",

    # FD74 ARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORM
    # ->   0637 ARABIC LETTER TAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD74}" => "\u{0637}\u{0645}\u{064A}",

    # FD75 ARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD75}" => "\u{0639}\u{062C}\u{0645}",

    # FD76 ARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD76}" => "\u{0639}\u{0645}\u{0645}",

    # FD77 ARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD77}" => "\u{0639}\u{0645}\u{0645}",

    # FD78 ARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD78}" => "\u{0639}\u{0645}\u{0649}",

    # FD79 ARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD79}" => "\u{063A}\u{0645}\u{0645}",

    # FD7A ARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD7A}" => "\u{063A}\u{0645}\u{064A}",

    # FD7B ARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD7B}" => "\u{063A}\u{0645}\u{0649}",

    # FD7C ARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD7C}" => "\u{0641}\u{062E}\u{0645}",

    # FD7D ARABIC LIGATURE FEH WITH KHAH WITH MEEM INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD7D}" => "\u{0641}\u{062E}\u{0645}",

    # FD7E ARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD7E}" => "\u{0642}\u{0645}\u{062D}",

    # FD7F ARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD7F}" => "\u{0642}\u{0645}\u{0645}",

    # FD80 ARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD80}" => "\u{0644}\u{062D}\u{0645}",

    # FD81 ARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD81}" => "\u{0644}\u{062D}\u{064A}",

    # FD82 ARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSURA FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD82}" => "\u{0644}\u{062D}\u{0649}",

    # FD83 ARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FD83}" => "\u{0644}\u{062C}\u{062C}",

    # FD84 ARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FD84}" => "\u{0644}\u{062C}\u{062C}",

    # FD85 ARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD85}" => "\u{0644}\u{062E}\u{0645}",

    # FD86 ARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD86}" => "\u{0644}\u{062E}\u{0645}",

    # FD87 ARABIC LIGATURE LAM WITH MEEM WITH HAH FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD87}" => "\u{0644}\u{0645}\u{062D}",

    # FD88 ARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD88}" => "\u{0644}\u{0645}\u{062D}",

    # FD89 ARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FD89}" => "\u{0645}\u{062D}\u{062C}",

    # FD8A ARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD8A}" => "\u{0645}\u{062D}\u{0645}",

    # FD8B ARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD8B}" => "\u{0645}\u{062D}\u{064A}",

    # FD8C ARABIC LIGATURE MEEM WITH JEEM WITH HAH INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FD8C}" => "\u{0645}\u{062C}\u{062D}",

    # FD8D ARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD8D}" => "\u{0645}\u{062C}\u{0645}",

    # FD8E ARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    #    + 062C ARABIC LETTER JEEM
    "\u{FD8E}" => "\u{0645}\u{062E}\u{062C}",

    # FD8F ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD8F}" => "\u{0645}\u{062E}\u{0645}",

    # FD92 ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    #    + 062E ARABIC LETTER KHAH
    "\u{FD92}" => "\u{0645}\u{062C}\u{062E}",

    # FD93 ARABIC LIGATURE HEH WITH MEEM WITH JEEM INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    "\u{FD93}" => "\u{0647}\u{0645}\u{062C}",

    # FD94 ARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD94}" => "\u{0647}\u{0645}\u{0645}",

    # FD95 ARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD95}" => "\u{0646}\u{062D}\u{0645}",

    # FD96 ARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD96}" => "\u{0646}\u{062D}\u{0649}",

    # FD97 ARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD97}" => "\u{0646}\u{062C}\u{0645}",

    # FD98 ARABIC LIGATURE NOON WITH JEEM WITH MEEM INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD98}" => "\u{0646}\u{062C}\u{0645}",

    # FD99 ARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD99}" => "\u{0646}\u{062C}\u{0649}",

    # FD9A ARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD9A}" => "\u{0646}\u{0645}\u{064A}",

    # FD9B ARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FD9B}" => "\u{0646}\u{0645}\u{0649}",

    # FD9C ARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD9C}" => "\u{064A}\u{0645}\u{0645}",

    # FD9D ARABIC LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FD9D}" => "\u{064A}\u{0645}\u{0645}",

    # FD9E ARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FD9E}" => "\u{0628}\u{062E}\u{064A}",

    # FD9F ARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FD9F}" => "\u{062A}\u{062C}\u{064A}",

    # FDA0 ARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062C ARABIC LETTER JEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA0}" => "\u{062A}\u{062C}\u{0649}",

    # FDA1 ARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDA1}" => "\u{062A}\u{062E}\u{064A}",

    # FDA2 ARABIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 062E ARABIC LETTER KHAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA2}" => "\u{062A}\u{062E}\u{0649}",

    # FDA3 ARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDA3}" => "\u{062A}\u{0645}\u{064A}",

    # FDA4 ARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   062A ARABIC LETTER TEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA4}" => "\u{062A}\u{0645}\u{0649}",

    # FDA5 ARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDA5}" => "\u{062C}\u{0645}\u{064A}",

    # FDA6 ARABIC LIGATURE JEEM WITH HAH WITH ALEF MAKSURA FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA6}" => "\u{062C}\u{062D}\u{0649}",

    # FDA7 ARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA7}" => "\u{062C}\u{0645}\u{0649}",

    # FDA8 ARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062E ARABIC LETTER KHAH
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDA8}" => "\u{0633}\u{062E}\u{0649}",

    # FDA9 ARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDA9}" => "\u{0635}\u{062D}\u{064A}",

    # FDAA ARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDAA}" => "\u{0634}\u{062D}\u{064A}",

    # FDAB ARABIC LIGATURE DAD WITH HAH WITH YEH FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDAB}" => "\u{0636}\u{062D}\u{064A}",

    # FDAC ARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDAC}" => "\u{0644}\u{062C}\u{064A}",

    # FDAD ARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDAD}" => "\u{0644}\u{0645}\u{064A}",

    # FDAE ARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDAE}" => "\u{064A}\u{062D}\u{064A}",

    # FDAF ARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDAF}" => "\u{064A}\u{062C}\u{064A}",

    # FDB0 ARABIC LIGATURE YEH WITH MEEM WITH YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDB0}" => "\u{064A}\u{0645}\u{064A}",

    # FDB1 ARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDB1}" => "\u{0645}\u{0645}\u{064A}",

    # FDB2 ARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDB2}" => "\u{0642}\u{0645}\u{064A}",

    # FDB3 ARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDB3}" => "\u{0646}\u{062D}\u{064A}",

    # FDB4 ARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FDB4}" => "\u{0642}\u{0645}\u{062D}",

    # FDB5 ARABIC LIGATURE LAM WITH HAH WITH MEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDB5}" => "\u{0644}\u{062D}\u{0645}",

    # FDB6 ARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDB6}" => "\u{0639}\u{0645}\u{064A}",

    # FDB7 ARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDB7}" => "\u{0643}\u{0645}\u{064A}",

    # FDB8 ARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FDB8}" => "\u{0646}\u{062C}\u{062D}",

    # FDB9 ARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDB9}" => "\u{0645}\u{062E}\u{064A}",

    # FDBA ARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDBA}" => "\u{0644}\u{062C}\u{0645}",

    # FDBB ARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDBB}" => "\u{0643}\u{0645}\u{0645}",

    # FDBC ARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDBC}" => "\u{0644}\u{062C}\u{0645}",

    # FDBD ARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    "\u{FDBD}" => "\u{0646}\u{062C}\u{062D}",

    # FDBE ARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDBE}" => "\u{062C}\u{062D}\u{064A}",

    # FDBF ARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORM
    # ->   062D ARABIC LETTER HAH
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDBF}" => "\u{062D}\u{062C}\u{064A}",

    # FDC0 ARABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDC0}" => "\u{0645}\u{062C}\u{064A}",

    # FDC1 ARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORM
    # ->   0641 ARABIC LETTER FEH
    #    + 0645 ARABIC LETTER MEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDC1}" => "\u{0641}\u{0645}\u{064A}",

    # FDC2 ARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    #    + 062D ARABIC LETTER HAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDC2}" => "\u{0628}\u{062D}\u{064A}",

    # FDC3 ARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDC3}" => "\u{0643}\u{0645}\u{0645}",

    # FDC4 ARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 062C ARABIC LETTER JEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDC4}" => "\u{0639}\u{062C}\u{0645}",

    # FDC5 ARABIC LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0645 ARABIC LETTER MEEM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDC5}" => "\u{0635}\u{0645}\u{0645}",

    # FDC6 ARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    #    + 062E ARABIC LETTER KHAH
    #    + 064A ARABIC LETTER YEH
    "\u{FDC6}" => "\u{0633}\u{062E}\u{064A}",

    # FDC7 ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    #    + 062C ARABIC LETTER JEEM
    #    + 064A ARABIC LETTER YEH
    "\u{FDC7}" => "\u{0646}\u{062C}\u{064A}",

    # FDF0 ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0644 ARABIC LETTER LAM
    #    + 06D2 ARABIC LETTER YEH BARREE
    "\u{FDF0}" => "\u{0635}\u{0644}\u{06D2}",

    # FDF1 ARABIC LIGATURE QALA USED AS KORANIC STOP SIGN ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    #    + 0644 ARABIC LETTER LAM
    #    + 06D2 ARABIC LETTER YEH BARREE
    "\u{FDF1}" => "\u{0642}\u{0644}\u{06D2}",

    # FDF2 ARABIC LIGATURE ALLAH ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    #    + 0644 ARABIC LETTER LAM
    #    + 0644 ARABIC LETTER LAM
    #    + 0647 ARABIC LETTER HEH
    "\u{FDF2}" => "\u{0627}\u{0644}\u{0644}\u{0647}",

    # FDF3 ARABIC LIGATURE AKBAR ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    #    + 0643 ARABIC LETTER KAF
    #    + 0628 ARABIC LETTER BEH
    #    + 0631 ARABIC LETTER REH
    "\u{FDF3}" => "\u{0627}\u{0643}\u{0628}\u{0631}",

    # FDF4 ARABIC LIGATURE MOHAMMAD ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    #    + 062D ARABIC LETTER HAH
    #    + 0645 ARABIC LETTER MEEM
    #    + 062F ARABIC LETTER DAL
    "\u{FDF4}" => "\u{0645}\u{062D}\u{0645}\u{062F}",

    # FDF5 ARABIC LIGATURE SALAM ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0644 ARABIC LETTER LAM
    #    + 0639 ARABIC LETTER AIN
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDF5}" => "\u{0635}\u{0644}\u{0639}\u{0645}",

    # FDF6 ARABIC LIGATURE RASOUL ISOLATED FORM
    # ->   0631 ARABIC LETTER REH
    #    + 0633 ARABIC LETTER SEEN
    #    + 0648 ARABIC LETTER WAW
    #    + 0644 ARABIC LETTER LAM
    "\u{FDF6}" => "\u{0631}\u{0633}\u{0648}\u{0644}",

    # FDF7 ARABIC LIGATURE ALAYHE ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    #    + 0644 ARABIC LETTER LAM
    #    + 064A ARABIC LETTER YEH
    #    + 0647 ARABIC LETTER HEH
    "\u{FDF7}" => "\u{0639}\u{0644}\u{064A}\u{0647}",

    # FDF8 ARABIC LIGATURE WASALLAM ISOLATED FORM
    # ->   0648 ARABIC LETTER WAW
    #    + 0633 ARABIC LETTER SEEN
    #    + 0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDF8}" => "\u{0648}\u{0633}\u{0644}\u{0645}",

    # FDF9 ARABIC LIGATURE SALLA ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    #    + 0644 ARABIC LETTER LAM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    "\u{FDF9}" => "\u{0635}\u{0644}\u{0649}",

    # FDFA ARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAM
    # ->   0635 ARABIC LETTER SAD
    #    + 0644 ARABIC LETTER LAM
    #    + 0649 ARABIC LETTER ALEF MAKSURA
    #    + 0020 SPACE
    #    + 0627 ARABIC LETTER ALEF
    #    + 0644 ARABIC LETTER LAM
    #    + 0644 ARABIC LETTER LAM
    #    + 0647 ARABIC LETTER HEH
    #    + 0020 SPACE
    #    + 0639 ARABIC LETTER AIN
    #    + 0644 ARABIC LETTER LAM
    #    + 064A ARABIC LETTER YEH
    #    + 0647 ARABIC LETTER HEH
    #    + 0020 SPACE
    #    + 0648 ARABIC LETTER WAW
    #    + 0633 ARABIC LETTER SEEN
    #    + 0644 ARABIC LETTER LAM
    #    + 0645 ARABIC LETTER MEEM
    "\u{FDFA}" => "\u{0635}\u{0644}\u{0649} \u{0627}\u{0644}\u{0644}\u{0647} \u{0639}\u{0644}\u{064A}\u{0647} \u{0648}\u{0633}\u{0644}\u{0645}",

    # FDFB ARABIC LIGATURE JALLAJALALOUHOU
    # ->   062C ARABIC LETTER JEEM
    #    + 0644 ARABIC LETTER LAM
    #    + 0020 SPACE
    #    + 062C ARABIC LETTER JEEM
    #    + 0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    #    + 0644 ARABIC LETTER LAM
    #    + 0647 ARABIC LETTER HEH
    "\u{FDFB}" => "\u{062C}\u{0644} \u{062C}\u{0644}\u{0627}\u{0644}\u{0647}",

    # FDFC RIAL SIGN
    # ->   0631 ARABIC LETTER REH
    #    + 06CC ARABIC LETTER FARSI YEH
    #    + 0627 ARABIC LETTER ALEF
    #    + 0644 ARABIC LETTER LAM
    "\u{FDFC}" => "\u{0631}\u{06CC}\u{0627}\u{0644}",

    # FE30 PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
    # ->   002E FULL STOP
    #    + 002E FULL STOP
    "\u{FE30}" => "..",

    # FE31 PRESENTATION FORM FOR VERTICAL EM DASH
    # ->   2014 EM DASH
    "\u{FE31}" => "\u{2014}",

    # FE32 PRESENTATION FORM FOR VERTICAL EN DASH
    # ->   2013 EN DASH
    "\u{FE32}" => "\u{2013}",

    # FE33 PRESENTATION FORM FOR VERTICAL LOW LINE
    # ->   005F LOW LINE
    "\u{FE33}" => "_",

    # FE34 PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
    # ->   005F LOW LINE
    "\u{FE34}" => "_",

    # FE35 PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
    # ->   0028 LEFT PARENTHESIS
    "\u{FE35}" => "(",

    # FE36 PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
    # ->   0029 RIGHT PARENTHESIS
    "\u{FE36}" => ")",

    # FE37 PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
    # ->   007B LEFT CURLY BRACKET
    "\u{FE37}" => "{",

    # FE38 PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
    # ->   007D RIGHT CURLY BRACKET
    "\u{FE38}" => "}",

    # FE39 PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
    # ->   3014 LEFT TORTOISE SHELL BRACKET
    "\u{FE39}" => "\u{3014}",

    # FE3A PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
    # ->   3015 RIGHT TORTOISE SHELL BRACKET
    "\u{FE3A}" => "\u{3015}",

    # FE3B PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
    # ->   3010 LEFT BLACK LENTICULAR BRACKET
    "\u{FE3B}" => "\u{3010}",

    # FE3C PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
    # ->   3011 RIGHT BLACK LENTICULAR BRACKET
    "\u{FE3C}" => "\u{3011}",

    # FE3D PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
    # ->   300A LEFT DOUBLE ANGLE BRACKET
    "\u{FE3D}" => "\u{300A}",

    # FE3E PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
    # ->   300B RIGHT DOUBLE ANGLE BRACKET
    "\u{FE3E}" => "\u{300B}",

    # FE3F PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
    # ->   3008 LEFT ANGLE BRACKET
    "\u{FE3F}" => "\u{3008}",

    # FE40 PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
    # ->   3009 RIGHT ANGLE BRACKET
    "\u{FE40}" => "\u{3009}",

    # FE41 PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
    # ->   300C LEFT CORNER BRACKET
    "\u{FE41}" => "\u{300C}",

    # FE42 PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
    # ->   300D RIGHT CORNER BRACKET
    "\u{FE42}" => "\u{300D}",

    # FE43 PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
    # ->   300E LEFT WHITE CORNER BRACKET
    "\u{FE43}" => "\u{300E}",

    # FE44 PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
    # ->   300F RIGHT WHITE CORNER BRACKET
    "\u{FE44}" => "\u{300F}",

    # FE47 PRESENTATION FORM FOR VERTICAL LEFT SQUARE BRACKET
    # ->   005B LEFT SQUARE BRACKET
    "\u{FE47}" => "[",

    # FE48 PRESENTATION FORM FOR VERTICAL RIGHT SQUARE BRACKET
    # ->   005D RIGHT SQUARE BRACKET
    "\u{FE48}" => "]",

    # FE49 DASHED OVERLINE
    # ->   0020 SPACE
    "\u{FE49}" => " ",

    # FE4A CENTRELINE OVERLINE
    # ->   0020 SPACE
    "\u{FE4A}" => " ",

    # FE4B WAVY OVERLINE
    # ->   0020 SPACE
    "\u{FE4B}" => " ",

    # FE4C DOUBLE WAVY OVERLINE
    # ->   0020 SPACE
    "\u{FE4C}" => " ",

    # FE4D DASHED LOW LINE
    # ->   005F LOW LINE
    "\u{FE4D}" => "_",

    # FE4E CENTRELINE LOW LINE
    # ->   005F LOW LINE
    "\u{FE4E}" => "_",

    # FE4F WAVY LOW LINE
    # ->   005F LOW LINE
    "\u{FE4F}" => "_",

    # FE50 SMALL COMMA
    # ->   002C COMMA
    "\u{FE50}" => ",",

    # FE51 SMALL IDEOGRAPHIC COMMA
    # ->   3001 IDEOGRAPHIC COMMA
    "\u{FE51}" => "\u{3001}",

    # FE52 SMALL FULL STOP
    # ->   002E FULL STOP
    "\u{FE52}" => ".",

    # FE54 SMALL SEMICOLON
    # ->   003B SEMICOLON
    "\u{FE54}" => ";",

    # FE55 SMALL COLON
    # ->   003A COLON
    "\u{FE55}" => ":",

    # FE56 SMALL QUESTION MARK
    # ->   003F QUESTION MARK
    "\u{FE56}" => "?",

    # FE57 SMALL EXCLAMATION MARK
    # ->   0021 EXCLAMATION MARK
    "\u{FE57}" => "!",

    # FE58 SMALL EM DASH
    # ->   2014 EM DASH
    "\u{FE58}" => "\u{2014}",

    # FE59 SMALL LEFT PARENTHESIS
    # ->   0028 LEFT PARENTHESIS
    "\u{FE59}" => "(",

    # FE5A SMALL RIGHT PARENTHESIS
    # ->   0029 RIGHT PARENTHESIS
    "\u{FE5A}" => ")",

    # FE5B SMALL LEFT CURLY BRACKET
    # ->   007B LEFT CURLY BRACKET
    "\u{FE5B}" => "{",

    # FE5C SMALL RIGHT CURLY BRACKET
    # ->   007D RIGHT CURLY BRACKET
    "\u{FE5C}" => "}",

    # FE5D SMALL LEFT TORTOISE SHELL BRACKET
    # ->   3014 LEFT TORTOISE SHELL BRACKET
    "\u{FE5D}" => "\u{3014}",

    # FE5E SMALL RIGHT TORTOISE SHELL BRACKET
    # ->   3015 RIGHT TORTOISE SHELL BRACKET
    "\u{FE5E}" => "\u{3015}",

    # FE5F SMALL NUMBER SIGN
    # ->   0023 NUMBER SIGN
    "\u{FE5F}" => "#",

    # FE60 SMALL AMPERSAND
    # ->   0026 AMPERSAND
    "\u{FE60}" => "&",

    # FE61 SMALL ASTERISK
    # ->   002A ASTERISK
    "\u{FE61}" => "*",

    # FE62 SMALL PLUS SIGN
    # ->   002B PLUS SIGN
    "\u{FE62}" => "+",

    # FE63 SMALL HYPHEN-MINUS
    # ->   002D HYPHEN-MINUS
    "\u{FE63}" => "-",

    # FE64 SMALL LESS-THAN SIGN
    # ->   003C LESS-THAN SIGN
    "\u{FE64}" => "<",

    # FE65 SMALL GREATER-THAN SIGN
    # ->   003E GREATER-THAN SIGN
    "\u{FE65}" => ">",

    # FE66 SMALL EQUALS SIGN
    # ->   003D EQUALS SIGN
    "\u{FE66}" => "=",

    # FE68 SMALL REVERSE SOLIDUS
    # ->   005C REVERSE SOLIDUS
    "\u{FE68}" => "\\",

    # FE69 SMALL DOLLAR SIGN
    # ->   0024 DOLLAR SIGN
    "\u{FE69}" => "\$",

    # FE6A SMALL PERCENT SIGN
    # ->   0025 PERCENT SIGN
    "\u{FE6A}" => "%",

    # FE6B SMALL COMMERCIAL AT
    # ->   0040 COMMERCIAL AT
    "\u{FE6B}" => "@",

    # FE70 ARABIC FATHATAN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE70}" => " ",

    # FE71 ARABIC TATWEEL WITH FATHATAN ABOVE
    # ->   0640 ARABIC TATWEEL
    "\u{FE71}" => "\u{0640}",

    # FE72 ARABIC DAMMATAN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE72}" => " ",

    # FE74 ARABIC KASRATAN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE74}" => " ",

    # FE76 ARABIC FATHA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE76}" => " ",

    # FE77 ARABIC FATHA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FE77}" => "\u{0640}",

    # FE78 ARABIC DAMMA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE78}" => " ",

    # FE79 ARABIC DAMMA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FE79}" => "\u{0640}",

    # FE7A ARABIC KASRA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE7A}" => " ",

    # FE7B ARABIC KASRA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FE7B}" => "\u{0640}",

    # FE7C ARABIC SHADDA ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE7C}" => " ",

    # FE7D ARABIC SHADDA MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FE7D}" => "\u{0640}",

    # FE7E ARABIC SUKUN ISOLATED FORM
    # ->   0020 SPACE
    "\u{FE7E}" => " ",

    # FE7F ARABIC SUKUN MEDIAL FORM
    # ->   0640 ARABIC TATWEEL
    "\u{FE7F}" => "\u{0640}",

    # FE80 ARABIC LETTER HAMZA ISOLATED FORM
    # ->   0621 ARABIC LETTER HAMZA
    "\u{FE80}" => "\u{0621}",

    # FE81 ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE81}" => "\u{0627}",

    # FE82 ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE82}" => "\u{0627}",

    # FE83 ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE83}" => "\u{0627}",

    # FE84 ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE84}" => "\u{0627}",

    # FE85 ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM
    # ->   0648 ARABIC LETTER WAW
    "\u{FE85}" => "\u{0648}",

    # FE86 ARABIC LETTER WAW WITH HAMZA ABOVE FINAL FORM
    # ->   0648 ARABIC LETTER WAW
    "\u{FE86}" => "\u{0648}",

    # FE87 ARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE87}" => "\u{0627}",

    # FE88 ARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE88}" => "\u{0627}",

    # FE89 ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FE89}" => "\u{064A}",

    # FE8A ARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FE8A}" => "\u{064A}",

    # FE8B ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FE8B}" => "\u{064A}",

    # FE8C ARABIC LETTER YEH WITH HAMZA ABOVE MEDIAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FE8C}" => "\u{064A}",

    # FE8D ARABIC LETTER ALEF ISOLATED FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE8D}" => "\u{0627}",

    # FE8E ARABIC LETTER ALEF FINAL FORM
    # ->   0627 ARABIC LETTER ALEF
    "\u{FE8E}" => "\u{0627}",

    # FE8F ARABIC LETTER BEH ISOLATED FORM
    # ->   0628 ARABIC LETTER BEH
    "\u{FE8F}" => "\u{0628}",

    # FE90 ARABIC LETTER BEH FINAL FORM
    # ->   0628 ARABIC LETTER BEH
    "\u{FE90}" => "\u{0628}",

    # FE91 ARABIC LETTER BEH INITIAL FORM
    # ->   0628 ARABIC LETTER BEH
    "\u{FE91}" => "\u{0628}",

    # FE92 ARABIC LETTER BEH MEDIAL FORM
    # ->   0628 ARABIC LETTER BEH
    "\u{FE92}" => "\u{0628}",

    # FE93 ARABIC LETTER TEH MARBUTA ISOLATED FORM
    # ->   0629 ARABIC LETTER TEH MARBUTA
    "\u{FE93}" => "\u{0629}",

    # FE94 ARABIC LETTER TEH MARBUTA FINAL FORM
    # ->   0629 ARABIC LETTER TEH MARBUTA
    "\u{FE94}" => "\u{0629}",

    # FE95 ARABIC LETTER TEH ISOLATED FORM
    # ->   062A ARABIC LETTER TEH
    "\u{FE95}" => "\u{062A}",

    # FE96 ARABIC LETTER TEH FINAL FORM
    # ->   062A ARABIC LETTER TEH
    "\u{FE96}" => "\u{062A}",

    # FE97 ARABIC LETTER TEH INITIAL FORM
    # ->   062A ARABIC LETTER TEH
    "\u{FE97}" => "\u{062A}",

    # FE98 ARABIC LETTER TEH MEDIAL FORM
    # ->   062A ARABIC LETTER TEH
    "\u{FE98}" => "\u{062A}",

    # FE99 ARABIC LETTER THEH ISOLATED FORM
    # ->   062B ARABIC LETTER THEH
    "\u{FE99}" => "\u{062B}",

    # FE9A ARABIC LETTER THEH FINAL FORM
    # ->   062B ARABIC LETTER THEH
    "\u{FE9A}" => "\u{062B}",

    # FE9B ARABIC LETTER THEH INITIAL FORM
    # ->   062B ARABIC LETTER THEH
    "\u{FE9B}" => "\u{062B}",

    # FE9C ARABIC LETTER THEH MEDIAL FORM
    # ->   062B ARABIC LETTER THEH
    "\u{FE9C}" => "\u{062B}",

    # FE9D ARABIC LETTER JEEM ISOLATED FORM
    # ->   062C ARABIC LETTER JEEM
    "\u{FE9D}" => "\u{062C}",

    # FE9E ARABIC LETTER JEEM FINAL FORM
    # ->   062C ARABIC LETTER JEEM
    "\u{FE9E}" => "\u{062C}",

    # FE9F ARABIC LETTER JEEM INITIAL FORM
    # ->   062C ARABIC LETTER JEEM
    "\u{FE9F}" => "\u{062C}",

    # FEA0 ARABIC LETTER JEEM MEDIAL FORM
    # ->   062C ARABIC LETTER JEEM
    "\u{FEA0}" => "\u{062C}",

    # FEA1 ARABIC LETTER HAH ISOLATED FORM
    # ->   062D ARABIC LETTER HAH
    "\u{FEA1}" => "\u{062D}",

    # FEA2 ARABIC LETTER HAH FINAL FORM
    # ->   062D ARABIC LETTER HAH
    "\u{FEA2}" => "\u{062D}",

    # FEA3 ARABIC LETTER HAH INITIAL FORM
    # ->   062D ARABIC LETTER HAH
    "\u{FEA3}" => "\u{062D}",

    # FEA4 ARABIC LETTER HAH MEDIAL FORM
    # ->   062D ARABIC LETTER HAH
    "\u{FEA4}" => "\u{062D}",

    # FEA5 ARABIC LETTER KHAH ISOLATED FORM
    # ->   062E ARABIC LETTER KHAH
    "\u{FEA5}" => "\u{062E}",

    # FEA6 ARABIC LETTER KHAH FINAL FORM
    # ->   062E ARABIC LETTER KHAH
    "\u{FEA6}" => "\u{062E}",

    # FEA7 ARABIC LETTER KHAH INITIAL FORM
    # ->   062E ARABIC LETTER KHAH
    "\u{FEA7}" => "\u{062E}",

    # FEA8 ARABIC LETTER KHAH MEDIAL FORM
    # ->   062E ARABIC LETTER KHAH
    "\u{FEA8}" => "\u{062E}",

    # FEA9 ARABIC LETTER DAL ISOLATED FORM
    # ->   062F ARABIC LETTER DAL
    "\u{FEA9}" => "\u{062F}",

    # FEAA ARABIC LETTER DAL FINAL FORM
    # ->   062F ARABIC LETTER DAL
    "\u{FEAA}" => "\u{062F}",

    # FEAB ARABIC LETTER THAL ISOLATED FORM
    # ->   0630 ARABIC LETTER THAL
    "\u{FEAB}" => "\u{0630}",

    # FEAC ARABIC LETTER THAL FINAL FORM
    # ->   0630 ARABIC LETTER THAL
    "\u{FEAC}" => "\u{0630}",

    # FEAD ARABIC LETTER REH ISOLATED FORM
    # ->   0631 ARABIC LETTER REH
    "\u{FEAD}" => "\u{0631}",

    # FEAE ARABIC LETTER REH FINAL FORM
    # ->   0631 ARABIC LETTER REH
    "\u{FEAE}" => "\u{0631}",

    # FEAF ARABIC LETTER ZAIN ISOLATED FORM
    # ->   0632 ARABIC LETTER ZAIN
    "\u{FEAF}" => "\u{0632}",

    # FEB0 ARABIC LETTER ZAIN FINAL FORM
    # ->   0632 ARABIC LETTER ZAIN
    "\u{FEB0}" => "\u{0632}",

    # FEB1 ARABIC LETTER SEEN ISOLATED FORM
    # ->   0633 ARABIC LETTER SEEN
    "\u{FEB1}" => "\u{0633}",

    # FEB2 ARABIC LETTER SEEN FINAL FORM
    # ->   0633 ARABIC LETTER SEEN
    "\u{FEB2}" => "\u{0633}",

    # FEB3 ARABIC LETTER SEEN INITIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    "\u{FEB3}" => "\u{0633}",

    # FEB4 ARABIC LETTER SEEN MEDIAL FORM
    # ->   0633 ARABIC LETTER SEEN
    "\u{FEB4}" => "\u{0633}",

    # FEB5 ARABIC LETTER SHEEN ISOLATED FORM
    # ->   0634 ARABIC LETTER SHEEN
    "\u{FEB5}" => "\u{0634}",

    # FEB6 ARABIC LETTER SHEEN FINAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    "\u{FEB6}" => "\u{0634}",

    # FEB7 ARABIC LETTER SHEEN INITIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    "\u{FEB7}" => "\u{0634}",

    # FEB8 ARABIC LETTER SHEEN MEDIAL FORM
    # ->   0634 ARABIC LETTER SHEEN
    "\u{FEB8}" => "\u{0634}",

    # FEB9 ARABIC LETTER SAD ISOLATED FORM
    # ->   0635 ARABIC LETTER SAD
    "\u{FEB9}" => "\u{0635}",

    # FEBA ARABIC LETTER SAD FINAL FORM
    # ->   0635 ARABIC LETTER SAD
    "\u{FEBA}" => "\u{0635}",

    # FEBB ARABIC LETTER SAD INITIAL FORM
    # ->   0635 ARABIC LETTER SAD
    "\u{FEBB}" => "\u{0635}",

    # FEBC ARABIC LETTER SAD MEDIAL FORM
    # ->   0635 ARABIC LETTER SAD
    "\u{FEBC}" => "\u{0635}",

    # FEBD ARABIC LETTER DAD ISOLATED FORM
    # ->   0636 ARABIC LETTER DAD
    "\u{FEBD}" => "\u{0636}",

    # FEBE ARABIC LETTER DAD FINAL FORM
    # ->   0636 ARABIC LETTER DAD
    "\u{FEBE}" => "\u{0636}",

    # FEBF ARABIC LETTER DAD INITIAL FORM
    # ->   0636 ARABIC LETTER DAD
    "\u{FEBF}" => "\u{0636}",

    # FEC0 ARABIC LETTER DAD MEDIAL FORM
    # ->   0636 ARABIC LETTER DAD
    "\u{FEC0}" => "\u{0636}",

    # FEC1 ARABIC LETTER TAH ISOLATED FORM
    # ->   0637 ARABIC LETTER TAH
    "\u{FEC1}" => "\u{0637}",

    # FEC2 ARABIC LETTER TAH FINAL FORM
    # ->   0637 ARABIC LETTER TAH
    "\u{FEC2}" => "\u{0637}",

    # FEC3 ARABIC LETTER TAH INITIAL FORM
    # ->   0637 ARABIC LETTER TAH
    "\u{FEC3}" => "\u{0637}",

    # FEC4 ARABIC LETTER TAH MEDIAL FORM
    # ->   0637 ARABIC LETTER TAH
    "\u{FEC4}" => "\u{0637}",

    # FEC5 ARABIC LETTER ZAH ISOLATED FORM
    # ->   0638 ARABIC LETTER ZAH
    "\u{FEC5}" => "\u{0638}",

    # FEC6 ARABIC LETTER ZAH FINAL FORM
    # ->   0638 ARABIC LETTER ZAH
    "\u{FEC6}" => "\u{0638}",

    # FEC7 ARABIC LETTER ZAH INITIAL FORM
    # ->   0638 ARABIC LETTER ZAH
    "\u{FEC7}" => "\u{0638}",

    # FEC8 ARABIC LETTER ZAH MEDIAL FORM
    # ->   0638 ARABIC LETTER ZAH
    "\u{FEC8}" => "\u{0638}",

    # FEC9 ARABIC LETTER AIN ISOLATED FORM
    # ->   0639 ARABIC LETTER AIN
    "\u{FEC9}" => "\u{0639}",

    # FECA ARABIC LETTER AIN FINAL FORM
    # ->   0639 ARABIC LETTER AIN
    "\u{FECA}" => "\u{0639}",

    # FECB ARABIC LETTER AIN INITIAL FORM
    # ->   0639 ARABIC LETTER AIN
    "\u{FECB}" => "\u{0639}",

    # FECC ARABIC LETTER AIN MEDIAL FORM
    # ->   0639 ARABIC LETTER AIN
    "\u{FECC}" => "\u{0639}",

    # FECD ARABIC LETTER GHAIN ISOLATED FORM
    # ->   063A ARABIC LETTER GHAIN
    "\u{FECD}" => "\u{063A}",

    # FECE ARABIC LETTER GHAIN FINAL FORM
    # ->   063A ARABIC LETTER GHAIN
    "\u{FECE}" => "\u{063A}",

    # FECF ARABIC LETTER GHAIN INITIAL FORM
    # ->   063A ARABIC LETTER GHAIN
    "\u{FECF}" => "\u{063A}",

    # FED0 ARABIC LETTER GHAIN MEDIAL FORM
    # ->   063A ARABIC LETTER GHAIN
    "\u{FED0}" => "\u{063A}",

    # FED1 ARABIC LETTER FEH ISOLATED FORM
    # ->   0641 ARABIC LETTER FEH
    "\u{FED1}" => "\u{0641}",

    # FED2 ARABIC LETTER FEH FINAL FORM
    # ->   0641 ARABIC LETTER FEH
    "\u{FED2}" => "\u{0641}",

    # FED3 ARABIC LETTER FEH INITIAL FORM
    # ->   0641 ARABIC LETTER FEH
    "\u{FED3}" => "\u{0641}",

    # FED4 ARABIC LETTER FEH MEDIAL FORM
    # ->   0641 ARABIC LETTER FEH
    "\u{FED4}" => "\u{0641}",

    # FED5 ARABIC LETTER QAF ISOLATED FORM
    # ->   0642 ARABIC LETTER QAF
    "\u{FED5}" => "\u{0642}",

    # FED6 ARABIC LETTER QAF FINAL FORM
    # ->   0642 ARABIC LETTER QAF
    "\u{FED6}" => "\u{0642}",

    # FED7 ARABIC LETTER QAF INITIAL FORM
    # ->   0642 ARABIC LETTER QAF
    "\u{FED7}" => "\u{0642}",

    # FED8 ARABIC LETTER QAF MEDIAL FORM
    # ->   0642 ARABIC LETTER QAF
    "\u{FED8}" => "\u{0642}",

    # FED9 ARABIC LETTER KAF ISOLATED FORM
    # ->   0643 ARABIC LETTER KAF
    "\u{FED9}" => "\u{0643}",

    # FEDA ARABIC LETTER KAF FINAL FORM
    # ->   0643 ARABIC LETTER KAF
    "\u{FEDA}" => "\u{0643}",

    # FEDB ARABIC LETTER KAF INITIAL FORM
    # ->   0643 ARABIC LETTER KAF
    "\u{FEDB}" => "\u{0643}",

    # FEDC ARABIC LETTER KAF MEDIAL FORM
    # ->   0643 ARABIC LETTER KAF
    "\u{FEDC}" => "\u{0643}",

    # FEDD ARABIC LETTER LAM ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    "\u{FEDD}" => "\u{0644}",

    # FEDE ARABIC LETTER LAM FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    "\u{FEDE}" => "\u{0644}",

    # FEDF ARABIC LETTER LAM INITIAL FORM
    # ->   0644 ARABIC LETTER LAM
    "\u{FEDF}" => "\u{0644}",

    # FEE0 ARABIC LETTER LAM MEDIAL FORM
    # ->   0644 ARABIC LETTER LAM
    "\u{FEE0}" => "\u{0644}",

    # FEE1 ARABIC LETTER MEEM ISOLATED FORM
    # ->   0645 ARABIC LETTER MEEM
    "\u{FEE1}" => "\u{0645}",

    # FEE2 ARABIC LETTER MEEM FINAL FORM
    # ->   0645 ARABIC LETTER MEEM
    "\u{FEE2}" => "\u{0645}",

    # FEE3 ARABIC LETTER MEEM INITIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    "\u{FEE3}" => "\u{0645}",

    # FEE4 ARABIC LETTER MEEM MEDIAL FORM
    # ->   0645 ARABIC LETTER MEEM
    "\u{FEE4}" => "\u{0645}",

    # FEE5 ARABIC LETTER NOON ISOLATED FORM
    # ->   0646 ARABIC LETTER NOON
    "\u{FEE5}" => "\u{0646}",

    # FEE6 ARABIC LETTER NOON FINAL FORM
    # ->   0646 ARABIC LETTER NOON
    "\u{FEE6}" => "\u{0646}",

    # FEE7 ARABIC LETTER NOON INITIAL FORM
    # ->   0646 ARABIC LETTER NOON
    "\u{FEE7}" => "\u{0646}",

    # FEE8 ARABIC LETTER NOON MEDIAL FORM
    # ->   0646 ARABIC LETTER NOON
    "\u{FEE8}" => "\u{0646}",

    # FEE9 ARABIC LETTER HEH ISOLATED FORM
    # ->   0647 ARABIC LETTER HEH
    "\u{FEE9}" => "\u{0647}",

    # FEEA ARABIC LETTER HEH FINAL FORM
    # ->   0647 ARABIC LETTER HEH
    "\u{FEEA}" => "\u{0647}",

    # FEEB ARABIC LETTER HEH INITIAL FORM
    # ->   0647 ARABIC LETTER HEH
    "\u{FEEB}" => "\u{0647}",

    # FEEC ARABIC LETTER HEH MEDIAL FORM
    # ->   0647 ARABIC LETTER HEH
    "\u{FEEC}" => "\u{0647}",

    # FEED ARABIC LETTER WAW ISOLATED FORM
    # ->   0648 ARABIC LETTER WAW
    "\u{FEED}" => "\u{0648}",

    # FEEE ARABIC LETTER WAW FINAL FORM
    # ->   0648 ARABIC LETTER WAW
    "\u{FEEE}" => "\u{0648}",

    # FEEF ARABIC LETTER ALEF MAKSURA ISOLATED FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FEEF}" => "\u{0649}",

    # FEF0 ARABIC LETTER ALEF MAKSURA FINAL FORM
    # ->   0649 ARABIC LETTER ALEF MAKSURA
    "\u{FEF0}" => "\u{0649}",

    # FEF1 ARABIC LETTER YEH ISOLATED FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FEF1}" => "\u{064A}",

    # FEF2 ARABIC LETTER YEH FINAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FEF2}" => "\u{064A}",

    # FEF3 ARABIC LETTER YEH INITIAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FEF3}" => "\u{064A}",

    # FEF4 ARABIC LETTER YEH MEDIAL FORM
    # ->   064A ARABIC LETTER YEH
    "\u{FEF4}" => "\u{064A}",

    # FEF5 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEF5}" => "\u{0644}\u{0627}",

    # FEF6 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEF6}" => "\u{0644}\u{0627}",

    # FEF7 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEF7}" => "\u{0644}\u{0627}",

    # FEF8 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEF8}" => "\u{0644}\u{0627}",

    # FEF9 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEF9}" => "\u{0644}\u{0627}",

    # FEFA ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEFA}" => "\u{0644}\u{0627}",

    # FEFB ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEFB}" => "\u{0644}\u{0627}",

    # FEFC ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    # ->   0644 ARABIC LETTER LAM
    #    + 0627 ARABIC LETTER ALEF
    "\u{FEFC}" => "\u{0644}\u{0627}",

    # FF01 FULLWIDTH EXCLAMATION MARK
    # ->   0021 EXCLAMATION MARK
    "\u{FF01}" => "!",

    # FF02 FULLWIDTH QUOTATION MARK
    # ->   0022 QUOTATION MARK
    "\u{FF02}" => "\"",

    # FF03 FULLWIDTH NUMBER SIGN
    # ->   0023 NUMBER SIGN
    "\u{FF03}" => "#",

    # FF04 FULLWIDTH DOLLAR SIGN
    # ->   0024 DOLLAR SIGN
    "\u{FF04}" => "\$",

    # FF05 FULLWIDTH PERCENT SIGN
    # ->   0025 PERCENT SIGN
    "\u{FF05}" => "%",

    # FF06 FULLWIDTH AMPERSAND
    # ->   0026 AMPERSAND
    "\u{FF06}" => "&",

    # FF07 FULLWIDTH APOSTROPHE
    # ->   0027 APOSTROPHE
    "\u{FF07}" => "'",

    # FF08 FULLWIDTH LEFT PARENTHESIS
    # ->   0028 LEFT PARENTHESIS
    "\u{FF08}" => "(",

    # FF09 FULLWIDTH RIGHT PARENTHESIS
    # ->   0029 RIGHT PARENTHESIS
    "\u{FF09}" => ")",

    # FF0A FULLWIDTH ASTERISK
    # ->   002A ASTERISK
    "\u{FF0A}" => "*",

    # FF0B FULLWIDTH PLUS SIGN
    # ->   002B PLUS SIGN
    "\u{FF0B}" => "+",

    # FF0C FULLWIDTH COMMA
    # ->   002C COMMA
    "\u{FF0C}" => ",",

    # FF0D FULLWIDTH HYPHEN-MINUS
    # ->   002D HYPHEN-MINUS
    "\u{FF0D}" => "-",

    # FF0E FULLWIDTH FULL STOP
    # ->   002E FULL STOP
    "\u{FF0E}" => ".",

    # FF0F FULLWIDTH SOLIDUS
    # ->   002F SOLIDUS
    "\u{FF0F}" => "/",

    # FF10 FULLWIDTH DIGIT ZERO
    # ->   0030 DIGIT ZERO
    "\u{FF10}" => "0",

    # FF11 FULLWIDTH DIGIT ONE
    # ->   0031 DIGIT ONE
    "\u{FF11}" => "1",

    # FF12 FULLWIDTH DIGIT TWO
    # ->   0032 DIGIT TWO
    "\u{FF12}" => "2",

    # FF13 FULLWIDTH DIGIT THREE
    # ->   0033 DIGIT THREE
    "\u{FF13}" => "3",

    # FF14 FULLWIDTH DIGIT FOUR
    # ->   0034 DIGIT FOUR
    "\u{FF14}" => "4",

    # FF15 FULLWIDTH DIGIT FIVE
    # ->   0035 DIGIT FIVE
    "\u{FF15}" => "5",

    # FF16 FULLWIDTH DIGIT SIX
    # ->   0036 DIGIT SIX
    "\u{FF16}" => "6",

    # FF17 FULLWIDTH DIGIT SEVEN
    # ->   0037 DIGIT SEVEN
    "\u{FF17}" => "7",

    # FF18 FULLWIDTH DIGIT EIGHT
    # ->   0038 DIGIT EIGHT
    "\u{FF18}" => "8",

    # FF19 FULLWIDTH DIGIT NINE
    # ->   0039 DIGIT NINE
    "\u{FF19}" => "9",

    # FF1A FULLWIDTH COLON
    # ->   003A COLON
    "\u{FF1A}" => ":",

    # FF1B FULLWIDTH SEMICOLON
    # ->   003B SEMICOLON
    "\u{FF1B}" => ";",

    # FF1C FULLWIDTH LESS-THAN SIGN
    # ->   003C LESS-THAN SIGN
    "\u{FF1C}" => "<",

    # FF1D FULLWIDTH EQUALS SIGN
    # ->   003D EQUALS SIGN
    "\u{FF1D}" => "=",

    # FF1E FULLWIDTH GREATER-THAN SIGN
    # ->   003E GREATER-THAN SIGN
    "\u{FF1E}" => ">",

    # FF1F FULLWIDTH QUESTION MARK
    # ->   003F QUESTION MARK
    "\u{FF1F}" => "?",

    # FF20 FULLWIDTH COMMERCIAL AT
    # ->   0040 COMMERCIAL AT
    "\u{FF20}" => "@",

    # FF21 FULLWIDTH LATIN CAPITAL LETTER A
    # ->   0041 LATIN CAPITAL LETTER A
    "\u{FF21}" => "A",

    # FF22 FULLWIDTH LATIN CAPITAL LETTER B
    # ->   0042 LATIN CAPITAL LETTER B
    "\u{FF22}" => "B",

    # FF23 FULLWIDTH LATIN CAPITAL LETTER C
    # ->   0043 LATIN CAPITAL LETTER C
    "\u{FF23}" => "C",

    # FF24 FULLWIDTH LATIN CAPITAL LETTER D
    # ->   0044 LATIN CAPITAL LETTER D
    "\u{FF24}" => "D",

    # FF25 FULLWIDTH LATIN CAPITAL LETTER E
    # ->   0045 LATIN CAPITAL LETTER E
    "\u{FF25}" => "E",

    # FF26 FULLWIDTH LATIN CAPITAL LETTER F
    # ->   0046 LATIN CAPITAL LETTER F
    "\u{FF26}" => "F",

    # FF27 FULLWIDTH LATIN CAPITAL LETTER G
    # ->   0047 LATIN CAPITAL LETTER G
    "\u{FF27}" => "G",

    # FF28 FULLWIDTH LATIN CAPITAL LETTER H
    # ->   0048 LATIN CAPITAL LETTER H
    "\u{FF28}" => "H",

    # FF29 FULLWIDTH LATIN CAPITAL LETTER I
    # ->   0049 LATIN CAPITAL LETTER I
    "\u{FF29}" => "I",

    # FF2A FULLWIDTH LATIN CAPITAL LETTER J
    # ->   004A LATIN CAPITAL LETTER J
    "\u{FF2A}" => "J",

    # FF2B FULLWIDTH LATIN CAPITAL LETTER K
    # ->   004B LATIN CAPITAL LETTER K
    "\u{FF2B}" => "K",

    # FF2C FULLWIDTH LATIN CAPITAL LETTER L
    # ->   004C LATIN CAPITAL LETTER L
    "\u{FF2C}" => "L",

    # FF2D FULLWIDTH LATIN CAPITAL LETTER M
    # ->   004D LATIN CAPITAL LETTER M
    "\u{FF2D}" => "M",

    # FF2E FULLWIDTH LATIN CAPITAL LETTER N
    # ->   004E LATIN CAPITAL LETTER N
    "\u{FF2E}" => "N",

    # FF2F FULLWIDTH LATIN CAPITAL LETTER O
    # ->   004F LATIN CAPITAL LETTER O
    "\u{FF2F}" => "O",

    # FF30 FULLWIDTH LATIN CAPITAL LETTER P
    # ->   0050 LATIN CAPITAL LETTER P
    "\u{FF30}" => "P",

    # FF31 FULLWIDTH LATIN CAPITAL LETTER Q
    # ->   0051 LATIN CAPITAL LETTER Q
    "\u{FF31}" => "Q",

    # FF32 FULLWIDTH LATIN CAPITAL LETTER R
    # ->   0052 LATIN CAPITAL LETTER R
    "\u{FF32}" => "R",

    # FF33 FULLWIDTH LATIN CAPITAL LETTER S
    # ->   0053 LATIN CAPITAL LETTER S
    "\u{FF33}" => "S",

    # FF34 FULLWIDTH LATIN CAPITAL LETTER T
    # ->   0054 LATIN CAPITAL LETTER T
    "\u{FF34}" => "T",

    # FF35 FULLWIDTH LATIN CAPITAL LETTER U
    # ->   0055 LATIN CAPITAL LETTER U
    "\u{FF35}" => "U",

    # FF36 FULLWIDTH LATIN CAPITAL LETTER V
    # ->   0056 LATIN CAPITAL LETTER V
    "\u{FF36}" => "V",

    # FF37 FULLWIDTH LATIN CAPITAL LETTER W
    # ->   0057 LATIN CAPITAL LETTER W
    "\u{FF37}" => "W",

    # FF38 FULLWIDTH LATIN CAPITAL LETTER X
    # ->   0058 LATIN CAPITAL LETTER X
    "\u{FF38}" => "X",

    # FF39 FULLWIDTH LATIN CAPITAL LETTER Y
    # ->   0059 LATIN CAPITAL LETTER Y
    "\u{FF39}" => "Y",

    # FF3A FULLWIDTH LATIN CAPITAL LETTER Z
    # ->   005A LATIN CAPITAL LETTER Z
    "\u{FF3A}" => "Z",

    # FF3B FULLWIDTH LEFT SQUARE BRACKET
    # ->   005B LEFT SQUARE BRACKET
    "\u{FF3B}" => "[",

    # FF3C FULLWIDTH REVERSE SOLIDUS
    # ->   005C REVERSE SOLIDUS
    "\u{FF3C}" => "\\",

    # FF3D FULLWIDTH RIGHT SQUARE BRACKET
    # ->   005D RIGHT SQUARE BRACKET
    "\u{FF3D}" => "]",

    # FF3E FULLWIDTH CIRCUMFLEX ACCENT
    # ->   005E CIRCUMFLEX ACCENT
    "\u{FF3E}" => "^",

    # FF3F FULLWIDTH LOW LINE
    # ->   005F LOW LINE
    "\u{FF3F}" => "_",

    # FF40 FULLWIDTH GRAVE ACCENT
    # ->   0060 GRAVE ACCENT
    "\u{FF40}" => "`",

    # FF41 FULLWIDTH LATIN SMALL LETTER A
    # ->   0061 LATIN SMALL LETTER A
    "\u{FF41}" => "a",

    # FF42 FULLWIDTH LATIN SMALL LETTER B
    # ->   0062 LATIN SMALL LETTER B
    "\u{FF42}" => "b",

    # FF43 FULLWIDTH LATIN SMALL LETTER C
    # ->   0063 LATIN SMALL LETTER C
    "\u{FF43}" => "c",

    # FF44 FULLWIDTH LATIN SMALL LETTER D
    # ->   0064 LATIN SMALL LETTER D
    "\u{FF44}" => "d",

    # FF45 FULLWIDTH LATIN SMALL LETTER E
    # ->   0065 LATIN SMALL LETTER E
    "\u{FF45}" => "e",

    # FF46 FULLWIDTH LATIN SMALL LETTER F
    # ->   0066 LATIN SMALL LETTER F
    "\u{FF46}" => "f",

    # FF47 FULLWIDTH LATIN SMALL LETTER G
    # ->   0067 LATIN SMALL LETTER G
    "\u{FF47}" => "g",

    # FF48 FULLWIDTH LATIN SMALL LETTER H
    # ->   0068 LATIN SMALL LETTER H
    "\u{FF48}" => "h",

    # FF49 FULLWIDTH LATIN SMALL LETTER I
    # ->   0069 LATIN SMALL LETTER I
    "\u{FF49}" => "i",

    # FF4A FULLWIDTH LATIN SMALL LETTER J
    # ->   006A LATIN SMALL LETTER J
    "\u{FF4A}" => "j",

    # FF4B FULLWIDTH LATIN SMALL LETTER K
    # ->   006B LATIN SMALL LETTER K
    "\u{FF4B}" => "k",

    # FF4C FULLWIDTH LATIN SMALL LETTER L
    # ->   006C LATIN SMALL LETTER L
    "\u{FF4C}" => "l",

    # FF4D FULLWIDTH LATIN SMALL LETTER M
    # ->   006D LATIN SMALL LETTER M
    "\u{FF4D}" => "m",

    # FF4E FULLWIDTH LATIN SMALL LETTER N
    # ->   006E LATIN SMALL LETTER N
    "\u{FF4E}" => "n",

    # FF4F FULLWIDTH LATIN SMALL LETTER O
    # ->   006F LATIN SMALL LETTER O
    "\u{FF4F}" => "o",

    # FF50 FULLWIDTH LATIN SMALL LETTER P
    # ->   0070 LATIN SMALL LETTER P
    "\u{FF50}" => "p",

    # FF51 FULLWIDTH LATIN SMALL LETTER Q
    # ->   0071 LATIN SMALL LETTER Q
    "\u{FF51}" => "q",

    # FF52 FULLWIDTH LATIN SMALL LETTER R
    # ->   0072 LATIN SMALL LETTER R
    "\u{FF52}" => "r",

    # FF53 FULLWIDTH LATIN SMALL LETTER S
    # ->   0073 LATIN SMALL LETTER S
    "\u{FF53}" => "s",

    # FF54 FULLWIDTH LATIN SMALL LETTER T
    # ->   0074 LATIN SMALL LETTER T
    "\u{FF54}" => "t",

    # FF55 FULLWIDTH LATIN SMALL LETTER U
    # ->   0075 LATIN SMALL LETTER U
    "\u{FF55}" => "u",

    # FF56 FULLWIDTH LATIN SMALL LETTER V
    # ->   0076 LATIN SMALL LETTER V
    "\u{FF56}" => "v",

    # FF57 FULLWIDTH LATIN SMALL LETTER W
    # ->   0077 LATIN SMALL LETTER W
    "\u{FF57}" => "w",

    # FF58 FULLWIDTH LATIN SMALL LETTER X
    # ->   0078 LATIN SMALL LETTER X
    "\u{FF58}" => "x",

    # FF59 FULLWIDTH LATIN SMALL LETTER Y
    # ->   0079 LATIN SMALL LETTER Y
    "\u{FF59}" => "y",

    # FF5A FULLWIDTH LATIN SMALL LETTER Z
    # ->   007A LATIN SMALL LETTER Z
    "\u{FF5A}" => "z",

    # FF5B FULLWIDTH LEFT CURLY BRACKET
    # ->   007B LEFT CURLY BRACKET
    "\u{FF5B}" => "{",

    # FF5C FULLWIDTH VERTICAL LINE
    # ->   007C VERTICAL LINE
    "\u{FF5C}" => "|",

    # FF5D FULLWIDTH RIGHT CURLY BRACKET
    # ->   007D RIGHT CURLY BRACKET
    "\u{FF5D}" => "}",

    # FF5E FULLWIDTH TILDE
    # ->   007E TILDE
    "\u{FF5E}" => "~",

    # FF5F FULLWIDTH LEFT WHITE PARENTHESIS
    # ->   2985 LEFT WHITE PARENTHESIS
    "\u{FF5F}" => "\u{2985}",

    # FF60 FULLWIDTH RIGHT WHITE PARENTHESIS
    # ->   2986 RIGHT WHITE PARENTHESIS
    "\u{FF60}" => "\u{2986}",

    # FF61 HALFWIDTH IDEOGRAPHIC FULL STOP
    # ->   3002 IDEOGRAPHIC FULL STOP
    "\u{FF61}" => "\u{3002}",

    # FF62 HALFWIDTH LEFT CORNER BRACKET
    # ->   300C LEFT CORNER BRACKET
    "\u{FF62}" => "\u{300C}",

    # FF63 HALFWIDTH RIGHT CORNER BRACKET
    # ->   300D RIGHT CORNER BRACKET
    "\u{FF63}" => "\u{300D}",

    # FF64 HALFWIDTH IDEOGRAPHIC COMMA
    # ->   3001 IDEOGRAPHIC COMMA
    "\u{FF64}" => "\u{3001}",

    # FF65 HALFWIDTH KATAKANA MIDDLE DOT
    # ->   30FB KATAKANA MIDDLE DOT
    "\u{FF65}" => "\u{30FB}",

    # FF66 HALFWIDTH KATAKANA LETTER WO
    # ->   30F2 KATAKANA LETTER WO
    "\u{FF66}" => "\u{30F2}",

    # FF67 HALFWIDTH KATAKANA LETTER SMALL A
    # ->   30A1 KATAKANA LETTER SMALL A
    "\u{FF67}" => "\u{30A1}",

    # FF68 HALFWIDTH KATAKANA LETTER SMALL I
    # ->   30A3 KATAKANA LETTER SMALL I
    "\u{FF68}" => "\u{30A3}",

    # FF69 HALFWIDTH KATAKANA LETTER SMALL U
    # ->   30A5 KATAKANA LETTER SMALL U
    "\u{FF69}" => "\u{30A5}",

    # FF6A HALFWIDTH KATAKANA LETTER SMALL E
    # ->   30A7 KATAKANA LETTER SMALL E
    "\u{FF6A}" => "\u{30A7}",

    # FF6B HALFWIDTH KATAKANA LETTER SMALL O
    # ->   30A9 KATAKANA LETTER SMALL O
    "\u{FF6B}" => "\u{30A9}",

    # FF6C HALFWIDTH KATAKANA LETTER SMALL YA
    # ->   30E3 KATAKANA LETTER SMALL YA
    "\u{FF6C}" => "\u{30E3}",

    # FF6D HALFWIDTH KATAKANA LETTER SMALL YU
    # ->   30E5 KATAKANA LETTER SMALL YU
    "\u{FF6D}" => "\u{30E5}",

    # FF6E HALFWIDTH KATAKANA LETTER SMALL YO
    # ->   30E7 KATAKANA LETTER SMALL YO
    "\u{FF6E}" => "\u{30E7}",

    # FF6F HALFWIDTH KATAKANA LETTER SMALL TU
    # ->   30C3 KATAKANA LETTER SMALL TU
    "\u{FF6F}" => "\u{30C3}",

    # FF70 HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
    "\u{FF70}" => "\u{30FC}",

    # FF71 HALFWIDTH KATAKANA LETTER A
    # ->   30A2 KATAKANA LETTER A
    "\u{FF71}" => "\u{30A2}",

    # FF72 HALFWIDTH KATAKANA LETTER I
    # ->   30A4 KATAKANA LETTER I
    "\u{FF72}" => "\u{30A4}",

    # FF73 HALFWIDTH KATAKANA LETTER U
    # ->   30A6 KATAKANA LETTER U
    "\u{FF73}" => "\u{30A6}",

    # FF74 HALFWIDTH KATAKANA LETTER E
    # ->   30A8 KATAKANA LETTER E
    "\u{FF74}" => "\u{30A8}",

    # FF75 HALFWIDTH KATAKANA LETTER O
    # ->   30AA KATAKANA LETTER O
    "\u{FF75}" => "\u{30AA}",

    # FF76 HALFWIDTH KATAKANA LETTER KA
    # ->   30AB KATAKANA LETTER KA
    "\u{FF76}" => "\u{30AB}",

    # FF77 HALFWIDTH KATAKANA LETTER KI
    # ->   30AD KATAKANA LETTER KI
    "\u{FF77}" => "\u{30AD}",

    # FF78 HALFWIDTH KATAKANA LETTER KU
    # ->   30AF KATAKANA LETTER KU
    "\u{FF78}" => "\u{30AF}",

    # FF79 HALFWIDTH KATAKANA LETTER KE
    # ->   30B1 KATAKANA LETTER KE
    "\u{FF79}" => "\u{30B1}",

    # FF7A HALFWIDTH KATAKANA LETTER KO
    # ->   30B3 KATAKANA LETTER KO
    "\u{FF7A}" => "\u{30B3}",

    # FF7B HALFWIDTH KATAKANA LETTER SA
    # ->   30B5 KATAKANA LETTER SA
    "\u{FF7B}" => "\u{30B5}",

    # FF7C HALFWIDTH KATAKANA LETTER SI
    # ->   30B7 KATAKANA LETTER SI
    "\u{FF7C}" => "\u{30B7}",

    # FF7D HALFWIDTH KATAKANA LETTER SU
    # ->   30B9 KATAKANA LETTER SU
    "\u{FF7D}" => "\u{30B9}",

    # FF7E HALFWIDTH KATAKANA LETTER SE
    # ->   30BB KATAKANA LETTER SE
    "\u{FF7E}" => "\u{30BB}",

    # FF7F HALFWIDTH KATAKANA LETTER SO
    # ->   30BD KATAKANA LETTER SO
    "\u{FF7F}" => "\u{30BD}",

    # FF80 HALFWIDTH KATAKANA LETTER TA
    # ->   30BF KATAKANA LETTER TA
    "\u{FF80}" => "\u{30BF}",

    # FF81 HALFWIDTH KATAKANA LETTER TI
    # ->   30C1 KATAKANA LETTER TI
    "\u{FF81}" => "\u{30C1}",

    # FF82 HALFWIDTH KATAKANA LETTER TU
    # ->   30C4 KATAKANA LETTER TU
    "\u{FF82}" => "\u{30C4}",

    # FF83 HALFWIDTH KATAKANA LETTER TE
    # ->   30C6 KATAKANA LETTER TE
    "\u{FF83}" => "\u{30C6}",

    # FF84 HALFWIDTH KATAKANA LETTER TO
    # ->   30C8 KATAKANA LETTER TO
    "\u{FF84}" => "\u{30C8}",

    # FF85 HALFWIDTH KATAKANA LETTER NA
    # ->   30CA KATAKANA LETTER NA
    "\u{FF85}" => "\u{30CA}",

    # FF86 HALFWIDTH KATAKANA LETTER NI
    # ->   30CB KATAKANA LETTER NI
    "\u{FF86}" => "\u{30CB}",

    # FF87 HALFWIDTH KATAKANA LETTER NU
    # ->   30CC KATAKANA LETTER NU
    "\u{FF87}" => "\u{30CC}",

    # FF88 HALFWIDTH KATAKANA LETTER NE
    # ->   30CD KATAKANA LETTER NE
    "\u{FF88}" => "\u{30CD}",

    # FF89 HALFWIDTH KATAKANA LETTER NO
    # ->   30CE KATAKANA LETTER NO
    "\u{FF89}" => "\u{30CE}",

    # FF8A HALFWIDTH KATAKANA LETTER HA
    # ->   30CF KATAKANA LETTER HA
    "\u{FF8A}" => "\u{30CF}",

    # FF8B HALFWIDTH KATAKANA LETTER HI
    # ->   30D2 KATAKANA LETTER HI
    "\u{FF8B}" => "\u{30D2}",

    # FF8C HALFWIDTH KATAKANA LETTER HU
    # ->   30D5 KATAKANA LETTER HU
    "\u{FF8C}" => "\u{30D5}",

    # FF8D HALFWIDTH KATAKANA LETTER HE
    # ->   30D8 KATAKANA LETTER HE
    "\u{FF8D}" => "\u{30D8}",

    # FF8E HALFWIDTH KATAKANA LETTER HO
    # ->   30DB KATAKANA LETTER HO
    "\u{FF8E}" => "\u{30DB}",

    # FF8F HALFWIDTH KATAKANA LETTER MA
    # ->   30DE KATAKANA LETTER MA
    "\u{FF8F}" => "\u{30DE}",

    # FF90 HALFWIDTH KATAKANA LETTER MI
    # ->   30DF KATAKANA LETTER MI
    "\u{FF90}" => "\u{30DF}",

    # FF91 HALFWIDTH KATAKANA LETTER MU
    # ->   30E0 KATAKANA LETTER MU
    "\u{FF91}" => "\u{30E0}",

    # FF92 HALFWIDTH KATAKANA LETTER ME
    # ->   30E1 KATAKANA LETTER ME
    "\u{FF92}" => "\u{30E1}",

    # FF93 HALFWIDTH KATAKANA LETTER MO
    # ->   30E2 KATAKANA LETTER MO
    "\u{FF93}" => "\u{30E2}",

    # FF94 HALFWIDTH KATAKANA LETTER YA
    # ->   30E4 KATAKANA LETTER YA
    "\u{FF94}" => "\u{30E4}",

    # FF95 HALFWIDTH KATAKANA LETTER YU
    # ->   30E6 KATAKANA LETTER YU
    "\u{FF95}" => "\u{30E6}",

    # FF96 HALFWIDTH KATAKANA LETTER YO
    # ->   30E8 KATAKANA LETTER YO
    "\u{FF96}" => "\u{30E8}",

    # FF97 HALFWIDTH KATAKANA LETTER RA
    # ->   30E9 KATAKANA LETTER RA
    "\u{FF97}" => "\u{30E9}",

    # FF98 HALFWIDTH KATAKANA LETTER RI
    # ->   30EA KATAKANA LETTER RI
    "\u{FF98}" => "\u{30EA}",

    # FF99 HALFWIDTH KATAKANA LETTER RU
    # ->   30EB KATAKANA LETTER RU
    "\u{FF99}" => "\u{30EB}",

    # FF9A HALFWIDTH KATAKANA LETTER RE
    # ->   30EC KATAKANA LETTER RE
    "\u{FF9A}" => "\u{30EC}",

    # FF9B HALFWIDTH KATAKANA LETTER RO
    # ->   30ED KATAKANA LETTER RO
    "\u{FF9B}" => "\u{30ED}",

    # FF9C HALFWIDTH KATAKANA LETTER WA
    # ->   30EF KATAKANA LETTER WA
    "\u{FF9C}" => "\u{30EF}",

    # FF9D HALFWIDTH KATAKANA LETTER N
    # ->   30F3 KATAKANA LETTER N
    "\u{FF9D}" => "\u{30F3}",

    # FFA0 HALFWIDTH HANGUL FILLER
    # ->   1160 HANGUL JUNGSEONG FILLER
    "\u{FFA0}" => "\u{1160}",

    # FFA1 HALFWIDTH HANGUL LETTER KIYEOK
    # ->   1100 HANGUL CHOSEONG KIYEOK
    "\u{FFA1}" => "\u{1100}",

    # FFA2 HALFWIDTH HANGUL LETTER SSANGKIYEOK
    # ->   1101 HANGUL CHOSEONG SSANGKIYEOK
    "\u{FFA2}" => "\u{1101}",

    # FFA3 HALFWIDTH HANGUL LETTER KIYEOK-SIOS
    # ->   11AA HANGUL JONGSEONG KIYEOK-SIOS
    "\u{FFA3}" => "\u{11AA}",

    # FFA4 HALFWIDTH HANGUL LETTER NIEUN
    # ->   1102 HANGUL CHOSEONG NIEUN
    "\u{FFA4}" => "\u{1102}",

    # FFA5 HALFWIDTH HANGUL LETTER NIEUN-CIEUC
    # ->   11AC HANGUL JONGSEONG NIEUN-CIEUC
    "\u{FFA5}" => "\u{11AC}",

    # FFA6 HALFWIDTH HANGUL LETTER NIEUN-HIEUH
    # ->   11AD HANGUL JONGSEONG NIEUN-HIEUH
    "\u{FFA6}" => "\u{11AD}",

    # FFA7 HALFWIDTH HANGUL LETTER TIKEUT
    # ->   1103 HANGUL CHOSEONG TIKEUT
    "\u{FFA7}" => "\u{1103}",

    # FFA8 HALFWIDTH HANGUL LETTER SSANGTIKEUT
    # ->   1104 HANGUL CHOSEONG SSANGTIKEUT
    "\u{FFA8}" => "\u{1104}",

    # FFA9 HALFWIDTH HANGUL LETTER RIEUL
    # ->   1105 HANGUL CHOSEONG RIEUL
    "\u{FFA9}" => "\u{1105}",

    # FFAA HALFWIDTH HANGUL LETTER RIEUL-KIYEOK
    # ->   11B0 HANGUL JONGSEONG RIEUL-KIYEOK
    "\u{FFAA}" => "\u{11B0}",

    # FFAB HALFWIDTH HANGUL LETTER RIEUL-MIEUM
    # ->   11B1 HANGUL JONGSEONG RIEUL-MIEUM
    "\u{FFAB}" => "\u{11B1}",

    # FFAC HALFWIDTH HANGUL LETTER RIEUL-PIEUP
    # ->   11B2 HANGUL JONGSEONG RIEUL-PIEUP
    "\u{FFAC}" => "\u{11B2}",

    # FFAD HALFWIDTH HANGUL LETTER RIEUL-SIOS
    # ->   11B3 HANGUL JONGSEONG RIEUL-SIOS
    "\u{FFAD}" => "\u{11B3}",

    # FFAE HALFWIDTH HANGUL LETTER RIEUL-THIEUTH
    # ->   11B4 HANGUL JONGSEONG RIEUL-THIEUTH
    "\u{FFAE}" => "\u{11B4}",

    # FFAF HALFWIDTH HANGUL LETTER RIEUL-PHIEUPH
    # ->   11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
    "\u{FFAF}" => "\u{11B5}",

    # FFB0 HALFWIDTH HANGUL LETTER RIEUL-HIEUH
    # ->   111A HANGUL CHOSEONG RIEUL-HIEUH
    "\u{FFB0}" => "\u{111A}",

    # FFB1 HALFWIDTH HANGUL LETTER MIEUM
    # ->   1106 HANGUL CHOSEONG MIEUM
    "\u{FFB1}" => "\u{1106}",

    # FFB2 HALFWIDTH HANGUL LETTER PIEUP
    # ->   1107 HANGUL CHOSEONG PIEUP
    "\u{FFB2}" => "\u{1107}",

    # FFB3 HALFWIDTH HANGUL LETTER SSANGPIEUP
    # ->   1108 HANGUL CHOSEONG SSANGPIEUP
    "\u{FFB3}" => "\u{1108}",

    # FFB4 HALFWIDTH HANGUL LETTER PIEUP-SIOS
    # ->   1121 HANGUL CHOSEONG PIEUP-SIOS
    "\u{FFB4}" => "\u{1121}",

    # FFB5 HALFWIDTH HANGUL LETTER SIOS
    # ->   1109 HANGUL CHOSEONG SIOS
    "\u{FFB5}" => "\u{1109}",

    # FFB6 HALFWIDTH HANGUL LETTER SSANGSIOS
    # ->   110A HANGUL CHOSEONG SSANGSIOS
    "\u{FFB6}" => "\u{110A}",

    # FFB7 HALFWIDTH HANGUL LETTER IEUNG
    # ->   110B HANGUL CHOSEONG IEUNG
    "\u{FFB7}" => "\u{110B}",

    # FFB8 HALFWIDTH HANGUL LETTER CIEUC
    # ->   110C HANGUL CHOSEONG CIEUC
    "\u{FFB8}" => "\u{110C}",

    # FFB9 HALFWIDTH HANGUL LETTER SSANGCIEUC
    # ->   110D HANGUL CHOSEONG SSANGCIEUC
    "\u{FFB9}" => "\u{110D}",

    # FFBA HALFWIDTH HANGUL LETTER CHIEUCH
    # ->   110E HANGUL CHOSEONG CHIEUCH
    "\u{FFBA}" => "\u{110E}",

    # FFBB HALFWIDTH HANGUL LETTER KHIEUKH
    # ->   110F HANGUL CHOSEONG KHIEUKH
    "\u{FFBB}" => "\u{110F}",

    # FFBC HALFWIDTH HANGUL LETTER THIEUTH
    # ->   1110 HANGUL CHOSEONG THIEUTH
    "\u{FFBC}" => "\u{1110}",

    # FFBD HALFWIDTH HANGUL LETTER PHIEUPH
    # ->   1111 HANGUL CHOSEONG PHIEUPH
    "\u{FFBD}" => "\u{1111}",

    # FFBE HALFWIDTH HANGUL LETTER HIEUH
    # ->   1112 HANGUL CHOSEONG HIEUH
    "\u{FFBE}" => "\u{1112}",

    # FFC2 HALFWIDTH HANGUL LETTER A
    # ->   1161 HANGUL JUNGSEONG A
    "\u{FFC2}" => "\u{1161}",

    # FFC3 HALFWIDTH HANGUL LETTER AE
    # ->   1162 HANGUL JUNGSEONG AE
    "\u{FFC3}" => "\u{1162}",

    # FFC4 HALFWIDTH HANGUL LETTER YA
    # ->   1163 HANGUL JUNGSEONG YA
    "\u{FFC4}" => "\u{1163}",

    # FFC5 HALFWIDTH HANGUL LETTER YAE
    # ->   1164 HANGUL JUNGSEONG YAE
    "\u{FFC5}" => "\u{1164}",

    # FFC6 HALFWIDTH HANGUL LETTER EO
    # ->   1165 HANGUL JUNGSEONG EO
    "\u{FFC6}" => "\u{1165}",

    # FFC7 HALFWIDTH HANGUL LETTER E
    # ->   1166 HANGUL JUNGSEONG E
    "\u{FFC7}" => "\u{1166}",

    # FFCA HALFWIDTH HANGUL LETTER YEO
    # ->   1167 HANGUL JUNGSEONG YEO
    "\u{FFCA}" => "\u{1167}",

    # FFCB HALFWIDTH HANGUL LETTER YE
    # ->   1168 HANGUL JUNGSEONG YE
    "\u{FFCB}" => "\u{1168}",

    # FFCC HALFWIDTH HANGUL LETTER O
    # ->   1169 HANGUL JUNGSEONG O
    "\u{FFCC}" => "\u{1169}",

    # FFCD HALFWIDTH HANGUL LETTER WA
    # ->   116A HANGUL JUNGSEONG WA
    "\u{FFCD}" => "\u{116A}",

    # FFCE HALFWIDTH HANGUL LETTER WAE
    # ->   116B HANGUL JUNGSEONG WAE
    "\u{FFCE}" => "\u{116B}",

    # FFCF HALFWIDTH HANGUL LETTER OE
    # ->   116C HANGUL JUNGSEONG OE
    "\u{FFCF}" => "\u{116C}",

    # FFD2 HALFWIDTH HANGUL LETTER YO
    # ->   116D HANGUL JUNGSEONG YO
    "\u{FFD2}" => "\u{116D}",

    # FFD3 HALFWIDTH HANGUL LETTER U
    # ->   116E HANGUL JUNGSEONG U
    "\u{FFD3}" => "\u{116E}",

    # FFD4 HALFWIDTH HANGUL LETTER WEO
    # ->   116F HANGUL JUNGSEONG WEO
    "\u{FFD4}" => "\u{116F}",

    # FFD5 HALFWIDTH HANGUL LETTER WE
    # ->   1170 HANGUL JUNGSEONG WE
    "\u{FFD5}" => "\u{1170}",

    # FFD6 HALFWIDTH HANGUL LETTER WI
    # ->   1171 HANGUL JUNGSEONG WI
    "\u{FFD6}" => "\u{1171}",

    # FFD7 HALFWIDTH HANGUL LETTER YU
    # ->   1172 HANGUL JUNGSEONG YU
    "\u{FFD7}" => "\u{1172}",

    # FFDA HALFWIDTH HANGUL LETTER EU
    # ->   1173 HANGUL JUNGSEONG EU
    "\u{FFDA}" => "\u{1173}",

    # FFDB HALFWIDTH HANGUL LETTER YI
    # ->   1174 HANGUL JUNGSEONG YI
    "\u{FFDB}" => "\u{1174}",

    # FFDC HALFWIDTH HANGUL LETTER I
    # ->   1175 HANGUL JUNGSEONG I
    "\u{FFDC}" => "\u{1175}",

    # FFE0 FULLWIDTH CENT SIGN
    # ->   00A2 CENT SIGN
    "\u{FFE0}" => "\u{A2}",

    # FFE1 FULLWIDTH POUND SIGN
    # ->   00A3 POUND SIGN
    "\u{FFE1}" => "\u{A3}",

    # FFE2 FULLWIDTH NOT SIGN
    # ->   00AC NOT SIGN
    "\u{FFE2}" => "\u{AC}",

    # FFE3 FULLWIDTH MACRON
    # ->   0020 SPACE
    "\u{FFE3}" => " ",

    # FFE4 FULLWIDTH BROKEN BAR
    # ->   00A6 BROKEN BAR
    "\u{FFE4}" => "\u{A6}",

    # FFE5 FULLWIDTH YEN SIGN
    # ->   00A5 YEN SIGN
    "\u{FFE5}" => "\u{A5}",

    # FFE6 FULLWIDTH WON SIGN
    # ->   20A9 WON SIGN
    "\u{FFE6}" => "\u{20A9}",

    # FFE8 HALFWIDTH FORMS LIGHT VERTICAL
    # ->   2502 BOX DRAWINGS LIGHT VERTICAL
    "\u{FFE8}" => "\u{2502}",

    # FFE9 HALFWIDTH LEFTWARDS ARROW
    # ->   2190 LEFTWARDS ARROW
    "\u{FFE9}" => "\u{2190}",

    # FFEA HALFWIDTH UPWARDS ARROW
    # ->   2191 UPWARDS ARROW
    "\u{FFEA}" => "\u{2191}",

    # FFEB HALFWIDTH RIGHTWARDS ARROW
    # ->   2192 RIGHTWARDS ARROW
    "\u{FFEB}" => "\u{2192}",

    # FFEC HALFWIDTH DOWNWARDS ARROW
    # ->   2193 DOWNWARDS ARROW
    "\u{FFEC}" => "\u{2193}",

    # FFED HALFWIDTH BLACK SQUARE
    # ->   25A0 BLACK SQUARE
    "\u{FFED}" => "\u{25A0}",

    # FFEE HALFWIDTH WHITE CIRCLE
    # ->   25CB WHITE CIRCLE
    "\u{FFEE}" => "\u{25CB}",

  }

end

