/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDISENSORSIMULATOR_HPP
  #define TEPDISENSORSIMULATOR_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIParameters.hpp"
  #include "TePDIInterpolator.hpp"
  
  #include <TeSharedPtr.h>

  /**
   * @brief This is the class for simulating an image from a sensor. 
   * Given an image generated by sensor 1, with known ifov and eifovs, this class generates another image, 
   * as if it was generated by sensor 2, with another ifov and eifovs values.
   * @author Eliana Pantaleo <elianap@dpi.inpe.br>
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup TePDIGeneralAlgoGroup
   *
   * @note The general required parameters:
   *
   * @param input_raster ( TePDITypes::TePDIRasterPtrType ) - The original input raster.
   * @param channels ( std::vector<int> ) - The band(s) to be simulated.
   * @param output_raster ( TePDITypes::TePDIRasterPtrType ) - The output raster.
   * @param ifov_in ( double ) - The ifov of the input raster.
   * @param eifov_in_across ( double ) - The eifov of the input raster across track.
   * @param eifov_in_long ( double ) - The eifov of the input raster long track.
   * @param ifov_out ( double ) - The ifov of the output raster.
   * @param eifov_out_across ( double ) - The eifov of the output raster across track.
   * @param eifov_out_long ( double ) - The eifov of the output raster long track.
   *
   * @note Optional parameters:
   *
   * @param interpolator (TePDIInterpolator::NNMethod) - The interpolation method to use when raster resampling is needed (default:TePDIInterpolator::BicubicMethod).
   *
   */
  class PDI_DLL TePDISensorSimulator : public TePDIAlgorithm {
    public :
      typedef TeSharedPtr< TePDISensorSimulator > pointer;
      typedef const TeSharedPtr< TePDISensorSimulator> const_pointer;
      
      /**
       * Default Constructor.
       *
       */
      TePDISensorSimulator();

      /**
       * Default Destructor
       */
      ~TePDISensorSimulator();
      
      /**
       * Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;      

    protected :
    
      /**
       * Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );    
     
      /**
       * Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();      
  };
  
/** @example TePDISensorSimulator_test.cpp
 *    Shows how to use this class.
 */    

#endif
