/* 
 * Copyright (C) 2001-2013 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.xiphias.dom;

import java.util.ArrayList;

import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DOMTraverser {

	private ArrayList<INodeVisitor> visitors = new ArrayList<INodeVisitor>();

    public DOMTraverser(INodeVisitor visitor) {
    	addVisitor(visitor);
    }
    
    public void addVisitor(INodeVisitor visitor) {
        visitors.add(visitor);
    }

    public void traverse(Node node) throws Exception {

    	if (node.getNodeType() == Node.ELEMENT_NODE) {
    		openTag(node);
    	}
    	
    	accept(node);
    	
        if ((node.getNodeType() == Node.ELEMENT_NODE) && node.hasAttributes()) {

        	NamedNodeMap attrs = ((Element) node).getAttributes();

            for (int i = 0; i < attrs.getLength(); i++) {
                Node attr = attrs.item(i);
                accept(attr);
            }
        }

        if (node.hasChildNodes()) {

        	NodeList children = node.getChildNodes();

            for (int i = 0; i < children.getLength(); i++) {
                Node child = children.item(i);
                traverse(child);
            }
        }

        if (node.getNodeType() == Node.ELEMENT_NODE) {
    		closeTag(node);
    	}
    	
    }

    private void openTag(Node node) throws Exception {

    	for (INodeVisitor visitor : visitors) {
    		visitor.openTag(node);
        }
    }

    private void accept(Node node) throws Exception {

    	for (INodeVisitor visitor : visitors) {
    		visitor.accept(node);
        }
    }
    
    private void closeTag(Node node) throws Exception {

    	for (INodeVisitor visitor : visitors) {
    		visitor.closeTag(node);
        }
    }
}
