/*

Ventrilo encryption/decryption algorithm 0.2
by Luigi Auriemma
e-mail: aluigi@autistici.org
web:    aluigi.org


INTRODUCTION
============
This algorithm is the method used by the chat program Ventrilo
(http://www.ventrilo.com) for encrypting the communication stream
between clients and servers.


FUNCTIONS
=========
- ventrilo_key_ctx:   not a function but the container of the client
                      and server keys, must be declared for each
                      client<-->server connection (one for client and
                      another for server)
- ventrilo_read_keys: required for reading the keys received from the
                      server, remember to use data+12 and len-12!!!
- ventrilo_first_dec: used ONLY for the first data block
- ventrilo_first_enc: used ONLY for the first data block
- ventrilo_dec:       decrypts a data block
- ventrilo_enc:       encrypts a data block

A Ventrilo data block is composed by 2 bytes (16 bit in network byte
order) that specifies the size of the data block and is then followed
by the data block:

  |##|#####################################|
  |  |
  |  data block
  size of data block (16 bit, network byte order)

For practical examples look the links below.


EXAMPLES
========
Ventrilo proxy with data manipulation example:
  http://aluigi.org/papers/ventrilo_encdec_ex.zip

Ventrilo proxy data decrypter:
  http://aluigi.org/papers/ventrilo_proxy.zip


THANX TO
========
Georg Hofstetter (http://www.g3gg0.de)


LICENSE
=======
    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

*/

#include <string.h>



typedef struct {
    unsigned char   key[256];
    unsigned int    pos;
    unsigned int    size;
} ventrilo_key_ctx;



int ventrilo_read_keys(ventrilo_key_ctx *client, ventrilo_key_ctx *server, unsigned char *data, int size) {
    ventrilo_key_ctx    *tmp;
    unsigned char       *p,
                        *del = NULL;

    for(p = data; size && *p; size--, p++) {
        if(del) continue;

        if(*p == ',') {         // old versions
            del = p;
        } else if(*p == '|') {  // version 2.3
            tmp    = server;    // exchange
            server = client;
            client = tmp;
            del = p;
        }
    }

    if(!del) return(-1);

    client->size = p - (del + 1);
    server->size = del - data;

    if((client->size > 256) || (server->size > 256)) {
         return(-1);
    }

    client->pos  = 0;
    memcpy(client->key, del + 1, client->size);
    server->pos  = 0;
    memcpy(server->key, data,    server->size);

    return(0);
}



void ventrilo_first_dec(unsigned char *data, int size) {
    const static unsigned char  first[] = "\xAA\x55\x22\xCC\x69\x7C\x38\x91\x88\xF5\xE1";
    int     i;

    for(i = 0; i < size; i++) {
        *data -= first[i % 11] + (i % 27);
        data++;
    }
}



void ventrilo_first_enc(unsigned char *data, int size) {
    const static unsigned char  first[] = "\xAA\x55\x22\xCC\x69\x7C\x38\x91\x88\xF5\xE1";
    int     i;

    for(i = 0; i < size; i++) {
        *data += first[i % 11] + (i % 27);
        data++;
    }
}



void ventrilo_dec(ventrilo_key_ctx *ctx, unsigned char *data, int size) {
    int     i;

    for(i = 0; i < size; i++) {
        *data -= ctx->key[ctx->pos] + (i % 45);
        data++;
        ctx->pos++;
        if(ctx->pos == ctx->size) ctx->pos = 0;
    }
}



void ventrilo_enc(ventrilo_key_ctx *ctx, unsigned char *data, int size) {
    int     i;

    for(i = 0; i < size; i++) {
        *data += ctx->key[ctx->pos] + (i % 45);
        data++;
        ctx->pos++;
        if(ctx->pos == ctx->size) ctx->pos = 0;
    }
}


