/*
    Copyright 2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <ctype.h>
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define LOADDLL         hLib = LoadLibrary(fname);                          \
                            if(!hLib) winerr();
    #define GETFUNC(x,y)    x  = GetProcAddress(hLib, y);                       \
                            if(!x) winerr();
    #define CLOSEDLL        FreeLibrary(hLib);
    HINSTANCE   hLib;

    void winerr(void);
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/time.h>
    #include <dlfcn.h>      // -ldl

    #define WINAPI
    #define LOADDLL         char    *error;                                     \
                            hLib = dlopen(fname, RTLD_LAZY);                    \
                            if(!hLib) {                                         \
                                fprintf(stderr, "\nError: %s\n\n", dlerror());  \
                                exit(1);                                        \
                            }
    #define GETFUNC(x,y)    x  = dlsym(hLib, y);                                \
                            error = dlerror();                                  \
                            if(error || !x) {                                   \
                            fprintf(stderr, "\nError: %s\n\n", error);          \
                                exit(1);                                        \
                            }
    #define CLOSEDLL        dlclose(hLib);
    void        *hLib;
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.3a"
#define BUFFSZ      0xffff
#define DLLINIT     "sudp_init"
#define DLLPCK      "sudp_pck"
#define DLLVIS      "sudp_vis"
#define TIMEOUT     30

#ifndef IP_TOS
    #define IP_TOS 3
#endif



typedef int (WINAPI*_sudp_init)(u8 *);      // initialization
typedef int (WINAPI*_sudp_pck)(u8 *, int);  // modification of the packet
typedef int (WINAPI*_sudp_vis)(u8 *, int);  // modification for visualization only
_sudp_init  sudp_init;
_sudp_pck   sudp_pck;
_sudp_vis   sudp_vis;



FILE *create_acp(u8 *acpfile);
struct clients_struct *check_fd(struct sockaddr_in *peer);
void loaddll(u8 *fname, u8 *par);
int little_endian_num(u8 *data, uint64_t num, int bits);
void acp_dump(FILE *fd, u8 *data, int len, struct sockaddr_in *src, struct sockaddr_in *dst);
u16 in_cksum(void *data, int len);
u32 resolv(char *host);
void std_err(void);



struct clients_struct {
    int     sd;
    struct  sockaddr_in peer;
    time_t  timez;
    struct  clients_struct  *next;
} *clients;

int     quiet = 0;
const static int
        on  = 1,
        tos = 0x10;



int main(int argc, char *argv[]) {
    struct clients_struct   *c,
                            *tmpc;
    struct  sockaddr_in peer,
                        tmpeer,
                        peeri,
                        peerl,
                        *psrc,
                        *pdst;
    struct  in_addr ip;
    struct  timeval tout;
    FILE    *fd      = NULL;
    fd_set  readset;
    int     sdl,
            sdi      = 0,
            i,
            len      = 0,
            sel,
            psz,
            hex      = 0,
            chr,
            everyone = 0,
            priority = 0;
    u16     sport,
            lport,
            inject   = 0;
    u8      *buff,
            *acpfile = NULL,
            *dll     = NULL,
            *dllpar  = NULL;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "Simple UDP proxy/pipe "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 4) {
        printf("\n"
            "Usage: %s [options] <server> <server_port> <local_port>\n"
            "\n"
            "Options:\n"
            "-x       show the hex dump of each packet\n"
            "-a FILE  create an ACP (tcpdump) file in which storing all the packets\n"
            "-b IP    bind only the interface identified with IP\n"
            "-l LIB   load a dll/so which will be used to process all the incoming\n"
            "         packets. The library must contain the following functions:\n"
            "           int sudp_init(u8 *data);          // if you need initialization\n"
            "           int sudp_pck(u8 *data, int len);  // each packet goes here\n"
            "           int sudp_vis(u8 *data, int len);  // for visualization only\n"
            "-L PAR   parameter for the initialization of the above function\n"
            "         if the plugin library supports parameters use -L \"\" for the help\n"
            "-e       forward each packet to anyone (clients and server) except the sender,\n"
            "         it works just like a chat or a broadcaster\n"
            "-i PORT  injection options, listen on the port PORT and each packet received\n"
            "         here is sent to the server from all the connected clients\n"
            "-p       increase process priority\n"
            "-q       quiet output\n"
            "\n", argv[0]);
        exit(1);
    }

    ip.s_addr = INADDR_ANY;

    argc -= 3;
    for(i = 1; i < argc; i++) {
        if(argv[i][0] != '-') continue;
        switch(argv[i][1]) {
            case 'x': hex       = 1;                    break;
            case 'a': acpfile   = argv[++i];            break;
            case 'b': ip.s_addr = resolv(argv[++i]);    break;
            case 'l': dll       = argv[++i];            break;
            case 'L': dllpar    = argv[++i];            break;
            case 'e': everyone  = 1;                    break;
            case 'i': inject    = atoi(argv[++i]);      break;
            case 'p': priority  = 1;                    break;
            case 'q': quiet     = 1;                    break;
            default: {
                fprintf(stderr, "\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    sport = atoi(argv[argc + 1]);
    lport = atoi(argv[argc + 2]);

    peer.sin_addr.s_addr  = resolv(argv[argc]); /* connect to server */
    peer.sin_port         = htons(sport);
    peer.sin_family       = AF_INET;

    peerl.sin_addr.s_addr = ip.s_addr;          /* listen for client */
    peerl.sin_port        = htons(lport);
    peerl.sin_family      = AF_INET;

    if(!quiet) {
        printf(
            "- server:   %s : %hu\n"
            "- bind UDP port %hu\n",
            inet_ntoa(peer.sin_addr), sport,
            lport);
    }

    if(priority) {
    #ifdef WIN32
        SetPriorityClass(GetCurrentProcess(), HIGH_PRIORITY_CLASS);
    //    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_HIGHEST);
    #else
        nice(-10);
    #endif
    }

    if(dll) {
        if(!quiet) printf("- load library %s\n", dll);
        loaddll(dll, dllpar);
    }

    if(acpfile) {
        if(!quiet) printf("- create ACP file %s\n", acpfile);
        fd = fopen(acpfile, "rb");
        if(fd) {
            fclose(fd);
            fprintf(stderr, "- do you want to overwrite (Y) or append (A) the file? (y/a/N)\n  ");
            fflush(stdin);
            chr = tolower(fgetc(stdin));
            if(chr == 'a') {
                fd = fopen(acpfile, "ab");
                if(!fd) std_err();
            } else if(chr == 'y') {
                fd = create_acp(acpfile);
            } else {
                return(0);
            }
        } else {
            fd = create_acp(acpfile);
        }
    }

    psrc = &peerl;
    pdst = &peer;

    sdl = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peerl, sizeof(struct sockaddr_in))
      < 0) std_err();
    setsockopt(sdl, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(tos));

    if(inject) {
        peeri.sin_addr.s_addr = ip.s_addr;      /* listen for injection */
        peeri.sin_port        = htons(inject);
        peeri.sin_family      = AF_INET;

        sdi = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sdi < 0) std_err();
        if(setsockopt(sdi, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
          < 0) std_err();
        if(bind(sdi, (struct sockaddr *)&peeri, sizeof(struct sockaddr_in))
          < 0) std_err();
        setsockopt(sdi, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(tos));
    }

    printf("- ready\n");
    FD_ZERO(&readset);      // wait first client's packet, this is NEEDED!
    FD_SET(sdl, &readset);
    if(select(sdl + 1, &readset, NULL, NULL, NULL)
      < 0) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    clients = NULL;

    for(;;) {
        FD_ZERO(&readset);
        FD_SET(sdl, &readset);
        sel = sdl;
        if(inject) {
            FD_SET(sdi, &readset);
            if(sdi > sel) sel = sdi;
        }
        for(c = clients; c; c = c->next) {
            FD_SET(c->sd, &readset);
            if(c->sd > sel) sel = c->sd;
        }
        sel++;

        tout.tv_sec  = TIMEOUT;     // this is useful if we want to free memory
        tout.tv_usec = 0;           // ...rarely used but I think it's good here
        chr = select(sel, &readset, NULL, NULL, &tout);
        if(chr < 0) std_err();

        // select seems to be able to return more than one file descriptor but
        // in the real world this happens never (I have tested it many times)

        if(!chr) {
            memset(&tmpeer, 0, sizeof(struct sockaddr_in));
            check_fd(&tmpeer);
            continue;
        }
//        if(select(sel, &readset, NULL, NULL, NULL)    // no timeout method, good too
//          < 0) std_err();

        psz = sizeof(struct sockaddr_in);
        memcpy(&tmpeer, &peerl, sizeof(struct sockaddr_in));

        if(inject) {
            if(FD_ISSET(sdi, &readset)) {
                len = recvfrom(sdi, buff, BUFFSZ, 0, (struct sockaddr *)&tmpeer, &psz);
                if(len < 0) continue;
                if(!quiet) {
                    printf("- packet injection from %s:%hu (%d bytes)\n",
                        inet_ntoa(tmpeer.sin_addr), ntohs(tmpeer.sin_port), len);
                }
                if(hex) {
                    show_dump(buff, len, stdout);
                }
                for(c = clients; c; c = c->next) {
                    sendto(c->sd, buff, len, 0, (struct sockaddr *)&peer, sizeof(struct sockaddr_in));
                    sleep(0);   // avoids packets congestion?
                }
                continue;
            }
        }

            /* CLIENTS */
        if(FD_ISSET(sdl, &readset)) {

                // this is the portal where all the clients arrive
            len = recvfrom(sdl, buff, BUFFSZ, 0, (struct sockaddr *)&tmpeer, &psz);
            if(len < 0) continue;

                // here we check if this is a new or existent client
            c = check_fd(&tmpeer);
            if(!c) continue;

            psrc = &c->peer;
            pdst = &peer;
            if(dll) {
                len = sudp_pck(buff, len);      // packets modification
            }

                // sends from the client's socket to the server (peer)
            sendto(c->sd, buff, len, 0, (struct sockaddr *)&peer, sizeof(struct sockaddr_in));

            if(everyone) {
                tmpc = c;
                for(c = clients; c; c = c->next) {
                    if(c == tmpc) continue;
                    sendto(sdl, buff, len, 0, (struct sockaddr *)&c->peer, sizeof(struct sockaddr_in));
                    sleep(0);   // avoids packets congestion?
                }
            }

        } else {

            /* SERVER */
            for(c = clients; c; c = c->next) {
                if(FD_ISSET(c->sd, &readset)) {

                        // this is the port where the server sends the data for a specific client
                    len = recvfrom(c->sd, buff, BUFFSZ, 0, (struct sockaddr *)&tmpeer, &psz);
                    if(len < 0) continue;

                    // anti server packet's injection, actually disabled for possible compatibility reasons
                /*
                    if((tmpeer.sin_addr.s_addr != peer.sin_addr.s_addr) || (tmpeer.sin_port != peer.sin_port)) {
                        if(!quiet) {
                            printf("- possible packet injection from %s:%hu\n",
                                inet_ntoa(tmpeer.sin_addr), ntohs(tmpeer.sin_port));
                        }
                        continue;
                    }
                */

                    psrc = &peer;
                    pdst = &c->peer;
                    if(dll) {
                        len = sudp_pck(buff, len);      // packets modification
                    }

                        // send from the receiving port to the specific client
                    sendto(sdl, buff, len, 0, (struct sockaddr *)&c->peer, sizeof(struct sockaddr_in));

                    if(everyone) {
                        tmpc = c;
                        for(c = clients; c; c = c->next) {
                            if(c == tmpc) continue;
                            sendto(sdl, buff, len, 0, (struct sockaddr *)&c->peer, sizeof(struct sockaddr_in));
                            sleep(0);   // avoids packets congestion?
                        }
                    }

                    break;
                }
            }
        }

        if(acpfile) acp_dump(fd, buff, len, psrc, pdst);

        if(dll)     len = sudp_vis(buff, len);

        if(hex) {
            if(!quiet) {
                printf("\n%s:%hu -> ",
                    inet_ntoa(psrc->sin_addr), ntohs(psrc->sin_port));
                printf("%s:%hu\n",
                    inet_ntoa(pdst->sin_addr), ntohs(pdst->sin_port));
            }
            show_dump(buff, len, stdout);
        }
    }

    close(sdl);
    if(inject)  close(sdi);
    if(acpfile) fclose(fd);
    if(dll) {
        CLOSEDLL;
    }
    free(buff);
    return(0);
}



FILE *create_acp(u8 *acpfile) {
    FILE    *fd;
    struct {
        u32     magic;
        u16     version_major;
        u16     version_minor;
        u32     thiszone;
        u32     sigfigs;
        u32     snaplen;
        u32     linktype;
    } acp_head;

    fd = fopen(acpfile, "wb");
    if(!fd) std_err();
    little_endian_num((u8 *)&acp_head.magic,         0xa1b2c3d4, 32);
    little_endian_num((u8 *)&acp_head.version_major, 2,          16);
    little_endian_num((u8 *)&acp_head.version_minor, 4,          16);
    little_endian_num((u8 *)&acp_head.thiszone,      0,          32);
    little_endian_num((u8 *)&acp_head.sigfigs,       0,          32);
    little_endian_num((u8 *)&acp_head.snaplen,       65535,      32);
    little_endian_num((u8 *)&acp_head.linktype,      1,          32);
    fwrite(&acp_head, sizeof(acp_head), 1, fd);
    fflush(fd);
    return(fd);
}



struct clients_struct *check_fd(struct sockaddr_in *peer) {
    struct clients_struct   *c,
                            *tmp,
                            *prev,
                            *ret;
    time_t  curr;
    int     sd;

    curr = time(NULL);
    prev = NULL;
    ret  = NULL;

    for(c = clients; c; ) {
        if((c->peer.sin_addr.s_addr == peer->sin_addr.s_addr) && (c->peer.sin_port == peer->sin_port)) {
            c->timez = curr;
            ret  = c;
            prev = c;
            c    = c->next;

        } else if((curr - c->timez) >= TIMEOUT) {
            if(!quiet) {
                printf("- remove %s:%hu\n",
                    inet_ntoa(c->peer.sin_addr), ntohs(c->peer.sin_port));
            }

            tmp = c->next;
            free(c);
            if(prev) {      // second, third and so on
                prev->next = tmp;
            } else {        // the first only
                clients    = tmp;
            }
            c = tmp;

        } else {
            prev = c;
            c    = c->next;
        }
    }

    if(ret) return(ret);
    if(!peer->sin_addr.s_addr || !peer->sin_port) return(NULL);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) return(NULL);

        // totally useless...
    setsockopt(sd, SOL_SOCKET, SO_BROADCAST, (char *)&on,  sizeof(on));
    setsockopt(sd, IPPROTO_IP, IP_TOS,       (char *)&tos, sizeof(tos));

    c = malloc(sizeof(struct clients_struct));
    if(!c) return(NULL);

    if(prev) {
        prev->next = c;
    } else {
        clients    = c;
    }

    c->sd    = sd;
    memcpy(&c->peer, peer, sizeof(struct sockaddr_in));
    c->timez = curr;
    c->next  = NULL;

    if(!quiet) {
        printf("- add %s:%hu\n",
            inet_ntoa(c->peer.sin_addr), ntohs(c->peer.sin_port));
    }

    return(c);
}




void loaddll(u8 *fname, u8 *par) {
    LOADDLL;

    GETFUNC(sudp_init, DLLINIT);

    GETFUNC(sudp_pck,  DLLPCK);

    GETFUNC(sudp_vis,  DLLVIS);

    if(sudp_init(par)) {
//        printf("\nError: plugin initialization failed\n\n");
        CLOSEDLL
        exit(1);
    }
}



int little_endian_num(u8 *data, uint64_t num, int bits) {
    u8      *p;

    p = data;
    *p++ = num & 0xff;
    if(bits >= 16) {
        *p++ = (num >>  8) & 0xff;
    }
    if(bits >= 32) {
        *p++ = (num >> 16) & 0xff;
        *p++ = (num >> 24) & 0xff;
    }
    if(bits >= 64) {
        *p++ = (num >> 32) & 0xff;
        *p++ = (num >> 40) & 0xff;
        *p++ = (num >> 48) & 0xff;
        *p++ = (num >> 56) & 0xff;
    }
    return(p - data);
}



void acp_dump(FILE *fd, u8 *data, int len, struct sockaddr_in *src, struct sockaddr_in *dst) {
    u8      *pseudobuff;
    const static u8 ethdata[14] =
                    "\x00\x00\x00\x00\x00\x00"  /* dest */
                    "\x00\x00\x00\x00\x00\x00"  /* source */
                    "\x08\x00";                 /* type */
    struct {
        struct timeval  ts;
        u32     caplen;
        u32     len;
    } acp_pck;
    struct {
        u8      ihl_version;
        u8      tos;
        u16     tot_len;
        u16     id;
        u16     frag_off;
        u8      ttl;
        u8      protocol;
        u16     check;
        u32     saddr;
        u32     daddr;
    } ip;
    struct {
        u16     source;
        u16     dest;
        u16     len;
        u16     check;
    } udp;
    struct pseudohdr {
        u32     saddr;
        u32     daddr;
        u8      zero;
        u8      protocol;
        u16     length;
    } *pseudo;

#define IPSZ        sizeof(ip)
#define UDPSZ       sizeof(udp)
#define PSEUDOSZ    sizeof(struct pseudohdr)
#define SIZE        (IPSZ + UDPSZ + len)
#define PSSIZE      (PSEUDOSZ + UDPSZ + len)

#ifdef WIN32
    little_endian_num((u8 *)&acp_pck.ts.tv_sec,  time(NULL),     32);
    little_endian_num((u8 *)&acp_pck.ts.tv_usec, GetTickCount(), 32);
#else
    gettimeofday(&(acp_pck.ts), NULL);
    little_endian_num((u8 *)&acp_pck.ts.tv_sec,  acp_pck.ts.tv_sec,  32);
    little_endian_num((u8 *)&acp_pck.ts.tv_usec, acp_pck.ts.tv_usec, 32);
#endif
    little_endian_num((u8 *)&acp_pck.caplen, sizeof(ethdata) + SIZE, 32);
    little_endian_num((u8 *)&acp_pck.len,    sizeof(ethdata) + SIZE, 32);

    ip.ihl_version   = 0x45;
    ip.tos           = 0;
    ip.tot_len       = htons(SIZE);
    ip.id            = htons(1);
    ip.frag_off      = htons(0);
    ip.ttl           = 128;
    ip.protocol      = IPPROTO_UDP;
    ip.check         = 0;
    ip.saddr         = src->sin_addr.s_addr;
    ip.daddr         = dst->sin_addr.s_addr;
    ip.check         = htons(in_cksum((u8 *)&ip, IPSZ));

    udp.source       = src->sin_port;
    udp.dest         = dst->sin_port;
    udp.check        = 0;
    udp.len          = htons(UDPSZ + len);

    pseudobuff       = malloc(PSSIZE);
    if(!pseudobuff) std_err();
    pseudo           = (struct pseudohdr *)pseudobuff;

    pseudo->saddr    = ip.saddr;
    pseudo->daddr    = ip.daddr;
    pseudo->zero     = 0;
    pseudo->protocol = IPPROTO_UDP;
    pseudo->length   = udp.len;

    memcpy(pseudobuff + PSEUDOSZ, &udp, UDPSZ);
    memcpy(pseudobuff + PSEUDOSZ + UDPSZ, data, len);
    udp.check        = htons(in_cksum(pseudobuff, PSSIZE));

    free(pseudobuff);

    fwrite(&acp_pck, sizeof(acp_pck), 1, fd);
    fwrite(ethdata,  sizeof(ethdata), 1, fd);
    fwrite(&ip,      IPSZ,            1, fd);
    fwrite(&udp,     UDPSZ,           1, fd);
    fwrite(data,     len,             1, fd);
    fflush(fd);

#undef IPSZ
#undef UDPSZ
#undef PSEUDOSZ
#undef SIZE
#undef PSSIZE
}



u16 in_cksum(void *data, int len) {
    u32     sum    = 0;
    int     i      = len >> 1,
            endian = 1; // big endian
    u16     crc,
            *p     = (u16 *)data;

    if(*(char *)&endian) endian = 0;
    while(i--) sum += *p++;
    if(len & 1) sum += *p & (endian ? 0xff00 : 0xff);
    crc = sum = (sum >> 16) + (sum & 0xffff);
    if(sum >>= 16) crc += sum;
    if(!endian) crc = (crc >> 8) | (crc << 8);
    return(~crc);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#else
void winerr(void) {
    char *error;

    if(!FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
        NULL,
        GetLastError(),
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
        (LPTSTR)&error,
        0,
        NULL)) {
        error = strerror(errno);
    }
    printf("\nError: %s\n", error);
    LocalFree(error);
    exit(1);
}
#endif


