/*
    Copyright 2006-2011 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifndef MYDOWN_GLOBAL_COOKIE
    #error "you must use -DMYDOWN_GLOBAL_COOKIE"    // optional (needed to me for remembering it)
#endif
#ifndef MYDOWN_SSL
    #error "you must use -DMYDOWN_SSL"  // optional (needed to me for remembering it)
#endif
#include "mydownlib.h"

#ifdef WIN32
    #include <windows.h>

    #define sleep       Sleep
    #define sleepms     sleep
    #define ONESEC      1000
#else
    #include <pthread.h>    // gcc -o mydown mydown.c mydownlib.c -lz -lpthread

    #define sleepms(x)  usleep(x * 1000)
    #define ONESEC      1
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;
typedef uint64_t    u64;



#define VER         "0.2.7a"
#define MAXERR      8
#define LOOPMS      500



typedef struct {
    int     num;
    char    *host;
    char    *filename;
    u32     from;
    u32     tot;
    mydown_options opt;
    int     done;
    u32     size;
} downthr_t;



quick_thread(client, downthr_t *down);
char *get_http_info(char *host, char *filename, int *filesize, mydown_options *opt);
void do_multidown(char *host, char *filename, int fsize, mydown_options *opt, int multinum, u8 *mirrors);
int get_num(char *str);
void std_err(void);



int main(int argc, char *argv[]) {
    mydown_options  opt;
    u32     filesize;
    int     i,
            multidown   = 0;
    char    *filename   = NULL,
            *multidowns = NULL,
            *host;

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "Mydown "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        fprintf(stderr,
            "\n"
            "Usage: %s [options] <URL>\n"
            "\n"
            "Options:\n"
            "-m NUM   enable multi-parts downloading of NUM pieces of the file at time\n"
            "         this option increases the downloading speed with big files\n"
            "-M URLs  function similar to the -m one but allows to download the file from\n"
            "         multiple locations/mirrors. URLs is the sequence of links from which\n"
            "         downloading the file separated by comma like the following example:\n"
            "         -M http://localhost/file,http://1.2.3.4/file/file,example.com/file\n"
            "-o FILE  force output filename, use - for stdout\n"
            "-f FROM  start download from offset FROM\n"
            "-t TOT   download max TOT bytes, if you use also -f -1 the downloading will\n"
            "         start TOT bytes from the end of the file)\n"
            "-r REF   use a specific Referer\n"
            "-u UA    use a specific User-Agent\n"
            "-c CK    use a specific cookie\n"
            "-s STR   add a new HTTP parameter (eg: \"Accept: */*\")\n"
            "-H STR   custom HTTP command like GET, POST, HEAD and so on\n"
            "-P STR   custom data to send as Content/Content-Length\n"
            "-U U P   user password authentication\n"
            "-a       auto resume based on current file size (does NOT work with -m/M)\n"
            "-e       shows the HTTP headers and exits\n"
            "-z       force the on-fly decompression of any compressed file\n"
            "-Z       disable the on-fly decompression despite the Content-Encoding (debug)\n"
            "-q       quiet output\n"
            "-v       verbose output\n"
            "\n", argv[0]);
        exit(1);
    }

    memset(&opt, 0, sizeof(opt));
    opt.verbose = 1;

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            fprintf(stderr, "\nError: recheck your options, %s is not valid\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'm': multidown         = get_num(argv[++i]);   break;
            case 'M': multidowns        = argv[++i];            break;
            case 'o': filename          = argv[++i];            break;
            case 'f': opt.from          = get_num(argv[++i]);   break;
            case 't': opt.tot           = get_num(argv[++i]);   break;
            case 'r': opt.referer       = argv[++i];            break;
            case 'u': opt.useragent     = argv[++i];            break;
            case 'c': opt.cookie        = argv[++i];            break;
            case 's': opt.more_http     = argv[++i];            break;
            case 'H': opt.get           = argv[++i];            break;
            case 'P': opt.content       = argv[++i];            break;
            case 'U': { opt.user = argv[++i]; opt.pass = argv[++i]; break; }
            case 'a': opt.resume        = 1;                    break;
            case 'e': opt.showhead      = 1;                    break;
            case 'z': opt.onflyunzip    = 1;                    break;
            case 'Z': opt.onflyunzip    = -1;                   break;
            case 'q': opt.verbose       = -1;                   break;
            case 'v': opt.verbose       = 2;                    break;
            case 'V': opt.verbose       = 3;                    break;
            default: {
                fprintf(stderr, "\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    host = argv[argc];

    filename = get_http_info(host, filename, &filesize, &opt);
    if(!filename) exit(1);

    if(multidown > 1) {
        do_multidown(host, filename, filesize, &opt, multidown, NULL);
    } else if(multidowns) {
        do_multidown(host, filename, filesize, &opt, -1, multidowns);
    } else {
        mydown(host, filename, &opt);
    }

    fprintf(stderr, "\n- done\n");
    return(0);
}



quick_thread(client, downthr_t *down) {
    FILE    *fd;
    int     ret,
            maxerr = 0;

    fprintf(stderr, "- thread %d started\n", down->num);
    fd = fopen(down->filename, "r+b");
    if(!fd) std_err();

    down->opt.resume   = 3;
    down->opt.ret_code = &down->size;

    for(;;) {
        down->opt.from = down->from + down->size;
        down->opt.tot  = down->tot  - down->size;
        if(!down->opt.tot) break;
        if((int)down->opt.tot < 0) exit(1);
        if(fseek(fd, down->opt.from, SEEK_SET)) std_err();

        ret = mydown(down->host, (void *)fd, &down->opt);
        if(ret != MYDOWN_ERROR) break;

        fprintf(stderr, "\n- an error during the downloading of the file through thread %d, continue\n", down->num);
        if(++maxerr > MAXERR) {
            fprintf(stderr, "\n"
                "Error: the maximum number of errors per download (thread %d) has been reached\n"
                "       retry using a lower -m value\n", down->num);
            exit(1);
        }
        sleep(ONESEC);
    }

    fclose(fd);
    down->done = 1;
    return(0);
}



int delimit(u8 *data) {
    u8      *p;

    for(p = data; *p && (*p != '\r') && (*p != '\n'); p++);
    *p = 0;
    return(p - data);
}



char *get_http_info(char *host, char *filename, int *filesize, mydown_options *opt) {
    FILE    *fd;
    int     old_showhead,
            fsize;
    char    ans[16];

    old_showhead = opt->showhead;
    if(!filename) opt->filedata = (void *)&filename;
    opt->showhead = 2;
    fsize = mydown(host, filename, opt);
    if(fsize == MYDOWN_ERROR) exit(1);
    opt->showhead = old_showhead;
    opt->filedata = NULL;

    if(filename) {
        delimit(filename);
    } else {
        filename = "noname.dat";    // don't pass it to delimit, it's a constant
    }

    fprintf(stderr, 
        "- file name   %s\n"
        "- file size   %u\n"
        "\n", filename, fsize);

    fd = fopen(filename, "rb");
    if(fd) {
        fclose(fd);
        fprintf(stderr, 
            "- the output file (%s) already exists\n"
            "  do you want to overwrite it (y/N)? ", filename);
        fgets(ans, sizeof(ans), stdin);
        if(tolower(ans[0]) != 'y') exit(1);
    }

    *filesize = fsize;
    return(filename);
}



u8 **get_mirrors(u8 *host, u8 *mirrors, int *multinum) {
    int     i;
    u8      **ret,
            *p,
            *l;

    *multinum = 1;
    for(p = mirrors;; p = l + 1) {
        l = strchr(p, ',');
        if(!l || (l[1] && (l[1] != ','))) {
            (*multinum)++;
        }
        if(!l) break;
    }

    ret = calloc(*multinum, sizeof(u8 *));
    if(!ret) std_err();

    i = 0;
    ret[i] = host;
    i++;

    for(p = mirrors; i < *multinum; p = l + 1) {
        l = strchr(p, ',');
        if(l) *l = 0;
        printf("- mirror %d: %s\n", i, p);
        ret[i] = p;
        i++;
    }
    return(ret);
}



void do_multidown(char *host, char *filename, int fsize, mydown_options *opt, int multinum, u8 *mirrors) {
    downthr_t *parts;
    FILE    *fd;
    u32     totsize,
            psize,
            oldsize;
    int     i,
            done;
    u8      **mirror    = NULL;

    if(mirrors) {
        mirror = get_mirrors(host, mirrors, &multinum);
    }

    psize = fsize / multinum;
    if((fsize <= 0) || !psize) {
        fprintf(stderr, "- the file is too small for multi downloading, I try with normal download\n");
        if(mydown(host, filename, opt) == MYDOWN_ERROR) exit(1);
        return;
    }

    fd = fopen(filename, "wb");
    if(!fd) std_err();
    fclose(fd);

    parts = malloc(sizeof(* parts) * multinum);
    if(!parts) std_err();

    opt->verbose = -1;

    for(i = 0; i < multinum; i++) {
        parts[i].num        = i + 1;
        if(mirrors) {
            parts[i].host   = mirror[i];
        } else {
            parts[i].host   = host;
        }
        parts[i].filename   = filename;
        if(!i) {
            parts[i].from   = 0;
            parts[i].tot    = psize + (fsize % multinum);
        } else {
            parts[i].from   = parts[i - 1].from + parts[i - 1].tot;
            parts[i].tot    = psize;
        }
        memcpy(&parts[i].opt, opt, sizeof(mydown_options));
        parts[i].done       = 0;
        parts[i].size       = 0;

        quick_threadx(client, &parts[i]);
        sleepms(200);
    }

    done = 0;
    oldsize = 0;
    while(done < multinum) {
        totsize = 0;
        for(i = 0; i < multinum; i++) {
            totsize += parts[i].size;
            if(parts[i].done == 1) {
                fprintf(stderr, "- thread %d completed\n", parts[i].num);
                parts[i].done = 2;
                done++;
            }
        }
        fprintf(stderr, "  %03d%% %8u kb/s\r", (u32)(((u64)totsize * (u64)100) / (u64)fsize), (u32)((((u64)(totsize - oldsize) * (u64)1000) / LOOPMS) / 1024));
        oldsize = totsize;
        sleepms(LOOPMS);
    }
}



int get_num(char *str) {
    int     num;

    if((str[0] == '0') && (tolower(str[1]) == 'x')) {
        sscanf(str + 2, "%x", &num);
    } else {
        sscanf(str, "%u", &num);
    }
    return(num);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


