﻿using System.Collections.Generic;
using System.Management;
using Seatbelt.Output.TextWriters;
using Seatbelt.Output.Formatters;
using System.Linq;
using Seatbelt.Util;


namespace Seatbelt.Commands.Windows
{
    internal class InterestingProcessesCommand : CommandBase
    {
        public override string Command => "InterestingProcesses";
        public override string Description => "\"Interesting\" processes - defensive products and admin tools";
        public override CommandGroup[] Group => new[] { CommandGroup.System, CommandGroup.Remote };
        public override bool SupportRemote => true;
        public Runtime ThisRunTime;

        public InterestingProcessesCommand(Runtime runtime) : base(runtime)
        {
            ThisRunTime = runtime;
        }

        public override IEnumerable<CommandDTOBase?> Execute(string[] args)
        {
            // TODO: check out https://github.com/harleyQu1nn/AggressorScripts/blob/master/ProcessColor.cna#L10

            // from https://github.com/threatexpress/red-team-scripts/blob/master/HostEnum.ps1#L985-L1033
            var defensiveProcesses = new Dictionary<string, string>()
            {
                { "mcshield", "McAfee"},
                { "windefend", "Windows Defender AV"},
                { "msascui", "Windows Defender AV"},
                { "msascuil", "Windows Defender AV"},
                { "msmpeng", "Windows Defender AV"},
                { "msmpsvc", "Windows Defender AV"},
                { "wrsa", "WebRoot AV"},
                { "savservice", "Sophos AV"},
                { "tmccsf", "Trend Micro AV"},
                { "symantec antivirus", "Symantec"},
                { "aexnsagent", "Symantec"},
                { "ccsvchst", "Symantec"},
                { "sisidsservice", "Symantec IDS"},
                { "sisipsservice", "Symantec IPS"},
                { "sisipsutil", "Symantec IPS"},
                { "kvoop", "Symantec DLP"},
                { "brkrprcs64", "Symantec DLP"},
                { "edpa", "Symantec DLP"},
                { "mbae", "MalwareBytes Anti-Exploit"},
                { "parity", "Bit9 application whitelisting"},
                { "cb", "Carbon Black behavioral analysis"},
                { "repux", "Carbon Black Defense"},
                { "repmgr", "Carbon Black Defense"},
                { "reputils", "Carbon Black Defense"},
                { "repwsc", "Carbon Black Defense"},
                { "cylancesvc", "Cylance Protect"},
                { "cylanceui", "Cylance Protect"},
                { "bds-vision", "BDS Vision behavioral analysis"},
                { "triumfant", "Triumfant behavioral analysis"},
                { "csfalcon", "CrowdStrike Falcon EDR"},
                { "csfalconcontainer", "CrowdStrike Falcon EDR"},
                { "csfalconservice", "CrowdStrike Falcon EDR"},
                { "ossec", "OSSEC intrusion detection"},
                { "tmpfw", "Trend Micro firewall"},
                { "dgagent", "Verdasys Digital Guardian DLP"},
                { "aawtray", "UNKNOWN"},
                { "ackwin32", "UNKNOWN"},
                { "ad-aware", "UNKNOWN"},
                { "adaware", "UNKNOWN"},
                { "advxdwin", "UNKNOWN"},
                { "agentsvr", "UNKNOWN"},
                { "agentw", "UNKNOWN"},
                { "alertsvc", "UNKNOWN"},
                { "alevir", "UNKNOWN"},
                { "alogserv", "UNKNOWN"},
                { "amon9x", "UNKNOWN"},
                { "anti-trojan", "UNKNOWN"},
                { "antivirus", "UNKNOWN"},
                { "ants", "UNKNOWN"},
                { "apimonitor", "UNKNOWN"},
                { "aplica32", "UNKNOWN"},
                { "apvxdwin", "UNKNOWN"},
                { "arr", "UNKNOWN"},
                { "atcon", "UNKNOWN"},
                { "atguard", "UNKNOWN"},
                { "atro55en", "UNKNOWN"},
                { "atupdater", "UNKNOWN"},
                { "atwatch", "UNKNOWN"},
                { "au", "UNKNOWN"},
                { "aupdate", "UNKNOWN"},
                { "auto-protect.nav80try", "UNKNOWN"},
                { "autodown", "UNKNOWN"},
                { "autoruns", "UNKNOWN"},
                { "autorunsc", "UNKNOWN"},
                { "autotrace", "UNKNOWN"},
                { "autoupdate", "UNKNOWN"},
                { "avconsol", "UNKNOWN"},
                { "ave32", "UNKNOWN"},
                { "avgcc32", "UNKNOWN"},
                { "avgctrl", "UNKNOWN"},
                { "avgemc", "UNKNOWN"},
                { "avgnt", "UNKNOWN"},
                { "avgrsx", "UNKNOWN"},
                { "avgserv", "UNKNOWN"},
                { "avgserv9", "UNKNOWN"},
                { "avguard", "UNKNOWN"},
                { "avgwdsvc", "UNKNOWN"},
                { "avgui", "UNKNOWN"},
                { "avgw", "UNKNOWN"},
                { "avkpop", "UNKNOWN"},
                { "avkserv", "UNKNOWN"},
                { "avkservice", "UNKNOWN"},
                { "avkwctl9", "UNKNOWN"},
                { "avltmain", "UNKNOWN"},
                { "avnt", "UNKNOWN"},
                { "avp", "UNKNOWN"},
                { "avp32", "UNKNOWN"},
                { "avpcc", "UNKNOWN"},
                { "avpdos32", "UNKNOWN"},
                { "avpm", "UNKNOWN"},
                { "avptc32", "UNKNOWN"},
                { "avpupd", "UNKNOWN"},
                { "avsched32", "UNKNOWN"},
                { "avsynmgr", "UNKNOWN"},
                { "avwin", "UNKNOWN"},
                { "avwin95", "UNKNOWN"},
                { "avwinnt", "UNKNOWN"},
                { "avwupd", "UNKNOWN"},
                { "avwupd32", "UNKNOWN"},
                { "avwupsrv", "UNKNOWN"},
                { "avxmonitor9x", "UNKNOWN"},
                { "avxmonitornt", "UNKNOWN"},
                { "avxquar", "UNKNOWN"},
                { "backweb", "UNKNOWN"},
                { "bargains", "UNKNOWN"},
                { "bd_professional", "UNKNOWN"},
                { "beagle", "UNKNOWN"},
                { "belt", "UNKNOWN"},
                { "besclient", "IBM BigFix"},
                { "bidef", "UNKNOWN"},
                { "bidserver", "UNKNOWN"},
                { "bipcp", "UNKNOWN"},
                { "bipcpevalsetup", "UNKNOWN"},
                { "bisp", "UNKNOWN"},
                { "blackd", "UNKNOWN"},
                { "blackice", "UNKNOWN"},
                { "blink", "UNKNOWN"},
                { "blss", "UNKNOWN"},
                { "bootconf", "UNKNOWN"},
                { "bootwarn", "UNKNOWN"},
                { "borg2", "UNKNOWN"},
                { "bpc", "UNKNOWN"},
                { "brasil", "UNKNOWN"},
                { "bs120", "UNKNOWN"},
                { "bundle", "UNKNOWN"},
                { "bvt", "UNKNOWN"},
                { "ccapp", "UNKNOWN"},
                { "ccevtmgr", "UNKNOWN"},
                { "ccpxysvc", "UNKNOWN"},
                { "cdp", "UNKNOWN"},
                { "cfd", "UNKNOWN"},
                { "cfgwiz", "UNKNOWN"},
                { "cfiadmin", "UNKNOWN"},
                { "cfiaudit", "UNKNOWN"},
                { "cfinet", "UNKNOWN"},
                { "cfinet32", "UNKNOWN"},
                { "claw95", "UNKNOWN"},
                { "claw95cf", "UNKNOWN"},
                { "clean", "UNKNOWN"},
                { "cleaner", "UNKNOWN"},
                { "cleaner3", "UNKNOWN"},
                { "cleanpc", "UNKNOWN"},
                { "cleanup", "UNKNOWN"},
                { "click", "UNKNOWN"},
                { "cmdagent", "UNKNOWN"},
                { "cmesys", "UNKNOWN"},
                { "cmgrdian", "UNKNOWN"},
                { "cmon016", "UNKNOWN"},
                { "connectionmonitor", "UNKNOWN"},
                { "cpd", "UNKNOWN"},
                { "cpf9x206", "UNKNOWN"},
                { "cpfnt206", "UNKNOWN"},
                { "ctrl", "UNKNOWN"},
                { "cv", "UNKNOWN"},
                { "cwnb181", "UNKNOWN"},
                { "cwntdwmo", "UNKNOWN"},
                { "cyprotect", "UNKNOWN"},
                { "cyupdate", "UNKNOWN"},
                { "cyserver", "UNKNOWN"},
                { "cytray", "UNKNOWN"},
                { "cyveraservice", "UNKNOWN"},
                { "datemanager", "UNKNOWN"},
                { "dcomx", "UNKNOWN"},
                { "defalert", "UNKNOWN"},
                { "defscangui", "UNKNOWN"},
                { "defwatch", "UNKNOWN"},
                { "deputy", "UNKNOWN"},
                { "divx", "UNKNOWN"},
                { "dgprompt", "UNKNOWN"},
                { "dgservice", "UNKNOWN"},
                { "dllcache", "UNKNOWN"},
                { "dllreg", "UNKNOWN"},
                { "doors", "UNKNOWN"},
                { "dpf", "UNKNOWN"},
                { "dpfsetup", "UNKNOWN"},
                { "dpps2", "UNKNOWN"},
                { "drwatson", "UNKNOWN"},
                { "drweb32", "UNKNOWN"},
                { "drwebupw", "UNKNOWN"},
                { "dssagent", "UNKNOWN"},
                { "dumpcap", "UNKNOWN"},
                { "dvp95", "UNKNOWN"},
                { "dvp95_0", "UNKNOWN"},
                { "ecengine", "UNKNOWN"},
                { "efpeadm", "UNKNOWN"},
                { "egui", "UNKNOWN"},
                { "ekrn", "UNKNOWN"},
                { "enstart64", "Encase (Forensics tool)"},
                { "emet_agent", "UNKNOWN"},
                { "emet_service", "UNKNOWN"},
                { "emsw", "UNKNOWN"},
                { "engineserver", "UNKNOWN"},
                { "ent", "UNKNOWN"},
                { "esafe", "UNKNOWN"},
                { "escanhnt", "UNKNOWN"},
                { "escanv95", "UNKNOWN"},
                { "espwatch", "UNKNOWN"},
                { "ethereal", "UNKNOWN"},
                { "etrustcipe", "UNKNOWN"},
                { "evpn", "UNKNOWN"},
                { "exantivirus-cnet", "UNKNOWN"},
                { "exe.avxw", "UNKNOWN"},
                { "expert", "UNKNOWN"},
                { "explore", "UNKNOWN"},
                { "f-agnt95", "UNKNOWN"},
                { "f-prot", "UNKNOWN"},
                { "f-prot95", "UNKNOWN"},
                { "f-stopw", "UNKNOWN"},
                { "fameh32", "UNKNOWN"},
                { "fast", "UNKNOWN"},
                { "fch32", "UNKNOWN"},
                { "fcagswd", "McAfee DLP Agent"},
                { "fcags", "McAfee DLP Agent"},
                { "fih32", "UNKNOWN"},
                { "findviru", "UNKNOWN"},
                { "firesvc", "McAfee Host Intrusion Prevention"},
                { "firetray", "UNKNOWN"},
                { "firewall", "UNKNOWN"},
                { "fnrb32", "UNKNOWN"},
                { "fp-win", "UNKNOWN"},
                { "fp-win_trial", "UNKNOWN"},
                { "fprot", "UNKNOWN"},
                { "frameworkservice", "UNKNOWN"},
                { "frminst", "UNKNOWN"},
                { "frw", "UNKNOWN"},
                { "fsaa", "UNKNOWN"},
                { "fsav", "UNKNOWN"},
                { "fsav32", "UNKNOWN"},
                { "fsav530stbyb", "UNKNOWN"},
                { "fsav530wtbyb", "UNKNOWN"},
                { "fsav95", "UNKNOWN"},
                { "fsgk32", "UNKNOWN"},
                { "fsm32", "UNKNOWN"},
                { "fsma32", "UNKNOWN"},
                { "fsmb32", "UNKNOWN"},
                { "gator", "UNKNOWN"},
                { "gbmenu", "UNKNOWN"},
                { "gbpoll", "UNKNOWN"},
                { "generics", "UNKNOWN"},
                { "gmt", "UNKNOWN"},
                { "guard", "UNKNOWN"},
                { "guarddog", "UNKNOWN"},
                { "hacktracersetup", "UNKNOWN"},
                { "hbinst", "UNKNOWN"},
                { "hbsrv", "UNKNOWN"},
                { "hijackthis", "UNKNOWN"},
                { "hipsvc", "UNKNOWN"},
                { "hipmgmt", "McAfee Host Intrusion Protection"},
                { "hotactio", "UNKNOWN"},
                { "hotpatch", "UNKNOWN"},
                { "htlog", "UNKNOWN"},
                { "htpatch", "UNKNOWN"},
                { "hwpe", "UNKNOWN"},
                { "hxdl", "UNKNOWN"},
                { "hxiul", "UNKNOWN"},
                { "iamapp", "UNKNOWN"},
                { "iamserv", "UNKNOWN"},
                { "iamstats", "UNKNOWN"},
                { "ibmasn", "UNKNOWN"},
                { "ibmavsp", "UNKNOWN"},
                { "icload95", "UNKNOWN"},
                { "icloadnt", "UNKNOWN"},
                { "icmon", "UNKNOWN"},
                { "icsupp95", "UNKNOWN"},
                { "icsuppnt", "UNKNOWN"},
                { "idle", "UNKNOWN"},
                { "iedll", "UNKNOWN"},
                { "iedriver", "UNKNOWN"},
                { "iface", "UNKNOWN"},
                { "ifw2000", "UNKNOWN"},
                { "inetlnfo", "UNKNOWN"},
                { "infus", "UNKNOWN"},
                { "infwin", "UNKNOWN"},
                { "init", "UNKNOWN"},
                { "intdel", "UNKNOWN"},
                { "intren", "UNKNOWN"},
                { "iomon98", "UNKNOWN"},
                { "istsvc", "UNKNOWN"},
                { "jammer", "UNKNOWN"},
                { "jdbgmrg", "UNKNOWN"},
                { "jedi", "UNKNOWN"},
                { "kavlite40eng", "UNKNOWN"},
                { "kavpers40eng", "UNKNOWN"},
                { "kavpf", "UNKNOWN"},
                { "kazza", "UNKNOWN"},
                { "keenvalue", "UNKNOWN"},
                { "kerio-pf-213-en-win", "UNKNOWN"},
                { "kerio-wrl-421-en-win", "UNKNOWN"},
                { "kerio-wrp-421-en-win", "UNKNOWN"},
                { "kernel32", "UNKNOWN"},
                { "keypass", "UNKNOWN"},
                { "killprocesssetup161", "UNKNOWN"},
                { "launcher", "UNKNOWN"},
                { "ldnetmon", "UNKNOWN"},
                { "ldpro", "UNKNOWN"},
                { "ldpromenu", "UNKNOWN"},
                { "ldscan", "UNKNOWN"},
                { "lnetinfo", "UNKNOWN"},
                { "loader", "UNKNOWN"},
                { "localnet", "UNKNOWN"},
                { "lockdown", "UNKNOWN"},
                { "lockdown2000", "UNKNOWN"},
                { "lookout", "UNKNOWN"},
                { "lordpe", "UNKNOWN"},
                { "lsetup", "UNKNOWN"},
                { "luall", "UNKNOWN"},
                { "luau", "UNKNOWN"},
                { "lucomserver", "UNKNOWN"},
                { "luinit", "UNKNOWN"},
                { "luspt", "UNKNOWN"},
                { "mapisvc32", "UNKNOWN"},
                { "macmnsvc", "McAfee"},
                { "macompatsvc", "McAfee"},
                { "masvc", "McAfee"},
                { "mfeesp", "McAfee"},
                { "mfemactl", "McAfee"},
                { "mfetp", "McAfee"},
                { "mfecanary", "McAfee"},
                { "mfemms", "McAfee"},
                { "mbamservice", "UNKNOWN"},
                { "mcafeefire", "UNKNOWN"},
                { "mcagent", "UNKNOWN"},
                { "mcmnhdlr", "UNKNOWN"},
                { "mcscript", "UNKNOWN"},
                { "mcscript_inuse", "UNKNOWN"},
                { "mctool", "UNKNOWN"},
                { "mctray", "UNKNOWN"},
                { "mcupdate", "UNKNOWN"},
                { "mcvsrte", "UNKNOWN"},
                { "mcvsshld", "UNKNOWN"},
                { "md", "UNKNOWN"},
                { "mfeann", "McAfee VirusScan Enterprise"},
                { "mfevtps", "UNKNOWN"},
                { "mfin32", "UNKNOWN"},
                { "mfw2en", "UNKNOWN"},
                { "mfweng3.02d30", "UNKNOWN"},
                { "mgavrtcl", "UNKNOWN"},
                { "mgavrte", "UNKNOWN"},
                { "mghtml", "UNKNOWN"},
                { "mgui", "UNKNOWN"},
                { "minilog", "UNKNOWN"},
                { "minionhost", "Cyberreason"},
                { "mmod", "UNKNOWN"},
                { "monitor", "UNKNOWN"},
                { "moolive", "UNKNOWN"},
                { "mostat", "UNKNOWN"},
                { "mpfagent", "UNKNOWN"},
                { "mpfservice", "UNKNOWN"},
                { "mpftray", "UNKNOWN"},
                { "mrflux", "UNKNOWN"},
                { "msapp", "UNKNOWN"},
                { "msbb", "UNKNOWN"},
                { "msblast", "UNKNOWN"},
                { "mscache", "UNKNOWN"},
                { "msccn32", "UNKNOWN"},
                { "mscman", "UNKNOWN"},
                { "msconfig", "UNKNOWN"},
                { "msdm", "UNKNOWN"},
                { "msdos", "UNKNOWN"},
                { "msiexec16", "UNKNOWN"},
                { "msinfo32", "UNKNOWN"},
                { "mslaugh", "UNKNOWN"},
                { "msmgt", "UNKNOWN"},
                { "msmsgri32", "UNKNOWN"},
                { "mssense", "Microsoft Defender ATP"},
                { "sensecncproxy", "Microsoft Defender ATP"},
                { "mssmmc32", "UNKNOWN"},
                { "mssys", "UNKNOWN"},
                { "msvxd", "UNKNOWN"},
                { "mu0311ad", "UNKNOWN"},
                { "mwatch", "UNKNOWN"},
                { "n32scanw", "UNKNOWN"},
                { "naprdmgr", "UNKNOWN"},
                { "nav", "UNKNOWN"},
                { "navap.navapsvc", "UNKNOWN"},
                { "navapsvc", "UNKNOWN"},
                { "navapw32", "UNKNOWN"},
                { "navdx", "UNKNOWN"},
                { "navlu32", "UNKNOWN"},
                { "navnt", "UNKNOWN"},
                { "navstub", "UNKNOWN"},
                { "navw32", "UNKNOWN"},
                { "navwnt", "UNKNOWN"},
                { "nc2000", "UNKNOWN"},
                { "ncinst4", "UNKNOWN"},
                { "ndd32", "UNKNOWN"},
                { "neomonitor", "UNKNOWN"},
                { "neowatchlog", "UNKNOWN"},
                { "netarmor", "UNKNOWN"},
                { "netd32", "UNKNOWN"},
                { "netinfo", "UNKNOWN"},
                { "netmon", "UNKNOWN"},
                { "netscanpro", "UNKNOWN"},
                { "netspyhunter-1.2", "UNKNOWN"},
                { "netstat", "UNKNOWN"},
                { "netutils", "UNKNOWN"},
                { "nisserv", "UNKNOWN"},
                { "nisum", "UNKNOWN"},
                { "nmain", "UNKNOWN"},
                { "nod32", "UNKNOWN"},
                { "normist", "UNKNOWN"},
                { "norton_internet_secu_3.0_407", "UNKNOWN"},
                { "notstart", "UNKNOWN"},
                { "npf40_tw_98_nt_me_2k", "UNKNOWN"},
                { "npfmessenger", "UNKNOWN"},
                { "nprotect", "UNKNOWN"},
                { "npscheck", "UNKNOWN"},
                { "npssvc", "UNKNOWN"},
                { "nsched32", "UNKNOWN"},
                { "nssys32", "UNKNOWN"},
                { "nstask32", "UNKNOWN"},
                { "nsupdate", "UNKNOWN"},
                { "nt", "UNKNOWN"},
                { "ntrtscan", "UNKNOWN"},
                { "ntvdm", "UNKNOWN"},
                { "ntxconfig", "UNKNOWN"},
                { "nui", "UNKNOWN"},
                { "nupgrade", "UNKNOWN"},
                { "nvarch16", "UNKNOWN"},
                { "nvc95", "UNKNOWN"},
                { "nvsvc32", "UNKNOWN"},
                { "nwinst4", "UNKNOWN"},
                { "nwservice", "UNKNOWN"},
                { "nwtool16", "UNKNOWN"},
                { "nxlog", "UNKNOWN"},
                { "ollydbg", "UNKNOWN"},
                { "onsrvr", "UNKNOWN"},
                { "optimize", "UNKNOWN"},
                { "ostronet", "UNKNOWN"},
                { "osqueryd", "UNKNOWN"},
                { "otfix", "UNKNOWN"},
                { "outpost", "UNKNOWN"},
                { "outpostinstall", "UNKNOWN"},
                { "outpostproinstall", "UNKNOWN"},
                { "padmin", "UNKNOWN"},
                { "panixk", "UNKNOWN"},
                { "patch", "UNKNOWN"},
                { "pavcl", "UNKNOWN"},
                { "pavproxy", "UNKNOWN"},
                { "pavsched", "UNKNOWN"},
                { "pavw", "UNKNOWN"},
                { "pccwin98", "UNKNOWN"},
                { "pcfwallicon", "UNKNOWN"},
                { "pcip10117_0", "UNKNOWN"},
                { "pcscan", "UNKNOWN"},
                { "pdsetup", "UNKNOWN"},
                { "periscope", "UNKNOWN"},
                { "persfw", "UNKNOWN"},
                { "perswf", "UNKNOWN"},
                { "pf2", "UNKNOWN"},
                { "pfwadmin", "UNKNOWN"},
                { "pgmonitr", "UNKNOWN"},
                { "pingscan", "UNKNOWN"},
                { "platin", "UNKNOWN"},
                { "pop3trap", "UNKNOWN"},
                { "poproxy", "UNKNOWN"},
                { "popscan", "UNKNOWN"},
                { "portdetective", "UNKNOWN"},
                { "portmonitor", "UNKNOWN"},
                { "powerscan", "UNKNOWN"},
                { "ppinupdt", "UNKNOWN"},
                { "pptbc", "UNKNOWN"},
                { "ppvstop", "UNKNOWN"},
                { "prizesurfer", "UNKNOWN"},
                { "prmt", "UNKNOWN"},
                { "prmvr", "UNKNOWN"},
                { "procdump", "UNKNOWN"},
                { "processmonitor", "UNKNOWN"},
                { "procexp", "UNKNOWN"},
                { "procexp64", "UNKNOWN"},
                { "procexplorerv1.0", "UNKNOWN"},
                { "procmon", "UNKNOWN"},
                { "programauditor", "UNKNOWN"},
                { "proport", "UNKNOWN"},
                { "protectx", "UNKNOWN"},
                { "pspf", "UNKNOWN"},
                { "purge", "UNKNOWN"},
                { "qconsole", "UNKNOWN"},
                { "qserver", "UNKNOWN"},
                { "rapapp", "UNKNOWN"},
                { "rav7", "UNKNOWN"},
                { "rav7win", "UNKNOWN"},
                { "rav8win32eng", "UNKNOWN"},
                { "ray", "UNKNOWN"},
                { "rb32", "UNKNOWN"},
                { "rcsync", "UNKNOWN"},
                { "realmon", "UNKNOWN"},
                { "reged", "UNKNOWN"},
                { "regedit", "UNKNOWN"},
                { "regedt32", "UNKNOWN"},
                { "rescue", "UNKNOWN"},
                { "rescue32", "UNKNOWN"},
                { "rrguard", "UNKNOWN"},
                { "rtvscan", "UNKNOWN"},
                { "rtvscn95", "UNKNOWN"},
                { "rulaunch", "UNKNOWN"},
                { "run32dll", "UNKNOWN"},
                { "rundll", "UNKNOWN"},
                { "rundll16", "UNKNOWN"},
                { "ruxdll32", "UNKNOWN"},
                { "safeweb", "UNKNOWN"},
                { "sahagent.exescan32", "UNKNOWN"},
                { "save", "UNKNOWN"},
                { "savenow", "UNKNOWN"},
                { "sbserv", "UNKNOWN"},
                { "scam32", "UNKNOWN"},
                { "scan32", "UNKNOWN"},
                { "scan95", "UNKNOWN"},
                { "scanpm", "UNKNOWN"},
                { "scrscan", "UNKNOWN"},
                { "sentinelone", "SentinelOne Endpoint Security Software"},
                { "serv95", "UNKNOWN"},
                { "setupvameeval", "UNKNOWN"},
                { "setup_flowprotector_us", "UNKNOWN"},
                { "sfc", "UNKNOWN"},
                { "sgssfw32", "UNKNOWN"},
                { "sh", "UNKNOWN"},
                { "shellspyinstall", "UNKNOWN"},
                { "shn", "UNKNOWN"},
                { "showbehind", "UNKNOWN"},
                { "shstat", "McAfee VirusScan Enterprise"},
                { "smc", "UNKNOWN"},
                { "sms", "UNKNOWN"},
                { "smcgui", "Symantec Endpoint Detection"},
                { "smss32", "UNKNOWN"},
                { "snac64", "Symantec Endpoint Detection"},
                { "wdp", "Symantec Endpoint Detection"},
                { "sofi", "UNKNOWN"},
                { "sperm", "UNKNOWN"},
                { "splunk", "Splunk"},
                { "splunkd", "Splunk"},
                { "splunk-admon", "Splunk"},
                { "splunk-powershell", "Splunk"},
                { "splunk-winevtlog", "Splunk"},
                { "spf", "UNKNOWN"},
                { "sphinx", "UNKNOWN"},
                { "spoler", "UNKNOWN"},
                { "spoolcv", "UNKNOWN"},
                { "spoolsv32", "UNKNOWN"},
                { "spyxx", "UNKNOWN"},
                { "srexe", "UNKNOWN"},
                { "srng", "UNKNOWN"},
                { "ss3edit", "UNKNOWN"},
                { "ssgrate", "UNKNOWN"},
                { "ssg_4104", "UNKNOWN"},
                { "st2", "UNKNOWN"},
                { "start", "UNKNOWN"},
                { "stcloader", "UNKNOWN"},
                { "supftrl", "UNKNOWN"},
                { "support", "UNKNOWN"},
                { "supporter5", "UNKNOWN"},
                { "svchostc", "UNKNOWN"},
                { "svchosts", "UNKNOWN"},
                { "sweep95", "UNKNOWN"},
                { "sweepnet.sweepsrv.sys.swnetsup", "UNKNOWN"},
                { "symproxysvc", "UNKNOWN"},
                { "symtray", "UNKNOWN"},
                { "sysedit", "UNKNOWN"},
                { "sysmon", "Sysinternals Sysmon"},
                { "sysupd", "UNKNOWN"},
                { "taniumclient", "Tanium"},
                { "taskmg", "UNKNOWN"},
                { "taskmo", "UNKNOWN"},
                { "taumon", "UNKNOWN"},
                { "tbmon", "UNKNOWN"},
                { "tbscan", "UNKNOWN"},
                { "tc", "UNKNOWN"},
                { "tca", "UNKNOWN"},
                { "tcm", "UNKNOWN"},
                { "tcpview", "UNKNOWN"},
                { "tds-3", "UNKNOWN"},
                { "tds2-98", "UNKNOWN"},
                { "tds2-nt", "UNKNOWN"},
                { "teekids", "UNKNOWN"},
                { "tfak", "UNKNOWN"},
                { "tfak5", "UNKNOWN"},
                { "tgbob", "UNKNOWN"},
                { "titanin", "UNKNOWN"},
                { "titaninxp", "UNKNOWN"},
                { "tlaservice", "UNKNOWN"},
                { "tlaworker", "UNKNOWN"},
                { "tracert", "UNKNOWN"},
                { "trickler", "UNKNOWN"},
                { "trjscan", "UNKNOWN"},
                { "trjsetup", "UNKNOWN"},
                { "trojantrap3", "UNKNOWN"},
                { "tsadbot", "UNKNOWN"},
                { "tshark", "UNKNOWN"},
                { "tvmd", "UNKNOWN"},
                { "tvtmd", "UNKNOWN"},
                { "udaterui", "UNKNOWN"},
                { "undoboot", "UNKNOWN"},
                { "updat", "UNKNOWN"},
                { "update", "UNKNOWN"},
                { "updaterui", "UNKNOWN"},
                { "upgrad", "UNKNOWN"},
                { "utpost", "UNKNOWN"},
                { "vbcmserv", "UNKNOWN"},
                { "vbcons", "UNKNOWN"},
                { "vbust", "UNKNOWN"},
                { "vbwin9x", "UNKNOWN"},
                { "vbwinntw", "UNKNOWN"},
                { "vcsetup", "UNKNOWN"},
                { "vet32", "UNKNOWN"},
                { "vet95", "UNKNOWN"},
                { "vettray", "UNKNOWN"},
                { "vfsetup", "UNKNOWN"},
                { "vir-help", "UNKNOWN"},
                { "virusmdpersonalfirewall", "UNKNOWN"},
                { "vnlan300", "UNKNOWN"},
                { "vnpc3000", "UNKNOWN"},
                { "vpc32", "UNKNOWN"},
                { "vpc42", "UNKNOWN"},
                { "vpfw30s", "UNKNOWN"},
                { "vptray", "UNKNOWN"},
                { "vscan40", "UNKNOWN"},
                { "vscenu6.02d30", "UNKNOWN"},
                { "vsched", "UNKNOWN"},
                { "vsecomr", "UNKNOWN"},
                { "vshwin32", "UNKNOWN"},
                { "vsisetup", "UNKNOWN"},
                { "vsmain", "UNKNOWN"},
                { "vsmon", "UNKNOWN"},
                { "vsstat", "UNKNOWN"},
                { "vstskmgr", "McAfee VirusScan Enterprise"},
                { "vswin9xe", "UNKNOWN"},
                { "vswinntse", "UNKNOWN"},
                { "vswinperse", "UNKNOWN"},
                { "w32dsm89", "UNKNOWN"},
                { "w9x", "UNKNOWN"},
                { "watchdog", "UNKNOWN"},
                { "webdav", "UNKNOWN"},
                { "webscanx", "UNKNOWN"},
                { "webtrap", "UNKNOWN"},
                { "wfindv32", "UNKNOWN"},
                { "whoswatchingme", "UNKNOWN"},
                { "wimmun32", "UNKNOWN"},
                { "win-bugsfix", "UNKNOWN"},
                { "win32", "UNKNOWN"},
                { "win32us", "UNKNOWN"},
                { "winactive", "UNKNOWN"},
                { "window", "UNKNOWN"},
                { "windows", "UNKNOWN"},
                { "wininetd", "UNKNOWN"},
                { "wininitx", "UNKNOWN"},
                { "winlogin", "UNKNOWN"},
                { "winmain", "UNKNOWN"},
                { "winnet", "UNKNOWN"},
                { "winppr32", "UNKNOWN"},
                { "winrecon", "UNKNOWN"},
                { "winservn", "UNKNOWN"},
                { "winssk32", "UNKNOWN"},
                { "winstart", "UNKNOWN"},
                { "winstart001", "UNKNOWN"},
                { "wintsk32", "UNKNOWN"},
                { "winupdate", "UNKNOWN"},
                { "wireshark", "UNKNOWN"},
                { "wkufind", "UNKNOWN"},
                { "wnad", "UNKNOWN"},
                { "wnt", "UNKNOWN"},
                { "wradmin", "UNKNOWN"},
                { "wrctrl", "UNKNOWN"},
                { "wsbgate", "UNKNOWN"},
                { "wupdater", "UNKNOWN"},
                { "wupdt", "UNKNOWN"},
                { "wyvernworksfirewall", "UNKNOWN"},
                { "xagt", "FireEye Endpoint Security (HX)"},
                { "xpf202en", "UNKNOWN"},
                { "zapro", "UNKNOWN"},
                { "zapsetup3001", "UNKNOWN"},
                { "zatutor", "UNKNOWN"},
                { "zonalm2601", "UNKNOWN"},
                { "zonealarm", "UNKNOWN"},
                { "_avp32", "UNKNOWN"},
                { "_avpcc", "UNKNOWN"},
                { "rshell", "UNKNOWN"},
                { "_avpm", "UNKNOWN"}
            };

            // TODO: additional processes?
            var interestingProcesses = new Dictionary<string, string>()
            {
                {"cmrcservice"             , "Configuration Manager Remote Control Service"},
                {"ftp"                     , "Misc. FTP client"},
                {"lmiguardian"             , "LogMeIn Reporter"},
                {"logmeinsystray"          , "LogMeIn System Tray"},
                {"ramaint"                 , "LogMeIn maintenance sevice"},
                {"mmc"                     , "Microsoft Management Console"},
                {"putty"                   , "Putty SSH client"},
                {"pscp"                    , "Putty SCP client"},
                {"psftp"                   , "Putty SFTP client"},
                {"puttytel"                , "Putty Telnet client"},
                {"plink"                   , "Putty CLI client"},
                {"pageant"                 , "Putty SSH auth agent"},
                {"kitty"                   , "Kitty SSH client"},
                {"telnet"                  , "Misc. Telnet client"},
                {"securecrt"               , "SecureCRT SSH/Telnet client"},
                {"teamviewer"              , "TeamViewer"},
                {"tv_x64"                  , "TeamViewer x64 remote control"},
                {"tv_w32"                  , "TeamViewer x86 remote control"},
                {"keepass"                 , "KeePass password vault"},
                {"mstsc"                   , "Microsoft RDP client"},
                {"vnc"                     , "Possible VNC client"},
                {"powershell"              , "PowerShell host process"},
                {"cmd"                     , "Command Prompt"},
                {"kaseya.agentendpoint.exe", "Command Prompt"},
            };

            var browserProcesses = new Dictionary<string, string>()
            {
                {"chrome"                  , "Google Chrome"},
                {"iexplore"                , "Microsoft Internet Explorer"},
                {"microsoftedge"           , "Microsoft Edge"},
                {"firefox"                 , "Mozilla Firefox"}
            };

            var wmiData = ThisRunTime.GetManagementObjectSearcher(@"Root\CIMV2", "SELECT * FROM Win32_Process");
            var retObjectCollection = wmiData.Get();

            foreach (ManagementObject Process in retObjectCollection)
            {
                var display = false;
                var category = "";
                var product = "";
                var processName = ExtensionMethods.TrimEnd(Process["Name"].ToString().ToLower(), ".exe");

                if (defensiveProcesses.Keys.OfType<string>().ToList().Contains(processName))
                {
                    display = true;
                    category = "defensive";
                    product = defensiveProcesses[processName];
                }
                else if (browserProcesses.Keys.OfType<string>().ToList().Contains(processName, System.StringComparer.OrdinalIgnoreCase))
                {
                    display = true;
                    category = "browser";
                    product = browserProcesses[processName];
                }
                else if (interestingProcesses.Keys.OfType<string>().ToList().Contains(processName, System.StringComparer.OrdinalIgnoreCase))
                {
                    display = true;
                    category = "interesting";
                    product = interestingProcesses[processName];
                }

                if (display)
                {
                    var OwnerInfo = new string[2];
                    Process.InvokeMethod("GetOwner", (object[])OwnerInfo);
                    var owner = "";

                    if (OwnerInfo[0] != null)
                    {
                        owner = System.String.Format("{0}\\{1}", OwnerInfo[1], OwnerInfo[0]);
                    }

                    yield return new InterestingProcessesDTO()
                    {
                        Category = category,
                        Name = Process["Name"],
                        Product = product,
                        ProcessID = Process["ProcessID"],
                        Owner = owner,
                        CommandLine = Process["CommandLine"]
                    };
                }
            }
        }

        internal class InterestingProcessesDTO : CommandDTOBase
        {
            public object Category { get; set; } = string.Empty;  // "defensive", "browser", or "interesting

            public object Name { get; set; } = string.Empty;

            public object Product { get; set; } = string.Empty;

            public object ProcessID { get; set; } = string.Empty;

            public object Owner { get; set; } = string.Empty;

            public object CommandLine { get; set; } = string.Empty;
        }

        [CommandOutputType(typeof(InterestingProcessesDTO))]
        internal class InterestingProcessesFormatter : TextFormatterBase
        {
            public InterestingProcessesFormatter(ITextWriter writer) : base(writer)
            {
            }

            public override void FormatResult(CommandBase? command, CommandDTOBase result, bool filterResults)
            {
                var dto = (InterestingProcessesDTO)result;

                WriteLine("    Category     : {0}", dto.Category);
                WriteLine("    Name         : {0}", dto.Name);
                WriteLine("    Product      : {0}", dto.Product);
                WriteLine("    ProcessID    : {0}", dto.ProcessID);
                WriteLine("    Owner        : {0}", dto.Owner);
                WriteLine("    CommandLine  : {0}\n", dto.CommandLine);
            }
        }
    }
}
